/* protocol.c -- understands and generates messages sent over the network
   Copyright (C) 2004 Maximiliano Pin

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define _POSIX_SOURCE

#include <string.h>		/* strlen, strncmp, strcpy, memcpy */
#include <netinet/in.h>		/* htonl, htons, ntohl, ntohs */
#include "protocol.h"
#include "transport.h"
#include "uconfig.h"
#include "user_iface.h"
#include "demux.h"
#include "misc.h"
#include "log.h"

/* Header of sent and received messages. */
/* ATTENTION: Changing this struct requires revision of hton_header(),
 *            ntoh_header(), and header_size(). */
struct msg_header {
	uint16_t type;                       /* message type */
	union {
		/* Hello message. Payload is the nick of the sender. */
		struct hdr_hello {
			uint16_t version;    /* protocol version */
			uint16_t port;       /* listen port */
		} hello;

		/* Text message. The text is on the payload. */
		struct hdr_text {
		} text;

		/* Beep message. */
		struct hdr_beep {
		} beep;

		/* ... */
	} c;
} __attribute__((packed));  /* TODO: this is gcc only */

/* We'll use this static header for all messages sent and received. */
static struct msg_header header;

/* Values of 'header.type'. */
enum { MSG_HELLO = 0, MSG_TEXT, MSG_BEEP /* ... */ };

/* Macro to calculate how many bytes of 'header' have to be sent or received. */
#define HEADER_SIZE(t) (sizeof(header.c.t) + sizeof(header) - sizeof(header.c))

/* Macros for byte order conversion. */
#define HTON16(x) x = htons(x)
#define HTON32(x) x = htonl(x)
#define NTOH16(x) x = ntohs(x)
#define NTOH32(x) x = ntohl(x)

/* Prototypes */
static int received_hello (contact_t *contact, struct hdr_hello *hdr,
                           const char *nick, int nick_len);
static int received_text (contact_t *contact, const char *text, int len);
static int received_beep (contact_t *contact);
static void hton_header ();
static void ntoh_header ();
static int header_size (uint16_t msg_type);

void
pr_send_text (contact_t *contact, const char *txt)
{
	size_t len = strlen (txt);

	header.type = MSG_TEXT;
	hton_header ();
	tr_send_msg (contact, (BYTE *)&header, HEADER_SIZE(text),
	             (BYTE *)txt, len);

	if (cfg.logging) {
		lg_log_text (cfg.nick, contact, txt, len);
	}
}

void
pr_send_hello (contact_t *contact)
{
	header.type = MSG_HELLO;
	header.c.hello.version = PROTOCOL_VERSION;
	header.c.hello.port = cfg.listen_port;
	hton_header ();
	tr_send_msg (contact, (BYTE *)&header, HEADER_SIZE(hello),
	             (BYTE *)cfg.nick, strlen (cfg.nick));
}

void
pr_send_beep (contact_t *contact)
{
	if (contact != BROADCAST && contact->state.hello_pending) {
		ui_output_err ("Contact not connected.");
		return;
	}
	header.type = MSG_BEEP;
	hton_header ();
	tr_send_msg (contact, (BYTE *)&header, HEADER_SIZE(beep),
	             (BYTE *)0, 0);
	ui_output_info ("Beeping %s.", contact == BROADCAST ? "everybody"
	                                                    : contact->nick);
}

int
pr_msg_received (contact_t *contact, const BYTE *msg, int len)
{
	uint16_t type;
	int hdr_size;
	const BYTE *payload;
	int pl_size;
	int r;

	CHECK_J (len >= sizeof(header.type), small_size);
	type = *(uint16_t *)msg;
	NTOH16 (type);
	hdr_size = header_size (type);
	CHECK_J (hdr_size != ERR, bad_type);
	CHECK_J (len >= hdr_size, small_size);
	memcpy (&header, msg, hdr_size);
	ntoh_header ();

	payload = msg + hdr_size;
	pl_size = len - hdr_size;

	switch (type) {

		case MSG_HELLO:
			r = received_hello (contact, &header.c.hello,
			                    (const char *)payload, pl_size);
			break;

		case MSG_TEXT:
			r = received_text (contact, (const char *)payload,
			                   pl_size);
			break;

		case MSG_BEEP:
			r = received_beep (contact);
			break;

		default:
			ui_output_err ("BUG: missing type in pr_msg_received.");
			r = ERR;
	}
	if (r < 0)
		return r;

	return OK;

small_size:
	ui_output_err ("Received message with incorrect size (%d).", len);
	return ERR;

bad_type:
	ui_output_err ("Received message of unknown type. You may need to "
	               "update IPChat.");
	/* return OK because there is no need to disconnect */
	return OK;
}

static int
received_hello (contact_t *contact, struct hdr_hello *hdr, const char *nick,
                int nick_len)
{
	BOOL modif, modif_nick, hello_was_pending;

	if (hdr->version < PROTOCOL_VERSION) {
		ui_output_err ("Connection from outdated client! (from %s)",
		               ipv4_to_string (contact->state.ip));
		return ERR;
	}

	if (hdr->version > PROTOCOL_VERSION) {
		ui_output_err ("You could need to upgrade IPChat, someone "
		               "(from %s) is using an incompatible newer "
		               "version.", ipv4_to_string (contact->state.ip));
		return ERR;
	}

	if (nick_len > (MAX_NICK - 1)) {
		ui_output_err ("Contact's nick is too long (from %s).",
		               ipv4_to_string (contact->state.ip));
		return ERR;
	}

	/* TODO We should check there are no control characters on the
	   nick, or some jokes can be done. For example, let /nick only
	   accept printable chars, and accept the same here. */
	if (nick_len == 0 || nick[0] == '\0') {
		ui_output_err ("Bad nick in incomming HELLO (from %s).",
		               ipv4_to_string (contact->state.ip));
		return ERR;
	}

	modif = modif_nick = FALSE;
	if (contact->port != hdr->port) {
		contact->port = hdr->port;
		modif = TRUE;
	}

	hello_was_pending = contact->state.hello_pending;
	if (hello_was_pending && contact->nick[0] != '\0') {
		ui_output_info ("Contact %s connected.", contact->nick);
	}

	if (strlen (contact->nick) != nick_len ||
	    strncmp (contact->nick, nick, nick_len) != 0) {
		char new_nick[MAX_NICK];
		memcpy (new_nick, nick, nick_len);
		new_nick[nick_len] = '\0';
		if (contact->nick[0] != '\0') {
			ui_output_info ("Contact %s is now known as %s.",
			                contact->nick, new_nick);
		}
		else {
			ui_output_info ("A new contact has connected: %s.",
			                new_nick);
		}
		strcpy (contact->nick, new_nick);
		modif = modif_nick = TRUE;
	}

	if (modif)
		cl_save ();

	if (hello_was_pending) {
		dmx_remove_timer (contact->state.hello_timeout);
		contact->state.hello_timeout = NULL;
		contact->state.hello_pending = FALSE;
	}

	if (hello_was_pending || modif_nick) {
		ui_redraw_contacts ();
	}

	return OK;
}

static int
received_text (contact_t *contact, const char *text, int len)
{
	/* TODO Check there are only printable characters and no '\n', etc.
	   Should we check for a maximum length? */

	ui_output_msg (contact, text, len);

	if (cfg.logging) {
		lg_log_text (contact->nick, contact, text, len);
	}

	return OK;
}

static int
received_beep (contact_t *contact)
{
	if (cfg.beep) {
		ui_output_info ("Beep from %s!", contact->nick);
		ui_beep ();
	}
	else {
		ui_output_info ("Beep from %s! (deactivated)", contact->nick);
	}

	return OK;
}

static void
hton_header ()
{
	switch (header.type) {

		case MSG_HELLO:
			HTON16 (header.c.hello.version);
			HTON16 (header.c.hello.port);
			break;

		case MSG_TEXT:
			break;

		case MSG_BEEP:
			break;

		/* ... */
	}

	HTON16 (header.type);
}

static void
ntoh_header ()
{
	NTOH16 (header.type);

	switch (header.type) {

		case MSG_HELLO:
			NTOH16 (header.c.hello.version);
			NTOH16 (header.c.hello.port);
			break;

		case MSG_TEXT:
			break;

		case MSG_BEEP:
			break;

		/* ... */
	}
}

static int
header_size (uint16_t msg_type)
{
	int sz;

	switch (msg_type) {
		case MSG_HELLO:
			sz = HEADER_SIZE(hello);
			break;
		case MSG_TEXT:
			sz = HEADER_SIZE(text);
			break;
		case MSG_BEEP:
			sz = HEADER_SIZE(beep);
			break;
		default:
			sz = ERR;
			break;
	}

	return sz;
}
