/* user_iface.c -- the ncurses user interface, main module
   Copyright (C) 2004 Maximiliano Pin

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#define _POSIX_SOURCE
#define _ISOC99_SOURCE		/* vsnprintf */

#include <signal.h>		/* SIG* */
#include <stdlib.h>		/* malloc, free */
#include <stdarg.h>		/* va_*, vsnprintf */
#include <string.h>		/* strcpy, memcpy, memset, strerror */
#include <errno.h>		/* errno */
#include "ui_common.h"
#include "demux.h"

extern int errno;

WINDOW *win_text;		/* window where messages appear */
WINDOW *win_input;		/* window for line edition */
WINDOW *win_contacts;		/* window for the list of contacts */
WINDOW *win_nick;		/* window where user's nick appears */

/* Position and size of ncurses windows. */
struct win_info i_text, i_input, i_contacts, i_nick;

contact_win *cwin[MAX_CONTACT_WINS];     /* contact windows */
int cwins = 0;                           /* number of contact windows */
contact_win *cur;                        /* current contact window */
contact_win *vcur;                       /* visualized contact window */
short seq_err_pair;                      /* color pair to show seq error */

/* Prototypes */
static contact_win *cwin_new (contact_t *contact);
static void cwin_del (int idx);
static WINDOW *newwin_i (struct win_info *wi);
static void hnd_winch (int sig);

/* Init module. Paint the interface for the first time. */
int
ui_init ()
{
	int i;
	
	cur = vcur = cwin_new (NULL);
	/* TODO if (!cur) ... */

	(void)initscr ();
	curs_set (1);
	(void)cbreak ();
	(void)noecho ();
	(void)nl ();

	if (has_colors ()) {
		start_color ();

		/* simple color assignment */
		/* TODO support transparent terminal emulators,
		   assume_default_colors(-1,-1) (not portable) */
		for (i = 1; i < 8; i++)
			init_pair (i, i, COLOR_BLACK);

		/* determine pair for error in sequence */
		if (COLOR_PAIRS >= 8) {
			seq_err_pair = 8;
			init_pair (seq_err_pair, COLOR_WHITE, COLOR_RED);
		}
		else {
			seq_err_pair = 0;
		}
	}

	recalc_windows ();

	if ((win_text = newwin_i (&i_text)) == NULL ||
	    (win_input = newwin_i (&i_input)) == NULL ||
	    (win_contacts = newwin_i (&i_contacts)) == NULL ||
	    (win_nick = newwin_i (&i_nick)) == NULL) {
		endwin ();
		return ERR;
	}

	keypad (win_input, TRUE);	/* enable keyboard mapping */
	leaveok (win_input, FALSE);
	scrollok (win_input, FALSE);
	nodelay (win_input, TRUE);	/* make wgetch() non-blocking */

	scrollok (win_text, TRUE);
	leaveok (win_text, FALSE);

	leaveok (win_contacts, TRUE);
	leaveok (win_nick, TRUE);

	paint_all ();
	ui_redraw_contacts (); /* TODO not needed (just for testing) */

	dmx_signal (SIGWINCH, hnd_winch);
	dmx_add_input_fd (0, cb_read_key, NULL);

	return OK;
}

/* Close curses mode, free used memory. */
void
ui_finish ()
{
	/* TODO free mem */
	endwin ();
}

/* Recreate the list of contacts. Create windows for connected contacts
   which don't have one. Call whenever a contact connects or disconnects. */
void
ui_redraw_contacts ()
{
	contact_win *cw;
	contact_t *c;
	int y, x;

	getsyx (y, x);

	/* find what changed */
	for (c = contacts; c; c = c->next) {
		cw = (contact_win *)(c->state.ui_info);
		if (!cw && !c->state.hello_pending && c->nick[0]) {
			/* the contact just connected */
			cwin_new (c);
		}
	}

	paint_contacts ();
	wnoutrefresh (win_contacts);
	setsyx (y, x);
	doupdate ();
}

/* Redraw user's nick. Call when the nick changes. */
void ui_redraw_nick ()
{
	remake_nick_windows ();
}

/* Close window of contact. */
void
ui_close_win (contact_t *contact)
{
	contact_win *cw = (contact_win *)(contact->state.ui_info);

	if (!cw) {
		ui_output_err ("There is no window for %s.", contact->nick);
	}
	else if (contact->state.hello_pending) {
		cwin_del (cw->pos);
	}
	else 
		ui_output_err ("Cannot close window for connected contact.");
}

/* Close current window. */
void
ui_close_cur_win ()
{
	if (!cur->contact) {
		ui_output_err ("You need this window. Not closing.");
		return;
	}
	if (cur->contact->state.hello_pending)
		cwin_del (cur->pos);
	else 
		ui_output_err ("Cannot close window for connected contact.");
}

/* Output a new message from a contact. */
void
ui_output_msg (contact_t *contact, const char *txt, int len)
{
	cur = (contact_win *)(contact->state.ui_info);

	if (cur) {
		/* build and add the line, and print it if current window */
		build_line (NICK_PREF, contact->nick, NICK_SUFF, txt,
		            strlen (NICK_PREF), strlen (contact->nick),
		            strlen (NICK_SUFF), len);
		cur = vcur;
	}
	else {
		cur = vcur;
		ui_output_err ("Message from %s, who has no open window. "
		               "This is a bug, please report it.",
		               contact->nick);
	}
}

/* Output an information message. Works like printf. */
void
ui_output_info (const char *fmt, ...)
{
	va_list ap;
	char str[1024]; /* TODO */

	/* TODO vwprintw... */
	va_start (ap, fmt);
	strcpy (str, INF_PREF);
	vsnprintf (str + strlen (INF_PREF), 1024 - strlen (INF_PREF), fmt, ap);
	va_end (ap);
	calc_align ();
	add_text (str, strlen (str));
}

/* Output an error message. Works like printf. */
void
ui_output_err (const char *fmt, ...)
{
	va_list ap;
	char    str[1024]; /* TODO */

	/* TODO vwprintw... */
	va_start (ap, fmt);
	strcpy (str, ERR_PREF);
	vsnprintf (str + strlen (ERR_PREF), 1024 - strlen (ERR_PREF), fmt, ap);
	va_end (ap);
	calc_align ();
	add_text (str, strlen (str));

	/* TODO this code aligns error messages like nicks */
#if 0
	int len_pref, align;

	len_pref = strlen (ERR_PREF);
	calc_align ();
	align = cur->align - 5; /* TODO */
	if (align < 0)
		align = 0;

	memcpy (str, ERR_PREF, len_pref);
	memset (str + len_pref, ' ', align);
	len_pref += align;

	va_start (ap, fmt);
	vsnprintf (str + len_pref, 1024 - len_pref, fmt, ap);
	va_end (ap);
	add_text (str, strlen (str));
#endif
}

/* Output the same as perror(). */
void
ui_perror (const char *txt)
{
	ui_output_err ("%s: %s", txt, strerror (errno));
}

/* Generate a beep. */
void
ui_beep ()
{
	beep ();
}

/* Get contact on current window, NULL if none. */
contact_t *
ui_current_contact ()
{
	return cur->contact;
}

/* Create a new contact window for referenced contact. */
static contact_win *
cwin_new (contact_t *contact)
{
	contact_win *cw;

	if (cwins == 1 && !cur->contact) {
		/* this will be the only window, use default win */
		cur->contact = contact;
		contact->state.ui_info = cur;
		return cur;
	}

	if (cwins == MAX_CONTACT_WINS) {
		ui_output_err ("Cannot create more windows.");
		return NULL;
	}

	cw = malloc (sizeof (contact_win));
	if (!cw) {
		ui_output_err ("No more memory.");
		return cw;
	}

	cw->ibuf_len = 0;
	cw->ishow = 0;
	cw->icursor = 0;
	cw->hist_tail = 0;
	cw->hist_head = 0;
	cw->hist_pos = 0;
	cw->hist[0] = NULL;
	cw->tbuf_tail = 0;
	cw->tbuf_head = 0;
	cw->align = MAX_NICK + NICK_PREF_SUFF_LEN;
	cw->alarm = FALSE;
	cw->pos = cwins;
	cw->contact = contact;

	if (contact)
		contact->state.ui_info = cw;

	return (cwin[cwins++] = cw);
}

/* Delete the contact window with index 'idx'. Change window if deleted
   window is the current window, otherwise redraw contact list. */
static void
cwin_del (int idx)
{
	int i;
	BOOL chg_win = FALSE; /* need to change window? */

	// TODO CHECK (idx < cwins);

	if (cwin[idx]->contact)
		cwin[idx]->contact->state.ui_info = NULL;

	if (cwins == 1) {
		/* we always leave one window */
		cwin[idx]->contact = NULL;
	}
	else {
		if (cur == cwin[idx])
			chg_win = TRUE;

		free (cwin[idx]);
		cwins--;
		for (i = idx; i < cwins; i++) {
			cwin[i] = cwin[i + 1];
			cwin[i]->pos--;
		}
		if (idx == cwins)
			idx--;
	}

	if (chg_win)
		change_window (idx);
	else
		ui_redraw_contacts ();
}

/* Create a curses window with the geometry specified in 'wi'. */
static WINDOW *
newwin_i (struct win_info *wi)
{
	return newwin (wi->nlines, wi->ncols, wi->pos_y, wi->pos_x);
}

/* Handle signal SIGWINCH, which arrives on terminal resize. */
static void
hnd_winch (int sig)
{
	/* TODO I don't know why, but I have to call this twice for it
	   to work well */
	remake_windows ();
	remake_windows ();
}
