/*
   This file is part of the XXCalc Library - version 3.2
   Copyright (C)  2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010,
   2011, 2012, 2013    Ivano Primi ( ivprimi@libero.it )    

   The XXCalc Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The XXCalc library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _VARLIST_H_
#define _VARLIST_H_

#include"compl.h"

#ifdef __cplusplus
extern "C" {
#endif

#define XX_VL_BLOCK_SIZE 1024U

  typedef struct
  {
    char *name;
    c_omplex value;
  }
    xx_vartype;

  typedef struct
  {
    xx_vartype** ppvar; /* PPVAR must be intended as an array of xx_vartype* */
    ui_nteger nblocks, length;
  }
    xx_varlist;

  /* 
     Whenever one of the functions below (except xx_vl_isempty()) 
     returns an int, 0 means failure, !0 successfull execution. 
     The function xx_vl_get() uses the last parameter for this aim.           
  */

  /* List of the functions that may be called on a list before xx_vl_sort()   */
  /* has been called on the same list:                                        */
  /* xx_vl_init() (must be called before xx_vl_sort()!)                       */
  /* xx_vl_isempty()                                                          */
  /* xx_vl_len()                                                              */
  /* xx_vl_nblocks()                                                          */
  /* xx_vl_cap()                                                              */
  /* xx_vl_add()                                                              */
  /* xx_vl_load()                                                             */
  /* xx_vl_assign()                                                           */
  /* xx_vl_at()                                                               */
  /* xx_vl_run()                                                              */
  /* xx_vl_destroy()                                                          */

  /* List of the functions that may be called only on empty or sorted lists   */
  /* xx_vl_ins()                                                              */
  /* xx_vl_del()                                                              */
  /* xx_vl_get()                                                              */
  /* xx_vl_set()                                                              */
  /* xx_vl_getpos()                                                           */
  /* xx_vl_getpos_r()                                                         */
  /* xx_vl_getrange()                                                         */
  /* xx_vl_merge()                                                            */

  /* 
     xx_vl_init() initializes the list pointed to by L. 
     It should be called on a list before performing any other operation.
  */
  void xx_vl_init (xx_varlist * l);

  /* xx_vl_add() adds the couple {"name", value} to the list pointed to by L. */
  int xx_vl_add (xx_varlist * l, const char *name, c_omplex value);

  /* 
     If the list pointed to by L is sorted according to the increasing        
     order of the variable names, this function adds to it the couple 
     {"name",value} in a such way that the list remains sorted.
     If already exists a variable named "name", then it is simply redefined,
     no addition is done.                                                    
     It can be used to create from 0 a sorted list.                           
  */
  int xx_vl_ins (xx_varlist * l, const char *name, c_omplex value);

  /* xx_vl_sort() sorts the list L according to the increasing order of   */
  /* the variable names.    Returns 0 if L is empty or too long (L.length */
  /* > LONG_MAX), otherwise 1.                                            */
  int xx_vl_sort(xx_varlist l);

  /* 
     xx_vl_merge() adds to the list pointed to by L1 all the elements of
     the list L2 in such a way that *L1 remains a sorted list.    
  */
  int xx_vl_merge(xx_varlist* l1, xx_varlist l2);

  /* xx_vl_getpos() returns a pointer to the element of the list L identified */
  /* by NAME, or -1 if such an element does not exist.                        */
  /* It returns -1 also if the list L has a length >= LONG_MAX.               */
  i_nteger xx_vl_getpos (xx_varlist l, const char* name);

  /* 
     xx_vl_getpos_r() returns a pointer to the element of the list L which is
     identified by NAME and is located in a position between
     FROM and TO, or -1 if such an element does not exist.                 
     It returns -1 also if the condition 0 <= FROM <= TO < L.length turns     
     out to be false.
  */
  i_nteger xx_vl_getpos_r(xx_varlist l, const char* name,
			  i_nteger from, i_nteger to); 

  /* 
     xx_vl_getrange() checks for the presence, inside the list L, of     
     variables whose name starts with PREFIX.                     
     If there is, inside the list L, at least one variable whose name       
     starts with PREFIX, then xx_vl_getrange() sets *FROM to the  
     index of the first variable and *TO to the index of the last         
     variable of L between those ones whose name starts with PREFIX. 
     If no variable in L has a name which starts with PREFIX, 
     then both *FROM and *TO are set to -1.                            
     *FROM and *TO are set to -1 also if the length of L is > LONG_MAX, since 
     in this case the function cannot work.                                 
     Mind that, if 'prefix' == "", then xx_vl_getrange() sets *FROM to 0 and  
     *TO to L.length .                                                         
  */
  void xx_vl_getrange(xx_varlist l, const char* prefix,
		      i_nteger* from, i_nteger* to);

  /* 
     xx_vl_at() returns the name of the element of the list L         
     at the position POS. *Z is set to the value of this element, if Z is
     not NULL. If pos >= length of L, then xx_vl_at() returns NULL and, if Z
     is not-NULL, sets *Z to zero.                                             
  */
  const char* xx_vl_at (xx_varlist l, ui_nteger pos, c_omplex* z);

  /* xx_vl_assign() sets to Z the value of the element of L at the position   */
  /* POS, or does nothing if POS >= length of L.                              */
  int xx_vl_assign(xx_varlist l, ui_nteger pos, c_omplex z);

  /* xx_vl_del() removes from the list pointed to by L the first variable     */
  /* identified by NAME.                                                      */
  int xx_vl_del (xx_varlist * l, const char *name);

  /* xx_vl_destroy() destroys the list pointed to by L.                       */
  void xx_vl_destroy (xx_varlist * l);

  /* 
    xx_vl_run() lets you run the list pointed to by L. The first time 
    xx_vl_run() is called on a list, a pointer to this one  must be  passed
    as first argument; after, the first argument must be NULL if you want  
    effectively run the list: if you pass, as first argument, a pointer  
    to an existent list, xx_vl_run() will start to run (or re-run) that list.
    xx_vl_run() returns a pointer to a variable-name till it reaches the end
    of the list pointed by L. When this happens, it returns NULL.          
  */
  const char *xx_vl_run (const xx_varlist * l, c_omplex * pz);

  /* xx_vl_set() sets to VALUE the value of the first variable                */
  /* in the list L which is identified by NAME.                               */
  int xx_vl_set (xx_varlist l, const char *name, c_omplex value);

  /* xx_vl_get() returns the value of the first variable in L which is        */
  /* indentified by NAME.                                                     */
  c_omplex xx_vl_get (xx_varlist l, const char *name, int *yesno);

  /* xx_vl_isempty() returns !0 (0) if L is (not) empty                       */
  int xx_vl_isempty (xx_varlist l);

  /* xx_vl_len() returns the length of L, i.e. the number of non-NULL nodes   */
  /* which form the list.                                                     */
  ui_nteger xx_vl_len (xx_varlist l);

  /* xx_vl_nblocks() returns the number of the blocks composing the list L.   */
  ui_nteger xx_vl_nblocks (xx_varlist l);

  /* xx_vl_cap() returns the capacity of the list L, i.e. the product of */
  /* XX_VL_BLOCK_SIZE for the number of the blocks of L.                 */
  ui_nteger xx_vl_cap (xx_varlist l);

  /* xx_vl_load() builds a list from the array PVAR; LENGTH is the size  */
  /* of the array.  The returned list is empty in case of error.         */
  /* This function calls automatically xx_vl_init() on the new list.     */
  /* In addition, xx_vl_load() is designed in such a way that it keeps   */
  /* the original order of the elements of the array PVAR.               */
  xx_varlist xx_vl_load (const xx_vartype * pvar, ui_nteger length);

#ifdef __cplusplus
}
#endif
#endif /* _VARLIST_H_ */
