/*
   This file is part of the XXCalc Library - version 3.2
   Copyright (C)  2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010,
   2011, 2012, 2013    Ivano Primi ( ivprimi@libero.it )    

   The XXCalc Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The XXCalc library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<stdlib.h>
#include<string.h>
#include"ioctype.h"
#include"compl.h"
#include"varlist.h"
#include"tokstack.h"
#include"utils.h"
#include"parser.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

#define END_REACHED  XX_FATAL_ERROR+1

static int is_implicit_multiplication_allowed = 0; /* FALSE by default */

void xx_allow_implicit_multiplication (void)
{
  is_implicit_multiplication_allowed = 1;  
}

void xx_forbid_implicit_multiplication (void)
{
  is_implicit_multiplication_allowed = 0;  
}

static int
read_item (const char *str, xx_tokstack * ts)
{
  static long parlevel;		/* level of parenthesis */
  static const char *sz = NULL;	/* iterator             */
  char *endp;
  c_omplex z;
  b_yte ttype;			/* token-type */

  if ((str))
    {
      sz = str;
      parlevel = 0;
    }
  if (*sz == '\0')
    {
      if (parlevel > 0)
	return XX_MISSCP;
      else
	return END_REACHED;
    }
  else if ( (io_isdigit (*sz)) || *sz == '.' )
    {
      switch (xx_str2r (sz, &endp, &z.re))
	{
	case XX_OFLOW:
	  return XX_OVER_FLOW;
	case XX_NAN:
	  return XX_INVNUM;
	default:		/* XX_UFLOW + XX_AGNUM */
	  z.im = 0;
	  if (!xx_ts_push (ts, XX_NUMBER, -1, z, "", 0))
	    return XX_FATAL_ERROR;
	  else
	    {
	      /* Added on March 21, 2013 */
	      if ((is_implicit_multiplication_allowed) && (io_isalpha (*endp)))
		{
		  /*
		    If an identifier follows immediately after a
		    number, then a multiplication is implicit
		  */
		  z = c_convert(0);
		  if (!xx_ts_push (ts, XX_TM, parlevel * XX_PR_SKIP, z, "", 0))
		    return XX_FATAL_ERROR;
		}
	      /* End of the part added on March 21, 2013 */
	      sz = endp; /* The position of the iterator should be updated */
	    }
	}
    }
  else if (*sz == '{')
    {
      switch (xx_str2c (sz, &endp, &z))
	{
	case XX_OFLOW:
	  return XX_OVER_FLOW;
	case XX_NAN:
	  return XX_INVNUM;
	default:	    /* XX_UFLOW + XX_AGNUM */
	  if (!xx_ts_push (ts, XX_NUMBER, -1, z, "", 0))
	    return XX_FATAL_ERROR;
	  else
	    sz = endp;	    /* The position of the iterator should be updated */
	}
    }
  else if ( (io_isalpha (*sz)) )
    {
      xx_str2id (sz, &endp);   
      /* The return-value of xx_str2id() can be ignored */
      ttype = xx_toktype (sz, endp-sz);      
      if ( ttype >= XX_FUNCTION && ttype < XX_FUNCTION + XX_FUNCNUM )
	/* The identifier is a function */
	{
	  z = c_convert (0);
	  if (!xx_ts_push (ts, ttype, parlevel * XX_PR_SKIP, z, "", 0))
	    return XX_FATAL_ERROR;
	  else
	    sz = endp;	    /* The position of the iterator should be updated */
	}
      else
	/* The identifier should be a variable */
	{
	  if (*endp == '[') /* Added on Oct 16, 2002 */
	    {
	      do
		{
		  for(++endp; *endp != ']'; endp++);
		  for(++endp; xx_maybeinid(*endp); endp++);
		}while (*endp == '[');
	      if (endp - sz + 1 > XX_MAX_NDIM)
		return XX_TOKEN2LONG;
	    }
	  else /* End of the part added on Oct 16, 2002 */
	    if (endp - sz + 1 > XX_MAX_VNSIZE)
	      return XX_VNAME2LONG;
	  /* The evaluation of the variable is postponed */
	  z = c_convert (0);
	  if (!xx_ts_push (ts, XX_NUMBER, -1, z, sz, endp-sz))
	    return XX_FATAL_ERROR;
	  else
	    sz = endp;	    /* The position of the iterator should be updated */
	}
    }
  else if ( (xx_maybeinop (*sz)) )
    {
      z = c_convert (0);
      xx_str2op (sz, &endp);    
      /* The return-value of xx_str2op() can be ignored */
      ttype = xx_toktype (sz, endp-sz);
      if (ttype == XX_UNDEFINED)
	return XX_INEXOP;
      else if (!xx_ts_push (ts, ttype, parlevel * XX_PR_SKIP, z, "", 0))
	return XX_FATAL_ERROR;
      else
	sz = endp;	/* The position of the iterator should be updated */
    }
  else if (*sz == '~')	/* Unary minus */
    {
      z = c_convert (0);
      if (!xx_ts_push (ts, XX_UNARYMINUS, parlevel * XX_PR_SKIP, z, "", 0))
	return XX_FATAL_ERROR;
      else
	sz++;
    }
  else if (*sz == '(')
    {
      parlevel++;
      if (parlevel > XX_MAX_PAR_LEVEL)
	return XX_OVERMLP;
      else
	{
	  z = c_convert (0);
	  if (!xx_ts_push (ts, XX_OPEN_PAR, -1, z, "", 0))
	    return XX_FATAL_ERROR;
	  else
	    sz++;	/* The position of the iterator should be updated */
	}
    }
  else if (*sz == ')')
    {
      parlevel--;
      if (parlevel < 0)
	return XX_TOOCP;
      else
	{
	  z = c_convert (0);
	  if (!xx_ts_push (ts, XX_CLOSE_PAR, -1, z, "", 0))
	    return XX_FATAL_ERROR;
	  else
	    sz++;	/* The position of the iterator should be updated */
	}
    }
  else if ( (xx_isspace (*sz)) ) /* This code skips any spaces */
    {				 /* Added on Aug, 26 2002      */
      do
	{
	  sz++;
	}
      while ((xx_isspace (*sz)));
    }
  else				/* This is executed also if *sz=='_' */
    return XX_UNRECTOK;
  return XX_OK;
}

/* The function below returns an array which is */
/* a down-top copy of TS.                       */
static xx_mathtoken *
remake (xx_tokstack * ts, int *errcode)
{
  ui_nteger i, stacklen;
  const xx_mathtoken *pctok;
  xx_mathtoken *ptok;

  stacklen = xx_ts_len (*ts);
  if (stacklen == 0)
    {
      *errcode = XX_EMPTYEXPR;
      return NULL;
    }
  else
    ptok = (xx_mathtoken *) malloc (stacklen * sizeof (xx_mathtoken));
  if (!ptok)
    {
      *errcode = XX_FATAL_ERROR;
      return NULL;
    }
  else
    {
      for (i = 1, pctok = xx_ts_run (ts); pctok != NULL;
	   pctok = xx_ts_run (NULL), i++)
	{
	  ptok[stacklen - i].type = pctok->type;
	  ptok[stacklen - i].pr = pctok->pr;
	  ptok[stacklen - i].value = pctok->value;
	  strncpy (ptok[stacklen - i].name, pctok->name, XX_MAX_NDIM - 1);
	  ptok[stacklen - i].name[XX_MAX_NDIM - 1] = '\0';
	}
      *errcode = XX_OK;
      return ptok;
    }
}

static xx_mathtoken *
check_syntax (xx_tokstack * ts, int *errcode)
{
  ui_nteger lp[XX_MAX_PAR_LEVEL + 1];	/* Will be used at the end */
  xx_mathtoken tok;
  xx_mathtoken *ptok;
  ui_nteger i, j, stacklen = xx_ts_len (*ts);
  b_yte ttype1, ttype2;

  if (stacklen == 0)
    {
      *errcode = XX_EMPTYEXPR;
      return NULL;
    }
  else if (!(ptok = (xx_mathtoken *) malloc(stacklen * sizeof (xx_mathtoken))))
    {
      *errcode = XX_FATAL_ERROR;
      return NULL;
    }
  else
    {
      for (i = 1; xx_ts_top (*ts, &tok) != 0; i++)
	{
	  ptok[stacklen - i].type = tok.type;
	  ptok[stacklen - i].pr = tok.pr;
	  ptok[stacklen - i].value = tok.value;
	  strncpy (ptok[stacklen - i].name, tok.name, XX_MAX_NDIM - 1);
	  ptok[stacklen - i].name[XX_MAX_NDIM - 1] = '\0';
	  xx_ts_pop (ts);
	}
    }
  /* In ptok[] we find now the tokens forming the expression that we */
  /* are analyzing in the same order as they appear in it.           */
  /* The syntax checking can start.                                  */
  if (ptok[0].type == XX_CLOSE_PAR || xx_isop (ptok[0].type))
    {
      free ((void *) ptok);
      *errcode = XX_BADTOKATSTART;
      return NULL;
    }
  else if (ptok[stacklen - 1].type != XX_CLOSE_PAR
	   && ptok[stacklen - 1].type != XX_NUMBER)
    {
      free ((void *) ptok);
      *errcode = XX_BADTOKATEND;
      return NULL;
    }
  else
    {
      for (i = 0; i < stacklen - 1; i++)
	{
	  ttype1 = ptok[i].type;
	  ttype2 = ptok[i + 1].type;
	  if (xx_isfn (ttype1))
	    {
	      if (ttype2 != XX_OPEN_PAR)
		{
		  free ((void *) ptok);
		  *errcode = XX_FUNCWOOPAR;
		  return NULL;
		}
	      ptok[i].pr += XX_FUNC_PR;
	    }
	  else if (xx_isop (ttype1))
	    {
	      if (ttype2 == XX_CLOSE_PAR || xx_isop (ttype2))
		{
		  free ((void *) ptok);
		  *errcode = XX_MISSVALUE;
		  return NULL;
		}
	      switch (ttype1)
		{
		case XX_AND:
		case XX_OR:
		case XX_XOR:
		  ptok[i].pr += XX_LOG_PR;
		  break;
		case XX_LT:
		case XX_LE:
		case XX_GT:
		case XX_GE:
		case XX_EQ:
		case XX_NE:
		  ptok[i].pr += XX_CMP_PR;
		  break;
		case XX_PL:
		case XX_MN:
		  ptok[i].pr += XX_SUM_PR;
		  break;
		case XX_TM:
		case XX_DV:
		case XX_PC:
		case XX_MD:
		case XX_ID:
		  ptok[i].pr += XX_PROD_PR;
		  break;
		case XX_PW:
		  ptok[i].pr += XX_POW_PR;
		  break;
		case XX_MIN_OP:
		case XX_MAX_OP:
		  ptok[i].pr += XX_MINMAX_PR;
		}
	    }
	  else if (ttype1 == XX_UNARYMINUS)
	    {
	      /* 
		 The syntax checking relative to the usage of the unary minus
		 has been done inside xx_preprocessor(). Thus, I do
	         not need to repeat it here.                     
	      */
	      ptok[i].pr += XX_MIN_PR;
	    }
	  else if (ttype1 == XX_NUMBER)
	    {
	      if (ttype2 != XX_CLOSE_PAR && !xx_isop (ttype2))
		{
		  free ((void *) ptok);
		  *errcode = XX_MISSOPER;
		  return NULL;
		}
	    }
	  else if (ttype1 == XX_OPEN_PAR)
	    {
	      if (ttype2 == XX_CLOSE_PAR || xx_isop (ttype2))
		{
		  free ((void *) ptok);
		  *errcode = XX_MISSVALUE;
		  return NULL;
		}
	    }
	  else if (ttype1 == XX_CLOSE_PAR)
	    {
	      if (ttype2 != XX_CLOSE_PAR && !xx_isop (ttype2))
		{
		  free ((void *) ptok);
		  *errcode = XX_MISSOPER;
		  return NULL;
		}
	    }
	  else
	    {
	      free ((void *) ptok);
	      *errcode = XX_UNRECTOK;
	      return NULL;
	    }
	}		/* End for(i=0 ; i<stacklen-1; i++) {...} */
    }			/* End else */
  /* After the syntax checking we do not need any parentheses. */
  /* We have to complete the attribution of the priority       */
  /* level to operators and functions.                         */
  for (j = 0; j <= XX_MAX_PAR_LEVEL; j++)
    lp[j] = XX_OFN4LP;		/* Every level of parenthesis may */
  /* contain at most XX_OFN4LP operators and functions.           */
  for (i = 0; i < stacklen; i++)
    {
      if (ptok[i].pr >= 0)
	{
	  j = ptok[i].pr / XX_PR_SKIP;
	  if (lp[j] < 0)
	    {
	      free ((void *) ptok);
	      *errcode = XX_OVERLOAD;
	      return NULL;
	    }
	  else
	    {
	      ptok[i].pr += lp[j];
	      lp[j]--;
	    }
	}
      if (ptok[i].type != XX_OPEN_PAR && ptok[i].type != XX_CLOSE_PAR)
	{
	  if (!xx_ts_push
	      (ts, ptok[i].type, ptok[i].pr, ptok[i].value, ptok[i].name, strlen(ptok[i].name)))
	    {
	      free ((void *) ptok);
	      *errcode = XX_FATAL_ERROR;
	      return NULL;
	    }
	}
    }
  free ((void *) ptok);
  ptok = remake (ts, errcode);
  return ptok;			/* It is NULL if *errcode != XX_OK */
}

/* See preprocessor.c */
extern int xx_preprocessor (char *s);

xx_mathtoken *
xx_parser (const char *expr, xx_tokstack * ts, ui_nteger * len, int *errcode)
{
  xx_mathtoken *toklist = NULL;
  char *str;
  str_s_ize i, lexpr;

  if ((expr))
    {
      lexpr = strlen (expr);
      *len = 0;			/* *len is the length of the toklist */
      if (!xx_ts_isempty (*ts))
	xx_ts_destroy (ts);
      for (i = 0; i < lexpr; i++)
	{
	  if (!xx_isacceptable (expr[i]))
	    {
	      *errcode = XX_ILLCHAR;
	      return NULL;
	    }
	}
      str = (char *) malloc (sizeof (char) * (lexpr + 1));
      if (!str)
	{
	  *errcode = XX_FATAL_ERROR;
	  return NULL;
	}
      else
	strcpy (str, expr);
      *errcode = xx_preprocessor (str);
      /* We check the errcode returned by xx_preprocessor() */
      if (*errcode != XX_OK)
	{
	  free ((void *) str);
	  return NULL;
	}
      else
	for (*errcode = read_item (str, ts); *errcode == XX_OK;
	     *errcode = read_item (NULL, ts));
      /* Now str is no more necessary */
      free ((void *) str);
      if (*errcode != END_REACHED)
	{
	  xx_ts_destroy (ts);
	  return NULL;
	}
      else if (!(toklist = check_syntax (ts, errcode)))
	{
	  xx_ts_destroy (ts);
	  return NULL;
	}
      else
	{
	  *len = xx_ts_len (*ts);
	  return toklist;
	}
    }				/* End  if((expr)) */
  else				/* expr == NULL */
    {
      if (!(toklist = remake (ts, errcode)))
	{
	  xx_ts_destroy (ts);
	  return NULL;
	}
      else
	{
	  *len = xx_ts_len (*ts);
	  return toklist;
	}
    }
}
