/*
   This file is part of the XXCalc Library - version 3.2
   Copyright (C)  2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010,
   2011, 2012, 2013    Ivano Primi ( ivprimi@libero.it )    

   The XXCalc Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The XXCalc library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include"heapsort.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

#define V_EQUAL(l,r)     ( strcmp(l->name,r->name)==0 )
#define V_LESSTHAN(l,r)  ( strcmp(l->name,r->name)< 0 )
#define C_EQUAL(l,r)     (l.pr == r.pr)
#define C_LESSTHAN(l,r)  (l.pr < r.pr)
#define S_EQUAL(l,r)     ( strcmp(l,r)==0 )
#define S_LESSTHAN(l,r)  ( strcmp(l,r)< 0 )

/* The function v_assign is a utility to copy a key to another one */
static void
v_assign (xx_vartype **lvalue, xx_vartype **rvalue)
{
  *lvalue = *rvalue;
}

/* 
   Compare the keys KEY1 and KEY2 and return:
   0  if they are equal,
   -1 if KEY1 is less than KEY2,
   +1 if KEY1 is greater than KEY2.
*/
static int
v_cfr (xx_vartype* key1, xx_vartype* key2)
{
  if V_EQUAL
    (key1, key2) return 0;
  else if V_LESSTHAN
    (key1, key2) return -1;
  else
    return 1;
}

/* 'A' should be an array, 'n' is the number of the elements of 'A' */

static void
v_heapify (xx_vartype ** A, long h, long i)
{
  long l, r, m;

  /* l : index of the left child of i       */
  /* r : index of the right child of i      */
  /* m : index of the maximum between i,l,r */

  xx_vartype *tmp;			/* auxiliary variable */

  /* We may assume h>i>=0 */
  l = 2 * i + 1;
  r = 2 * i + 2;
  if ((l < h) && (v_cfr (A[l], A[i]) > 0))
    m = l;
  else
    m = i;
  if ((r < h) && (v_cfr (A[r], A[m]) > 0))
    m = r;			/* It is obvious that h>i>=0 ==> h>m>=0 */
  if (m != i)
    {
      v_assign (&tmp, &A[i]);
      v_assign (&A[i], &A[m]);
      v_assign (&A[m], &tmp);
      v_heapify (A, h, m);
    }
}

static void
v_buildheap (xx_vartype ** A, long n)
{
  long i;			/* Counter */

  /* 
     We can assume n > 0.
     The elements with index > [n/2]-1 will be the leafs of the
     heap; therefore, they are not going to be processed.
  */
  for (i = n / 2 - 1; i >= 0; i--)
    v_heapify (A, n, i);	/* n > i >= 0 */
}

int
xx_vheapsort (xx_vartype ** A, long n)
{
  xx_vartype *tmp;		/* auxiliary variable */
  long i;			/* Counter */

  if (n <= 0)
    {
      fprintf (stderr, "\"%s\", %u: 2nd arg is <= 0 (xx_vheapsort)\n\n",
	       __FILE__, __LINE__);
      return -1;
    }
  v_buildheap (A, n);
  for (i = n - 1; i >= 1; i--)
    {
      v_assign (&tmp, &A[i]);
      v_assign (&A[i], &A[0]);
      v_assign (&A[0], &tmp);
      v_heapify (A, i, 0);
    }
  return 0;
}

/* *** */

/* The function c_assign is a utility to copy a key to another one */
static void
c_assign (xx_couple * lvalue, const xx_couple * rvalue)
{
  *lvalue = *rvalue;
}

/* 
   Compare the keys KEY1 and KEY2 and return:
   0  if they are equal,
   -1 if KEY1 is less than KEY2,
   +1 if KEY1 is greater than KEY2.
*/
static int
c_cfr (xx_couple key1, xx_couple key2)
{
  if C_EQUAL
    (key1, key2) return 0;
  else if C_LESSTHAN
    (key1, key2) return -1;
  else
    return 1;
}

/* 'A' should be an array, 'n' is the number of the elements of 'A' */

static void
c_heapify (xx_couple * A, long h, long i)
{
  long l, r, m;

  /* l : index of the left child of i       */
  /* r : index of the right child of i      */
  /* m : index of the maximum between i,l,r */

  xx_couple tmp;			/* auxiliary variable */

  /* We can assume h>i>=0 */
  l = 2 * i + 1;
  r = 2 * i + 2;
  if ((l < h) && (c_cfr (A[l], A[i]) > 0))
    m = l;
  else
    m = i;
  if ((r < h) && (c_cfr (A[r], A[m]) > 0))
    m = r;			/* It is obvious that h>i>=0 ==> h>m>=0 */
  if (m != i)
    {
      c_assign (&tmp, &A[i]);
      c_assign (&A[i], &A[m]);
      c_assign (&A[m], &tmp);
      c_heapify (A, h, m);
    }
}

static void
c_buildheap (xx_couple * A, long n)
{
  long i;			/* Counter */

  /* 
     We can assume n > 0.
     The elements with index > [n/2]-1 will be the leafs of the
     heap; therefore, they are not going to be processed.
  */
  for (i = n / 2 - 1; i >= 0; i--)
    c_heapify (A, n, i);	/* n > i >= 0 */
}

int
xx_cheapsort (xx_couple * A, long n)
{
  xx_couple tmp;		/* auxiliary variable */
  long i;			/* Counter */

  if (n <= 0)
    {
      fprintf (stderr, "\"%s\", %u: 2nd arg is <= 0 (xx_cheapsort)\n\n",
	       __FILE__, __LINE__);
      return -1;
    }
  c_buildheap (A, n);
  for (i = n - 1; i >= 1; i--)
    {
      c_assign (&tmp, &A[i]);
      c_assign (&A[i], &A[0]);
      c_assign (&A[0], &tmp);
      c_heapify (A, i, 0);
    }
  return 0;
}

/* *** */

/* The function s_assign is a utility to copy a key to another one */
static void
s_assign (const char** lvalue, const char** rvalue)
{
  *lvalue = *rvalue;
}

/* 
   Compare the keys KEY1 and KEY2 and return:
   0  if they are equal,
   -1 if KEY1 is less than KEY2,
   +1 if KEY1 is greater than KEY2.
*/
static int
s_cfr (const char* key1, const char* key2)
{
  if S_EQUAL
    (key1, key2) return 0;
  else if S_LESSTHAN
    (key1, key2) return -1;
  else
    return 1;
}

/* 'A' should be an array of dinamycally allocated strings, 
   'n' is the number of the elements of 'A' */

static void
s_heapify (const char** A, long h, long i)
{
  long l, r, m;

  /* l : index of the left child of i       */
  /* r : index of the right child of i      */
  /* m : index of the maximum between i,l,r */

  const char* tmp = NULL;	/* auxiliary variable */

  /* We can assume h>i>=0 */
  l = 2 * i + 1;
  r = 2 * i + 2;
  if ((l < h) && (s_cfr (A[l], A[i]) > 0))
    m = l;
  else
    m = i;
  if ((r < h) && (s_cfr (A[r], A[m]) > 0))
    m = r;			/* It is obvious that h>i>=0 ==> h>m>=0 */
  if (m != i)
    {
      s_assign (&tmp, &A[i]);
      s_assign (&A[i], &A[m]);
      s_assign (&A[m], &tmp);
      s_heapify (A, h, m);
    }
}

static void
s_buildheap (const char** A, long n)
{
  long i;			/* Counter */

  /* 
     We can assume n > 0.
     The elements with index > [n/2]-1 will be the leafs of the
     heap; therefore, they are not going to be processed.
  */
  for (i = n / 2 - 1; i >= 0; i--)
    s_heapify (A, n, i);	/* n > i >= 0 */
}

int
xx_sheapsort (const char** A, long n)
{
  const char* tmp = NULL;	/* auxiliary variable */
  long i;			/* Counter */

  if (n <= 0)
    {
      fprintf (stderr, "\"%s\", %u: 2nd arg is <= 0 (xx_sheapsort)\n\n",
	       __FILE__, __LINE__);
      return -1;
    }
  s_buildheap (A, n);
  for (i = n - 1; i >= 1; i--)
    {
      s_assign (&tmp, &A[i]);
      s_assign (&A[i], &A[0]);
      s_assign (&A[0], &tmp);
      s_heapify (A, i, 0);
    }
  return 0;
}
