/*
   This file is part of the XXCalc Library - version 3.2
   Copyright (C)  2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010,
   2011, 2012, 2013    Ivano Primi ( ivprimi@libero.it )    

   The XXCalc Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The XXCalc library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<string.h>
#include<math.h>
#include<stdarg.h>
#include"compl.h"

/* 
   Ok, what follows is not the best solution, but I was
   forced to do so, since C++ compilers do not
   love the header file <tgmath.h>.
   On the other hand, this solution should increase the
   speed of execution of the code.                      
*/
#ifdef USE_LONG_DOUBLE
#define __FABS  fabsl
#define __ATAN2 atan2l
#define __SQRT  sqrtl
#define __FLOOR floorl
#define __CEIL  ceill
#define __POW   powl
#define __COS   cosl
#define __SIN   sinl
#define __EXP   expl
#define __LOG   logl
#define __LOG10 log10l
#else
#define __FABS  fabs
#define __ATAN2 atan2
#define __SQRT  sqrt
#define __FLOOR floor
#define __CEIL  ceil
#define __POW   pow
#define __COS   cos
#define __SIN   sin
#define __EXP   exp
#define __LOG   log
#define __LOG10 log10
#endif

#ifndef STOP_AT_ERROR

int c_errno;

const int c_nerr = 5;

const char *c_errlist[] = {
  "No error",
  "Division by 0",
  "Out of domain",
  "Bad exponent",
  "Complex-valued result"
};

#else

#include<stdlib.h>

#endif /* STOP_AT_ERROR */

#ifdef DMALLOC
#include <dmalloc.h>
#endif

const c_omplex c_0 = { 0.0, 0.0 };
const c_omplex c_1 = { 1.0, 0.0 };
const c_omplex c_i = { 0.0, 1.0 };

#define VSV  1.0e-4
#define VGV  1.0e+2

/* Some simple utility functions */

static int
sprint (char *s, str_s_ize size, const char *fmt, ...)
{
  /* Call vsnpritf(s,size,fmt,...) if vsnprintf() is available, */
  /* else call vsprintf(s,size,fmt,...).                        */
  int retval;
  va_list ap;

  va_start (ap, fmt);
#ifdef HAVE_VSNPRINTF
  retval = vsnprintf (s, size, fmt, ap);
#else
  retval = vsprintf (s, fmt, ap);
#endif
  va_end (ap);
  return retval;
}

/* 
   Round x to the nearest integer.
   If the fractional part of x is exactly equal     
   to 1/2, then round x away from zero.
*/
static r_eal
rinteger (r_eal x)
{
  if (x >= 0)
    {
      if (x - __FLOOR (x) < 0.5)
	return __FLOOR (x);
      else
	return __CEIL (x);
    }
  else
    {
      if (x - __FLOOR (x) <= 0.5)
	return __FLOOR (x);
      else
	return __CEIL (x);
    }
}

/* 
   Round x to the nearest integer 
   not larger in absolute value.  
*/
static r_eal
rtruncate (r_eal x)
{
  if (x >= 0)
    return __FLOOR (x);
  else
    return __CEIL (x);
}

static r_eal
ARG (c_omplex z)
{
  if (z.re < 0 && __FABS (z.im) < EPS)
    return MATH_PI;
  else
    return __ATAN2 (z.im, z.re);
}

c_omplex
c_reset (r_eal re, r_eal im)
{
  c_omplex z;

  z.re = re;
  z.im = im;
  return z;
}

c_omplex
c_convert (r_eal x)
{
  c_omplex z;

  z.re = x;
  z.im = 0;
  return z;
}

c_omplex
c_init (void)
{
  c_omplex z;

  z.re = z.im = 0;
  return z;
}

gauss_int
gi_reset (i_nteger re, i_nteger im)
{
  gauss_int w;

  w.re = re;
  w.im = im;
  return w;
}

gauss_int
gi_convert (i_nteger n)
{
  gauss_int w;

  w.re = n;
  w.im = 0;
  return w;
}

gauss_int
gi_init (void)
{
  gauss_int w;

  w.re = w.im = 0;
  return w;
}

c_omplex
c_fromgi (gauss_int w)
{
  c_omplex z;

  z.re = (r_eal) w.re;
  z.im = (r_eal) w.im;
  return z;
}

gauss_int
gi_fromc (c_omplex z)
{
  gauss_int w;

  w.re = (i_nteger) z.re;
  w.im = (i_nteger) z.im;
  return w;
}

r_eal
c_re (c_omplex z)
{
  return z.re;
}

r_eal
c_im (c_omplex z)
{
  return z.im;
}

r_eal
c_abs (c_omplex z)
{
  return __SQRT (z.re * z.re + z.im * z.im);
}

r_eal
c_arg (c_omplex z)
{
  r_eal mod = __SQRT (z.re * z.re + z.im * z.im);

#if !defined(STOP_AT_ERROR)
  if (mod == 0)
    {
      c_errno = C_EDOM;
      return 0;
    }
#else
  if (mod == 0)
    {
      fprintf (stderr, "\"%s\", %u: Out of domain(c_arg)\n\n",
	       __FILE__, __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  return ARG (z);
}

c_omplex
c_sum (c_omplex z1, c_omplex z2)
{
  c_omplex z;

  z.re = z1.re + z2.re;
  z.im = z1.im + z2.im;
  return z;
}

c_omplex
c_diff (c_omplex z1, c_omplex z2)
{
  c_omplex z;

  z.re = z1.re - z2.re;
  z.im = z1.im - z2.im;
  return z;
}

c_omplex
c_prod (c_omplex z1, c_omplex z2)
{
  c_omplex z;

  z.re = z1.re * z2.re - z1.im * z2.im;
  z.im = z1.im * z2.re + z1.re * z2.im;
  return z;
}

c_omplex
c_div (c_omplex z1, c_omplex z2)
{
  c_omplex z;
  r_eal mod2 = z2.re * z2.re + z2.im * z2.im;

#if !defined(STOP_AT_ERROR)
  if (mod2 == 0)
    {
      c_errno = C_EDIV;
      z.re = z.im = 0;
      return z;
    }
#else
  if (mod2 == 0)
    {
      fprintf (stderr, "\"%s\", %u: Division by zero\n\n", __FILE__,
	       __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  z.re = (z1.re * z2.re + z1.im * z2.im) / mod2;
  z.im = (z2.re * z1.im - z2.im * z1.re) / mod2;
  return z;
}

c_omplex
c_idiv (c_omplex z1, c_omplex z2)
{
  c_omplex w1, w2, z;
  r_eal mod2;

  w1.re = rinteger (z1.re);
  w1.im = rinteger (z1.im);
  w2.re = rinteger (z2.re);
  w2.im = rinteger (z2.im);
#if !defined(STOP_AT_ERROR)
  if ((w2.re == 0) && (w2.im == 0))
    {
      c_errno = C_EDIV;
      z.re = z.im = 0;
      return z;
    }
#else
  if ((w2.re == 0) && (w2.im == 0))
    {
      fprintf (stderr, "\"%s\", %u: Division by zero\n\n", __FILE__,
	       __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  mod2 = w2.re * w2.re + w2.im * w2.im;
  z.re = rinteger ((w1.re * w2.re + w1.im * w2.im) / mod2);
  z.im = rinteger ((w1.im * w2.re - w1.re * w2.im) / mod2);
  return z;
}

c_omplex
c_idiv_ (c_omplex z1, c_omplex z2)
{
  c_omplex w1, w2, z;
  r_eal mod2;

  w1.re = rinteger (z1.re);
  w1.im = rinteger (z1.im);
  w2.re = rinteger (z2.re);
  w2.im = rinteger (z2.im);
#if !defined(STOP_AT_ERROR)
  if ((w2.re == 0) && (w2.im == 0))
    {
      c_errno = C_EDIV;
      z.re = z.im = 0;
      return z;
    }
#else
  if ((w2.re == 0) && (w2.im == 0))
    {
      fprintf (stderr, "\"%s\", %u: Division by zero\n\n", __FILE__,
	       __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  mod2 = w2.re * w2.re + w2.im * w2.im;
  z.re = rtruncate ((w1.re * w2.re + w1.im * w2.im) / mod2);
  z.im = rtruncate ((w1.im * w2.re - w1.re * w2.im) / mod2);
  return z;
}

c_omplex
c_mod (c_omplex z1, c_omplex z2)
{
  c_omplex w1, w2, w, z;
  r_eal mod2;

  w1.re = rinteger (z1.re);
  w1.im = rinteger (z1.im);
  w2.re = rinteger (z2.re);
  w2.im = rinteger (z2.im);
#if !defined(STOP_AT_ERROR)
  if ((w2.re == 0) && (w2.im == 0))
    {
      c_errno = C_EDIV;
      z.re = z.im = 0;
      return z;
    }
#else
  if ((w2.re == 0) && (w2.im == 0))
    {
      fprintf (stderr, "\"%s\", %u: Division by zero\n\n", __FILE__,
	       __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  mod2 = w2.re * w2.re + w2.im * w2.im;
  w.re = rinteger ((w1.re * w2.re + w1.im * w2.im) / mod2);
  w.im = rinteger ((w1.im * w2.re - w1.re * w2.im) / mod2);
  z.re = rinteger (w1.re - w.re * w2.re + w.im * w2.im);
  z.im = rinteger (w1.im - w.im * w2.re - w.re * w2.im);
  return z;
}

c_omplex
c_mod_ (c_omplex z1, c_omplex z2)
{
  c_omplex w1, w2, w, z;
  r_eal mod2;

  w1.re = rinteger (z1.re);
  w1.im = rinteger (z1.im);
  w2.re = rinteger (z2.re);
  w2.im = rinteger (z2.im);
#if !defined(STOP_AT_ERROR)
  if ((w2.re == 0) && (w2.im == 0))
    {
      c_errno = C_EDIV;
      z.re = z.im = 0;
      return z;
    }
#else
  if ((w2.re == 0) && (w2.im == 0))
    {
      fprintf (stderr, "\"%s\", %u: Division by zero\n\n", __FILE__,
	       __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  mod2 = w2.re * w2.re + w2.im * w2.im;
  w.re = rtruncate ((w1.re * w2.re + w1.im * w2.im) / mod2);
  w.im = rtruncate ((w1.im * w2.re - w1.re * w2.im) / mod2);
  z.re = rinteger (w1.re - w.re * w2.re + w.im * w2.im);
  z.im = rinteger (w1.im - w.im * w2.re - w.re * w2.im);
  return z;
}

c_omplex
c_ipow (c_omplex z, i_nteger n)
{
  r_eal mod, arg;
  c_omplex w;

  mod = __SQRT (z.re * z.re + z.im * z.im);
  if (mod == 0)
    {
#if !defined(STOP_AT_ERROR)
      if (n <= 0)
	c_errno = C_EBADEXP;
      else
	c_errno = 0;
      w.re = w.im = 0;
      return w;
#else
      if (n <= 0)
	{
	  fprintf (stderr,
		   "\"%s\", %u: Cannot raise zero to a power n <= 0\n\n",
		   __FILE__, __LINE__);
	  exit (EXIT_FAILURE);
	}
      else
	{
	  w.re = w.im = 0;
	  return w;
	}
#endif
    }
  arg = ARG (z);
  mod = __POW (mod, n);
  arg *= n;
  w.re = mod * __COS (arg);
  w.im = mod * __SIN (arg);
  return w;
}

c_omplex
c_sqr (c_omplex z)
{
  c_omplex w;

  w.re = z.re * z.re - z.im * z.im;
  w.im = 2 * z.re * z.im;
  return w;
}

c_omplex
c_pow (c_omplex z1, c_omplex z2)
{
  r_eal mod, arg, a, b;
  c_omplex w;

  mod = __SQRT (z1.re * z1.re + z1.im * z1.im);
  if (mod == 0)
    {
#if !defined(STOP_AT_ERROR)
      if (z2.re <= 0)
	c_errno = C_EBADEXP;
      else
	c_errno = 0;
      w.re = w.im = 0;
      return w;
#else
      if (z2.re <= 0)
	{
	  fprintf (stderr,
		   "\"%s\", %u: Cannot raise zero to a power z with real part <= 0\n\n",
		   __FILE__, __LINE__);
	  exit (EXIT_FAILURE);
	}
      else
	{
	  w.re = w.im = 0;
	  return w;
	}
#endif
    }
  arg = ARG (z1);
  a = z2.re * __LOG (mod) - z2.im * arg;
  b = z2.re * arg + z2.im * __LOG (mod);
  w.re = __EXP (a) * __COS (b);
  w.im = __EXP (a) * __SIN (b);
  return w;
}

c_omplex
c_root (c_omplex z, i_nteger i, i_nteger n)
{
  r_eal mod, arg;
  c_omplex w;

  if (n == 0)
    {
#if !defined(STOP_AT_ERROR)
      c_errno = C_EBADEXP;
      w.re = w.im = 0;
      return w;
#else
      fprintf (stderr, "\"%s\", %u: Root 0th is not defined\n\n", __FILE__,
	       __LINE__);
      exit (EXIT_FAILURE);
#endif
    }
  mod = __SQRT (z.re * z.re + z.im * z.im);
  if (mod == 0)
    {
#if !defined(STOP_AT_ERROR)
      if (n < 0)
	c_errno = C_EBADEXP;
      else
	c_errno = 0;
      w.re = w.im = 0;
      return w;
#else
      if (n < 0)
	{
	  fprintf (stderr,
		   "\"%s\", %u: Cannot raise zero to 1/n with n < 0\n\n",
		   __FILE__, __LINE__);
	  exit (EXIT_FAILURE);
	}
      else
	{
	  w.re = w.im = 0;
	  return w;
	}
#endif
    }
  arg = ARG (z);
  mod = __POW (mod, 1.0 / n);
  arg = (arg + i * 2 * MATH_PI) / n;
  w.re = mod * __COS (arg);
  w.im = mod * __SIN (arg);
  return w;
}

#define SGN(x) ((x) >= 0 ? 1 : -1)

c_omplex
c_sqrt (c_omplex z)
{
  r_eal mod;
  c_omplex w;

  mod = __SQRT (z.re * z.re + z.im * z.im);
  w.re = __SQRT ((mod + z.re) * 0.5);
  w.im = SGN (z.im) * __SQRT ((mod - z.re) * 0.5);
  return w;
}

i_nteger
c_is0 (c_omplex z)
{
  return ((z.re == 0) && (z.im == 0)) ? 1 : 0;
}

i_nteger
c_not0 (c_omplex z)
{
  return ((z.re != 0) || (z.im != 0)) ? 1 : 0;
}

i_nteger
c_eq (c_omplex z1, c_omplex z2)
{
  return ((z1.re == z2.re) && (z1.im == z2.im)) ? 1 : 0;
}

i_nteger
c_neq (c_omplex z1, c_omplex z2)
{
  return ((z1.re != z2.re) || (z1.im != z2.im)) ? 1 : 0;
}

i_nteger
c_gt (c_omplex z1, c_omplex z2)
{
  return z1.re > z2.re ? 1 : 0;
}

i_nteger
c_ge (c_omplex z1, c_omplex z2)
{
  return z1.re >= z2.re ? 1 : 0;
}

i_nteger
c_lt (c_omplex z1, c_omplex z2)
{
  return z1.re < z2.re ? 1 : 0;
}

i_nteger
c_le (c_omplex z1, c_omplex z2)
{
  return z1.re <= z2.re ? 1 : 0;
}

c_omplex
c_neg (c_omplex z)
{
  c_omplex w;

  w.re = -z.re;
  w.im = -z.im;
  return w;
}

c_omplex
c_conj (c_omplex z)
{
  c_omplex w;

  w.re = z.re;
  w.im = -z.im;
  return w;
}

c_omplex
c_inv (c_omplex z)
{
  c_omplex w;
  r_eal mod2 = z.re * z.re + z.im * z.im;

#if !defined(STOP_AT_ERROR)
  if (mod2 == 0)
    {
      c_errno = C_EDIV;
      w.re = w.im = 0;
      return w;
    }
#else
  if (mod2 == 0)
    {
      fprintf (stderr, "\"%s\", %u: Division by zero\n\n", __FILE__,
	       __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  w.re = z.re / mod2;
  w.im = -z.im / mod2;
  return w;
}

c_omplex
c_swap (c_omplex z)
{
  c_omplex w;

  w.re = z.im;
  w.im = z.re;
  return w;
}

c_omplex
c_exp (c_omplex z)
{
  c_omplex w;

  w.re = __EXP (z.re) * __COS (z.im);
  w.im = __EXP (z.re) * __SIN (z.im);
  return w;
}

c_omplex
c_exp10 (c_omplex z)
{
  c_omplex w;

  w.re = __EXP (__LOG (10) * z.re) * __COS (__LOG (10) * z.im);
  w.im = __EXP (__LOG (10) * z.re) * __SIN (__LOG (10) * z.im);
  return w;
}

c_omplex
c_log (c_omplex z)
{
  r_eal mod = __SQRT (z.re * z.re + z.im * z.im);
  c_omplex w;

#if !defined(STOP_AT_ERROR)
  if (mod == 0)
    {
      c_errno = C_EDOM;
      w.re = w.im = 0;
      return w;
    }
#else
  if (mod == 0)
    {
      fprintf (stderr, "\"%s\", %u: Out of domain(c_log)\n\n",
	       __FILE__, __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  w.re = __LOG (mod);
  w.im = ARG (z);
  return w;
}

c_omplex
c_log10 (c_omplex z)
{
  r_eal mod = __SQRT (z.re * z.re + z.im * z.im);
  c_omplex w;

#if !defined(STOP_AT_ERROR)
  if (mod == 0)
    {
      c_errno = C_EDOM;
      w.re = w.im = 0;
      return w;
    }
#else
  if (mod == 0)
    {
      fprintf (stderr, "\"%s\", %u: Out of domain(c_log10)\n\n",
	       __FILE__, __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  w.re = __LOG10 (mod);
  w.im = (ARG (z)) / __LOG (10);
  return w;
}

c_omplex
c_sin (c_omplex z)
{
  c_omplex w1, w2;

  w1.re = -z.im;
  w1.im = z.re;			/* Now w1= i*z,  where i={0,1} */
  w2.re = z.im;
  w2.im = -z.re;		/* Now w2= -i*z, where i={0,1} */
  w1 = c_diff (c_exp (w1), c_exp (w2));
  w2.re = w1.im * 0.5;
  w2.im = -w1.re * 0.5;		/* Now w2= (exp(i*z)-exp(-i*z))/2i */
  return w2;
}

c_omplex
c_cos (c_omplex z)
{
  c_omplex w1, w2;

  w1.re = -z.im;
  w1.im = z.re;			/* Now w1= i*z,  where i={0,1} */
  w2.re = z.im;
  w2.im = -z.re;		/* Now w2= -i*z, where i={0,1} */
  w1 = c_sum (c_exp (w1), c_exp (w2));
  w2.re = w1.re * 0.5;
  w2.im = w1.im * 0.5;
  return w2;
}

c_omplex
c_tan (c_omplex z)
{
  c_omplex w;

#if !defined(STOP_AT_ERROR)
  if (__COS (z.re) == 0 && z.im == 0)
    {
      c_errno = C_EDOM;
      w.re = w.im = 0;
      return w;
    }
#else
  if (__COS (z.re) == 0 && z.im == 0)
    {
      fprintf (stderr, "\"%s\", %u: Out of domain(c_tan)\n\n", __FILE__,
	       __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  w = c_div (c_sin (z), c_cos (z));
  return w;
}

c_omplex
c_sinh (c_omplex z)
{
  c_omplex w;

  w = c_diff (c_exp (z), c_exp (c_neg (z)));
  w.re *= 0.5;
  w.im *= 0.5;
  return w;
}

c_omplex
c_cosh (c_omplex z)
{
  c_omplex w;

  w = c_sum (c_exp (z), c_exp (c_neg (z)));
  w.re *= 0.5;
  w.im *= 0.5;
  return w;
}

c_omplex
c_tanh (c_omplex z)
{
  c_omplex w;

#if !defined(STOP_AT_ERROR)
  if (z.re == 0 && __COS (z.im) == 0)
    {
      c_errno = C_EDOM;
      w.re = w.im = 0;
      return w;
    }
#else
  if (z.re == 0 && __COS (z.im) == 0)
    {
      fprintf (stderr, "\"%s\", %u: Out of domain(c_tanh)\n\n", __FILE__,
	       __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  w = c_div (c_sinh (z), c_cosh (z));
  return w;
}

c_omplex
c_asin (c_omplex z)
{
  c_omplex w;
  c_omplex one = c_reset (1, 0);
  c_omplex i = c_reset (0, 1);

  w = c_sqrt (c_diff (one, c_sqr (z)));
  if (c_abs (c_sum (c_prod (i, z), w)) < VSV * c_abs (z))
    {
      w = c_log (c_diff (w, c_prod (i, z)));
      w = c_prod (i, w);
    }
  else
    {
      w = c_log (c_sum (c_prod (i, z), w));
      w = c_prod (c_neg (i), w);
    }
  return w;
}

c_omplex
c_acos (c_omplex z)
{
  c_omplex w;
  c_omplex one = c_reset (1, 0);
  c_omplex i = c_reset (0, 1);

  w = c_sqrt (c_diff (one, c_sqr (z)));
  if (c_abs (c_sum (z, c_prod (i, w))) < VSV * c_abs(z))
    {
      w = c_log (c_diff (z, c_prod (i, w)));
      w = c_prod (i, w);
    }
  else
    {
      w = c_log (c_sum (z, c_prod (i, w)));
      w = c_prod (c_neg (i), w);
    }
  return w;
}

c_omplex
c_atan (c_omplex z)
{
  c_omplex w;
  c_omplex one = c_reset (1, 0);
  c_omplex i = c_reset (0, 1);

#if !defined(STOP_AT_ERROR)
  if (c_eq (z, i) || c_eq (z, c_neg (i)))
    {
      c_errno = C_EDOM;
      w.re = w.im = 0;
      return w;
    }
#else
  if (c_eq (z, i) || c_eq (z, c_neg (i)))
    {
      fprintf (stderr, "\"%s\", %u: Out of domain(c_atan)\n\n", __FILE__,
	       __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  else
    {
      /* In this way, c_atan() works fine also with complex numbers */
      /* very far from the origin.                                  */
      if (c_abs (z) > VGV)
	{
	  w = c_sqrt (c_sum (one, c_sqr (z)));
	  w = c_div (c_sum (one, c_prod (i, z)), w);
	  w = c_prod (c_neg (i), c_log (w));
	}
      else
	{
	  w = c_div (c_sum (one, c_prod (i, z)), c_diff (one, c_prod (i, z)));
	  w = c_prod (c_neg (i), c_log (c_sqrt (w)));
	}
      return w;
    }
}

c_omplex
c_asinh (c_omplex z)
{
  c_omplex w;
  c_omplex one = c_reset (1, 0);

  /* In this way, c_asinh() works fine also with real numbers */
  /* very near to -oo.                                       */
  w = c_sqrt (c_sum (one, c_sqr (z)));
  if (c_abs (c_sum (z, w)) < VSV * c_abs(z))
    w = c_neg (c_log (c_diff (w, z)));
  else
    w = c_log (c_sum (z, w));
  return w;
}

c_omplex
c_acosh (c_omplex z)
{
  c_omplex w;
  c_omplex one = c_reset (1, 0);

  w = c_sqrt (c_diff (c_sqr (z), one));
  if (c_abs (c_sum (z, w)) < VSV * c_abs(z))
    w = c_neg (c_log (c_diff (z, w)));
  else
    w = c_log (c_sum (z, w));
  return w;
}

c_omplex
c_atanh (c_omplex z)
{
  c_omplex w;
  c_omplex one = c_reset (1, 0);

#if !defined(STOP_AT_ERROR)
  if (c_eq (z, one) || c_eq (z, c_neg (one)))
    {
      c_errno = C_EDOM;
      w.re = w.im = 0;
      return w;
    }
#else
  if (c_eq (z, one) || c_eq (z, c_neg (one)))
    {
      fprintf (stderr, "\"%s\", %u: Out of domain(c_atanh)\n\n", __FILE__,
	       __LINE__);
      exit (EXIT_FAILURE);
    }
#endif
  else
    {
      w = c_div (c_sum (one, z), c_diff (one, z));
      w = c_log (c_sqrt (w));
      return w;
    }
}

c_omplex
c_floor (c_omplex z)
{
  c_omplex w;

  w.re = __FLOOR (z.re);
  w.im = __FLOOR (z.im);
  return w;
}

c_omplex
c_ceil (c_omplex z)
{
  c_omplex w;

  w.re = __CEIL (z.re);
  w.im = __CEIL (z.im);
  return w;
}

c_omplex
c_round (c_omplex z)
{
  c_omplex w;

  w.re = rinteger (z.re);
  w.im = rinteger (z.im);
  return w;
}

c_omplex
c_fix (c_omplex z)
{
  c_omplex w;

  w.re = rtruncate (z.re);
  w.im = rtruncate (z.im);
  return w;
}

c_omplex
c_frac (c_omplex z)
{
  c_omplex w;

  w.re = z.re - rtruncate (z.re);
  w.im = z.im - rtruncate (z.im);
  return w;
}

/* --> 1 if z.im == 0  and  0 <= z.re <= 1, else 0 */
c_omplex c_chcc (c_omplex z)
{
  if( z.im==0.0 && z.re>=0.0 && z.re<=1.0 )
    return c_1;
  else
    return c_0;
}

/* --> 1 if z.im == 0  and  0 <= z.re <  1, else 0 */
c_omplex c_chco (c_omplex z)
{
  if( z.im==0.0 && z.re>=0.0 && z.re<1.0 )
    return c_1;
  else
    return c_0;
}

/* --> 1 if z.im == 0  and  0 <  z.re <= 1, else 0 */
c_omplex c_choc (c_omplex z)
{
  if( z.im==0.0 && z.re>0.0 && z.re<=1.0 )
    return c_1;
  else
    return c_0;
}

/* --> 1 if z.im == 0  and  0 <  z.re <  1, else 0 */
c_omplex c_choo (c_omplex z)
{
  if( z.im==0.0 && z.re>0.0 && z.re<1.0 )
    return c_1;
  else
    return c_0;
}

/* --> 1 if z.im == 0  and  z.re <=  0, else 0 */
c_omplex c_chlc (c_omplex z)
{
  if( z.im==0.0 && z.re<=0.0 )
    return c_1;
  else
    return c_0;
}

/* --> 1 if z.im == 0  and  z.re <   0, else 0 */
c_omplex c_chlo (c_omplex z)
{
  if( z.im==0.0 && z.re<0.0 )
    return c_1;
  else
    return c_0;
}

/* --> 1 if z.im == 0  and  z.re >=  0, else 0 */
c_omplex c_chrc (c_omplex z)
{
  if( z.im==0.0 && z.re>=0.0 )
    return c_1;
  else
    return c_0;
}

/* --> 1 if z.im == 0  and  z.re >   0, else 0 */
c_omplex c_chro (c_omplex z)
{
  if( z.im==0.0 && z.re>0.0 )
    return c_1;
  else
    return c_0;
}

#if  defined(_BSD_SOURCE) || defined(_SVID_SOURCE) || defined(_ISOC99_SOURCE) || _XOPEN_SOURCE - 0 >= 600

#ifdef USE_LONG_DOUBLE
#define r_erf erfl
#else
#define r_erf erf
#endif

#else

static r_eal r_erf (r_eal x)
{
  int sign = 1;
  r_eal one, coeff, C, N, u, h, t, f0, f1, f2, f3, f4, prod, sum;
  int i, j, k, n;

#ifdef USE_LONG_DOUBLE
  coeff = 2.0L;
  one = 1.0L;
  N = 7.4L;
  C = 2.0L;
  n = 43;
#else
  coeff = 2.0;
  one = 1.0;
  N = 6.1;
  C = 2.0;
  n = 34;
#endif
  coeff /= __SQRT (MATH_PI);
  
  if ( x < 0.0 )
    {
      sign = -1;
      x = -x;
    }
  /* Now x is eqal to abs(x) and sign contains the sign of x */
  if ( x >= N )
    {
      /* 
	 The absolute difference between the returned value
	 and the exact one is less or equal than 
	 E(N) := (2/sqrt(PI)) * (exp(-N*N)/N).
	 With the precedent choices of N we have that

	 E ~ 10^-25   in case r_eal == long double
	 E ~ 10^-17   if      r_eal == double
      */
      return sign * one;
    }
  else
    {
      /* x in [0, N) */
      u = (x > C + 0.3 ? C : x);

      sum = 0.0;
      for (k = 0; k <= n; k++)
	{
	  prod = one;
	  for (j = 1; j <= k; j++)
	    prod *= (-u*u)/ j;
	  sum += u * prod / (2*k +1);
	}
      /*
	Mind the formula:

	erf(u) = sign * coeff * sum_{k = 0}^{inf} (-1)^k |u|^(2k+1)/((2k+1)*k!)

	where the computation of the series is truncated at the 'n'th term.
	The absolute difference between the returned value
	and the exact one is less or equal than 
	e(N, n) := N^(2n+3) / ((2n+3) * (n+1)!)
	With the precedent choices of N and n we have that

	e ~ 10^-25   in case r_eal == long double
	e ~ 10^-17   if      r_eal == double	
      */
      if ( x > u )
	{
#ifdef USE_LONG_DOUBLE
	  n = 400;
#else
	  n = 100;
#endif
	  /* 
	     Here we use Boole's rule to compute

	     int_{u}^{x} exp(-t*t) dt
	  */
	  h = (x - u) / n;
	  for (i = 0; i < n; i++)
	    {
	      t = u + i * h;
	      f0 = exp (-t*t);
	      f1 = exp (-(t+h/4)*(t+h/4));
	      f2 = exp (-(t+h/2)*(t+h/2));
	      f3 = exp (-(t+h * 0.75)*(t+h * 0.75));
	      f4 = exp (-(t+h)*(t+h));
	      sum += (7*f0 + 32*f1 + 12*f2 + 32*f3 + 7*f4) * h / 90.0;
	    }
	}
      return sign * coeff * sum;
    }
}

#endif

/*
  The complex error function and its complementary function are
  defined differently as done below. However, I do not have time
  now to write the proper implementation :(
*/

c_omplex c_erf (c_omplex z)
{
  c_omplex w;

  w.re = r_erf (z.re);
  w.im = 0.0;
  return w;
}

c_omplex c_erfc (c_omplex z)
{
  c_omplex w;

  w.re = 1.0 - r_erf (z.re);
  w.im = 0.0;
  return w;
}

#ifdef USE_LONG_DOUBLE
#define R_FMT_FX     "%.*Lf"
#define R_FMT_SC     "%.*LE"
#define R_FMT_FX_S   "%+.*Lf"
#define R_FMT_SC_S   "%+.*LE"
#define R_FMT_FX_PW  "%*.*Lf"
#define R_FMT_SC_PW  "%*.*LE"
#define R_FMT_FX_SPW "%+*.*Lf"
#define R_FMT_SC_SPW "%+*.*LE"
#define R_FMT_FX_NW  "%-*.*Lf"
#define R_FMT_SC_NW  "%-*.*LE"
#define R_FMT_FX_SNW "%+-*.*Lf"
#define R_FMT_SC_SNW "%+-*.*LE"
#else
#define R_FMT_FX     "%.*f"
#define R_FMT_SC     "%.*E"
#define R_FMT_FX_S   "%+.*f"
#define R_FMT_SC_S   "%+.*E"
#define R_FMT_FX_PW  "%*.*f"
#define R_FMT_SC_PW  "%*.*E"
#define R_FMT_FX_SPW "%+*.*f"
#define R_FMT_SC_SPW "%+*.*E"
#define R_FMT_FX_NW  "%-*.*f"
#define R_FMT_SC_NW  "%-*.*E"
#define R_FMT_FX_SNW "%+-*.*f"
#define R_FMT_SC_SNW "%+-*.*E"
#endif

#define C_L_DELIM  "{"
#define C_R_DELIM  "}"
#define C_SEP      " , "
#define DEF_PREC   6

static int __fmt = C_FMT_NICE;
static int __not = C_OUT_FIXED;
static int __sf  = 0;
static int __wd  = 0;
static int __pr  = DEF_PREC;

int c_setoutflags (int  fmt, int  notat, int  sf, int  wd, int  pr)
{
  if ( fmt != C_FMT_NOCHANGE )
    {
      if ( fmt < 0 || fmt > 2 )
	return -1;
      else
	__fmt = fmt;
    }
  __not = notat; __sf = sf; __wd = wd; __pr = pr;
  return 0;
} 

void c_getoutflags (int* fmt, int* notat, int *sf, int *wd, int *pr)
{
  *fmt = __fmt; *notat = __not; *sf = __sf; *wd = __wd; *pr = __pr;
}

int
r_fprint (FILE * pf, r_eal x)
{
  const char* format_string;

  if ( __pr < 0 )
    __pr = DEF_PREC;
  else if ( __pr > R_MAX_PREC )
    __pr = R_MAX_PREC;
  if ( !__wd )
    {
      if ( !__sf )
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX : R_FMT_SC;
      else
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX_S : R_FMT_SC_S;
      return fprintf (pf, format_string, __pr, x);
    }
  else if ( __wd > 0 )
    {
      if ( !__sf )
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX_PW : R_FMT_SC_PW;
      else
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX_SPW : R_FMT_SC_SPW;
      return fprintf (pf, format_string, __wd, __pr, x);
    }
  else /* __wd < 0 */
    {
      if ( !__sf )
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX_NW : R_FMT_SC_NW;
      else
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX_SNW : R_FMT_SC_SNW;
      return fprintf (pf, format_string,-__wd, __pr, x);
    }
}

int
r_print (r_eal x)
{
  return r_fprint (stdout, x);
}

int
r_sprint (char *s, str_s_ize n, r_eal x)
{
  const char* format_string;

  if ( __pr < 0 )
    __pr = DEF_PREC;
  else if ( __pr > R_MAX_PREC )
    __pr = R_MAX_PREC;
  if ( !__wd )
    {
      if ( !__sf )
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX : R_FMT_SC;
      else
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX_S : R_FMT_SC_S;
      return sprint (s, n, format_string, __pr, x);
    }
  else if ( __wd > 0 )
    {
      if ( !__sf )
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX_PW : R_FMT_SC_PW;
      else
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX_SPW : R_FMT_SC_SPW;
      return sprint (s, n, format_string, __wd, __pr, x);
    }
  else /* __wd < 0 */
    {
      if ( !__sf )
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX_NW : R_FMT_SC_NW;
      else
	format_string = ( __not == C_OUT_FIXED ) ? R_FMT_FX_SNW : R_FMT_SC_SNW;
      return sprint (s, n, format_string,-__wd, __pr, x);
    }
}

int
c_fprint (FILE * pf, c_omplex z)
{
  const char* real_fmt;
  char format_string[25];

  if ( __pr < 0 )
    __pr = DEF_PREC;
  else if ( __pr > R_MAX_PREC )
    __pr = R_MAX_PREC;
  if ( !__wd )
    {
      if ( !__sf )
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX : R_FMT_SC;
      else
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX_S : R_FMT_SC_S;
    }
  else if ( __wd > 0 )
    {
      if ( !__sf )
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX_PW : R_FMT_SC_PW;
      else
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX_SPW : R_FMT_SC_SPW;
    }
  else /* __wd < 0 */
    {
      if ( !__sf )
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX_NW : R_FMT_SC_NW;
      else
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX_SNW : R_FMT_SC_SNW;
    }
  memset (format_string, '\0', 25);
  if ( __fmt == C_FMT_SIMPLE )
    {
      strcpy (format_string, real_fmt);
      strcat (format_string, "\n");
      strcat (format_string, real_fmt);
    }
  else if ( __fmt == C_FMT_STD )
    {
      strcpy (format_string, real_fmt);
      strcat (format_string, " ");
      if ( (__sf) )
	strcat (format_string, real_fmt);
      else
	{
	  strcat (format_string, "%+");
	  strcat (format_string, real_fmt+1);
	}
      strcat (format_string, "i");
    }
  else
    {
      strcpy (format_string, C_L_DELIM);
      strcat (format_string, real_fmt);
      strcat (format_string, C_SEP);
      strcat (format_string, real_fmt);
      strcat (format_string, C_R_DELIM);
    }
  if ( !__wd )
    return fprintf (pf, format_string, __pr, z.re, __pr, z.im);
  else if ( __wd > 0 )
    return fprintf (pf, format_string, __wd, __pr, z.re, __wd, __pr, z.im);
  else
    return fprintf (pf, format_string, -__wd, __pr, z.re, -__wd, __pr, z.im);
}

/* c_print() does the same as c_fprint(), but it writes on stdout */
int
c_print (c_omplex z)
{
  return c_fprint (stdout, z);
}

int
c_sprint (char *s, str_s_ize n, c_omplex z)
{
  const char* real_fmt;
  char format_string[25];

  if ( __pr < 0 )
    __pr = DEF_PREC;
  else if ( __pr > R_MAX_PREC )
    __pr = R_MAX_PREC;
  if ( !__wd )
    {
      if ( !__sf )
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX : R_FMT_SC;
      else
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX_S : R_FMT_SC_S;
    }
  else if ( __wd > 0 )
    {
      if ( !__sf )
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX_PW : R_FMT_SC_PW;
      else
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX_SPW : R_FMT_SC_SPW;
    }
  else /* __wd < 0 */
    {
      if ( !__sf )
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX_NW : R_FMT_SC_NW;
      else
	real_fmt = ( __not == C_OUT_FIXED ) ? R_FMT_FX_SNW : R_FMT_SC_SNW;
    }
  memset (format_string, '\0', 25);
  if ( __fmt == C_FMT_SIMPLE )
    {
      strcpy (format_string, real_fmt);
      strcat (format_string, "\n");
      strcat (format_string, real_fmt);
    }
  else if ( __fmt == C_FMT_STD )
    {
      strcpy (format_string, real_fmt);
      strcat (format_string, " ");
      if ( (__sf) )
	strcat (format_string, real_fmt);
      else
	{
	  strcat (format_string, "%+");
	  strcat (format_string, real_fmt+1);	  
	}
      strcat (format_string, "i");
    }
  else
    {
      strcpy (format_string, C_L_DELIM);
      strcat (format_string, real_fmt);
      strcat (format_string, C_SEP);
      strcat (format_string, real_fmt);
      strcat (format_string, C_R_DELIM);
    }
  if ( !__wd )
    return sprint (s, n, format_string, __pr, z.re, __pr, z.im);
  else if ( __wd > 0 )
    return sprint (s, n, format_string, __wd, __pr, z.re, __wd, __pr, z.im);
  else
    return sprint (s, n, format_string, -__wd, __pr, z.re, -__wd, __pr, z.im);
}

int
c_fwrt (FILE * pf, int stdfmt, int prec, c_omplex z)
{
  char fmt1[10], fmt2[10], fmt[30];

  if (prec < 0)
    prec = DEF_PREC;		/* Use the default precision */
  else if (prec > R_MAX_PREC)
    prec = R_MAX_PREC;		/* Use the best precision */
#ifdef USE_LONG_DOUBLE

  if (__FABS (z.re) >= 10.0E10)
    sprintf (fmt1, "%s%d%s", "%.", prec, "LE");
  else
    sprintf (fmt1, "%s%d%s", "%15.", prec, "Lf");	/* Don't worry, fmt1 is
							   long enough!        */
  if (__FABS (z.im) >= 10.0E10)
    sprintf (fmt2, "%s%d%s", "%+.", prec, "LE");
  else
    sprintf (fmt2, "%s%d%s", "%+15.", prec, "Lf");	/* Don't worry, fmt2 is
							   long enough!        */
#else /* USE_LONG_DOUBLE */

  if (__FABS (z.re) >= 10.0E10)
    sprintf (fmt1, "%s%d%s", "%.", prec, "E");
  else
    sprintf (fmt1, "%s%d%s", "%15.", prec, "f");	/* Don't worry, fmt1 is
							   long enough!        */
  if (__FABS (z.im) >= 10.0E10)
    sprintf (fmt2, "%s%d%s", "%+.", prec, "E");
  else
    sprintf (fmt2, "%s%d%s", "%+15.", prec, "f");	/* Don't worry, fmt2 is
							   long enough!        */
#endif /* USE_LONG_DOUBLE */
  if (!stdfmt)
    sprintf (fmt, "{%s , %s}", fmt1, fmt2);
  else
    sprintf (fmt, "%s%si", fmt1, fmt2);
  return fprintf (pf, fmt, z.re, z.im);
}

int
c_wrt (int stdfmt, int prec, c_omplex z)
{
  char fmt1[10], fmt2[10], fmt[30];

  if (prec < 0)
    prec = DEF_PREC;		/* Use the default precision */
  else if (prec > R_MAX_PREC)
    prec = R_MAX_PREC;		/* Use the best precision */
#ifdef USE_LONG_DOUBLE

  if (__FABS (z.re) >= 10.0E10)
    sprintf (fmt1, "%s%d%s", "%.", prec, "LE");
  else
    sprintf (fmt1, "%s%d%s", "%15.", prec, "Lf");	/* Don't worry, fmt1 is
							   long enough!        */
  if (__FABS (z.im) >= 10.0E10)
    sprintf (fmt2, "%s%d%s", "%+.", prec, "LE");
  else
    sprintf (fmt2, "%s%d%s", "%+15.", prec, "Lf");	/* Don't worry, fmt2 is
							   long enough!        */
#else /* USE_LONG_DOUBLE */

  if (__FABS (z.re) >= 10.0E10)
    sprintf (fmt1, "%s%d%s", "%.", prec, "E");
  else
    sprintf (fmt1, "%s%d%s", "%15.", prec, "f");	/* Don't worry, fmt1 is
							   long enough!        */
  if (__FABS (z.im) >= 10.0E10)
    sprintf (fmt2, "%s%d%s", "%+.", prec, "E");
  else
    sprintf (fmt2, "%s%d%s", "%+15.", prec, "f");	/* Don't worry, fmt2 is
							   long enough!        */
#endif /* USE_LONG_DOUBLE */
  if (!stdfmt)
    sprintf (fmt, "{%s , %s}", fmt1, fmt2);
  else
    sprintf (fmt, "%s%si", fmt1, fmt2);
  return printf (fmt, z.re, z.im);
}

int
c_swrt (char *s, str_s_ize n, int stdfmt, int prec, c_omplex z)
{
  char fmt1[10], fmt2[10], fmt[30];

  if (prec < 0)
    prec = DEF_PREC;		/* Use the default precision */
  else if (prec > R_MAX_PREC)
    prec = R_MAX_PREC;		/* Use the best precision */
#ifdef USE_LONG_DOUBLE
  if (__FABS (z.re) >= 10.0E10)
    sprintf (fmt1, "%s%d%s", "%.", prec, "LE");
  else
    sprintf (fmt1, "%s%d%s", "%15.", prec, "Lf");	/* Don't worry, fmt1 is
							   long enough!        */
  if (__FABS (z.im) >= 10.0E10)
    sprintf (fmt2, "%s%d%s", "%+.", prec, "LE");
  else
    sprintf (fmt2, "%s%d%s", "%+15.", prec, "Lf");	/* Don't worry, fmt2 is
							   long enough!        */
#else /* USE_LONG_DOUBLE */
  if (__FABS (z.re) >= 10.0E10)
    sprintf (fmt1, "%s%d%s", "%.", prec, "E");
  else
    sprintf (fmt1, "%s%d%s", "%15.", prec, "f");	/* Don't worry, fmt1 is
							   long enough!        */
  if (__FABS (z.im) >= 10.0E10)
    sprintf (fmt2, "%s%d%s", "%+.", prec, "E");
  else
    sprintf (fmt2, "%s%d%s", "%+15.", prec, "f");	/* Don't worry, fmt2 is
							   long enough!        */
#endif /* USE_LONG_DOUBLE */
  if (!stdfmt)
    sprintf (fmt, "{%s , %s}", fmt1, fmt2);
  else
    sprintf (fmt, "%s%si", fmt1, fmt2);
  return sprint (s, n, fmt, z.re, z.im);
}

int
c_fscan (FILE * pf, c_omplex * z)
{
#ifdef USE_LONG_DOUBLE
  return fscanf (pf, "%Lf %Lf", &z->re, &z->im);
#else
  return fscanf (pf, "%lf %lf", &z->re, &z->im);
#endif
}

/* Like the precedent function but it reads from stdin */
int
c_scan (c_omplex * z)
{
#ifdef USE_LONG_DOUBLE
  return scanf ("%Lf %Lf", &z->re, &z->im);
#else
  return scanf ("%lf %lf", &z->re, &z->im);
#endif
}

/* Like the precedent function but it reads from a given string */
int
c_sscan (const char* s, c_omplex * z)
{
#ifdef USE_LONG_DOUBLE
  return sscanf (s, "%Lf %Lf", &z->re, &z->im);
#else
  return sscanf (s, "%lf %lf", &z->re, &z->im);
#endif
}

#if !defined(STOP_AT_ERROR)

const char *
strc_error (int errnum)
{
  static const char *last = "Undocumented error";

  if ((errnum >= 0) && (errnum < c_nerr))
    return c_errlist[errnum];
  else
    return last;
}

void
pc_error (const char *s)
{
  if ((s) && (*s != '\0'))
    fprintf (stderr, "%s: %s\n", s, c_errlist[c_errno]);
  else
    fprintf (stderr, "%s\n", c_errlist[c_errno]);
}

#endif
