/*
   This file is part of the XXCalc Library - version 3.2
   Copyright (C)  2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010,
   2011, 2012, 2013    Ivano Primi ( ivprimi@libero.it )    

   The XXCalc Library is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   The XXCalc library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<stdlib.h>
#include<string.h>
#include"compl.h"
#include"mathtok.h"
#include"parser.h"
#include"heapsort.h"
#include"expander.h"
#include"calc.h"
#ifdef DMALLOC
#include <dmalloc.h>
#endif

#ifdef USE_LONG_DOUBLE
#define NOT_ZERO(x) ((x) >= 1.0e-36L || (x) <= -1.0e-36L)
#else
#define NOT_ZERO(x) ((x) >= 1.0e-18 || (x) <= -1.0e-18)
#endif


#if !defined(STOP_AT_ERROR)
extern int c_errno;
#endif

extern const c_omplex c_0;
extern const c_omplex c_1;

/* See specfunc.c */
extern c_omplex c_dms (c_omplex z);
extern c_omplex c_deg (c_omplex z);
extern c_omplex c_rtod (c_omplex z);
extern c_omplex c_dtor (c_omplex z);
extern c_omplex c_fact (c_omplex z);

/* log_and(), log_or() and log_xor() are the usual logical operators. */
/* Thus, their return-values must be thought as boolean values.       */

static c_omplex
log_and (c_omplex z1, c_omplex z2)
{
  return (c_not0 (z1) && c_not0 (z2)) ? c_1 : c_0;
}

static c_omplex
log_or (c_omplex z1, c_omplex z2)
{
  return (c_not0 (z1) || c_not0 (z2)) ? c_1 : c_0;
}

static c_omplex
log_xor (c_omplex z1, c_omplex z2)
{
  if ((c_not0 (z1) && c_is0 (z2)) || (c_is0 (z1) && c_not0 (z2)))
    return c_1;
  else
    return c_0;
}

static int
call_function (b_yte func, c_omplex z, c_omplex * w, int rm)
{
#if !defined (STOP_AT_ERROR)
  c_errno = 0;
#endif
  switch (func)
    {
    case XX_RE:
      *w = c_convert (c_re (z));
      break;
    case XX_IM:
      *w = c_convert (c_im (z));
      break;
    case XX_ABS:
      *w = c_convert (c_abs (z));
      break;
    case XX_ARG:
      *w = c_convert (c_arg (z));
      break;
    case XX_NEG:
      *w = c_neg (z);
      break;
    case XX_CONJ:
      *w = c_conj (z);
      break;
    case XX_INV:
      *w = c_inv (z);
      break;
    case XX_SWAP:
      *w = c_swap (z);
      break;
    case XX_SQR:
      *w = c_sqr (z);
      break;
    case XX_SQRT:
      *w = c_sqrt (z);
      break;
    case XX_CBRT:
      if (z.re >= 0)
	*w = c_root (z, 0, 3);
      else
	*w = c_root (z, 1, 3);
      break;
    case XX_EXP:
      *w = c_exp (z);
      break;
    case XX_EXP10:
      *w = c_exp10 (z);
      break;
    case XX_LOG:
      *w = c_log (z);
      break;
    case XX_LOG10:
      *w = c_log10 (z);
      break;
    case XX_SIN:
      *w = c_sin (z);
      break;
    case XX_COS:
      *w = c_cos (z);
      break;
    case XX_TAN:
      *w = c_tan (z);
      break;
    case XX_ASIN:
      *w = c_asin (z);
      break;
    case XX_ACOS:
      *w = c_acos (z);
      break;
    case XX_ATAN:
      *w = c_atan (z);
      break;
    case XX_SINH:
      *w = c_sinh (z);
      break;
    case XX_COSH:
      *w = c_cosh (z);
      break;
    case XX_TANH:
      *w = c_tanh (z);
      break;
    case XX_ASINH:
      *w = c_asinh (z);
      break;
    case XX_ACOSH:
      *w = c_acosh (z);
      break;
    case XX_ATANH:
      *w = c_atanh (z);
      break;
    case XX_FLOOR:
      *w = c_floor (z);
      break;
    case XX_CEIL:
      *w = c_ceil (z);
      break;
    case XX_ROUND:
      *w = c_round (z);
      break;
    case XX_FIX:
      *w = c_fix (z);
      break;
    case XX_FRAC:
      *w = c_frac (z);
      break;
    case XX_CHCC:
      *w = c_chcc (z);
      break;
    case XX_CHCO:
      *w = c_chco (z);
      break;
    case XX_CHOC:
      *w = c_choc (z);
      break;
    case XX_CHOO:
      *w = c_choo (z);
      break;
    case XX_CHLC:
      *w = c_chlc (z);
      break;
    case XX_CHLO:
      *w = c_chlo (z);
      break;
    case XX_CHRC:
    case XX_STEP:
      *w = c_chrc (z);
      break;
    case XX_CHRO:
      *w = c_chro (z);
      break;
    case XX_ERF:
      *w = c_erf (z);
      break;
    case XX_ERFC:
      *w = c_erfc (z);
      break;
    case XX_DMS:
      *w = c_dms (z);
      break;
    case XX_DEG:
      *w = c_deg (z);
      break;
    case XX_RTOD:
      *w = c_rtod (z);
      break;
    case XX_DTOR:
      *w = c_dtor (z);
      break;
    case XX_FACT:
      *w = c_fact (z);
      break;
    }
#if !defined (STOP_AT_ERROR)
  if ((rm) && NOT_ZERO(w->im))
    {
	c_errno = C_ECVRES; 
    }
  switch (c_errno)
    {
    case C_EDIV:
      return XX_DIVBYZERO;
    case C_EDOM:
    case C_ECVRES:
      return XX_OUTOFDOMAIN;
    case C_EBADEXP:
      return XX_BADEXPONENT;
    default:
      return XX_OK;
    }
#else
  if ((rm) && NOT_ZERO(w->im))
    {
      fprintf (stderr, "\"%s\", %u: Out of domain (complex-valued result)\n\n",
               __FILE__, __LINE__);
      exit (EXIT_FAILURE);
    }
  return XX_OK;
#endif
}

static int
call_operator (b_yte oper, c_omplex z1, c_omplex z2, c_omplex * w, int rm)
{
#if !defined (STOP_AT_ERROR)
  c_errno = 0;
#endif
  switch (oper)
    {
    case XX_PL:
      *w = c_sum (z1, z2);
      break;
    case XX_MN:
      *w = c_diff (z1, z2);
      break;
    case XX_TM:
      *w = c_prod (z1, z2);
      break;
    case XX_DV:
      *w = c_div (z1, z2);
      break;
    case XX_PC:
      *w = c_prod (c_div (z1, c_convert (100)), z2);
      break;
    case XX_MD:
      *w = c_mod_ (z1, z2);
      break;
    case XX_ID:
      *w = c_idiv_ (z1, z2);
      break;
    case XX_PW:
      *w = c_pow (z1, z2);
      break;
    case XX_LT:
      if ((c_lt (z1, z2)))
	*w = c_1;
      else
	*w = c_0;
      break;
    case XX_GT:
      if ((c_gt (z1, z2)))
	*w = c_1;
      else
	*w = c_0;
      break;
    case XX_LE:
      if ((c_le (z1, z2)))
	*w = c_1;
      else
	*w = c_0;
      break;
    case XX_GE:
      if ((c_ge (z1, z2)))
	*w = c_1;
      else
	*w = c_0;
      break;
    case XX_EQ:
      if ((c_eq (z1, z2)))
	*w = c_1;
      else
	*w = c_0;
      break;
    case XX_NE:
      if ((c_neq (z1, z2)))
	*w = c_1;
      else
	*w = c_0;
      break;
    case XX_AND:
      *w = log_and (z1, z2);
      break;
    case XX_OR:
      *w = log_or (z1, z2);
      break;
    case XX_XOR:
      *w = log_xor (z1, z2);
      break;
    case XX_MIN_OP:
      *w = ( c_le (z1, z2) == 1 ) ? z1 : z2;
      break;
    case XX_MAX_OP:
      *w = ( c_ge (z1, z2) == 1 ) ? z1 : z2;
    }
#if !defined (STOP_AT_ERROR)
  if ((rm) && NOT_ZERO(w->im))
    {
	c_errno = C_ECVRES; 
    }
  switch (c_errno)
    {
    case C_EDIV:
      return XX_DIVBYZERO;
    case C_EDOM:
    case C_ECVRES:
      return XX_OUTOFDOMAIN;
    case C_EBADEXP:
      return XX_BADEXPONENT;
    default:
      return XX_OK;
    }
#else
  if ((rm) && NOT_ZERO(w->im))
    {
      fprintf (stderr, "\"%s\", %u: Out of domain (complex-valued result)\n\n",
               __FILE__, __LINE__);
      exit (EXIT_FAILURE);
    }
  return XX_OK;
#endif
}

int
xx_calc (xx_mathtoken * toklist, ui_nteger tllen, xx_varlist vl,
	 c_omplex * w, int rm)
{
  /* It supposes tllen <= LONG_MAX. */
  c_omplex z, z1, z2;
  xx_couple *v;
  long h, i, j, k;		/* Counters */
  int errcode, yesno;
  char* expvname;

  /* 
     First we must perform the evaluation of variables.
     Mind that, if a variable is not defined,
     0 is taken as default value for it (unless
     the macro BE_STRICT has been defined).      
  */
  for (i = 0; i < tllen; i++)
    if (toklist[i].type == XX_NUMBER && toklist[i].name[0] != '\0')
      {
	if( (strchr(toklist[i].name, '[')) ) /* Added in date 16/10/2002 */
	  {
	    if( !(expvname = xx_expander (toklist[i].name, vl, &errcode)) )
	      {
		*w = c_0;
		return errcode;
	      }
	    else
	      /* toklist[i].name is long enough! */
	      strcpy (toklist[i].name, expvname);
	  } /* End Added in date 16/10/2002 */
	toklist[i].value = xx_vl_get (vl, toklist[i].name, &yesno);
#ifdef BE_STRICT
	if(!yesno)
	  {
	    *w = c_0;
	    return XX_UNDEFVAR;
	  }
#endif
      }
  /* k = number of the elements of toklist for which the field .pr >= 0 */
  for (i = 0, k = 0; i < tllen; i++)
    {
      if (toklist[i].pr >= 0)
	k++;
    }
  if (k == 0)
    {
      /* There is only one element and it is a number */
      if (tllen == 1)
	{
	  *w = toklist[0].value;
	  return XX_OK;
	}
      else			/* tllen > 1. */
	{
	  *w = c_0;
	  return XX_MISSOPER;
	}
    }
  v = (xx_couple *) malloc (k * sizeof (xx_couple));
  if (!v)
    {
      *w = c_0;
      return XX_FATAL_ERROR;
    }
  else
    {
      for (i = 0, j = 0; i < tllen; i++)
	{
	  if (toklist[i].pr >= 0)	/* i.e. toklist[i] is */
	    {			/* a function or an operator. */
	      v[j].pr = toklist[i].pr;
	      v[j++].index = i;
	    }
	}
    }				
  /* Now v contains indexes and priority-levels of all      */
  /* functions and operators listed in toklist.             */
  xx_cheapsort (v, k);
  /* Now v is ordered according to increasing priority      */
  for (i = k - 1; i >= 0; i--)
    {
      h = v[i].index;
      if (toklist[h].type == XX_UNARYMINUS)
	{
	  for (j = h + 1; j < tllen && toklist[j].type != XX_NUMBER; j++)
	    ;
	  if (j == tllen)
	    {
	      free ((void *) v);
	      return XX_MISSVALUE;
	    }
	  else
	    {
	      z = toklist[j].value;
	      toklist[j].value = c_0;
	      toklist[j].type = XX_NULLTOK;	/* .pr holds the value -1 */
	    }
	  errcode = call_function (XX_NEG, z, w, rm);
	}
      else if (xx_isop (toklist[h].type))
	{
	  for (j = h - 1; j >= 0 && toklist[j].type != XX_NUMBER; j--)
	    ;
	  if (j == -1)
	    {
	      free ((void *) v);
	      return XX_MISSVALUE;
	    }
	  else
	    {
	      z1 = toklist[j].value;
	      toklist[j].value = c_0;
	      toklist[j].type = XX_NULLTOK;	/* .pr holds the value -1 */
	    }
	  for (j = h + 1; j < tllen && toklist[j].type != XX_NUMBER; j++)
	    ;
	  if (j == tllen)
	    {
	      free ((void *) v);
	      return XX_MISSVALUE;
	    }
	  else
	    {
	      z2 = toklist[j].value;
	      toklist[j].value = c_0;
	      toklist[j].type = XX_NULLTOK;	/* .pr holds the value -1 */
	    }
	  errcode = call_operator (toklist[h].type, z1, z2, w, rm);
	}
      else			/* toklist[h] may be only a function */
	{
	  for (j = h + 1; j < tllen && toklist[j].type != XX_NUMBER; j++)
	    ;
	  if (j == tllen)
	    {
	      free ((void *) v);
	      return XX_MISSVALUE;
	    }
	  else
	    {
	      z = toklist[j].value;
	      toklist[j].value = c_0;
	      toklist[j].type = XX_NULLTOK;	/* .pr holds the value -1 */
	    }
	  errcode = call_function (toklist[h].type, z, w, rm);
	}
      if (errcode != XX_OK)
	{
	  free ((void *) v);
	  return errcode;
	}
      else
	{
	  toklist[h].type = XX_NUMBER;
	  toklist[h].value = *w;
	  toklist[h].pr = -1;
	}
    }	/* end of for(i= k-1; i>=0; i--) {...} */
  /* *w is the result of the evaluation of the toklist. */
  /* v is not any longer useful. */
  free ((void *) v);
  /* 
     If we arrive here, in toklist[] there are only null
     tokens and numbers. If the numbers are two or more, then 
     we can say that we miss one or more operators. 
  */
  for (i = 0, j = 0; i < tllen; i++)
    {
      if (toklist[i].type != XX_NULLTOK)
	j++;
    }
  if (j > 1)
    return XX_MISSOPER;
  else				/* j == 1 */
    return XX_OK;
}
