/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<string.h>
#include<xxcalc-3.2.0/evaluate.h>
#include"par2d.h"
#include"gnuplot.h"
#include"utils.h"

#define ABS(x) ((x) >= 0 ? (x) : -(x))

par2d_object par2d_new (unsigned x_eval, unsigned y_eval)
{
  par2d_object obj;

  obj.x_eq = obj.y_eq = NULL;
  obj.x_func = x_eval;
  obj.y_func = y_eval;
  obj.inf_s = obj.sup_s = obj.inf_t = obj.sup_t = 0.0;
  obj.nsteps_s = obj.nsteps_t = 0;
  obj.style = GRSTYLE_NONE;
  obj.color = 0;
  obj.caption = NULL;
  obj.type = GRMATH_EMPTY;
  return obj;
}

grint par2d_is_empty (par2d_object obj)
{
  if ( gstrcmp(NULL, obj.x_eq) == 0 &&  gstrcmp(NULL, obj.y_eq) == 0 &&
       obj.inf_s == 0.0 &&  obj.sup_s == 0.0 && obj.nsteps_s == 0 &&
       obj.inf_t == 0.0 &&  obj.sup_t == 0.0 && obj.nsteps_t == 0 &&
       gstrcmp(NULL, obj.caption) == 0 )
    return 1;
  else
    return 0;
}

/*
  Return GRMATH_CURVE if the object pointed to by POBJ is a curve, 
  GRMATH_SURFACE if it is a surface,
  GRMATH_EMPTY if it is empty, and GRMATH_BAD if it is badly defined.
  In addition, update POBJ->TYPE.
  Return GRMATH_EMPTY if POBJ == NULL.
  
  Rem.: A parametric object is a curve if and only if:
 
        1. the strings pointed to by X_EQ and Y_EQ are valid 
	   mathematical equations containing at most the parameter t,

	2. SUP_T - INF_T > GR_EPSILON,

	3. NSTEPS_T > 0, and

	4. STYLE != GRSTYLE_NONE.

	A parametric object is a surface if and only if:

	1. the strings pointed to by X_EQ and Y_EQ are valid mathematical 
	   equations and collectively contain both the parameters s and t,

	2. SUP_S - INF_S > GR_EPSILON, SUP_T - INF_T > GR_EPSILON,

	3. both NSTEPS_S and NSTEPS_T are greater than zero, and

	4. STYLE != GRSTYLE_NONE.
*/
static grmathtype par2d_set_type (par2d_object* pobj)
{
  gruint var_x, var_y, vars;
  int errcode;

  if ( pobj == NULL || (par2d_is_empty(*pobj)) )
    return GRMATH_EMPTY;
  if (pobj->style == GRSTYLE_NONE || pobj->x_eq == NULL || pobj->y_eq == NULL)
    {
      return (pobj->type = GRMATH_BAD);
    }
  /* If we arrive here then POBJ->STYLE != GRSTYLE_NONE */
  if ((errcode = evaluator_set (pobj->x_func, pobj->x_eq)) != XX_OK && 
      errcode != XX_DIVBYZERO &&
      errcode != XX_OUTOFDOMAIN &&
      errcode != XX_BADEXPONENT)
    {
      notify_math_error ("An error has been found in the definition\n" \
			 "of the mathematical object.\nx: ", xx_strerror(errcode), 
			 "\n\nA correction is required to proceed.");
      return (pobj->type = GRMATH_BAD);
    }
  if ((errcode = evaluator_set (pobj->y_func, pobj->y_eq)) != XX_OK && 
      errcode != XX_DIVBYZERO &&
      errcode != XX_OUTOFDOMAIN &&
      errcode != XX_BADEXPONENT)
    {
      notify_math_error ("An error has been found in the definition\n" \
			 "of the mathematical object.\ny: ", xx_strerror(errcode), 
			 "\n\nA correction is required to proceed.");
      return (pobj->type = GRMATH_BAD);
    }

  var_x = which_variables (pobj->x_func); 
  var_y = which_variables (pobj->y_func);
  vars = var_x | var_y;
  switch(vars)
    {
    case 0x0:  /* Neither s nor t appear in the set of equations   */
    case 0x10: /* only s appears in the set of equations           */
      notify_math_error ("An error has been found in the definition\n" \
			 "of the mathematical object:\n", 
			 "the parameter t does not appear in any equation",
			 "\n\nA correction is required to proceed.");
      return (pobj->type = GRMATH_BAD); 
    case 0x01: /* only t appears in the set of equations           */
      return (pobj->type = GRMATH_CURVE);	  
    case 0x11: /* Both s and t have been found in the set of eqts. */
      return (pobj->type = GRMATH_SURFACE);	  
    default:
      notify_math_error ("An error has been found in the definition\n" \
			 "of the mathematical object:\n", 
			 "the equations cannot contain other parameters than s and t",
			 "\n\nA correction is required to proceed.");
      return (pobj->type = GRMATH_BAD); 	  
      /* At least one of the equations contains more than two variables */
    }
}

grexitstatus par2d_define (par2d_object* pobj,
			   grchar* x_eq, grchar* y_eq,
			   double inf_s, double sup_s, gruint nsteps_s,
			   double inf_t, double sup_t, gruint nsteps_t,
			   grstyle style, grint color, grchar* caption)
{
  double absinf_s, absinf_t, abssup_s, abssup_t;
  grmathtype type;

  if (pobj == NULL)
    return GR_NO_ACTION;
  absinf_s = ABS(pobj->inf_s - inf_s);
  absinf_t = ABS(pobj->inf_t - inf_t);
  abssup_s = ABS(pobj->sup_s - sup_s);
  abssup_t = ABS(pobj->sup_t - sup_t);
  if (gstrcmp(pobj->x_eq, x_eq) == 0 && gstrcmp(pobj->y_eq, y_eq) == 0 &&
      absinf_s < GR_EPSILON &&  absinf_t < GR_EPSILON && abssup_s < GR_EPSILON &&  abssup_t < GR_EPSILON &&
      pobj->nsteps_s == nsteps_s && pobj->nsteps_t == nsteps_t && pobj->style == style && pobj->color == color &&
      gstrcmp(pobj->caption, caption) == 0)
    return GR_NO_ACTION;
  xfree (&(pobj->x_eq));
  xfree (&(pobj->y_eq));
  if ((x_eq))
    pobj->x_eq = xstrdup (x_eq);
  if ((y_eq))
    pobj->y_eq = xstrdup (y_eq);
  pobj->inf_s = inf_s;
  pobj->inf_t = inf_t;
  pobj->sup_s = sup_s;
  pobj->sup_t = sup_t;
  pobj->nsteps_s = nsteps_s;
  pobj->nsteps_t = nsteps_t;
  pobj->style = style;
  pobj->color = color;
  xfree (&(pobj->caption));
  if ((caption))
    pobj->caption = xstrdup (caption);
  type = par2d_set_type (pobj);
  if ( type == GRMATH_BAD )
    return GR_FAILURE;
  else if (type == GRMATH_CURVE && (sup_t - inf_t < GR_EPSILON))
    {
      notify_math_error ("An error has been found in the definition\n" \
			 "of the mathematical object:\n", 
			 "Max should always be greater than Min",
			 "\n\nA correction is required to proceed.");
      return GR_FAILURE;
    }
  else if (type == GRMATH_CURVE && nsteps_t == 0)
    {
      notify_math_error ("An error has been found in the definition\n" \
			 "of the mathematical object:\n", 
			 "the number of subintervals should always be greater than zero",
			 "\n\nA correction is required to proceed.");
      return GR_FAILURE;
    }
  else if (type == GRMATH_SURFACE && 
	   (sup_s - inf_s < GR_EPSILON || sup_t - inf_t < GR_EPSILON))
    {
      notify_math_error ("An error has been found in the definition\n"	\
			 "of the mathematical object:\n", 
			 "Max should always be greater than Min",
			 "\n\nA correction is required to proceed.");
      return GR_FAILURE;
    }
  else if (type == GRMATH_SURFACE && (nsteps_s == 0 || nsteps_t == 0))
    {
      notify_math_error ("An error has been found in the definition\n" \
			 "of the mathematical object:\n", 
			 "the number of subintervals should always be greater than zero",
			 "\n\nA correction is required to proceed.");
      return GR_FAILURE;
    }
  else
    return GR_OK;
}

#define DOTTED_CURVE     0
#define STROKED_CURVE    1
#define DOTTED_SURFACE   2
#define STROKED_SURFACE  3
#define FILLED_SURFACE   4

grexitstatus par2d_add_to_gr2d_object (gr2d_object* pgr2d_obj,
				       par2d_object obj)
{
  const char* var_names[2] = {
    "t", "s"
  };
  double var_values[2] = {
    0.0, 0.0
  };
  gruint btype, i, j, m, n;
  double inf_s, sup_s, inf_t, sup_t;
  gr2d_point P, Q, R;
  double s1, s2, t1, t2; /* used only in case of filled surfaces */

  if (pgr2d_obj == NULL || obj.type == GRMATH_BAD || obj.type == GRMATH_EMPTY)
    return GR_NO_ACTION;
  if (obj.type == GRMATH_CURVE)
    btype = (obj.style == GRSTYLE_DOTTED ? DOTTED_CURVE : STROKED_CURVE);
  else
    {
      if (obj.style == GRSTYLE_DOTTED)
	btype = DOTTED_SURFACE;
      else if (obj.style == GRSTYLE_STROKED)
	btype = STROKED_SURFACE;
      else
	btype= FILLED_SURFACE;
    }
  switch (btype)
    {
    case DOTTED_CURVE:
      n = obj.nsteps_t;
      inf_t = obj.inf_t;
      sup_t = obj.sup_t;
      for (i = 0; i <= n; i++)
	{
	  var_values[0] = inf_t + i * ((sup_t - inf_t) / n);
	  P.x = evaluator_evaluate (obj.x_func, 1, var_names, var_values);
	  P.y = evaluator_evaluate (obj.y_func, 1, var_names, var_values);
	  if ( gr2d_point_is_finite(P) )
	    {
	      /* Do not write any caption for the moment */
	      if ( gr2d_add_point (pgr2d_obj, obj.color, NULL, P.x, P.y) < 0 )
		return GR_FAILURE; /*out of memory */
	    }
	}
      break;
    case STROKED_CURVE:
      n = obj.nsteps_t;
      inf_t = obj.inf_t;
      sup_t = obj.sup_t;
      for (i = 0; i < n; i++)
	{
	  var_values[0] = inf_t + i * ((sup_t - inf_t) / n);
	  P.x = evaluator_evaluate (obj.x_func, 1, var_names, var_values);
	  P.y = evaluator_evaluate (obj.y_func, 1, var_names, var_values);
	  var_values[0] = inf_t + (i+1) * ((sup_t - inf_t) / n);
	  Q.x = evaluator_evaluate (obj.x_func, 1, var_names, var_values);
	  Q.y = evaluator_evaluate (obj.y_func, 1, var_names, var_values);
	  if ( gr2d_point_is_finite(P) && gr2d_point_is_finite(Q) )
	    {
	      /* Do not write any caption for the moment */
	      if ( gr2d_add_segment (pgr2d_obj, obj.color, NULL, P.x, P.y, Q.x, Q.y) < 0 )
		return GR_FAILURE; /*out of memory */
	    }
	}
      break;
    case DOTTED_SURFACE:
      m = obj.nsteps_s;
      inf_s = obj.inf_s;
      sup_s = obj.sup_s;
      n = obj.nsteps_t;
      inf_t = obj.inf_t;
      sup_t = obj.sup_t;
      for (i = 0; i <= n; i++)
	{
	  var_values[0] = inf_t + i * ((sup_t - inf_t) / n);
	  for (j = 0; j <= m; j++)
	    {
	      var_values[1] = inf_s + j * ((sup_s - inf_s) / m);
	      P.x = evaluator_evaluate (obj.x_func, 2, var_names, var_values);
	      P.y = evaluator_evaluate (obj.y_func, 2, var_names, var_values);
	      if ( gr2d_point_is_finite(P) )
		{
		  /* Do not write any caption for the moment */
		  if ( gr2d_add_point (pgr2d_obj, obj.color, NULL, P.x, P.y) < 0 )
		    return GR_FAILURE; /*out of memory */
		}
	    }
	}
      break;
    case STROKED_SURFACE:
      m = obj.nsteps_s;
      inf_s = obj.inf_s;
      sup_s = obj.sup_s;
      n = obj.nsteps_t;
      inf_t = obj.inf_t;
      sup_t = obj.sup_t;
      for (i = 0; i <= n; i++)
	{
	  var_values[0] = inf_t + i * ((sup_t - inf_t) / n);
	  for (j = 0; j < m; j++)
	    {
	      var_values[1] = inf_s + j * ((sup_s - inf_s) / m);
	      P.x = evaluator_evaluate (obj.x_func, 2, var_names, var_values);
	      P.y = evaluator_evaluate (obj.y_func, 2, var_names, var_values);
	      var_values[1] = inf_s + (j+1) * ((sup_s - inf_s) / m);
	      Q.x = evaluator_evaluate (obj.x_func, 2, var_names, var_values);
	      Q.y = evaluator_evaluate (obj.y_func, 2, var_names, var_values);
	      if ( gr2d_point_is_finite(P) && gr2d_point_is_finite(Q) )
		{
		  /* Do not write any caption for the moment */
		  if ( gr2d_add_segment (pgr2d_obj, obj.color, NULL, P.x, P.y, Q.x, Q.y) < 0 )
		    return GR_FAILURE; /*out of memory */
		}
	    }
	}
      for (j = 0; j <= m; j++)
	{
	  var_values[1] = inf_s + j * ((sup_s - inf_s) / m);
	  for (i = 0; i < n; i++)
	    {
	      var_values[0] = inf_t + i * ((sup_t - inf_t) / n);
	      P.x = evaluator_evaluate (obj.x_func, 2, var_names, var_values);
	      P.y = evaluator_evaluate (obj.y_func, 2, var_names, var_values);
	      var_values[0] = inf_t + (i+1) * ((sup_t - inf_t) / n);
	      Q.x = evaluator_evaluate (obj.x_func, 2, var_names, var_values);
	      Q.y = evaluator_evaluate (obj.y_func, 2, var_names, var_values);
	      if ( gr2d_point_is_finite(P) && gr2d_point_is_finite(Q) )
		{
		  /* Do not write any caption for the moment */
		  if ( gr2d_add_segment (pgr2d_obj, obj.color, NULL, P.x, P.y, Q.x, Q.y) < 0 )
		    return GR_FAILURE; /*out of memory */
		}
	    }
	}
      break;
    case FILLED_SURFACE:
      m = obj.nsteps_s;
      inf_s = obj.inf_s;
      sup_s = obj.sup_s;
      n = obj.nsteps_t;
      inf_t = obj.inf_t;
      sup_t = obj.sup_t;
      for (i = 0; i < n; i++)
	{      
	  t1 = inf_t + i * ((sup_t - inf_t) / n);
	  t2 = inf_t + (i+1) * ((sup_t - inf_t) / n);
	  for (j = 0; j < m; j++)
	    {
	      s1 = inf_s + j * ((sup_s - inf_s) / m);
	      s2 = inf_s + (j+1) * ((sup_s - inf_s) / m);
	      var_values[0] = t1;
	      var_values[1] = s1;
	      P.x = evaluator_evaluate (obj.x_func, 2, var_names, var_values);
	      P.y = evaluator_evaluate (obj.y_func, 2, var_names, var_values);
	      var_values[0] = t2;
	      Q.x = evaluator_evaluate (obj.x_func, 2, var_names, var_values);
	      Q.y = evaluator_evaluate (obj.y_func, 2, var_names, var_values);
	      var_values[1] = s2;
	      R.x = evaluator_evaluate (obj.x_func, 2, var_names, var_values);
	      R.y = evaluator_evaluate (obj.y_func, 2, var_names, var_values);
	      if ( gr2d_point_is_finite(P) && gr2d_point_is_finite(Q) && gr2d_point_is_finite(R) )
		{
		  /* Do not write any caption for the moment */
		  if ( gr2d_add_triangle (pgr2d_obj, obj.color, NULL, P.x, P.y, Q.x, Q.y, R.x, R.y) < 0 )
		    return GR_FAILURE; /*out of memory */
		}
	      var_values[0] = t1;
	      Q.x = evaluator_evaluate (obj.x_func, 2, var_names, var_values);
	      Q.y = evaluator_evaluate (obj.y_func, 2, var_names, var_values);
	      if ( gr2d_point_is_finite(P) && gr2d_point_is_finite(Q) && gr2d_point_is_finite(R) )
		{
		  /* Do not write any caption for the moment */
		  if ( gr2d_add_triangle (pgr2d_obj, obj.color, NULL, R.x, R.y, P.x, P.y, Q.x, Q.y) < 0 )
		    return GR_FAILURE; /*out of memory */
		}
	    }
	}
      break;
    default:
      /*
	We should never reach this point actually
      */
      return GR_NO_ACTION;
    }
  return GR_OK;
}


void par2d_clear (par2d_object* pobj)
{
  if ( pobj != NULL && !par2d_is_empty(*pobj) )
    {
      xfree (&(pobj->x_eq));
      xfree (&(pobj->y_eq));
      /* Leave POBJ->X_FUNC and POBJ->Y_FUNC unchanged */
      pobj->inf_s = pobj->inf_t = pobj->sup_s = pobj->sup_t = 0.0;
      pobj->nsteps_s = pobj->nsteps_t = 0;
      pobj->style = GRSTYLE_NONE;
      pobj->color = 0;
      xfree (&(pobj->caption));
      pobj->type = GRMATH_EMPTY;
    }
}

#define FLOAT_FORMAT "%+30.12G\n"

grexitstatus par2d_save (par2d_object obj, FILE* fp)
{
  int rv;

  if ( par2d_is_empty(obj) )
    return GR_NO_ACTION;
  rv = fprintf (fp, "# GtkMathPlot object definition file, do not edit!\n");
  if (rv > 0)
    rv = fprintf (fp, "Object type: %d\n", obj.type);
  if (rv > 0)
    {
      if ((obj.x_eq))
	rv = fprintf (fp, "x: %s\n", obj.x_eq);
      else
	rv = fprintf (fp, "x: ---\n");
    }
  if (rv > 0)
    {
      if ((obj.y_eq))
	rv = fprintf (fp, "y: %s\n", obj.y_eq);
      else
	rv = fprintf (fp, "y: ---\n");
    }

  if (rv > 0)
    rv = fprintf (fp, "inf_t= " \
		  FLOAT_FORMAT, obj.inf_t);
  if (rv > 0)
    rv = fprintf (fp, "sup_t= " \
		  FLOAT_FORMAT, obj.sup_t);
  if (rv > 0)
    rv = fprintf (fp, "number of subintervals for t= %lu\n", obj.nsteps_t);

  if (obj.type == GRMATH_SURFACE)
    {
      if (rv > 0)
	rv = fprintf (fp, "inf_s= "		\
		      FLOAT_FORMAT, obj.inf_s);
      if (rv > 0)
	rv = fprintf (fp, "sup_s= "		\
		      FLOAT_FORMAT, obj.sup_s);
      if (rv > 0)
	rv = fprintf (fp, "number of subintervals for s= %lu\n", obj.nsteps_s);
    }
  
  if (rv > 0)
    rv = fprintf (fp, "style= %d\n", obj.style);

  if (rv > 0)
    rv = fprintf (fp, "color= %ld\n", obj.color);

  if (rv > 0)
    {
      if ((obj.caption) && (*obj.caption))
	rv = fprintf (fp, "caption: %s\n", obj.caption);
      else
	rv = fprintf (fp, "caption: ---\n");
    }

  if (rv > 0)
    rv = fprintf (fp, "# End of definition file\n");

  return (rv > 0 ? GR_OK : GR_FAILURE);
}

#define FLOAT2_FORMAT "  %+24.8G  %+24.8G\n"

grexitstatus par2d_eval_save (par2d_object obj, FILE* fp)
{
  static const char* style_cmd[] = {
    "# ",
    DOTTED_COMM,
    STROKED_COMM,
    FILLED_COMM,
    NULL
  };
  const char* var_names[2] = {
    "t", "s"
  };
  double var_values[2] = {
    0.0, 0.0
  };
  gruint i, j, m, n;
  double inf_s, sup_s, inf_t, sup_t;
  gr2d_point P;

  if (obj.type == GRMATH_BAD || obj.type == GRMATH_EMPTY)
    return GR_NO_ACTION;

  if ( fprintf (fp, "# Created by GtkMathPlot, do not edit!\n") <= 0 )
    return GR_FAILURE; /* I/O error */      

  if ( (obj.caption) && (*obj.caption) && 
       fprintf (fp, "%s %s\n", CAPTION_COMM, obj.caption) <= 0 )
    return GR_FAILURE; /* I/O error */      

  if ( fprintf (fp, "%s\n#\n", style_cmd[obj.style]) <= 0 )
    return GR_FAILURE; /* I/O error */

  if (obj.type == GRMATH_CURVE)
    {
      n = obj.nsteps_t;
      inf_t = obj.inf_t;
      sup_t = obj.sup_t;
      for (i = 0; i <= n; i++)
	{
	  var_values[0] = inf_t + i * ((sup_t - inf_t) / n);
	  P.x = evaluator_evaluate (obj.x_func, 1, var_names, var_values);
	  P.y = evaluator_evaluate (obj.y_func, 1, var_names, var_values);
	  if ( gr2d_point_is_finite(P) )
	    {
	      if ( fprintf (fp, FLOAT2_FORMAT, P.x, P.y) < 0 )
		return GR_FAILURE; /* I/O error */
	    }
	}
    }
  else
    {
      /* obj.type == GRMATH_SURFACE */
      m = obj.nsteps_s;
      inf_s = obj.inf_s;
      sup_s = obj.sup_s;
      n = obj.nsteps_t;
      inf_t = obj.inf_t;
      sup_t = obj.sup_t;
      for (i = 0; i <= n; i++)
	{
	  var_values[0] = inf_t + i * ((sup_t - inf_t) / n);
	  for (j = 0; j <= m; j++)
	    {
	      var_values[1] = inf_s + j * ((sup_s - inf_s) / m);
	      P.x = evaluator_evaluate (obj.x_func, 2, var_names, var_values);
	      P.y = evaluator_evaluate (obj.y_func, 2, var_names, var_values);
	      if ( gr2d_point_is_finite(P) )
		{
		  if ( fprintf (fp, FLOAT2_FORMAT, P.x, P.y) < 0 )
		    return GR_FAILURE; /* I/O error */
		}
	    }
	  if ( i < n && fputc ('\n', fp) == EOF )
	    return GR_FAILURE; /* I/O error */
	}      
    }
  return GR_OK;
}

#define BUFF_SIZE 10240

grmathtype par2d_load (par2d_object* pobj, FILE* fp)
{
  grchar linebuf[BUFF_SIZE];
  grchar x_eq[BUFF_SIZE];
  grchar y_eq[BUFF_SIZE];
  grchar caption[BUFF_SIZE];

  if (pobj == NULL)
    return GRMATH_BAD;
  else
    par2d_clear (pobj);
  
  if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
       gstrcmp(linebuf, "# GtkMathPlot object definition file, do not edit!\n") != 0)
    return GRMATH_BAD;

  if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
       sscanf (linebuf, "Object type: %d\n", &(pobj->type)) < 1 ||
       pobj->type < GRMATH_BAD ||  pobj->type > GRMATH_SURFACE )
    return GRMATH_BAD;

  if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
       read_from_string (linebuf, "x: ", x_eq, BUFF_SIZE-1) < 1 )
    return GRMATH_BAD;

  if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
       read_from_string (linebuf, "y: ", y_eq, BUFF_SIZE-1) < 1 )
    return GRMATH_BAD;

  if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
       sscanf (linebuf, "inf_t= %lf\n", &(pobj->inf_t)) < 1 )
    return GRMATH_BAD;

  if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
       sscanf (linebuf, "sup_t= %lf\n", &(pobj->sup_t)) < 1 )
    return GRMATH_BAD;

  if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
       sscanf (linebuf, "number of subintervals for t= %lu\n", &(pobj->nsteps_t)) < 1 )
    return GRMATH_BAD;

  if (pobj->type == GRMATH_SURFACE)
    {
      if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
	   sscanf (linebuf, "inf_s= %lf\n", &(pobj->inf_s)) < 1 )
	return GRMATH_BAD;
      
      if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
	   sscanf (linebuf, "sup_s= %lf\n", &(pobj->sup_s)) < 1 )
	return GRMATH_BAD;
      
      if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
	   sscanf (linebuf, "number of subintervals for s= %lu\n", &(pobj->nsteps_s)) < 1 )
	return GRMATH_BAD;
    }

  if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
       sscanf (linebuf, "style= %u\n", &(pobj->style)) < 1 ||
       pobj->style < GRSTYLE_NONE ||  pobj->style > GRSTYLE_FILLED )
    return GRMATH_BAD;

  if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
       sscanf (linebuf, "color= %ld\n", &(pobj->color)) < 1 )
    return GRMATH_BAD;

  if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
       read_from_string (linebuf, "caption: ", caption, BUFF_SIZE-1) < 1 )
    return GRMATH_BAD;

  if ( strncmp (caption, "---", 3) == 0 )
    *caption = '\0';

  if ( fgets (linebuf, BUFF_SIZE, fp) == NULL ||
       gstrcmp(linebuf, "# End of definition file\n") != 0)
    return GRMATH_BAD;
  
  if ( fgets (linebuf, BUFF_SIZE, fp) != NULL ) /* The file got corrupted */
    return GRMATH_BAD;

  xfree (&(pobj->x_eq));
  xfree (&(pobj->y_eq));
  pobj->x_eq = xstrdup (x_eq);
  pobj->y_eq = xstrdup (y_eq);
  xfree (&(pobj->caption));
  pobj->caption = xstrdup (caption);
  return par2d_set_type (pobj);
}
