/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<float.h>
#include<math.h>
#include"gr3d.h"
#include"utils.h" 

#define FMT_STRING  " x = %+30.12G\n y = %+30.12G\n z = %+30.12G\n"
#define FMT_STRING2 " X = %+30.12G\n Y = %+30.12G\n"
#define FMT_STRING3 "%+30.12G\n"
#define FMT_STRING4 " x = %+19.8G, y = %+19.8G, z = %+19.8G\n"
#define FMT_STRING5 " x = %lG, y = %lG, z = %lG\n"  /* coupled to FMT_STRING4 */
#define FMT_STRING6 "%+19.8G\n"
#define FMT_STRING7 "%lG\n"                         /* coupled to FMT_STRING6 */

#define CHUNK_ALLOC 1024

static int compare (const void* ppshape1, const void* ppshape2)
{
  gr3d_basic_shape* sh1 = *((gr3d_basic_shape**) ppshape1);
  gr3d_basic_shape* sh2 = *((gr3d_basic_shape**) ppshape2);
  double d1 = sh1->depth;
  double d2 = sh2->depth;

  if (d1 < d2)
    return -1;
  else if (d1 > d2)
    return 1;
  else
    return 0;
}

grint gr3d_point_is_finite (gr3d_point P)
{
  return (is_value_finite(P.x) && is_value_finite(P.y) && is_value_finite(P.z) ? 1 : 0);
}

gr3d_object gr3d_new (grint color1, grint color2)
{
  gr3d_object obj;

  obj.first_color = color1;
  obj.last_color = color2;
  obj.size = obj.len = obj.rlen = 0;
  obj.R = 0.0;
  obj.list = (gr3d_basic_shape**) malloc (sizeof(gr3d_basic_shape*));
  if ((obj.list))
    {
      obj.list[0] = (gr3d_basic_shape*) malloc (sizeof(gr3d_basic_shape));
      if ((obj.list[0]))
	{
	  (obj.list[0])->type = GR_NONE;
	  (obj.list[0])->color = 0;
	  (obj.list[0])->caption = NULL;
	  (obj.list[0])->A.x = (obj.list[0])->A.y = (obj.list[0])->A.z = 0.0;
	  (obj.list[0])->B.x = (obj.list[0])->B.y = (obj.list[0])->B.z = 0.0;
	  (obj.list[0])->C.x = (obj.list[0])->C.y = (obj.list[0])->C.z = 0.0;
	  (obj.list[0])->M.x = (obj.list[0])->M.y = (obj.list[0])->M.z = 0.0;
	  (obj.list[0])->depth = (obj.list[0])->length = 0.0;
	  (obj.list[0])->XA = (obj.list[0])->YA = 0.0;
	  (obj.list[0])->XB = (obj.list[0])->YB = 0.0;
	  (obj.list[0])->XC = (obj.list[0])->YC = 0.0;
	  obj.size = 1;
	}
      else
	{
	  free((void*)obj.list);
	  obj.list = NULL;
	}
    }
  return obj; /* In case of memory issue, this is an empty object */
}

int gr3d_is_empty (gr3d_object obj)
{
  return (obj.list == NULL ? 1 : 0);
}

static
grint gr3d_add_shape (gr3d_object* pobj, 
		      grtype type,
		      grint color,
		      const grchar* caption,
		      double length,
		      double xA, double yA, double zA,
		      double xB, double yB, double zB,
		      double xC, double yC, double zC)
{
  gr3d_basic_shape **list, *shape;
  gr3d_point barycentre;
  grint size, len, rlen, i;

  if (pobj == NULL || gr3d_is_empty(*pobj) != 0)
    return -1;
  else
    {
      list = pobj->list; /* This is different from NULL */
      size = pobj->size; /* This is a positive number */
      len = pobj->len;   /* 0 <= len <= size - 1      */
      rlen = pobj->rlen; /* 0 <= rlen <= len          */
    }
  if (len == size - 1)
    {
      /* We need to allocate more memory for the array */
      size += CHUNK_ALLOC;
      pobj->list = 
	(gr3d_basic_shape**)realloc ((void*)list, size * sizeof(gr3d_basic_shape*));
      if (!pobj->list)
	{
	  /* Leave everything unchanged and return -1 */
	  pobj->list = list;
	  return -1;
	}
      else
	{
	  for (i = pobj->size; i < size; pobj->list[i] = NULL, i++);
	  pobj->size = size;
	}
    }
  /* Now len < size - 1 */
  shape = (gr3d_basic_shape*) malloc (sizeof(gr3d_basic_shape));
  if (!shape)
    return -1; /* Do not update pobj->len, just return -1 */
  else
    {
      shape->type = type;
      shape->color = color;
      shape->caption = xstrdup (caption);
      shape->A.x = xA;
      shape->A.y = yA;
      shape->A.z = zA;
      shape->B.x = xB;
      shape->B.y = yB;
      shape->B.z = zB;
      shape->C.x = xC;
      shape->C.y = yC;
      shape->C.z = zC;
      shape->M.x = (xA+xB+xC)/type;
      shape->M.y = (yA+yB+yC)/type;
      shape->M.z = (zA+zB+zC)/type;
      shape->length = length;
      shape->depth = 0.0; /* no depth for the moment */
      shape->XA = shape->YA = 0.0;
      shape->XB = shape->YB = 0.0;
      shape->XC = shape->YC = 0.0; /* These are also 0.0 for the moment */
      if (shape->color >= pobj->first_color &&
	  shape->color <= pobj->last_color)
	{
	  barycentre = (pobj->list[0])->M;
	  barycentre.x = (barycentre.x * rlen + shape->M.x)/(rlen+1);
	  barycentre.y = (barycentre.y * rlen + shape->M.y)/(rlen+1);
	  barycentre.z = (barycentre.z * rlen + shape->M.z)/(rlen+1);
	  (pobj->list[0])->M = barycentre;      
	  (pobj->list[0])->length = ((pobj->list[0])->length * rlen + length)/(rlen+1);
	  pobj->rlen++;
	}
      pobj->len++;
      len = pobj->len;
      pobj->list[len] = shape;
      return len;
    }
}

grint gr3d_add_triangle (gr3d_object* pobj, 
			 grint color,
			 const grchar* caption,
			 double xA, double yA, double zA,
			 double xB, double yB, double zB,
			 double xC, double yC, double zC)
{
  double length = sqrt ((xB-xA)*(xB-xA)+(yB-yA)*(yB-yA)+(zB-zA)*(zB-zA))+
    sqrt ((xC-xA)*(xC-xA)+(yC-yA)*(yC-yA)+(zC-zA)*(zC-zA))+
    sqrt ((xC-xB)*(xC-xB)+(yC-yB)*(yC-yB)+(zC-zB)*(zC-zB));

  return gr3d_add_shape (pobj, GR_TRIANGLE, color, caption,
			 length, xA, yA, zA, xB, yB, zB, xC, yC, zC);
}

grint gr3d_add_segment (gr3d_object* pobj, 
			grint color,
			const grchar* caption,
			double xA, double yA, double zA,
			double xB, double yB, double zB)
{
  double length = sqrt ((xB-xA)*(xB-xA)+(yB-yA)*(yB-yA)+(zB-zA)*(zB-zA));

  return gr3d_add_shape (pobj, GR_SEGMENT, color, caption,
			 length, xA, yA, zA, xB, yB, zB, 0.0, 0.0, 0.0);
}

grint gr3d_add_point (gr3d_object* pobj, 
		      grint color,
		      const grchar* caption,
		      double x, double y, double z)
{
  return gr3d_add_shape (pobj, GR_POINT, color, caption,
			 0.0, x, y, z, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0);
}

static 
void gr3d_get_projection (const gr3d_point* P, const gr3d_point* O,
			  double lambda, double phi, double mu,
			  double* px, double* py)
{
  double X, Y;

  X=cos(lambda)*(P->y - O->y)-sin(lambda)*(P->x - O->x);
  Y=(P->z - O->z)*cos(phi)-sin(phi)*
    (cos(lambda)*(P->x - O->x)+sin(lambda)*(P->y - O->y));

  *px =  cos(mu)*X+sin(mu)*Y;
  *py = -sin(mu)*X+cos(mu)*Y;
}

static void gr3d_get_userdef_coordinates (gr3d_object* pobj,
					  grint recompute_all,
					  grint color, gr3d_point O,
					  double longitude, double latitude,
					  double rot_angle)
{
  if (pobj != NULL && !gr3d_is_empty(*pobj))
    {
      double xdir, ydir, zdir;
      gr3d_basic_shape* shape;
      gr3d_point barycentre, point;
      grtype shape_type;
      grint i, len;

      xdir = cos(longitude) * cos(latitude);
      ydir = sin(longitude) * cos(latitude);
      zdir = sin(latitude);

      barycentre = O;
      len = pobj->len;
      for (i = 1; i <= len; i++)
	{
	  shape = pobj->list[i];
	  if ( (recompute_all) || (shape->color == color) )
	    {
	      shape_type = shape->type;
	      point = shape->M;
	      shape->depth = (point.x - barycentre.x) * xdir +
		(point.y - barycentre.y) * ydir + (point.z - barycentre.z) * zdir;
	      
	      gr3d_get_projection (&(shape->A), &O,
				   longitude, latitude, rot_angle,
				   &(shape->XA),
				   &(shape->YA));

	      if (shape_type >= GR_SEGMENT)
		{
		  gr3d_get_projection (&(shape->B), &O,
				       longitude, latitude, rot_angle,
				       &(shape->XB),
				       &(shape->YB));
		}

	      if (shape_type == GR_TRIANGLE)
		{
		  gr3d_get_projection (&(shape->C), &O,
				       longitude, latitude, rot_angle,
				       &(shape->XC),
				       &(shape->YC));
		}
	    } /* end external if */
	} /* end for */
    }
}

void gr3d_sort_shapes (gr3d_object* pobj)
{
  if (pobj != NULL && !gr3d_is_empty(*pobj))
    qsort (&(pobj->list[1]), pobj->len, sizeof(gr3d_basic_shape*), compare);
}

void gr3d_view_from_direction (gr3d_object* pobj, gr3d_point O,
			       double longitude, double latitude,
			       double rot_angle)
{
  gr3d_get_userdef_coordinates (pobj, 1, 0, O, longitude, latitude, rot_angle);
  gr3d_sort_shapes (pobj);
}

void gr3d_partial_upgrade (gr3d_object* pobj, grint color,
			   gr3d_point O,
			   double longitude, double latitude, 
			   double rot_angle)
{
  gr3d_get_userdef_coordinates (pobj, 0, color, O, longitude, latitude, rot_angle);
}

void gr3d_compute_bounding_sphere (gr3d_object* pobj, double threshold)
{
  if (pobj != NULL && !gr3d_is_empty(*pobj))
    {
      grint i, len = pobj->len;
      gr3d_point barycentre = (pobj->list[0])->M;
      double bx, by, bz;
      gr3d_basic_shape* shape;
      double R, RB, RC, Ax, Ay, Az, Bx, By, Bz, Cx, Cy, Cz;

      bx = barycentre.x;
      by = barycentre.y;
      bz = barycentre.z;
      pobj->R = 0.0;
      if (threshold < 0.0)
	threshold = DBL_MAX;
      for (i = 1; i <= len; i++)
	{
	  shape = pobj->list[i];
	  if (shape->length <= threshold &&
	      shape->color >= pobj->first_color &&
	      shape->color <= pobj->last_color)
	    {	  
	      Ax = shape->A.x-bx;
	      Ay = shape->A.y-by;
	      Az = shape->A.z-bz;
	      R  = sqrt(Ax*Ax+Ay*Ay+Az*Az);

	      if (shape->type >= GR_SEGMENT)
		{
		  Bx = shape->B.x-bx;
		  By = shape->B.y-by;
		  Bz = shape->B.z-bz;
		  RB = sqrt(Bx*Bx+By*By+Bz*Bz);
		}
	      else
		RB = 0.0;

	      if (shape->type == GR_TRIANGLE)
		{
		  Cx = shape->C.x-bx;
		  Cy = shape->C.y-by;
		  Cz = shape->C.z-bz;
		  RC = sqrt(Cx*Cx+Cy*Cy+Cz*Cz);
		}
	      else
		RC = 0.0;

	      if (RB > R)
		R = RB;
	      if (RC > R)
		R = RC;
	      if (R > pobj->R)
		pobj->R = R;
	    }
	} /* end for */
    }
}

const gr3d_point Zero_point_3d = {0.0, 0.0, 0.0};

gr3d_point gr3d_get_middle_point (gr3d_object obj)
{
  if ( (gr3d_is_empty (obj)) )
    return Zero_point_3d;
  else
    return (obj.list[0])->M;
}

double gr3d_get_mean_length (gr3d_object obj, double factor)
{
  if ( (gr3d_is_empty (obj)) )
    return 0.0;
  else if (factor <= 0.0)
    return (obj.list[0])->length;
  else
    {
      gr3d_basic_shape* shape;
      double threshold = (obj.list[0])->length * factor;
      double sum = 0.0;
      grint i, n, len = obj.len;

      for (i = 1, n = 0; i <= len; i++)
	{
	  shape = obj.list[i];
	  if (shape->length <= threshold &&
	      shape->color >= obj.first_color &&
	      shape->color <= obj.last_color)
	    {
	      sum += shape->length;
	      n++;
	    }
	}
      return (n > 0 ? sum / n : 0.0);
    }
}

double gr3d_get_mean_length_by_color (gr3d_object obj, grint color)
{
  if ( (gr3d_is_empty (obj)) )
    return 0.0;
  else
    {
      gr3d_basic_shape* shape;
      double sum = 0.0;
      grint i, n, len = obj.len;

      for (i = 1, n = 0; i <= len; i++)
	{
	  shape = obj.list[i];
	  if (shape->color == color)
	    {
	      sum += shape->length;
	      n++;
	    }
	}
      return (n > 0 ? sum / n : 0.0);
    }
}

grint gr3d_print (gr3d_object obj, FILE* fp)
{
  int rv;
  grtype shape_type;

  if ( gr3d_is_empty(obj) )
    rv = fprintf (fp, "len = 0, size = 0, first_color = %ld, last_color = %ld, rlen = 0, list = <Empty>\n", obj.first_color, obj.last_color);
  else
    {
      grint len, size, i;
      gr3d_basic_shape* shape;
      gr3d_point barycentre;

      len = obj.len;
      size = obj.size;
      rv = fprintf (fp, "len = %lu, size = %lu\n", len, size);
      if (rv > 0)
	{
	  rv = fprintf (fp, "first_color = %ld, last_color = %ld, rlen = %lu\n", obj.first_color, obj.last_color, obj.rlen);
	}
      if (rv > 0)
	{
	  if (obj.R > 0.0)
	    rv = fprintf (fp, "Radius of the bounding sphere = %+30.12G\n",
			  obj.R);
	  else
	    rv = fprintf (fp, "Radius of the bounding sphere : Not computed\n");
	}
      if (rv > 0)
	{
	  if (len > 0)
	    {
	      barycentre = (obj.list[0])->M;
	      rv = fprintf (fp, 
			    "Coordinates of the middle point of the object:\n"\
			    FMT_STRING, 
			    barycentre.x, barycentre.y, barycentre.z);
	      if (rv > 0)
		{
		  rv = fprintf (fp, 
				"Mean length of the object = " \
				FMT_STRING3, 
				(obj.list[0])->length);
		}
	    }
	}
      if (rv > 0)
	{
	  for (i = 1; i <= len; i++)
	    {
	      if (rv > 0)
		{
		  shape = obj.list[i];
		  shape_type = shape->type;
		  if (shape_type == GR_POINT)
		    rv = fprintf (fp,
				  "\nShape no. %lu >> \n"	\
				  "Type: point\n"               \
				  "Color: %ld\n"                \
				  "Caption: \"%s\"\n"           \
				  "Position:\n"	                \
				  FMT_STRING			\
				  FMT_STRING2			\
				  "Middle point:\n"		\
				  FMT_STRING			\
				  "Length = "                   \
				  FMT_STRING3                   \
				  "Current depth = "            \
				  FMT_STRING3, i,
				  shape->color, shape->caption,
				  shape->A.x, shape->A.y, shape->A.z,
				  shape->XA, shape->YA,
				  shape->M.x, shape->M.y, shape->M.z,
				  shape->length, shape->depth);
		  else if (shape_type == GR_SEGMENT)
		    rv = fprintf (fp,
				  "\nShape no. %lu >> \n"	\
				  "Type: segment\n"             \
				  "Color: %ld\n"                \
				  "Caption: \"%s\"\n"           \
				  "First endpoint:\n"		\
				  FMT_STRING			\
				  FMT_STRING2			\
				  "Second endpoint:\n"		\
				  FMT_STRING			\
				  FMT_STRING2			\
				  "Middle point:\n"		\
				  FMT_STRING			\
				  "Length = "                   \
				  FMT_STRING3                   \
				  "Current depth = "            \
				  FMT_STRING3, i,
				  shape->color, shape->caption,
				  shape->A.x, shape->A.y, shape->A.z,
				  shape->XA, shape->YA,
				  shape->B.x, shape->B.y, shape->B.z,
				  shape->XB, shape->YB,
				  shape->M.x, shape->M.y, shape->M.z,
				  shape->length, shape->depth);
		  else
		    rv = fprintf (fp,
				  "\nShape no. %lu >> \n"	\
				  "Type: triangle\n"            \
				  "Color: %ld\n"                \
				  "Caption: \"%s\"\n"           \
				  "First vertex:\n"		\
				  FMT_STRING			\
				  FMT_STRING2			\
				  "Second vertex:\n"		\
				  FMT_STRING			\
				  FMT_STRING2			\
				  "Third vertex:\n"		\
				  FMT_STRING			\
				  FMT_STRING2			\
				  "Middle point:\n"		\
				  FMT_STRING			\
				  "Length = "                   \
				  FMT_STRING3                   \
				  "Current depth = "            \
				  FMT_STRING3, i,
				  shape->color, shape->caption,
				  shape->A.x, shape->A.y, shape->A.z,
				  shape->XA, shape->YA,
				  shape->B.x, shape->B.y, shape->B.z,
				  shape->XB, shape->YB,
				  shape->C.x, shape->C.y, shape->C.z,
				  shape->XC, shape->YC,
				  shape->M.x, shape->M.y, shape->M.z,
				  shape->length, shape->depth);
		}
	      else
		break;
	    }
	}
      if (rv > 0)
	{
	  for (i = len+1; i < size; i++)
	    {
	      if (rv > 0)
		rv = fprintf (fp, "\nShape no. %lu >> NULL\n", i);
	      else
		break;
	    }
	}
    }
  return rv;
}

grint gr3d_save (gr3d_object obj, FILE* fp)
{
  int rv;

  if ( gr3d_is_empty(obj) )
    rv = fprintf (fp, "##len= 0, size= 0, rlen= 0, first_color= %ld, last_color= %ld, R= 0.0\n",
		  obj.first_color, obj.last_color);
  else
    {
      grint len, size, i;
      gr3d_basic_shape* shape;
      
      len = obj.len;
      size = obj.size;
      rv = fprintf (fp, "##len= %ld, size= %ld, rlen= %ld, first_color= %ld, last_color= %ld, R= "	\
		    FMT_STRING6, len, size, obj.rlen, obj.first_color, obj.last_color, obj.R);

      for (i = 1; i <= len && rv > 0; i++)
	{
	  shape = obj.list[i];
	  switch (shape->type)
	    {
	    case GR_POINT:
	      rv = fprintf (fp,
			    "#:%ld, "			\
			    "type: %u, "		\
			    "color: %ld\n"		\
			    "$:%s\n"  		        \
			    "A:"			\
			    FMT_STRING4, i, shape->type,
			    shape->color, shape->caption,
			    shape->A.x, shape->A.y, shape->A.z);
	      break;
	    case GR_SEGMENT:
	      rv = fprintf (fp,
			    "#:%ld, "			\
			    "type: %u, "		\
			    "color: %ld\n"		\
			    "$:%s\n"  		        \
			    "A:"			\
			    FMT_STRING4			\
			    "B:"			\
			    FMT_STRING4, i, shape->type,
			    shape->color, shape->caption,
			    shape->A.x, shape->A.y, shape->A.z,
			    shape->B.x, shape->B.y, shape->B.z);
	      break;
	    case GR_TRIANGLE:
	      rv = fprintf (fp,
			    "#:%ld, "			\
			    "type: %u, "		\
			    "color: %ld\n"		\
			    "$:%s\n"  		        \
			    "A:"			\
			    FMT_STRING4			\
			    "B:"			\
			    FMT_STRING4			\
			    "C:"			\
			    FMT_STRING4, i, shape->type,
			    shape->color, shape->caption,
			    shape->A.x, shape->A.y, shape->A.z,
			    shape->B.x, shape->B.y, shape->B.z,
			    shape->C.x, shape->C.y, shape->C.z);
	      break;
	    default:   /* This should never happen */
	      rv = -1; 
	    } /* end switch */
	} /* end for */
    } /* end !gr3d_is_empty(obj) */
  return rv;
}

#define BUFF_SIZE 1024

grint gr3d_load (gr3d_object* pobj, FILE* fp)
{
  char buffer[BUFF_SIZE], caption[BUFF_SIZE];
  grint len, size;
  int rv;

  if ( pobj == NULL )
    return -1;

  if ( fgets (buffer, BUFF_SIZE, fp) == NULL )
    return -1;
  rv = sscanf (buffer, "##len= %ld, size= %ld, rlen= %ld, first_color= %ld, last_color= %ld, R= " \
	       FMT_STRING7, &len, &size, &pobj->rlen, &pobj->first_color, &pobj->last_color, &pobj->R);
  if (rv < 6)
    return -1;
  else if (len == 0)
    return 0;
  else
    {
      grint i, j;
      gr3d_basic_shape shape;

      for (i = 1; i <= len; i++)
	{
	  if ( fgets (buffer, BUFF_SIZE, fp) == NULL )
	    return -1;
	  rv = sscanf (buffer, "#:%ld, type: %u, color: %ld\n",
		       &j, &shape.type, &shape.color);
	  if (rv < 3 || j != i)
	    return -1;

	  if ( fgets (buffer, BUFF_SIZE, fp) == NULL || strncmp (buffer, "$:", 2) != 0 )
	    return -1;
	  for (j = 2; buffer[j] != '\0' && buffer[j] != '\n'; caption[j-2] = buffer[j], j++);
	  caption[j-2] = '\0';

	  if ( fgets (buffer, BUFF_SIZE, fp) == NULL )
	    return -1;
	  rv = sscanf (buffer, "A:" \
		       FMT_STRING5,
		       &shape.A.x, &shape.A.y, &shape.A.z);
	  if (rv < 3)
	    return -1;

	  if (shape.type >= GR_SEGMENT)
	    {
	      if ( fgets (buffer, BUFF_SIZE, fp) == NULL )
		return -1;
	      rv = sscanf (buffer, "B:"		\
			   FMT_STRING5,
			   &shape.B.x, &shape.B.y, &shape.B.z);
	      if (rv < 3)
		return -1;
	    }

	  if (shape.type == GR_TRIANGLE)
	    {
	      if ( fgets (buffer, BUFF_SIZE, fp) == NULL )
		return -1;
	      rv = sscanf (buffer, "C:"		\
			   FMT_STRING5,
			   &shape.C.x, &shape.C.y, &shape.C.z);
	      if (rv < 3)
		return -1;
	    }
	  
	  if (shape.type == GR_POINT)
	    rv = gr3d_add_point (pobj, shape.color, caption, shape.A.x, shape.A.y, shape.A.z);
	  else if (shape.type == GR_SEGMENT)
	    rv = gr3d_add_segment (pobj, shape.color, caption, shape.A.x, shape.A.y, shape.A.z, shape.B.x, shape.B.y, shape.B.z);
	  else
	    rv = gr3d_add_triangle (pobj, shape.color, caption, shape.A.x, shape.A.y, shape.A.z, shape.B.x, shape.B.y, shape.B.z, 
				    shape.C.x, shape.C.y, shape.C.z);

	  if (rv < 0)
	    return -1;
	} /* end for */
      return len;
    }
}

grint gr3d_remove_items_with_color (gr3d_object* pobj, grint color, 
				    double threshold, gr3d_object* trashcan)
{
  if (pobj != NULL && !gr3d_is_empty(*pobj))
    {
      grint len, rlen, i, j;
      grint trashcan_len = 0;
      gr3d_basic_shape **shlist, *shape;
      double Mx, My, Mz, mean_length;

      len = pobj->len;
      rlen = pobj->rlen;
      shlist = pobj->list;
      Mx = (shlist[0]->M).x;
      My = (shlist[0]->M).y;
      Mz = (shlist[0]->M).z;
      mean_length = shlist[0]->length;
      for (i = 1; i <= len; i++)
	{
	  shape = shlist[i];
	  if (shape->color == color && shape->length > threshold)
	    {
	      /*
		First put the shape in the object pointed to by
		TRASHCAN, if TRASHCAN != NULL and *TRASHCAN is
		not the empty object...
	      */
              if ((trashcan) && !gr3d_is_empty(*trashcan))
                {
                  trashcan_len = gr3d_add_shape (trashcan, 
                                              shape->type, shape->color,
                                              shape->caption, shape->length,
                                              shape->A.x, shape->A.y, shape->A.z,
                                              shape->B.x, shape->B.y, shape->B.z,
                                              shape->C.x, shape->C.y, shape->C.z);
                  /* return immediately in case of failed memory allocation */
                  if (trashcan_len == -1)
                      return -1; 
                }

	      /* then recompute the coordinate of the barycentre of the */
	      /* object without the shape and update RLEN, if this is   */
	      /* necessary,...                                          */
	      if (color >= pobj->first_color &&
		  color <= pobj->last_color)
		{
		  if (rlen > 1)
		    {
		      Mx = (rlen * Mx - shape->M.x) / (rlen-1);
		      My = (rlen * My - shape->M.y) / (rlen-1);
		      Mz = (rlen * Mz - shape->M.z) / (rlen-1);
		      mean_length = (rlen * mean_length - shape->length) / (rlen-1);
		    }
		  else
		    Mx = My = Mz = mean_length = 0.0;
		  rlen--;
		}
	      /* ...and remove the shape from the object */
	      free ((void*)(shape->caption));
	      free ((void*)shape);
	      for (j = i; j < len; j++)
		shlist[j] = shlist[j+1];
	      shlist[len] = NULL;	      
	      len--;
	      i--;
	    }
	}
      /* Set properly the barycentre of the object... */
      (shlist[0]->M).x = Mx;
      (shlist[0]->M).y = My;
      (shlist[0]->M).z = Mz;
      /* ...its mean length... */
      shlist[0]->length = mean_length;
      /* ...its length */
      pobj->len = len;
      /* ...and its reduced length */
      pobj->rlen = rlen;
      /* We do not change pobj->size                  */
      /* since we do not do any realloc on pobj->list */
      return trashcan_len;
    }
  else
      return 0;
}

grint gr3d_add_object (gr3d_object* pobj, gr3d_object obj)
{
  if (pobj != NULL && !gr3d_is_empty(*pobj) && !gr3d_is_empty(obj))
    {
      grint i, new_len, objlen = obj.len;	
      gr3d_basic_shape **shlist, *shape;	
	
      shlist = obj.list;	
      for (i = 1; i <= objlen; i++)
	{
	  shape = shlist[i];	
	  new_len = gr3d_add_shape (pobj, 
				    shape->type, shape->color,
				    shape->caption, shape->length,
				    shape->A.x, shape->A.y, shape->A.z,
				    shape->B.x, shape->B.y, shape->B.z,
				    shape->C.x, shape->C.y, shape->C.z);
	}
      return new_len;
    }
  else
    return -1;
}

void gr3d_delete (gr3d_object* pobj)
{
  if (pobj != NULL && !gr3d_is_empty(*pobj))
    {
      grint len, i;
      
      len = pobj->len;
      for (i = 1; i <= len; i++)
	{
	  free ((void*)((pobj->list[i])->caption));
	  free ((void*)(pobj->list[i]));
	}
      free ((void*)(pobj->list[0])); /* For this first element caption is NULL */
      free ((void*)pobj->list);
      pobj->rlen = 0;
      pobj->len = 0;
      pobj->size = 0;
      pobj->R = 0.0;
      pobj->list = NULL;
    }
}
