/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __GR2D__
#define __GR2D__

#include<stdio.h> /* for FILE */
#include"gr.h"

typedef struct {
  double x, y; /* Coordinates of a point in a 2D space */
} gr2d_point;

typedef struct {
  grtype type;
  grint  color;       /* This can be used to mark the single basic shapes */
                      /* as parts of a bigger shape                       */ 
  grchar* caption;    /* A caption to be used as identifier               */
  gr2d_point A, B, C; /* vertices. If type is POINT, then only A is used. */
                      /* If type is SEGMENT, then only A and B are used.  */
                      /* Not used vertices are set to {0.0, 0.0}.         */

  gr2d_point M;       /* middle point, always computed from the vertices. */
                      /* M must be handled as a private field.            */

  double length;      /* The length or perimeter of the shape             */

  grint depth;        /* depth of the shape: the layer to which the shape  */
                      /* belongs to. This must be considered a private     */
                      /* field. It is set to 0 by gr2d_add*() and can be   */
                      /* changed only by gr2d_set_depth().                 */

  double XA, YA; /* Coordinates of A, B and C with respect to a user-defined */
  double XB, YB; /* coordinate system.                                       */
  double XC, YC; /* These must be handled as private fields.                 */
} gr2d_basic_shape;

typedef struct {
  gr2d_basic_shape** list;     /* This is an array of pointers to shapes */
                     /* If LIST == NULL, we say that the object is empty */
  grint len, size;   /* LEN is the number of shapes stored in the array,
                        SIZE is the number of available slots in the array */ 
  double R;          /* The radius of the smallest circle centered at
                        list[0]->M which contains the whole object, or
			alternatively all the shapes of the object whose length
			is below a given threshold, see gr2d_compute_bounding_circle().
                        After the creation of an object this is set to 0.0 
                        and can only be changed by calling 
                        gr2d_compute_bounding_circle() on the object.      */
  /*
	 After adding one or more shapes to an object by using gr2d_add_*(),
	 LIST[0]->M.x and LIST[0]->M.y are the coordinates
	 of the middle point of the object, LIST[0]->length is the mean
	 length of the shapes forming the object. They are both computed
	 by considering only the shapes whose COLOR lies in the range
	 FIRST_COLOR..LAST_COLOR.
  */
  grint first_color, last_color;

  /*
    Number of the shapes of the object whose COLOR lies in the range
    FIRST_COLOR..LAST_COLOR.
  */
  grint rlen;
} gr2d_object; 



/*
  Return 1 if P.x and P.y are finite values, otherwise 0. 
*/
grint gr2d_point_is_finite (gr2d_point P);


/* 
   Create a new gr2d object with FIRST_COLOR = COLOR1 and
   LAST_COLOR = COLOR2. Return an empty object if no memory is available 
   to create the object.

   Rem.: The new object has R = 0.0, RLEN = LEN = 0, SIZE = 1 and LIST 
         is formed by just one slot, LIST[0]. 
	 This is a pointer to a shape with TYPE = GR_NONE,
	 COLOR = 0, CAPTION = NULL,
	 A.x = A.y = B.x = B.y = C.x = C.y = 0.0,
	 M.x = M.y = 0.0, and XA = YA = XB = YB = XC = YC = 0.0.  
*/
gr2d_object gr2d_new (grint color1, grint color2);

/*
  Return 1 if OBJ is an empty object, otherwise 0.

  Remark: empty means that OBJ.LIST == NULL.
*/
int gr2d_is_empty (gr2d_object obj);

/* 
   Add a triangle to an existing gr2d object. Return the new number
   of shapes forming the object, or -1 if it was not possible
   to add the new shape due to memory issues. Return -1 also
   if POBJ == NULL or *POBJ is an empty object.

   Rem.: POBJ is a pointer to an existing object, the other parameters
         are the coordinates of the three vertices A, B and C of the
	 triangle to be added.
*/
grint gr2d_add_triangle (gr2d_object* pobj, 
			 grint color,
			 const grchar* caption,
			 double xA, double yA,
			 double xB, double yB,
			 double xC, double yC);

/* 
   Add a segment to an existing gr2d object. Return the new number
   of shapes forming the object, or -1 if it was not possible
   to add the new shape due to memory issues. Return -1 also
   if POBJ == NULL or *POBJ is an empty object.

   Rem.: POBJ is a pointer to an existing object, the other parameters
         are the coordinates of the vertices A, B of the
	 segment to be added.
*/
grint gr2d_add_segment (gr2d_object* pobj, 
			grint color,
			const grchar* caption,
			double xA, double yA,
			double xB, double yB);

/* 
   Add a point to an existing gr2d object. Return the new number
   of shapes forming the object, or -1 if it was not possible
   to add the new shape due to memory issues. Return -1 also
   if POBJ == NULL or *POBJ is an empty object.

   Rem.: POBJ is a pointer to an existing object, the other parameters
         are the coordinates of the point to be added.
*/
grint gr2d_add_point (gr2d_object* pobj, 
		      grint color,			
		      const grchar* caption,
		      double x, double y);

/* 
   Compute the coordinates of the points forming the object 
   with respect to the orthogonal reference system 
   having the point O as origin and defined by the angle
   ROT_ANGLE.

   Rem.: POBJ should be a pointer to an existing object. 
         Do nothing if POBJ == NULL or *POBJ is an empty object.
*/
void gr2d_compute_userdef_coordinates (gr2d_object* pobj, gr2d_point O,
				       double rot_angle); 

/*
  This function works exactly like gr2d_compute_userdef_coordinates()
  but, instead of computing the user-defined coordinates of every point 
  of the object, it does this computations only for the basic shapes
  marked by COLOR. 

  Rem.: POBJ should be a pointer to an existing object. 
        Nothing is done if POBJ == NULL or *POBJ is an empty object.

        This function is useful whenever the basic shapes marked
        by COLOR have been removed using gr2d_remove_items_with_color()
	and then new shapes with the same color have been added
	by means of gr2d_add*(). If this is the case, gr2d_partial_upgrade()
	allows to save time compared to gr2d_compute_userdef_coordinates().
*/
void gr2d_partial_upgrade (gr2d_object* pobj, grint color,
			   gr2d_point O, double rot_angle);

/*
  If THRESHOLD >= 0, then set POBJ->R to the radius of the smallest 
  circle centered at (POBJ->LIST[0])->M which contains all the shapes 
  of the object pointed to by POBJ whose length does not exceed the 
  given THRESHOLD and whose color lies in the range
  POBJ->FIRST_COLOR..POBJ->LAST_COLOR.
  If THRESHOLD < 0, simply set POBJ->R to the radius of the smallest 
  circle centered at (POBJ->LIST[0])->M which contains all the shapes 
  of the object pointed to by POBJ whose color lies in the range
  POBJ->FIRST_COLOR..POBJ->LAST_COLOR.

  Rem.: POBJ should be a pointer to an existing object. 
        Do nothing if POBJ == NULL or *POBJ is an empty object.
*/
void gr2d_compute_bounding_circle (gr2d_object* pobj, double threshold);

/*
  Return the middle point of the object OBJ, i.e. OBJ.LIST[0]->M.

  Remark: if OBJ is an empty object, then {0.0, 0.0} is returned.
*/
gr2d_point gr2d_get_middle_point (gr2d_object obj);

/*
  If FACTOR z <= 0.0, then return the mean length of the object
  OBJ, i.e. OBJ.LIST[0]->LENGTH.
  Otherwise, return the mean length of all the shapes from OBJ
  whose length does not exceed FACTOR * OBJ.LIST[0]->LENGTH and
  whose color lies in the range OBJ.FIRST_COLOR..OBJ.LAST_COLOR. 

  Remark: if OBJ is an empty object, then 0.0 is returned.
*/
double gr2d_get_mean_length (gr2d_object obj, double factor);

/*
  Return the mean length of all basic shapes from OBJ marked
  by COLOR. If OBJ is an empty object, return zero.
*/
double gr2d_get_mean_length_by_color (gr2d_object obj, grint color);

/*
  Print to the file pointed to by FP the shapes forming the
  object OBJ. Return the number of printed shapes in case of
  success (0 if OBJ is an empty object), -1 in case of failure.
*/
grint gr2d_print (gr2d_object obj, FILE* fp);

/*
  Save in the file pointed to by FP the shapes forming the
  object OBJ. Return a non-negative number in case of
  success (0 if OBJ is an empty object), -1 in case of failure.
*/
grint gr2d_save (gr2d_object obj, FILE* fp);

/*
  Load an object from the file pointed to by FP and
  stores it in the structure pointed to by POBJ.
  Return the number of loaded shapes in case of
  success, -1 in case of failure (for example,
  if POBJ == NULL or the contents of the file
  could not be completely read).

  Remark: POBJ should be a pointer to an existing object. 
          The shapes read from file are appended to this object.
*/
grint gr2d_load (gr2d_object* pobj, FILE* fp);

/*
  Remove from the object pointed to by POBJ all basic shapes
  marked by COLOR whose length/perimeter is greater than
  THRESHOLD. If TRASHCAN != NULL and *TRASHCAN is not an empty object,
  put the removed shapes in the object pointed to by TRASHCAN.
  Do nothing but return 0 if POBJ == NULL or *POBJ is an empty object.

  Return the number of shapes forming the object pointed to by TRASHCAN
  if TRASHCAN != NULL , *TRASHCAN is not an empty object, and the shapes
  removed from *POBJ have been successfully added to *TRASHCAN, otherwise
  return either 0, to mean that no shape has been actually added to *TRASHCAN,
  or -1, to indicate a failure while adding shapes to *TRASHCAN.

  Remarks: POBJ should be a pointer to an existing object. 
	   This function recomputes (POBJ->LIST[0]).M,
	   (POBJ->LIST[0]).LENGTH and POBJ->RLEN if necessary.
	   If a memory failure occurs while trying to add new shapes to
	   the object pointed to by TRASHCAN, then this function
           immediately stops and return -1.
*/
grint gr2d_remove_items_with_color (gr2d_object* pobj, 
				    grint color, double threshold,
				    gr2d_object* trashcan);

/*
  Append to the object pointed to by POBJ the basic shapes from OBJ.
  Return the new number of shapes forming the object pointed to by POBJ.
  Do nothing if POBJ == NULL, *POBJ is an empty object, or OBJ is empty.
  In this case returns -1.

  Remark: POBJ should be a pointer to an existing object.
          If the number returned is less than the sum of OBJ.LEN
	  with the initial value of POBJ->LEN, then a memory
	  failure occurred while trying to add a new shape.
*/
grint gr2d_add_object (gr2d_object* pobj, gr2d_object obj);

/*
  For all basic shapes of the object pointed to by POBJ which
  have the given COLOR set to DEPTH the value of the depth field.
  Do nothing if POBJ == NULL or *POBJ is an empty object.

  Rem.: POBJ should be a pointer to an existing object. 
*/
void gr2d_set_depth (gr2d_object* pobj, grint color, grint depth);

/*
  Sort (POBJ->LIST[1],...,POBJ->LIST[POBJ->LEN])
  in ascending order according to the value of the depth field.
  Do nothing if POBJ == NULL or *POBJ is an empty object.

  Rem.: POBJ should be a pointer to an existing object. 
*/
void gr2d_sort_shapes (gr2d_object* pobj);

/*
  Delete the object pointed to by POBJ. Do nothing if POBJ == NULL
  or *POBJ is an empty object.

  Rem.: POBJ->FIRST_COLOR and POBJ->LAST_COLOR are left unchanged,
        all other fields are zeroed.
*/
void gr2d_delete (gr2d_object* pobj);

#endif /* __GR2D__ */
