/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<stdio.h>
#include<stdlib.h>
#include<string.h> /* for strncmp() */
#include"gr2d.h"
#include"gr3d.h"
#include"controls.h"
#include"gnuplot.h"
#include"utils.h"

#define CHUNK_ALLOC  1024
#define BUFFER_SIZE   513
#define CAPTION_SIZE  513 
#define COMMENT_PREFIX '#'

typedef gr3d_point point;

typedef struct {
  point* array;
  size_t length, size;
} list_of_points;

static const char dotted_style_cmd[]=DOTTED_COMM;
static const char stroked_style_cmd[]=STROKED_COMM;
static const char filled_style_cmd[]=FILLED_COMM;
static const char set_caption_cmd[]=CAPTION_COMM;

static grstyle g_style = GRSTYLE_STROKED;
static grchar g_caption[CAPTION_SIZE] = {'\0'};

static grchar* g_captions_2d[NUMBER_OF_2DOBJECTS] = {NULL};
static grchar* g_captions_3d[NUMBER_OF_3DOBJECTS] = {NULL};

#ifdef GNUPLOT_TEST
static char last_line_read[BUFFER_SIZE] = {'\0'};
#endif /* GNUPLOT_TEST */

static
void reinit_caption_and_style(void)
{
  g_style = GRSTYLE_STROKED;
  g_caption[0] = '\0';
}

void reinit_captions (graphic_mode mode)
{
  int i;

  if (mode == GRAPHIC_2D)
    {
      for (i = 0; i < NUMBER_OF_2DOBJECTS; i++)
	xfree (&g_captions_2d[i]);
    }
  if (mode == GRAPHIC_3D)
    {
      for (i = 0; i < NUMBER_OF_3DOBJECTS; i++)
	xfree (&g_captions_3d[i]);
    }
}

static
void save_object_caption (const char* filepath, graphic_mode mode, int id)
{
  if (mode == GRAPHIC_2D)
    {
      if (id >= 0 && id < NUMBER_OF_2DOBJECTS)
	{
	  g_captions_2d[id] = xstralloc (xstrlen(filepath)+xstrlen(g_caption)+3);
	  strcpy (g_captions_2d[id], filepath);
	  if ( !(is_string_blank(g_caption)) )
	    {
	      strcat (g_captions_2d[id], ", ");
	      strcat (g_captions_2d[id], g_caption);
	    }
	}
    }
  else
    {
      if (id >= 0 && id < NUMBER_OF_3DOBJECTS)
	{
	  g_captions_3d[id] = xstralloc (xstrlen(filepath)+xstrlen(g_caption)+3);
	  strcpy (g_captions_3d[id], filepath);
	  if ( !(is_string_blank(g_caption)) )
	    {
	      strcat (g_captions_3d[id], ", ");
	      strcat (g_captions_3d[id], g_caption);
	    }
	}
    }
}

const grchar* get_object_caption (graphic_mode mode, int id)
{
  if (mode == GRAPHIC_2D)
    {
      if (id >= 0 && id < NUMBER_OF_2DOBJECTS)
	return g_captions_2d[id];
      else
	return NULL;
    }
  else if (mode == GRAPHIC_3D)
    {
      if (id >= 0 && id < NUMBER_OF_3DOBJECTS)
	return g_captions_3d[id];
      else
	return NULL;
    }
  else
    return NULL;
}

static
void dispose_object_caption (graphic_mode mode, int id)
{
  if (mode == GRAPHIC_2D)
    {
      if (id >= 0 && id < NUMBER_OF_2DOBJECTS)
	xfree (&g_captions_2d[id]);
    }
  else
    {
      if (id >= 0 && id < NUMBER_OF_3DOBJECTS)
	xfree (&g_captions_3d[id]);
    }
}

static
void lp_init (list_of_points* plist)
{
  if ((plist))
    {
      plist->array = NULL;
      plist->length = plist->size = 0;
    }
}

static
size_t lp_add_point (list_of_points* plist, point p)
{
  if (!plist)
    return (size_t)(-1);
  if (plist->length == plist->size)
    {
      if (plist->size == 0)
	{
	  plist->array = (point*) calloc (CHUNK_ALLOC, sizeof(point));
	  if (!plist->array)
	    return 0;
	}
      else
	{
	  point* ptmp;

	  ptmp = (point*) realloc (plist->array,
				  (CHUNK_ALLOC + plist->size)*sizeof(point));
	  if (!ptmp)
	    return 0;
	  else
	    plist->array = ptmp;
	}
      plist->size += CHUNK_ALLOC;
    }
  plist->array[plist->length] = p;
  plist->length++;
  return plist->length;
}

/* NOT NEEDED FOR NOW
  static
  void lp_clear (list_of_points* plist)
  {
    if ((plist))
    {
      plist->length = 0;
    }
  }
*/

static
void lp_free (list_of_points* plist)
{
  if ((plist))
    {
      if ((plist->array))
	{
	  free ((void*)plist->array);
	  plist->array = NULL;
	}
      plist->length = plist->size = 0;
    }
}

static
int is_a_comment_line (const char* line_buffer)
{
  const char* ptr;

  for (ptr = line_buffer; (is_space(*ptr)); ptr++);
  return (*ptr == COMMENT_PREFIX ? 1 : 0);
}

static
void handle_special_comments (const char* line_buffer)
{
  static const size_t nd = sizeof(dotted_style_cmd)-1;
  static const size_t ns = sizeof(stroked_style_cmd)-1;
  static const size_t nf = sizeof(filled_style_cmd)-1;
  static const size_t nsc = sizeof(set_caption_cmd)-1;
  size_t n;

  if (strncmp (dotted_style_cmd, line_buffer, nd) == 0)
    {
      for (n = nd; (is_space (line_buffer[n])); n++);
      if (line_buffer[n] == '\0')
	g_style = GRSTYLE_DOTTED;
    }

  if (strncmp (stroked_style_cmd, line_buffer, ns) == 0)
    {
      for (n = ns; (is_space (line_buffer[n])); n++);
      if (line_buffer[n] == '\0')
	g_style = GRSTYLE_STROKED;
    }

  if (strncmp (filled_style_cmd, line_buffer, nf) == 0)
    {
      for (n = nf; (is_space (line_buffer[n])); n++);
      if (line_buffer[n] == '\0')
	g_style = GRSTYLE_FILLED;
    }
  if (strncmp (set_caption_cmd, line_buffer, nsc) == 0)
    {
      size_t i;
      
      for (n = nsc; (is_space (line_buffer[n])); n++);
      for (i = 0; i < CAPTION_SIZE - 1 && line_buffer[n] != '\0'; i++, n++)
	{
	  g_caption[i] = line_buffer[n];
	}
      g_caption[i] = '\0';
    }
}

/*
  The following function reads a line from the file pointed to by FP
  and stores the first LINE_BUFFER_SIZE - 1 characters read into
  the buffer pointed to by LINE_BUFFER (whose size must be greater or
  equal to LINE_BUFFER_SIZE). It returns one of the following values:

  -1 --> The end of the file pointed to by FP has been reached
         and no characters from this file have been stored into LINE_BUFFER.
   0 --> The end of the file pointed to by FP has been reached,
         the buffer pointed to by LINE_BUFFER contains new characters from
	 the file.
   1 --> The end of the file pointed to by FP has not been reached,
         thus there is still at least one line in this file that has
	 to be read (after the one already stored in LINE_BUFFER).

   Remark: The newline at the end of a line is read and neglected.
*/
static
int read_line_into_buffer (FILE* fp, char* line_buffer, size_t line_buffer_size)
{
  int ch;
  size_t n;

  for (n = 0, ch = fgetc(fp); ch != EOF && '\n' != (char)ch; ch = fgetc(fp))
    {
      if (n < line_buffer_size)
	{
	  line_buffer[n] = ch;
	  n++;
	}
    }
  line_buffer[n] = '\0';

#ifdef GNUPLOT_TEST
  strncpy (last_line_read, line_buffer, BUFFER_SIZE - 1);
  last_line_read[BUFFER_SIZE - 1] = '\0';
#endif /* GNUPLOT_TEST */

  if (ch == '\n')
    return 1;
  else
    return (n > 0 ? 0 : -1);
}

/*
  The following function reads the next non-comment line 
  from the file pointed to by FP and stores the first LINE_BUFFER_SIZE - 1 
  of this line into the buffer pointed to by LINE_BUFFER 
  (whose size must be greater or equal to LINE_BUFFER_SIZE). 
  If INTERPRET_SPECIAL_COMMENTS is zero, then any encountered line of comment
  (i.e. any line starting by #) is neglected. Otherwise,
  the function checks for special comments and for every
  special comment found executes the corresponding action
  (see function handle_special_comment()).
  The return value is

   1 --> The buffer pointed to by LINE_BUFFER contains new characters read from
	 the file pointed to by FP.
   0 --> The end of the file pointed to by FP has been reached before
         it was possible to read any character from the file.
*/
static
int next_line_from_file (FILE* fp, char* line_buffer, size_t line_buffer_size,
			 int interpret_special_comments)
{
  int rv;
  
  while ( (rv = read_line_into_buffer (fp, line_buffer, line_buffer_size)) >= 0 
	  && (is_a_comment_line(line_buffer)) )
    {
      if ((interpret_special_comments))
	handle_special_comments (line_buffer);
    }
  return (rv >= 0 ? 1 : 0);
}

/*
  The following function reads from the file pointed to by FP
  a sequence of lines till the first non-comment line comes.
  If the end-of-file comes before such a line can be read,
  then -1 is returned.
  Otherwise:

  * if the line read contains only white spaces,
    the function returns 0;

  * if the line read starts with a list of three
    space separated floating point values, then
    the function returns 3 after having stored
    the extracted values in PP->X, PP->Y and PP->Z,
    respectively;

  * if the line read starts with a list of two
    space separated floating point values, then
    the function returns 2 after having stored
    the extracted values in PP->X and PP->Y
    respectively, and having set PP->Z to zero;

  * In any other case the function returns -2.
    -2 is also returned if PP is NULL.
*/
static
int read_first_point (FILE* fp, point* pp)
{
  static char buffer[BUFFER_SIZE] = {'\0'};
  char* ptr;
  int read_line;
  
  if (!pp)
    return -2; /* safety check */
  read_line = next_line_from_file (fp, buffer, BUFFER_SIZE, 1);
  if (!read_line)
    return -1; /* end-of-file reached */
  for (ptr = buffer; (is_space(*ptr)); ptr++);
  if (*ptr == '\0')
    return 0; /* empty line read */
  else if ( sscanf (buffer, " %lf %lf %lf ", &pp->x, &pp->y, &pp->z) == 3 )
    return 3;
  else if ( sscanf (buffer, " %lf %lf ", &pp->x, &pp->y) == 2 )
    {
      pp->z = 0.0;
      return 2;
    }
  else
    return -2;
}

/*
  The following function reads from the file pointed to by FP
  a sequence of lines till the first non-comment line comes.
  If the end-of-file comes before such a line can be read,
  then -1 is returned.
  Otherwise:

  * if the line read contains only white spaces,
    the function returns 0;

  * if the line read starts with a list of three
    space separated floating point values and DIMENSION
    is equal to 3, then the function returns 3 after having stored
    the extracted values in PP->X, PP->Y and PP->Z,
    respectively;

  * if the line read starts with a list of two
    space separated floating point values and DIMENSION
    is equal to 2, then the function returns 2 after having stored
    the extracted values in PP->X and PP->Y
    respectively, and having set PP->Z to zero;

  * In any other case the function returns -2.
    -2 is also returned if PP is NULL, or
    DIMENSION is neither 2 nor 3.
*/
static
int read_next_point (FILE* fp, point* pp, int dimension)
{
  static char buffer[BUFFER_SIZE] = {'\0'};
  char* ptr;
  int read_line;

  if (!pp || dimension < 2 || dimension > 3)
    return -2;
  read_line = next_line_from_file (fp, buffer, BUFFER_SIZE, 0);
  if (!read_line)
    return -1; /* end-of-file reached */
  for (ptr = buffer; (is_space(*ptr)); ptr++);
  if (*ptr == '\0')
    return 0; /* empty line read */
  if (dimension == 3)
    {
      if ( sscanf (buffer, " %lf %lf %lf ", &pp->x, &pp->y, &pp->z) == 3 )
	return 3;
      else
	return -2; 
    }
  else
    {
      /* dimension == 2 */
      if ( sscanf (buffer, " %lf %lf ", &pp->x, &pp->y) == 2 )
	{
	  pp->z = 0.0;
	  return 2;
	}
      else
	return -2; 
    }
}

static
int load_first_array (FILE* fp, list_of_points* plist)
{
  point p;
  int dim, rv;
  size_t n = 0;

  if (!plist)
    return 0;
  for (dim = rv = read_first_point (fp, &p); 
       rv > 0;
       rv = read_next_point (fp, &p, dim))
    {
      if ((n = lp_add_point (plist, p)) == 0)
	return -1;
    }
  if (rv < -1) 
    return -1;   /* error */
  else if (n == 0)
    return 0;    /* Eof or empty line read before any point could be read */
  else if (rv == -1)
    return -dim; /* Eof reached after having read one or more points */
  else /* rv == 0: an empty line has been found after having read one or more points */
    return dim;   
}

static
int load_next_array (FILE* fp, list_of_points* plist, int dim)
{
  point p;
  int rv;
  size_t n = 0;

  if (!plist)
    return 0;
  for (rv = read_next_point (fp, &p, dim); 
       rv > 0;
       rv = read_next_point (fp, &p, dim))
    {
      if ((n = lp_add_point (plist, p)) == 0)
	return -1;
    }
  if (rv < -1) 
    return -1;   /* error */
  else if (n == 0)
    return 0;    /* Eof or empty line read before any point could be read */
  else if (rv == -1)
    return -dim; /* Eof reached after having read one or more points */
  else /* rv == 0: an empty line has been found after having read one or more points */
    return dim;   
}

static
int add_2dcurve_from_array (gr2d_object* pobj2d, list_of_points list,
			    grint id, grstyle style, const grchar* caption)
{
  size_t n;
  
  if (!pobj2d || style == GRSTYLE_NONE || style == GRSTYLE_FILLED)
    return -1; /* error */
  /* We assume that the pipeline pointed to by POBJ2D */
  /* has been properly initialized */
  if (style == GRSTYLE_DOTTED)
    {
      for (n = 0; n < list.length; n++)
	{
	  if ( gr2d_add_point(pobj2d, id, caption, 
			      list.array[n].x, list.array[n].y) < 0 )
	    return -1;
	  caption = NULL; /* only the first point may have a caption */
	}
    }
  else
    {
      for (n = 0; n + 1 < list.length; n++)
	{
	  if ( gr2d_add_segment (pobj2d, id, caption, 
				 list.array[n].x, list.array[n].y,
				 list.array[n+1].x, list.array[n+1].y) < 0 )
	    return -1;
	  caption = NULL; /* only the first segment may have a caption */
	}      
    }
  return 0;
}

static
int add_3dcurve_from_array (gr3d_object* pobj3d, list_of_points list,
			    grint id, grstyle style, const grchar* caption)
{
  size_t n;
  
  if (!pobj3d || style == GRSTYLE_NONE || style == GRSTYLE_FILLED)
    return -1; /* error */
  /* We assume that the pipeline pointed to by POBJ3D */
  /* has been properly initialized */
  if (style == GRSTYLE_DOTTED)
    {
      for (n = 0; n < list.length; n++)
	{
	  if ( gr3d_add_point(pobj3d, id, caption, 
			      list.array[n].x, 
			      list.array[n].y, 
			      list.array[n].z) < 0 )
	    return -1;
	  caption = NULL; /* only the first point may have a caption */
	}
    }
  else
    {
      for (n = 0; n + 1 < list.length; n++)
	{
	  if ( gr3d_add_segment (pobj3d, id, caption, 
				 list.array[n].x, 
				 list.array[n].y, 
				 list.array[n].z,
				 list.array[n+1].x, 
				 list.array[n+1].y,
				 list.array[n+1].z) < 0 )
	    return -1;
	  caption = NULL; /* only the first segment may have a caption */
	}      
    }
  return 0;
}

static
int add_2dsurface_from_array (gr2d_object* pobj2d, list_of_points list,
			      grint id, grstyle style, const grchar* caption)
{
  size_t n;
  
  if (!pobj2d || style == GRSTYLE_NONE)
    return -1; /* error */
  /* We assume that the pipeline pointed to by POBJ2D */
  /* has been properly initialized */
  if (style == GRSTYLE_DOTTED)
    {
      for (n = 0; n < list.length; n++)
	{
	  if ( gr2d_add_point(pobj2d, id, caption, 
			      list.array[n].x, list.array[n].y) < 0 )
	    return -1;
	  caption = NULL; /* only the first point may have a caption */
	}
      return 0;
    }
  else if (style == GRSTYLE_STROKED)
    {
      for (n = 0; n + 1 < list.length; n++)
	{
	  if ( gr2d_add_segment (pobj2d, id, caption, 
				 list.array[n].x, list.array[n].y,
				 list.array[n+1].x, list.array[n+1].y) < 0 )
	    return -1;
	  caption = NULL; /* only the first segment may have a caption */
	}    
      return 0;  
    }
  else /* GRSTYLE_FILLED */
    return 0;
}

static
int add_3dsurface_from_array (gr3d_object* pobj3d, list_of_points list,
			      grint id, grstyle style, const grchar* caption)
{
  size_t n;
  
  if (!pobj3d || style == GRSTYLE_NONE)
    return -1; /* error */
  /* We assume that the pipeline pointed to by POBJ3D */
  /* has been properly initialized */
  if (style == GRSTYLE_DOTTED)
    {
      for (n = 0; n < list.length; n++)
	{
	  if ( gr3d_add_point(pobj3d, id, caption, 
			      list.array[n].x, 
			      list.array[n].y, 
			      list.array[n].z) < 0 )
	    return -1;
	  caption = NULL; /* only the first point may have a caption */
	}
      return 0;
    }
  else if (style == GRSTYLE_STROKED)
    {
      for (n = 0; n + 1 < list.length; n++)
	{
	  if ( gr3d_add_segment (pobj3d, id, caption, 
				 list.array[n].x, 
				 list.array[n].y, 
				 list.array[n].z,
				 list.array[n+1].x, 
				 list.array[n+1].y,
				 list.array[n+1].z) < 0 )
	    return -1;
	  caption = NULL; /* only the first segment may have a caption */
	} 
      return 0;
    }
  else /* GRSTYLE_FILLED */
    return 0;
}

static
int add_next_2dsurface_from_arrays (gr2d_object* pobj2d, 
				    list_of_points list1, list_of_points list2, 
				    grint id, grstyle style, 
				    const grchar* caption) 
{
  size_t n;
  
  if (!pobj2d || style == GRSTYLE_NONE || list1.length != list2.length)
    return -1; /* error */
  /* We assume that the pipeline pointed to by POBJ2D */
  /* has been properly initialized */
  if (style == GRSTYLE_DOTTED)
    {
      for (n = 0; n < list2.length; n++)
	{
	  if ( gr2d_add_point(pobj2d, id, NULL, 
			      list2.array[n].x, list2.array[n].y) < 0 )
	    return -1;
	}
      return 0;
    }
  else if (style == GRSTYLE_STROKED)
    {
      for (n = 0; n + 1 < list2.length; n++)
	{
	  if ( gr2d_add_segment (pobj2d, id, NULL, 
				 list2.array[n].x, list2.array[n].y,
				 list2.array[n+1].x, list2.array[n+1].y) < 0 )
	    return -1;
	}    
      for (n = 0; n < list2.length; n++)
	{
	  if ( gr2d_add_segment (pobj2d, id, NULL, 
				 list1.array[n].x, list1.array[n].y,
				 list2.array[n].x, list2.array[n].y) < 0 )
	    return -1;
	}    
      return 0;
    }
  else
    {
      if (list2.length < 2)
	return -1;
      /* style == GRSTYLE_FILLED */
      for (n = 0; n + 1 < list2.length; n++)
	{
	  if ( gr2d_add_triangle (pobj2d, id, caption, 
				  list1.array[n].x, list1.array[n].y,
				  list2.array[n].x, list2.array[n].y,
				  list2.array[n+1].x, list2.array[n+1].y) < 0 )
	    return -1;
	  caption = NULL; /* only the first triangle may have a caption */
	}
      for (n = 1; n  < list2.length; n++)
	{
	  if ( gr2d_add_triangle (pobj2d, id, NULL, 
				  list2.array[n].x, list2.array[n].y,
				  list1.array[n-1].x, list1.array[n-1].y,
				  list1.array[n].x, list1.array[n].y) < 0 )
	    return -1;
	}
      return 0;
    }
}

static
int add_next_3dsurface_from_arrays (gr3d_object* pobj3d, 
				    list_of_points list1, list_of_points list2, 
				    grint id, grstyle style,
				    const grchar* caption) 
{
  size_t n;
  
  if (!pobj3d || style == GRSTYLE_NONE || list1.length != list2.length)
    return -1; /* error */
  /* We assume that the pipeline pointed to by POBJ3D */
  /* has been properly initialized */
  if (style == GRSTYLE_DOTTED)
    {
      for (n = 0; n < list2.length; n++)
	{
	  if ( gr3d_add_point(pobj3d, id, NULL, 
			      list2.array[n].x, 
			      list2.array[n].y,
			      list2.array[n].z) < 0 )
	    return -1;
	}
      return 0;
    }
  else if (style == GRSTYLE_STROKED)
    {
      for (n = 0; n + 1 < list2.length; n++)
	{
	  if ( gr3d_add_segment (pobj3d, id, NULL, 
				 list2.array[n].x, 
				 list2.array[n].y,
				 list2.array[n].z,
				 list2.array[n+1].x, 
				 list2.array[n+1].y,
				 list2.array[n+1].z) < 0 )
	    return -1;
	}    
      for (n = 0; n < list2.length; n++)
	{
	  if ( gr3d_add_segment (pobj3d, id, NULL, 
				 list1.array[n].x, 
				 list1.array[n].y,
				 list1.array[n].z,
				 list2.array[n].x, 
				 list2.array[n].y,
				 list2.array[n].z) < 0 )
	    return -1;
	}    
      return 0;
    }
  else
    {
      if (list2.length < 2)
	return -1;
      /* style == GRSTYLE_FILLED */
      for (n = 0; n + 1 < list2.length; n++)
	{
	  if ( gr3d_add_triangle (pobj3d, id, caption, 
				  list1.array[n].x, 
				  list1.array[n].y,
				  list1.array[n].z,
				  list2.array[n].x, 
				  list2.array[n].y,
				  list2.array[n].z,
				  list2.array[n+1].x, 
				  list2.array[n+1].y,
				  list2.array[n+1].z) < 0 )
	    return -1;
	  caption = NULL; /* only the first triangle may have a caption */
	}
      for (n = 1; n  < list2.length; n++)
	{
	  if ( gr3d_add_triangle (pobj3d, id, NULL, 
				  list2.array[n].x, 
				  list2.array[n].y,
				  list2.array[n].z,
				  list1.array[n-1].x, 
				  list1.array[n-1].y,
				  list1.array[n-1].z,
				  list1.array[n].x, 
				  list1.array[n].y,
				  list1.array[n].z) < 0 )
	    return -1;
	}
      return 0;
    }
}

static
int load_object_from_file (FILE* fp, gr2d_object* pobj2d, gr3d_object* pobj3d, 
			   grint id, int first_object_in_file, int* pobj_dim)
{
  int retval, dim, more_data_are_needed = 0;
  list_of_points list1, list2;

  if (!pobj2d || !pobj3d)
    return -1; /* error */
  /* We assume that the pipeline pointed to by POBJ2D (POBJ3D) */
  /* has been properly initialized */
  lp_init (&list1);
  lp_init (&list2);
  dim = load_first_array (fp, &list1);

  /* The first object in a file can be preceded by an empty line */
  if ((first_object_in_file) && dim == 0)
    dim = load_first_array (fp, &list1);

  if ((pobj_dim))
    *pobj_dim = (dim < 0 ? -dim : dim);
  switch (dim)
    {
    case -2:
      retval = add_2dcurve_from_array (pobj2d, list1, id, g_style, g_caption);
      lp_free (&list1);
      if (retval != 0)
	return -1; /* error */
      else
	return 0;  /* EoF reached after having successfully loaded an object */
    case -3:
      retval = add_3dcurve_from_array (pobj3d, list1, id, g_style, g_caption); 
      lp_free (&list1);
      if ( retval != 0)
	return -1; /* error */
      else
	return 0;  /* EoF reached after having successfully loaded an object */
    case 0:  
      /* An eof or an empty line has been read before any valid point   */
      /* could be read. We need to distinguish between two cases:       */
      /* - Eof has been reached: 0 must be returned, to mean that we    */
      /*   should stop reading from file;                               */
      /* - Eof has not been reached: the file is not properly formatted */
      /*   and -1 should be returned.                                   */
      lp_free (&list1);
      return ( (feof(fp)) ? 0 : -1);
    case 2:
      if ( add_2dsurface_from_array (pobj2d, list1, id, g_style, g_caption) != 0 )
	{
	  lp_free (&list1);
	  return -1; /* error */
	}
      if (g_style == GRSTYLE_FILLED)
	more_data_are_needed = 1;
      break;
    case 3:
      if ( add_3dsurface_from_array (pobj3d, list1, id, g_style, g_caption) != 0 )
	{
	  lp_free (&list1);
	  return -1; /* error */
	}
      if (g_style == GRSTYLE_FILLED)
	more_data_are_needed = 1;
      break;
    default: /* -1 == error */
      lp_free (&list1);
      return -1;
    }
  if (dim == 2)
    {
      int errcode = 0;
      int rv = 0;

      while ( (rv = load_next_array (fp, &list2, dim)) > 0 && errcode == 0)
	{
	  more_data_are_needed = 0;
	  errcode = add_next_2dsurface_from_arrays (pobj2d, list1, list2, 
						    id, g_style, g_caption);
	  lp_free (&list1);
	  list1 = list2;
	  lp_init (&list2);
	}
      if ((errcode) || rv == -1)
	{
	  lp_free (&list1);
	  lp_free (&list2);
	  return -1; /* error */
	}
      if (rv < 0)
	{
	  more_data_are_needed = 0;
	  errcode = add_next_2dsurface_from_arrays (pobj2d, list1, list2, 
						    id, g_style, g_caption);
	  lp_free (&list1);
	  lp_free (&list2);
	  if( (errcode) )
	    return -1; /* error */
	  else
	    return 0;  /* EoF reached after having successfully loaded an object */
	}
      /* only case left: rv == 0, i.e. Eof or empty line read before */
      /* any valid point could be read.                              */
      lp_free (&list1);
      lp_free (&list2);
      if ((more_data_are_needed))
	return -1; /* Error */
      else if ( (feof(fp)) )
	return 0;  /* EoF reached after having successfully loaded an object */
      else
	return 1;  /* Object successfully loaded, it is possible that another one follows */
    }
  else
    {
      /* dim == 3 */
      int errcode = 0;
      int rv = 0;

      while ( (rv = load_next_array (fp, &list2, dim)) > 0 && errcode == 0)
	{
	  more_data_are_needed = 0;
	  errcode = add_next_3dsurface_from_arrays (pobj3d, list1, list2, 
						    id, g_style, g_caption);
	  lp_free (&list1);
	  list1 = list2;
	  lp_init (&list2);
	}
      if ((errcode) || rv == -1)
	{
	  lp_free (&list1);
	  lp_free (&list2);
	  return -1; /* error */
	}
      if (rv < 0)
	{
	  more_data_are_needed = 0;
	  errcode = add_next_3dsurface_from_arrays (pobj3d, list1, list2, 
						    id, g_style, g_caption);
	  lp_free (&list1);
	  lp_free (&list2);
	  if( (errcode) )
	    return -1; /* error */
	  else
	    return 0;  /* EoF reached after having successfully loaded an object */
	}
      /* only case left: rv == 0, i.e. Eof or empty line read before */
      /* any valid point could be read.                              */
      lp_free (&list1);
      lp_free (&list2);
      if ((more_data_are_needed))
	return -1; /* Error */
      else if ( (feof(fp)) )
	return 0; /* EoF reached after having successfully loaded an object */
      else
	return 1; /* Object successfully loaded, it is possible that another one follows */
    }
}

static int flags_2d[NUMBER_OF_2DOBJECTS] = {0};
static int flags_3d[NUMBER_OF_3DOBJECTS] = {0};

void refresh_lists_available_ids(graphic_mode mode)
{
  /*
    Meaning of the flags: 1 == available, 0 == used.
  */
  int id;

  if (mode == GRAPHIC_2D)
    {
      int *f2d;

      f2d = get_free_ids (GRAPHIC_2D);
      for (id = 0; id < NUMBER_OF_2DOBJECTS; id++)
	flags_2d[id] = f2d[id];
    }
  if (mode == GRAPHIC_3D)
    {
      int *f3d;

      f3d = get_free_ids (GRAPHIC_3D);
      for (id = 0; id < NUMBER_OF_3DOBJECTS; id++)
	flags_3d[id] = f3d[id];      
    }
}

static
int get_first_available_id (void)
{
  /*
    Meaning of the flags: 1 == available, 0 == used.
  */
  int id;
  graphic_mode mode;
  
  if ( (mode = get_graphic_mode()) == GRAPHIC_2D )
    {
      for (id = 0; id < NUMBER_OF_2DOBJECTS && flags_2d[id] == 0; id++);
      if (id == NUMBER_OF_2DOBJECTS)
	return -1; /* No availiable id */
      else
	{
	  /* After using it, ID will not be available any longer */
	  flags_2d[id] = 0; 
	  return id;
	}
    }
  else
    {
      for (id = 0; id < NUMBER_OF_3DOBJECTS && flags_3d[id] == 0; id++);
      if (id == NUMBER_OF_3DOBJECTS)
	return -1; /* No availiable id */
      else
	{
	  /* After using it, ID will not be available any longer */
	  flags_3d[id] = 0;
	  return id;
	}
    }
}

static
void make_id_available (int id)
{
  if (id >= 0)
    {
      graphic_mode mode;

      if ( (mode = get_graphic_mode()) == GRAPHIC_2D )
	{
	  if (id < NUMBER_OF_2DOBJECTS)
	    flags_2d[id] = 1; 
	}
      else
	{
	  if (id < NUMBER_OF_3DOBJECTS)
	    flags_3d[id] = 1; 	  
	}
    }
}

void mark_all_ids_as_used (void)
{
  int id;

  for (id = 0; id < NUMBER_OF_2DOBJECTS; id++)
    flags_2d[id] = 0;
  for (id = 0; id < NUMBER_OF_3DOBJECTS; id++)
    flags_3d[id] = 0;      
}

int load_objects_from_file (const char* filepath, FILE* fp, 
			    gr2d_object* pobj2d, gr3d_object* pobj3d)
{
  grint id;
  int used_by_this_file[MAX_NUMBER_OF_OBJECTS] = {0};
  int dim, rv, check_dim;

  reinit_caption_and_style();
  if ( (id = get_first_available_id()) < 0 )
    return -2; /* too many objects in the file */

  /* We assume that the pipeline pointed to by POBJ2D (POBJ3D) */
  /* has been properly initialized                             */
  if ( (rv = load_object_from_file (fp, pobj2d, pobj3d, id, 1, &dim)) == -1 )
    {
      make_id_available (id); /* ID is not going to be used now */
      return -1;
    }
  else
    {
      /* rv is either 0 or 1 */
      check_dim = dim;
      save_object_caption (filepath, dim, id);
      used_by_this_file[id] = 1;
      while (rv > 0 && check_dim == dim)
	{
	  if ( (id = get_first_available_id()) >= 0 )
	    {
	      rv = load_object_from_file (fp, pobj2d, pobj3d, 
					  id, 0, &check_dim);
	      save_object_caption (filepath, dim, id);
	      used_by_this_file[id] = 1;
	    }
	  else
	    {
	      /* too many objects in the file */
	      for (id = 0; id < MAX_NUMBER_OF_OBJECTS; id++)
		{
		  if ( (used_by_this_file[id]) )
		    {
		      /* ID is not going to be used now */
		      make_id_available (id); 
		      dispose_object_caption (dim, id);
		    }
		}
	      return -2; 
	    }
	}
      if (rv != 0 || check_dim != dim || dim != get_graphic_mode())
	{
	  for (id = 0; id < MAX_NUMBER_OF_OBJECTS; id++)
	    {
	      if ( (used_by_this_file[id]) )
		{
		  /* ID is not going to be used now */
		  make_id_available (id); 
		  dispose_object_caption (dim, id);
		}
	    }
	  return -1; /* Error occurred */ 
	}
      else
	return dim; /* File successfully loaded: return the dimension of the loaded objects */
    }
}

#ifdef GNUPLOT_TEST

int main (int argc, char* argv[])
{
  FILE* fp;
  gr2d_object pipeline_2d;
  gr3d_object pipeline_3d;

  if (argc != 2)
    {
      fprintf (stderr, "Usage: %s FILE\n", argv[0]);
      return 1;
    }
  if ( !(fp = fopen (argv[1], "r")) )
    {
      fprintf (stderr, "Cannot open %s:\n", argv[1]);
      perror (NULL);
    }
  pipeline_2d = gr2d_new (0, 9);
  pipeline_3d = gr3d_new (0, 4);
  if ( load_objects_from_file(argv[1], fp, &pipeline_2d, &pipeline_3d) < 0 )
    {
      fprintf (stderr, "Error while loading data from %s\n\n", argv[1]);
      if (pipeline_2d.len > 0)
	gr2d_print (pipeline_2d, stdout);
      else
	gr3d_print (pipeline_3d, stdout);
      gr2d_delete (&pipeline_2d);
      gr3d_delete (&pipeline_3d);
      return 1;
    }
  else
    {
      printf ("File %s successfully loaded\n\n", argv[1]);
      if (pipeline_2d.len > 0)
	gr2d_print (pipeline_2d, stdout);
      else
	gr3d_print (pipeline_3d, stdout);
      gr2d_delete (&pipeline_2d);
      gr3d_delete (&pipeline_3d);
      return 0;
    }
}

#endif /*GNUPLOT_TEST */
