/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<stdlib.h>
#include<assert.h>
#include<math.h>
#include<string.h>
#include<xxcalc-3.2.0/evaluate.h>
#include"cairo_driver.h" /* for get_formatted_caption() */
#include"controls.h"
#include"gnuplot.h"
#include"utils.h"
#include"gui.h"

#define BUFF_SIZE             513

static graphic_mode gmode;

static struct {
  int current_object;
  par2d_object math_object[NUMBER_OF_2DOBJECTS];
  update_flag math_object_flag[NUMBER_OF_2DOBJECTS];
} math2d_input;

static struct {
  int current_object;
  par3d_object math_object[NUMBER_OF_3DOBJECTS];
  update_flag math_object_flag[NUMBER_OF_3DOBJECTS];
} math3d_input;

static struct {
  gr2d_point origin, O;
  double rotation_angle, zoom;
  int show_main_directions;
  int show_grid;
  int show_tics;
  int show_discontinuities;
  double R, grid_step;
  int basic_prop_to_update; /* 0 = no update needed, 1 = update needed */
} graphic2d_state;

static struct {
  gr3d_point origin, O;
  double longitude, latitude, rotation_angle;
  double zoom, smoke_density;
  int show_main_directions;
  int show_grid;
  int show_tics;
  int show_discontinuities;
  double R, grid_step;
  int basic_prop_to_update; /* 0 = no update needed, 1 = update needed */
} graphic3d_state;

static gr2d_object graphic_pipeline_2d, trashcan_2d;
static gr3d_object graphic_pipeline_3d, trashcan_3d;

const c_omplex c_zero = { 0.0, 0.0 };
const c_omplex c_pi = {MATH_PI, 0.0};
const c_omplex c_e = {MATH_E, 0.0};

void init_internal_structs (void)
{
  int i;
  const char* vnames[] = { "s", "t", "pi", "e", NULL };
  const char* cnames[] = { "pi", "e", NULL };
  c_omplex vvalues[] = { c_zero, c_zero, c_pi, c_e };
  c_omplex cvalues[] = { c_pi, c_e };

  /* First we initialize the structs of the XXCalc library */
  if ( !xx_mkstacks (1 + 2*NUMBER_OF_2DOBJECTS+3*NUMBER_OF_3DOBJECTS + 3) )
    {
      fputs ("WARNING!!! No memory available for new allocations,\n", stderr);
      fputs ("           the program is going to terminate now!!!\n", stderr);
      exit (EXIT_FAILURE);
    }
  if ( !xx_mklists(2) || 
       !xx_addvars (1, vnames, vvalues) ||
       !xx_addvars (0, cnames, cvalues) )
    {
      fputs ("WARNING!!! No memory available for new allocations,\n", stderr);
      fputs ("           the program is going to terminate now!!!\n", stderr);
      exit (EXIT_FAILURE);
    }
  xx_allow_implicit_multiplication ();

  /* Then we initialize the structures needed for the graphics */
  gmode = DEFAULT_GRAPHIC_MODE;
  math2d_input.current_object = DEFAULT_CURRENT_OBJECT;
  math3d_input.current_object = DEFAULT_CURRENT_OBJECT;
  for (i = 0; i < NUMBER_OF_2DOBJECTS; i++)
    {
      math2d_input.math_object_flag[i] = DEFAULT_UPDATE_FLAG;
      math2d_input.math_object[i] = par2d_new(2*i+1, 2*i+2);
      math2d_input.math_object[i].color = i;
      math2d_input.math_object[i].inf_s = math2d_input.math_object[i].inf_t = DEFAULT_INF;
      math2d_input.math_object[i].sup_s = math2d_input.math_object[i].sup_t = DEFAULT_SUP;
      math2d_input.math_object[i].nsteps_s = 0;
      math2d_input.math_object[i].nsteps_t = DEFAULT_NSTEPS; 
    }
  for (i = 0; i < NUMBER_OF_3DOBJECTS; i++)
    {
      math3d_input.math_object_flag[i] = DEFAULT_UPDATE_FLAG;
      math3d_input.math_object[i] = par3d_new(2*NUMBER_OF_2DOBJECTS+3*i+1,
					      2*NUMBER_OF_2DOBJECTS+3*i+2,
					      2*NUMBER_OF_2DOBJECTS+3*i+3);     
      math3d_input.math_object[i].color = i;
      math3d_input.math_object[i].inf_s = math3d_input.math_object[i].inf_t = DEFAULT_INF;
      math3d_input.math_object[i].sup_s = math3d_input.math_object[i].sup_t = DEFAULT_SUP;
      math3d_input.math_object[i].nsteps_s = math3d_input.math_object[i].nsteps_t = DEFAULT_NSTEPS_ST;
    }

  graphic2d_state.O = graphic2d_state.origin = DEFAULT_ORIGIN_2D;
  graphic2d_state.rotation_angle = DEFAULT_ROTATION_ANGLE * CONV_FACTOR_D2R;
  graphic2d_state.zoom = DEFAULT_ZOOM;
  graphic2d_state.show_main_directions = DEFAULT_SHOW_MAIN_DIRS;
  graphic2d_state.show_grid = DEFAULT_SHOW_GRID;
  graphic2d_state.show_tics = DEFAULT_SHOW_TICS;
  graphic2d_state.show_discontinuities = DEFAULT_SHOW_DISC;
  graphic2d_state.R = DEFAULT_R;
  graphic2d_state.grid_step = DEFAULT_GRID_STEP;
  graphic2d_state.basic_prop_to_update = 0;

  graphic3d_state.O = graphic3d_state.origin = DEFAULT_ORIGIN_3D;
  graphic3d_state.longitude = DEFAULT_LONGITUDE * CONV_FACTOR_D2R;
  graphic3d_state.latitude = DEFAULT_LATITUDE * CONV_FACTOR_D2R;
  graphic3d_state.rotation_angle = DEFAULT_ROTATION_ANGLE * CONV_FACTOR_D2R;
  graphic3d_state.zoom = DEFAULT_ZOOM;
  graphic3d_state.smoke_density = DEFAULT_SMOKE_DENSITY;
  graphic3d_state.show_main_directions = DEFAULT_SHOW_MAIN_DIRS;
  graphic3d_state.show_grid = DEFAULT_SHOW_GRID;
  graphic3d_state.show_tics = DEFAULT_SHOW_TICS;
  graphic3d_state.show_discontinuities = DEFAULT_SHOW_DISC;
  graphic3d_state.R = DEFAULT_R;
  graphic3d_state.grid_step = DEFAULT_GRID_STEP;
  graphic3d_state.basic_prop_to_update = 0;

  graphic_pipeline_2d =gr2d_new(0, NUMBER_OF_2DOBJECTS-1);
  trashcan_2d = gr2d_new(0, NUMBER_OF_2DOBJECTS - 1);
  graphic_pipeline_3d =gr3d_new(0, NUMBER_OF_3DOBJECTS -1);
  trashcan_3d = gr3d_new(0, NUMBER_OF_3DOBJECTS - 1);
}

void reset_internal_structs (void)
{
  int i;

  xx_rmstacks ();
  xx_rmlists ();
  gmode = DEFAULT_GRAPHIC_MODE;
  math2d_input.current_object = DEFAULT_CURRENT_OBJECT;
  math3d_input.current_object = DEFAULT_CURRENT_OBJECT;
  for (i = 0; i < NUMBER_OF_2DOBJECTS; i++)
    {
      math2d_input.math_object_flag[i] = DEFAULT_UPDATE_FLAG;
      par2d_clear (&math2d_input.math_object[i]);
      math2d_input.math_object[i].color = i;
      math2d_input.math_object[i].inf_s = math2d_input.math_object[i].inf_t = DEFAULT_INF;
      math2d_input.math_object[i].sup_s = math2d_input.math_object[i].sup_t = DEFAULT_SUP;
      math2d_input.math_object[i].nsteps_s = 0;
      math2d_input.math_object[i].nsteps_t = DEFAULT_NSTEPS; 
    }
  for (i = 0; i < NUMBER_OF_3DOBJECTS; i++)
    {
      math3d_input.math_object_flag[i] = DEFAULT_UPDATE_FLAG;
      par3d_clear (&math3d_input.math_object[i]);
      math3d_input.math_object[i].color = i;
      math3d_input.math_object[i].inf_s = math3d_input.math_object[i].inf_t = DEFAULT_INF;
      math3d_input.math_object[i].sup_s = math3d_input.math_object[i].sup_t = DEFAULT_SUP;
      math3d_input.math_object[i].nsteps_s = math3d_input.math_object[i].nsteps_t = DEFAULT_NSTEPS_ST;
    }

  graphic2d_state.O = graphic2d_state.origin = DEFAULT_ORIGIN_2D;
  graphic2d_state.rotation_angle = DEFAULT_ROTATION_ANGLE * CONV_FACTOR_D2R;
  graphic2d_state.zoom = DEFAULT_ZOOM;
  graphic2d_state.show_main_directions = DEFAULT_SHOW_MAIN_DIRS;
  graphic2d_state.show_grid = DEFAULT_SHOW_GRID;
  graphic2d_state.show_tics = DEFAULT_SHOW_TICS;
  graphic2d_state.show_discontinuities = DEFAULT_SHOW_DISC;
  graphic2d_state.R = DEFAULT_R;
  graphic2d_state.grid_step = DEFAULT_GRID_STEP;
  graphic2d_state.basic_prop_to_update = 0;

  graphic3d_state.O = graphic3d_state.origin = DEFAULT_ORIGIN_3D;
  graphic3d_state.longitude = DEFAULT_LONGITUDE * CONV_FACTOR_D2R;
  graphic3d_state.latitude = DEFAULT_LATITUDE * CONV_FACTOR_D2R;
  graphic3d_state.rotation_angle = DEFAULT_ROTATION_ANGLE * CONV_FACTOR_D2R;
  graphic3d_state.zoom = DEFAULT_ZOOM;
  graphic3d_state.smoke_density = DEFAULT_SMOKE_DENSITY;
  graphic3d_state.show_main_directions = DEFAULT_SHOW_MAIN_DIRS;
  graphic3d_state.show_grid = DEFAULT_SHOW_GRID;
  graphic3d_state.show_tics = DEFAULT_SHOW_TICS;
  graphic3d_state.show_discontinuities = DEFAULT_SHOW_DISC;
  graphic3d_state.R = DEFAULT_R;
  graphic3d_state.grid_step = DEFAULT_GRID_STEP;
  graphic3d_state.basic_prop_to_update = 0;

  gr2d_delete (&graphic_pipeline_2d);
  gr2d_delete (&trashcan_2d);
  gr3d_delete (&graphic_pipeline_3d);
  gr3d_delete (&trashcan_3d);
}

void set_graphic_mode (graphic_mode gm)
{
  gmode = gm;
}

graphic_mode get_graphic_mode (void)
{
  return gmode;
}

void set_current_object (int id)
{
  if ( gmode == GRAPHIC_2D )
    {
      assert (id >= 0 && id < NUMBER_OF_2DOBJECTS);
      math2d_input.current_object = id;
    }
  else
    {
      assert (id >= 0 && id < NUMBER_OF_3DOBJECTS);
      math3d_input.current_object = id;
    }
}

int get_current_object (void)
{
  return (gmode == GRAPHIC_2D ? math2d_input.current_object : math3d_input.current_object);
}

int* get_free_ids (graphic_mode for_which_graphic_mode)
{
  static int flags_2d[NUMBER_OF_2DOBJECTS] = {0};
  static int flags_3d[NUMBER_OF_3DOBJECTS] = {0};
  int id;

  if (for_which_graphic_mode == GRAPHIC_2D)
    {
      for (id = 0; id < NUMBER_OF_2DOBJECTS; id++)
	{
	  if ( !math2d_input.math_object[id].x_eq &&
	       !math2d_input.math_object[id].y_eq )
	    flags_2d[id] = 1; /* not used */
	  else
	    flags_2d[id] = 0; /* used */
	}
      return flags_2d;
    }
  else
    {
      for (id = 0; id < NUMBER_OF_3DOBJECTS; id++)
	{
	  if ( !math3d_input.math_object[id].x_eq &&
	       !math3d_input.math_object[id].y_eq &&
	       !math3d_input.math_object[id].z_eq )
	    flags_3d[id] = 1; /* not used */
	  else
	    flags_3d[id] = 0; /* used */
	}
      return flags_3d;
    }
}

gchar* get_formatted_captions (void)
{
  gchar *mgr_caption = NULL, *legend = NULL;
  grchar* caption_2d[NUMBER_OF_2DOBJECTS+1] = {NULL}; /* +1 is needed by g_strjoinv() */
  grchar* caption_3d[NUMBER_OF_3DOBJECTS+1] = {NULL}; /* +1 is needed by g_strjoinv() */
  int id;

  /* First check if the plotted data comes from a MGR file */
  if ( *(mgr_caption = (gchar*) mgr_filename (NULL)) != '\0' )
    {
      /* 
	 If yes, then you should just format the MGR_CAPTION,
	 which points to the path of the MGR file.
      */
      legend = wrap_string (mgr_caption, 0);
    }
  else if (gmode == GRAPHIC_2D)
    {
      for (id = 0; id < NUMBER_OF_2DOBJECTS; id++)
	{
	  caption_2d[id] = (char*) get_object_caption (gmode, id);
	  if ( caption_2d[id] == NULL )
	    caption_2d[id] = math2d_input.math_object[id].caption;
	  caption_2d[id] = get_formatted_caption (caption_2d[id], gmode, id); 
	  if (!caption_2d[id])
	    {
	      /*
		Not enough memory!
	      */
	      for (id--; id >= 0; id--)
		g_free (caption_2d[id]);
	      return NULL;
	    }
	}
      legend = g_strjoinv ("\n\n", caption_2d);
    }
  else
    {
      /* gmode == GRAPHIC_3D*/
      for (id = 0; id < NUMBER_OF_3DOBJECTS; id++)
	{
	  caption_3d[id] = (char*) get_object_caption (gmode, id);
	  if ( caption_3d[id] == NULL )
	    caption_3d[id] = math3d_input.math_object[id].caption;	
	  caption_3d[id] = get_formatted_caption (caption_3d[id], gmode, id); 
	  if (!caption_3d[id])
	    {
	      /*
		Not enough memory!
	      */
	      for (id--; id >= 0; id--)
		g_free (caption_3d[id]);
	      return NULL;
	    }
	}
      legend = g_strjoinv ("\n\n", caption_3d);
    }
  return legend;
}

static
repr_type get_representation_type (int id)
{
  grchar *x_eq;

  if (gmode == GRAPHIC_2D)
    {
      grmathtype type;

      type = math2d_input.math_object[id].type;
      x_eq = math2d_input.math_object[id].x_eq;
      if ( type == GRMATH_CURVE && (no_other_char_in_string(x_eq, 't')) )
	return CARTESIAN;
      else
	return PARAMETRIC;
    }
  else
    {
      grchar *y_eq;

      x_eq = math3d_input.math_object[id].x_eq;
      y_eq = math3d_input.math_object[id].y_eq;
      if ( (no_other_char_in_string(x_eq, 's')) &&
	   (no_other_char_in_string(y_eq, 't')) )
	return CARTESIAN;
      else
	return PARAMETRIC;
    }
}

static
int is_definition_empty (const char* x_eq, const char* y_eq,
			 const char* z_eq)
{
  int rx, ry, rz;

  rx = is_string_blank (x_eq);
  ry = is_string_blank (y_eq);
  if (gmode == GRAPHIC_2D)
    return ((rx) && (ry) ? 1 : 0);
  else
    {
      rz = is_string_blank (z_eq);      
      return ((rx) && (ry) && (rz) ? 1 : 0);
    }
}

grexitstatus define_math_object (const char* x_eq, const char* y_eq,
				 const char* z_eq,
				 const char* inf_s, const char* sup_s, 
				 const char* nsteps_s,
				 const char* inf_t, const char* sup_t, 
				 const char* nsteps_t,
				 grstyle style, const char* caption,
				 int cartesian_mode_is_active)
{
  grint id;
  double Inf_s, Sup_s, Inf_t, Sup_t;
  gruint Nsteps_s, Nsteps_t;
  update_flag uf = UP_TO_DATE;
  const char* errmsg;
  grexitstatus rv;

  if ( gstrcmp ((const grchar*)inf_s, NULL) == 0 )
    Inf_s = 0.0;
  else
    {
      if ( (errmsg = str_2_double ((const grchar*)inf_s, &Inf_s)) )
	{
	  uf = WRONG_UPDATE;
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("y, Min: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("s, Min: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	}
    }

  if ( gstrcmp ((const grchar*)sup_s, NULL) == 0 )
    Sup_s = 0.0;
  else
    {
      if ( (errmsg = str_2_double ((const grchar*)sup_s, &Sup_s)) )
	{
	  uf = WRONG_UPDATE;
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("y, Max: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("s, Max: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	}
    }

  if ( gstrcmp ((const grchar*)inf_t, NULL) == 0 )
    Inf_t = 0.0;
  else
    {
      if ( (errmsg = str_2_double ((const grchar*)inf_t, &Inf_t)) )
	{
	  uf = WRONG_UPDATE;
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("x, Min: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("t, Min: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  return GR_FAILURE;
	}
    }

  if ( gstrcmp ((const grchar*)sup_t, NULL) == 0 )
    Sup_t = 0.0;
  else
    {
      if ( (errmsg = str_2_double ((const grchar*)sup_t, &Sup_t)) )
	{
	  uf = WRONG_UPDATE;
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("x, Max: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("t, Max: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	}
    }

  if ( gstrcmp ((const grchar*)nsteps_s, NULL) == 0 )
    Nsteps_s = 0;
  else
    {
      if ( str_2_ulng ((const grchar*)nsteps_s, &Nsteps_s) != STR_2_OK )
	{
	  uf = WRONG_UPDATE;
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("y, Number of subintervals: ", 
				 "Invalid number",
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("s, Number of subintervals: ", 
				 "Invalid number",
				 "\n\nA correction is required to proceed.");
	    }
	}
    }

  if ( gstrcmp ((const grchar*)nsteps_t, NULL) == 0 )
    Nsteps_t = 0;
  else
    {
      if ( str_2_ulng ((const grchar*)nsteps_t, &Nsteps_t) != STR_2_OK )
	{
	  uf = WRONG_UPDATE;
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("x, Number of subintervals: ", 
				 "Invalid number",
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("t, Number of subintervals: ", 
				 "Invalid number",
				 "\n\nA correction is required to proceed.");
	    }
	}
    }

  if ( gmode == GRAPHIC_2D )
    {
      id = math2d_input.current_object;
      if (uf == WRONG_UPDATE)
	{
	  math2d_input.math_object_flag[id] |= UP_TO_DATE;
	  return GR_FAILURE;
	}
      if ( (is_definition_empty(x_eq, y_eq, NULL)) )
	{
	  graphic2d_state.basic_prop_to_update = 1;
	  math2d_input.math_object_flag[id] = TO_UPDATE;
	  par2d_clear (&math2d_input.math_object[id]);
	  math2d_input.math_object[id].type = GRMATH_EMPTY;
	  math2d_input.math_object[id].color = id;
	  math2d_input.math_object[id].inf_s = 
	    math2d_input.math_object[id].inf_t = DEFAULT_INF;
	  math2d_input.math_object[id].sup_s = 
	    math2d_input.math_object[id].sup_t = DEFAULT_SUP;
	  math2d_input.math_object[id].nsteps_s = 0;
	  math2d_input.math_object[id].nsteps_t = DEFAULT_NSTEPS;
	  rv = GR_OK;
	}
      else
	{
	  rv = par2d_define (&math2d_input.math_object[id],
			     (grchar*)x_eq, (grchar*) y_eq,
			     Inf_s, Sup_s, Nsteps_s,
			     Inf_t, Sup_t, Nsteps_t,
			     style, id, (grchar*) caption);
	  if (rv == GR_NO_ACTION)
	    {
	      math2d_input.math_object_flag[id] |= UP_TO_DATE;
	      if (math2d_input.math_object_flag[id] == WRONG_UPDATE)
		rv = GR_FAILURE;
	    }
	  else if (rv == GR_FAILURE)
	    math2d_input.math_object_flag[id] = WRONG_UPDATE;
	  else
	    math2d_input.math_object_flag[id] = TO_UPDATE;
	}
      return rv;
    }
  else
    {
      /* gmode == GRAPHIC_3D */
      id = math3d_input.current_object;
      if (uf == WRONG_UPDATE)
	{
	  math3d_input.math_object_flag[id] |= UP_TO_DATE;
	  return GR_FAILURE;
	}

      if ( (is_definition_empty(x_eq, y_eq, z_eq)) )
	{
	  graphic3d_state.basic_prop_to_update = 1;
	  math3d_input.math_object_flag[id] = TO_UPDATE;
	  par3d_clear (&math3d_input.math_object[id]);
	  math3d_input.math_object[id].type = GRMATH_EMPTY;
	  math3d_input.math_object[id].color = id;
	  math3d_input.math_object[id].inf_s = 
	    math3d_input.math_object[id].inf_t = DEFAULT_INF;
	  math3d_input.math_object[id].sup_s = 
	    math3d_input.math_object[id].sup_t = DEFAULT_SUP;
	  math3d_input.math_object[id].nsteps_s = 
	    math3d_input.math_object[id].nsteps_t = DEFAULT_NSTEPS_ST;	  
	  rv = GR_OK;
	}
      else
	{
	  rv = par3d_define (&math3d_input.math_object[id],
			     (grchar*)x_eq, (grchar*) y_eq, (grchar*) z_eq,
			     Inf_s, Sup_s, Nsteps_s,
			     Inf_t, Sup_t, Nsteps_t,
			     style, id, (grchar*) caption);
	  if (rv == GR_NO_ACTION)
	    {
	      math3d_input.math_object_flag[id] |= UP_TO_DATE;
	      if (math3d_input.math_object_flag[id] == WRONG_UPDATE)
		rv = GR_FAILURE;
	    }
	  else if (rv == GR_FAILURE)
	    math3d_input.math_object_flag[id] = WRONG_UPDATE;
	  else
	    math3d_input.math_object_flag[id] = TO_UPDATE;
	}
      return rv;
    }
}


static char inf_s_buff[BUFF_SIZE], sup_s_buff[BUFF_SIZE], nsteps_s_buff[BUFF_SIZE];
static char inf_t_buff[BUFF_SIZE], sup_t_buff[BUFF_SIZE], nsteps_t_buff[BUFF_SIZE];

static const char empty_buff[1] = "";

mathobj_description get_description_2dmath_object (const par2d_object* p2d)
{
  mathobj_description description;

  description.x_eq = (p2d->x_eq != NULL ? (const char*) p2d->x_eq : empty_buff);
  description.y_eq = (p2d->y_eq != NULL ? (const char*) p2d->y_eq : empty_buff);
  description.z_eq = (const char*) empty_buff;
  sprintf (inf_s_buff, FORMAT_STRING_DBL, p2d->inf_s);
  description.inf_s = (const char*) inf_s_buff;
  sprintf (sup_s_buff, FORMAT_STRING_DBL, p2d->sup_s);
  description.sup_s = (const char*) sup_s_buff;
  sprintf (nsteps_s_buff, FORMAT_STRING_UINT, p2d->nsteps_s);
  description.nsteps_s = (const char*) nsteps_s_buff;
  sprintf (inf_t_buff, FORMAT_STRING_DBL, p2d->inf_t);
  description.inf_t = (const char*) inf_t_buff;
  sprintf (sup_t_buff, FORMAT_STRING_DBL, p2d->sup_t);
  description.sup_t = (const char*) sup_t_buff;
  sprintf (nsteps_t_buff, FORMAT_STRING_UINT, p2d->nsteps_t);
  description.nsteps_t = (const char*) nsteps_t_buff;
  description.caption = (p2d->caption != NULL ? (const char*) p2d->caption : empty_buff);
  description.style = p2d->style;
  description.type = p2d->type;
  return description;
}

mathobj_description get_description_3dmath_object (const par3d_object* p3d)
{
  mathobj_description description;

  description.x_eq = (p3d->x_eq != NULL ? (const char*) p3d->x_eq : empty_buff);
  description.y_eq = (p3d->y_eq != NULL ? (const char*) p3d->y_eq : empty_buff);
  description.z_eq = (p3d->z_eq != NULL ? (const char*) p3d->z_eq : empty_buff);
  sprintf (inf_s_buff, FORMAT_STRING_DBL, p3d->inf_s);
  description.inf_s = (const char*) inf_s_buff;
  sprintf (sup_s_buff, FORMAT_STRING_DBL, p3d->sup_s);
  description.sup_s = (const char*) sup_s_buff;
  sprintf (nsteps_s_buff, FORMAT_STRING_UINT, p3d->nsteps_s);
  description.nsteps_s = (const char*) nsteps_s_buff;
  sprintf (inf_t_buff, FORMAT_STRING_DBL, p3d->inf_t);
  description.inf_t = (const char*) inf_t_buff;
  sprintf (sup_t_buff, FORMAT_STRING_DBL, p3d->sup_t);
  description.sup_t = (const char*) sup_t_buff;
  sprintf (nsteps_t_buff, FORMAT_STRING_UINT, p3d->nsteps_t);
  description.nsteps_t = (const char*) nsteps_t_buff;
  description.caption = (p3d->caption != NULL ? (const char*) p3d->caption : empty_buff);
  description.style = p3d->style;
  description.type = p3d->type;
  return description;
}

mathobj_description retrieve_math_object (void)
{
  grint id;
  const par2d_object *p2d;
  const par3d_object *p3d;

  if ( gmode == GRAPHIC_2D )
    {
      id = math2d_input.current_object;
      p2d = &math2d_input.math_object[id];
      return get_description_2dmath_object (p2d);
    }
  else
    {
      /* gmode == GRAPHIC_3D */
      id = math3d_input.current_object;
      p3d = &math3d_input.math_object[id];
      return get_description_3dmath_object (p3d);
    }
}

static double adjust_grid_step (double grid_step)
{
  int i;
  double min_grid_step = 2 * pow (10.0, MIN_EXP);
  double max_grid_step = pow (10.0, MAX_EXP) * 0.5;

  if (grid_step < min_grid_step || grid_step > max_grid_step)
    return 0.0; /* Error: we are working with a too small/large intervall */
  if (grid_step < 1.0)
    {
      for (i = 1; grid_step * pow(10.0, i) < 1; i++);
      if (grid_step * pow(10.0, i) >= 5.0)
	return 5.0 * pow(10.0, -i);
      else
	return pow(10.0, -i);
    }
  else
    {
      for (i = -1; grid_step * pow(10.0, i) >= 1; i--);
      if (grid_step * pow(10.0, i) >= 0.5)
	return 0.5 * pow(10.0, -i);
      else
	return 0.1 * pow(10.0, -i);
    }
}

static void set_basic_2d_graphic_state (void)
{
  double adj_grid_step, L;

  if ( (graphic2d_state.basic_prop_to_update) )
    {
      graphic2d_state.O = graphic2d_state.origin = gr2d_get_middle_point (graphic_pipeline_2d); 
      L = gr2d_get_mean_length (graphic_pipeline_2d, TRUNC_FACTOR);
      gr2d_compute_bounding_circle (&graphic_pipeline_2d, TRUNC_FACTOR*L);
      graphic2d_state.R = graphic_pipeline_2d.R * DEFAULT_R_MAG_FACTOR;
      graphic2d_state.grid_step = (100.0 * graphic2d_state.R) / (DEFAULT_DIVISOR_2D * graphic2d_state.zoom);
      /* assert ( (adj_grid_step =  adjust_grid_step(graphic2d_state.grid_step)) != 0.0 ); */
      adj_grid_step =  adjust_grid_step(graphic2d_state.grid_step);
      graphic2d_state.grid_step = adj_grid_step;
      graphic2d_state.basic_prop_to_update = 0; /* Update just made */
    }
}

static void erase_main_directions_2d (void)
{
  gr2d_remove_items_with_color (&graphic_pipeline_2d, X_COLOR, -1.0, NULL);
  gr2d_remove_items_with_color (&graphic_pipeline_2d, Y_COLOR, -1.0, NULL);
}

static void draw_main_directions_2d (void)
{
  int i, nsteps = DEFAULT_NSTEPS;
  double R, Ox, Oy;

  /*
    If GRAPHIC2D_STATE.ZOOM > 100.0, then this function must be called
    after each translation of GRAPHIC2D_STATE.O before recomputing
    the user-defined coordinates.
  */
  Ox = graphic2d_state.O.x;
  Oy = graphic2d_state.O.y;

#ifdef _DEBUG1_
  fprintf (stderr, "graphic state:\n  origin = (%f, %f)\n",
	   graphic2d_state.origin.x,
	   graphic2d_state.origin.y);
  fprintf (stderr, "  O = (%f, %f)\n",
	   graphic2d_state.O.x,
	   graphic2d_state.O.y);
  fprintf (stderr, "  rotation angle = %f\n",
	   graphic2d_state.rotation_angle);
  fprintf (stderr, "  zoom = %f\n",
	   graphic2d_state.zoom);	   
  fprintf (stderr, "  show main dirs = %d\n",
	   graphic2d_state.show_main_directions);
  fprintf (stderr, "  show grid = %d\n",
	   graphic2d_state.show_grid);
  fprintf (stderr, "  show tics = %d\n",
	   graphic2d_state.show_tics);
  fprintf (stderr, "  show discontinuities = %d\n",
	   graphic2d_state.show_discontinuities);
  fprintf (stderr, "  R = %f\n",
	   graphic2d_state.R);
  fprintf (stderr, "  grid step = %f\n",
	   graphic2d_state.grid_step);	     
#endif

  R = graphic2d_state.R;
  for (i = 0; i < nsteps; i++)
    {
      gr2d_add_segment (&graphic_pipeline_2d,
			X_COLOR, NULL,
			Ox - R + (2*R/nsteps)*i, Oy,
			Ox - R + (2*R/nsteps)*(i+1), Oy);
      gr2d_add_segment (&graphic_pipeline_2d,
			Y_COLOR, NULL,
			Ox, Oy -R + (2*R/nsteps)*i,
			Ox, Oy -R + (2*R/nsteps)*(i+1));
    }
  gr2d_add_point (&graphic_pipeline_2d,
		  X_COLOR, "X",
		  Ox + 0.95 * R, Oy - 0.05 * R);
  gr2d_add_point (&graphic_pipeline_2d,
		  Y_COLOR, "Y",
		  Ox - 0.05 * R, Oy + 0.95 * R);
#ifdef _DEBUG1_
  fprintf (stderr, "\nThe graphic pipeline contains:\n");
  (void)gr2d_print (graphic_pipeline_2d, stderr);
#endif

  gr2d_set_depth (&graphic_pipeline_2d, X_COLOR, 0);
  gr2d_set_depth (&graphic_pipeline_2d, Y_COLOR, 1);

#ifdef _DEBUG1_
  fprintf (stderr, "\nAfter setting the depth, the graphic pipeline contains:\n");
  (void)gr2d_print (graphic_pipeline_2d, stderr);
#endif
}

static void erase_2d_grid (void)
{
  gr2d_remove_items_with_color (&graphic_pipeline_2d, GRID_COLOR, -1.0, NULL);
}

/* 
   Round x to the nearest integer 
   not larger in absolute value.  

   This function works like trunc(), but it
   supports also compilers which are not C99-compliant.
*/
static double
r_trunc (double x)
{
  if (x >= 0.0)
    return floor (x);
  else
    return ceil (x);
}

static void draw_2d_grid (void)
{
  int i, nsteps;
  double adj_grid_step = graphic2d_state.grid_step;
  double zf, Ox, Oy;

  /*
    If GRAPHIC2D_STATE.ZOOM > 100.0, then this function must be called
    after each translation of GRAPHIC2D_STATE.O before recomputing
    the user-defined coordinates.
  */
  Ox = graphic2d_state.O.x;
  Oy = graphic2d_state.O.y;
  zf = graphic2d_state.zoom > 100.0 ? 100.0 / graphic2d_state.zoom : 1.0;

  nsteps = r_trunc(zf * graphic2d_state.R / adj_grid_step);
  for (i = -nsteps; i <= nsteps; i++)
    {
      gr2d_add_segment (&graphic_pipeline_2d,
			GRID_COLOR, NULL,
			Ox + i * adj_grid_step, Oy - graphic2d_state.R,
			Ox + i * adj_grid_step, Oy + graphic2d_state.R);
      gr2d_add_segment (&graphic_pipeline_2d,
			GRID_COLOR, NULL,
			Ox - graphic2d_state.R, Oy + i * adj_grid_step,
			Ox + graphic2d_state.R, Oy + i * adj_grid_step);
    }
  gr2d_set_depth (&graphic_pipeline_2d, GRID_COLOR, -1);	  
}

static void erase_2d_tics (void)
{
  gr2d_remove_items_with_color (&graphic_pipeline_2d, TICS_COLOR, -1.0, NULL);
}

static void draw_2d_tics (void)
{
  int i, nsteps, interval;
  double tics_coord, adj_grid_step = graphic2d_state.grid_step;
  grchar caption[LABEL_SIZE];
  double x_dev_factor, y_dev_factor, common_dev_factor, zf;
  double Ox, Oy;

  /*
    If GRAPHIC2D_STATE.ZOOM > 100.0, then this function must be called
    after each translation of GRAPHIC2D_STATE.O before recomputing
    the user-defined coordinates.
  */
  Ox = graphic2d_state.O.x;
  Oy = graphic2d_state.O.y;
  zf = graphic2d_state.zoom > 100.0 ? 100.0 / graphic2d_state.zoom : 1.0;
  x_dev_factor = 1.15 * zf;  
  y_dev_factor = 1.10 * zf;  
  common_dev_factor = 1.025 * zf;
  nsteps = r_trunc(zf * graphic2d_state.R / adj_grid_step);
  /* First we draw the tics on the x-axis... */
  interval =  nsteps >= DEFAULT_TIC_DIVISOR ? nsteps / DEFAULT_TIC_DIVISOR : 1;
  for (i = interval; i <= nsteps; i += interval)
    {
      tics_coord = smart_round (Ox + i * adj_grid_step, adj_grid_step, 1);
      sprintf (caption, STRING_FMT_TICS, tics_coord);
      gr2d_add_point (&graphic_pipeline_2d,
		      TICS_COLOR, caption,
		      Ox + i * adj_grid_step,
		      Oy - graphic2d_state.R * y_dev_factor);
      gr2d_add_point (&graphic_pipeline_2d,
		      TICS_COLOR, NULL,
		      Ox + i * adj_grid_step,
		      Oy - graphic2d_state.R * common_dev_factor);
    }
  for (i = 0; i >= -nsteps; i -= interval)
    {
      tics_coord = smart_round (Ox + i * adj_grid_step, adj_grid_step, 1);
      sprintf (caption, STRING_FMT_TICS, tics_coord);
      gr2d_add_point (&graphic_pipeline_2d,
		      TICS_COLOR, caption,
		      Ox + i * adj_grid_step,
		      Oy - graphic2d_state.R * y_dev_factor);
      gr2d_add_point (&graphic_pipeline_2d,
		      TICS_COLOR, NULL,
		      Ox + i * adj_grid_step,
		      Oy - graphic2d_state.R * common_dev_factor);
    }
  /* then the tics on the y-axis */
  interval =  nsteps >= DEFAULT_YTIC_DIVISOR ? nsteps / DEFAULT_YTIC_DIVISOR : 1;
  for (i = interval; i <= nsteps; i += interval)
    {
      tics_coord = smart_round (Oy + i * adj_grid_step, adj_grid_step, 1);
      sprintf (caption, STRING_FMT_TICS, tics_coord);
      gr2d_add_point (&graphic_pipeline_2d,
		      TICS_COLOR, caption,
		      Ox - graphic2d_state.R * x_dev_factor,
		      Oy + i * adj_grid_step);
      gr2d_add_point (&graphic_pipeline_2d,
		      TICS_COLOR, NULL,
		      Ox - graphic2d_state.R * common_dev_factor,
		      Oy + i * adj_grid_step);
    }
  for (i = 0; i >= -nsteps; i -= interval)
    {
      tics_coord = smart_round (Oy + i * adj_grid_step, adj_grid_step, 1);
      sprintf (caption, STRING_FMT_TICS, tics_coord);
      gr2d_add_point (&graphic_pipeline_2d,
		      TICS_COLOR, caption,
		      Ox - graphic2d_state.R * x_dev_factor,
		      Oy + i * adj_grid_step);
      gr2d_add_point (&graphic_pipeline_2d,
		      TICS_COLOR, NULL,
		      Ox - graphic2d_state.R * common_dev_factor,
		      Oy + i * adj_grid_step);
    }
  gr2d_set_depth (&graphic_pipeline_2d, TICS_COLOR, TICS_COLOR);	  
}

static void remove_discontinuities_from_2dmathobj (int id)
{
  double L;

  L = gr2d_get_mean_length_by_color (graphic_pipeline_2d, id);
  if ( gr2d_remove_items_with_color (&graphic_pipeline_2d, id, DISC_FACTOR * L, &trashcan_2d) < 0 )
    {
      handle_fatal_error ("The program has not enough memory to keep on working\n" \
                          "please quit now.\n");
    }
}

static void ignore_discontinuities_2d (void)
{
  gr2d_add_object (&graphic_pipeline_2d, trashcan_2d);
  gr2d_delete (&trashcan_2d);
  trashcan_2d = gr2d_new(0, NUMBER_OF_2DOBJECTS - 1);
} 

static void compute_local_2dcoordinates (int after_loading_from_file, int graphic_state_changed)
{
  int id;
    
  if ((after_loading_from_file))
    {
      gr2d_compute_userdef_coordinates (&graphic_pipeline_2d,
                                        graphic2d_state.O,
                                        graphic2d_state.rotation_angle);   
      /* After loading an image from file, all mathematical objects
         must be re-evaluated                                       */      
      for (id = 0; id < NUMBER_OF_2DOBJECTS; id++)
        math2d_input.math_object_flag[id] = TO_UPDATE;
    }
  else
    {
      if ((graphic_state_changed))
	{
	  for (id = 0; id < NUMBER_OF_2DOBJECTS; id++)
	    {
	      gr2d_partial_upgrade (&graphic_pipeline_2d, id, 
				    graphic2d_state.O,
				    graphic2d_state.rotation_angle);
	      math2d_input.math_object_flag[id] = UP_TO_DATE;
	    }
	  if ( (graphic2d_state.show_main_directions) )
	    {
	      gr2d_partial_upgrade (&graphic_pipeline_2d, X_COLOR, 
				    graphic2d_state.O,
				    graphic2d_state.rotation_angle);
	      gr2d_partial_upgrade (&graphic_pipeline_2d, Y_COLOR, 
				    graphic2d_state.O,
				    graphic2d_state.rotation_angle);
	      gr2d_partial_upgrade (&graphic_pipeline_2d, Z_COLOR, 
				    graphic2d_state.O,
				    graphic2d_state.rotation_angle);
	    }      
	  if ( (graphic2d_state.show_grid) )
	    {
	      gr2d_partial_upgrade (&graphic_pipeline_2d, GRID_COLOR, 
				    graphic2d_state.O,
				    graphic2d_state.rotation_angle);
	    }
	  if ( (graphic2d_state.show_tics) )
	    {
	      gr2d_partial_upgrade (&graphic_pipeline_2d, TICS_COLOR, 
				    graphic2d_state.O,
				    graphic2d_state.rotation_angle);
	    }
	} /* End if ((graphic_state_changed)) */
    }
}

static grexitstatus upgrade_2dpipeline (void)
{
  int id, update_graphic_state = graphic2d_state.basic_prop_to_update;
  update_flag uf;

  gr2d_delete (&trashcan_2d);
  trashcan_2d = gr2d_new(0, NUMBER_OF_2DOBJECTS - 1);
  for (id = 0; id < NUMBER_OF_2DOBJECTS; id++)
    {
      uf = math2d_input.math_object_flag[id];
      if (uf == WRONG_UPDATE)
	return GR_FAILURE; /* Out of memory */
      if (uf == TO_UPDATE)
	{
	  update_graphic_state = 1;
	  gr2d_remove_items_with_color(&graphic_pipeline_2d, id, -1.0, NULL);
	  if (par2d_add_to_gr2d_object (&graphic_pipeline_2d, math2d_input.math_object[id]) ==
	      GR_FAILURE)
	    return GR_FAILURE; /* Out of memory */
	  else
	    {
	      if ( (graphic2d_state.show_discontinuities) )
		remove_discontinuities_from_2dmathobj (id);
	      gr2d_set_depth (&graphic_pipeline_2d, id, 2+id);
	    }
	}
    }
  graphic2d_state.basic_prop_to_update = update_graphic_state;
  set_basic_2d_graphic_state(); /* graphic2d_state.basic_prop_to_update set to 0 */
  if ((update_graphic_state))
    {
      if ( (graphic2d_state.show_main_directions) )
	{
	  erase_main_directions_2d();
	  draw_main_directions_2d();
	}
      if (graphic2d_state.grid_step > 0.0)
	{
	  if ( (graphic2d_state.show_grid) )
	    {
	      /* Draw grid if requested */
	      erase_2d_grid();
	      draw_2d_grid();
	    }
	  if ( (graphic2d_state.show_tics) )
	    {
	      /* Draw tics if requested */
	      erase_2d_tics();
	      draw_2d_tics();
	    }
	}
    }
  /* 
     Now we compute the local coordinates of the 
     objects in the graphic pipeline, we set the
     update flags to UP_TO_DATE (if needed)...
  */
  compute_local_2dcoordinates(0, update_graphic_state);
  /*
    ...and then we sort the pipeline.
  */
  if ((update_graphic_state))
    gr2d_sort_shapes (&graphic_pipeline_2d);
  return GR_OK;
}

static void set_basic_3d_graphic_state (void)
{
  double adj_grid_step, L;

  if ( (graphic3d_state.basic_prop_to_update) )
    {
      graphic3d_state.O = graphic3d_state.origin = gr3d_get_middle_point (graphic_pipeline_3d); 
      L = gr3d_get_mean_length (graphic_pipeline_3d, TRUNC_FACTOR);
      gr3d_compute_bounding_sphere (&graphic_pipeline_3d, TRUNC_FACTOR*L);
      graphic3d_state.R = graphic_pipeline_3d.R * DEFAULT_R_MAG_FACTOR;
      graphic3d_state.grid_step = (100.0 * graphic3d_state.R) / (DEFAULT_DIVISOR_3D * graphic3d_state.zoom);
      adj_grid_step =  adjust_grid_step(graphic3d_state.grid_step);
      graphic3d_state.grid_step = adj_grid_step;
      graphic3d_state.basic_prop_to_update = 0; /* Update just made */
    }
}

static void erase_main_directions_3d (void)
{
  gr3d_remove_items_with_color (&graphic_pipeline_3d, X_COLOR, -1.0, NULL);
  gr3d_remove_items_with_color (&graphic_pipeline_3d, Y_COLOR, -1.0, NULL);
  gr3d_remove_items_with_color (&graphic_pipeline_3d, Z_COLOR, -1.0, NULL);
}

static void draw_main_directions_3d (void)
{
  int i, nsteps = DEFAULT_NSTEPS;
  double R;

  R = graphic3d_state.R;
  for (i = 0; i < nsteps; i++)
    {
      gr3d_add_segment (&graphic_pipeline_3d,
			X_COLOR, NULL,
			graphic3d_state.origin.x-R + (2*R/nsteps)*i, graphic3d_state.origin.y - R, graphic3d_state.origin.z - R,
			graphic3d_state.origin.x-R + (2*R/nsteps)*(i+1), graphic3d_state.origin.y - R, graphic3d_state.origin.z - R);
      gr3d_add_segment (&graphic_pipeline_3d,
			Y_COLOR, NULL,
			graphic3d_state.origin.x - R, graphic3d_state.origin.y -R + (2*R/nsteps)*i, graphic3d_state.origin.z - R,
			graphic3d_state.origin.x - R, graphic3d_state.origin.y -R + (2*R/nsteps)*(i+1), graphic3d_state.origin.z - R);
      gr3d_add_segment (&graphic_pipeline_3d,
			Z_COLOR, NULL,
			graphic3d_state.origin.x - R, graphic3d_state.origin.y - R, graphic3d_state.origin.z -R + (2*R/nsteps)*i,
			graphic3d_state.origin.x - R, graphic3d_state.origin.y - R, graphic3d_state.origin.z -R + (2*R/nsteps)*(i+1));
    }
  gr3d_add_point (&graphic_pipeline_3d,
		  X_COLOR, "X",
		  graphic3d_state.origin.x + 0.5 * R, 
		  graphic3d_state.origin.y - 0.88 * R, 
		  graphic3d_state.origin.z - 0.88 * R);
  gr3d_add_point (&graphic_pipeline_3d,
		  Y_COLOR, "Y",
		  graphic3d_state.origin.x - 0.88 * R, 
		  graphic3d_state.origin.y + 0.5 * R, 
		  graphic3d_state.origin.z - 0.88 * R);
  gr3d_add_point (&graphic_pipeline_3d,
		  Z_COLOR, "Z",
		  graphic3d_state.origin.x - 0.88 * R, 
		  graphic3d_state.origin.y - 0.88 * R, 
		  graphic3d_state.origin.z + 0.5 * R);
}

static void erase_3d_grid (void)
{
  gr3d_remove_items_with_color (&graphic_pipeline_3d, XYPLANE_GRID_COLOR, -1.0, NULL);
  gr3d_remove_items_with_color (&graphic_pipeline_3d, XZPLANE_GRID_COLOR, -1.0, NULL);
  gr3d_remove_items_with_color (&graphic_pipeline_3d, YZPLANE_GRID_COLOR, -1.0, NULL);
}

static void draw_3d_grid (void)
{
  int i, j, nsteps;
  double adj_grid_step = graphic3d_state.grid_step;

  nsteps = r_trunc(graphic3d_state.R / adj_grid_step);
  for (i = -nsteps; i <= nsteps; i++)
    {
      for (j = -nsteps; j <= nsteps; j++)
  	{
  	  gr3d_add_point (&graphic_pipeline_3d,
  			  XYPLANE_GRID_COLOR, NULL,
  			  graphic3d_state.origin.x + i * adj_grid_step,
  			  graphic3d_state.origin.y + j * adj_grid_step,
  			  graphic3d_state.origin.z - graphic3d_state.R);
  	}
    }
  for (i = -nsteps; i <= nsteps; i++)
    {
      for (j = -nsteps; j <= nsteps; j++)
  	{
  	  gr3d_add_point (&graphic_pipeline_3d,
  			  XZPLANE_GRID_COLOR, NULL,
  			  graphic3d_state.origin.x + i * adj_grid_step,
  			  graphic3d_state.origin.y - graphic3d_state.R,
  			  graphic3d_state.origin.z + j * adj_grid_step);
  	}
    }
  for (i = -nsteps; i <= nsteps; i++)
    {
      for (j = -nsteps; j <= nsteps; j++)
  	{
  	  gr3d_add_point (&graphic_pipeline_3d,
  			  YZPLANE_GRID_COLOR, NULL,
  			  graphic3d_state.origin.x - graphic3d_state.R,
  			  graphic3d_state.origin.y + i * adj_grid_step,
  			  graphic3d_state.origin.z + j * adj_grid_step);
  	}
    }
}

static void erase_3d_tics (void)
{
  gr3d_remove_items_with_color (&graphic_pipeline_3d, TICS_COLOR, -1.0, NULL);
}

static void draw_3d_tics (void)
{
  int i, interval, nsteps;
  double tics_coord, adj_grid_step = graphic3d_state.grid_step;
  grchar caption[LABEL_SIZE];
  const double half_sqrt_2 = 0.707106781187;

  nsteps = r_trunc(graphic3d_state.R / adj_grid_step);
  /* First we draw the tics on the x-axis... */
  interval =  nsteps >= DEFAULT_TIC_DIVISOR ? nsteps / DEFAULT_TIC_DIVISOR : 1;
  for (i = interval; i <= nsteps; i += interval)
    {
      tics_coord = smart_round (graphic3d_state.origin.x + i * adj_grid_step, adj_grid_step, 1);
      sprintf (caption, STRING_FMT_TICS, tics_coord);
      gr3d_add_point (&graphic_pipeline_3d,
		      TICS_COLOR, caption,
		      graphic3d_state.origin.x + i * adj_grid_step,
		      graphic3d_state.origin.y - graphic3d_state.R - adj_grid_step,
		      graphic3d_state.origin.z - graphic3d_state.R);
    }
  for (i = 0; i >= -nsteps; i -= interval)
    {
      tics_coord = smart_round (graphic3d_state.origin.x + i * adj_grid_step, adj_grid_step, 1);
      sprintf (caption, STRING_FMT_TICS, tics_coord);
      gr3d_add_point (&graphic_pipeline_3d,
		      TICS_COLOR, caption,
		      graphic3d_state.origin.x + i * adj_grid_step,
		      graphic3d_state.origin.y - graphic3d_state.R - adj_grid_step,
		      graphic3d_state.origin.z - graphic3d_state.R);
    }

  for (i = interval; i <= nsteps; i += interval)
    {
      tics_coord = smart_round (graphic3d_state.origin.y + i * adj_grid_step, adj_grid_step, 1);
      sprintf (caption, STRING_FMT_TICS, tics_coord);
      gr3d_add_point (&graphic_pipeline_3d,
		      TICS_COLOR, caption,
		      graphic3d_state.origin.x - graphic3d_state.R - adj_grid_step,
		      graphic3d_state.origin.y + i * adj_grid_step,
		      graphic3d_state.origin.z - graphic3d_state.R);
    }
  for (i = 0; i >= -nsteps; i -= interval)
    {
      tics_coord = smart_round (graphic3d_state.origin.y + i * adj_grid_step, adj_grid_step, 1);
      sprintf (caption, STRING_FMT_TICS, tics_coord);
      gr3d_add_point (&graphic_pipeline_3d,
		      TICS_COLOR, caption,
		      graphic3d_state.origin.x - graphic3d_state.R - adj_grid_step,
		      graphic3d_state.origin.y + i * adj_grid_step,
		      graphic3d_state.origin.z - graphic3d_state.R);
    }

  for (i = interval; i <= nsteps; i += interval)
    {
      tics_coord = smart_round (graphic3d_state.origin.z + i * adj_grid_step, adj_grid_step, 1);
      sprintf (caption, STRING_FMT_TICS, tics_coord);
      gr3d_add_point (&graphic_pipeline_3d,
		      TICS_COLOR, caption,
		      graphic3d_state.origin.x - graphic3d_state.R - adj_grid_step * half_sqrt_2,
		      graphic3d_state.origin.y - graphic3d_state.R - adj_grid_step * half_sqrt_2,
		      graphic3d_state.origin.z + i * adj_grid_step);
    }
  for (i = 0; i >= -nsteps; i -= interval)
    {
      tics_coord = smart_round (graphic3d_state.origin.z + i * adj_grid_step, adj_grid_step, 1);
      sprintf (caption, STRING_FMT_TICS, tics_coord);
      gr3d_add_point (&graphic_pipeline_3d,
		      TICS_COLOR, caption,
		      graphic3d_state.origin.x - graphic3d_state.R - adj_grid_step * half_sqrt_2,
		      graphic3d_state.origin.y - graphic3d_state.R - adj_grid_step * half_sqrt_2,
		      graphic3d_state.origin.z + i * adj_grid_step);
    }
}

static void remove_discontinuities_from_3dmathobj (int id)
{
  double L;

  L = gr3d_get_mean_length_by_color (graphic_pipeline_3d, id);
  if ( gr3d_remove_items_with_color (&graphic_pipeline_3d, id, DISC_FACTOR * L, &trashcan_3d) < 0 )
    {
      handle_fatal_error ("The program has not enough memory to keep on working\n" \
			  "please quit now.\n");
    }
}

static void ignore_discontinuities_3d (void)
{
  gr3d_add_object (&graphic_pipeline_3d, trashcan_3d);
  gr3d_delete (&trashcan_3d);
  trashcan_3d = gr3d_new(0, NUMBER_OF_3DOBJECTS - 1);
}

static void compute_local_3dcoordinates (int after_loading_from_file, int graphic_state_changed)
{
  int id;
    
  if ((after_loading_from_file))
    {
      gr3d_view_from_direction (&graphic_pipeline_3d,
				graphic3d_state.O,
				graphic3d_state.longitude,
				graphic3d_state.latitude,
				graphic3d_state.rotation_angle);
      /* After loading an image from file, all mathematical
         objects must be re-evaluated                       */      
      for (id = 0; id < NUMBER_OF_3DOBJECTS; id++)
        math3d_input.math_object_flag[id] = TO_UPDATE;
    }
  else
    {
      if ((graphic_state_changed))
	{
	  for (id = 0; id < NUMBER_OF_3DOBJECTS; id++)
	    {
              gr3d_partial_upgrade (&graphic_pipeline_3d, id, 
                                    graphic3d_state.O,
                                    graphic3d_state.longitude,
                                    graphic3d_state.latitude,
                                    graphic3d_state.rotation_angle);
              math3d_input.math_object_flag[id] = UP_TO_DATE;
	    }
	  if ( (graphic3d_state.show_main_directions) )
	    {
	      gr3d_partial_upgrade (&graphic_pipeline_3d, X_COLOR, 
				    graphic3d_state.O,
				    graphic3d_state.longitude,
				    graphic3d_state.latitude,
				    graphic3d_state.rotation_angle);
	      gr3d_partial_upgrade (&graphic_pipeline_3d, Y_COLOR, 
				    graphic3d_state.O,
				    graphic3d_state.longitude,
				    graphic3d_state.latitude,
				    graphic3d_state.rotation_angle);
	      gr3d_partial_upgrade (&graphic_pipeline_3d, Z_COLOR, 
				    graphic3d_state.O,
				    graphic3d_state.longitude,
				    graphic3d_state.latitude,
				    graphic3d_state.rotation_angle);
	    }      
	  if ( (graphic3d_state.show_grid) )
	    {
	      gr3d_partial_upgrade (&graphic_pipeline_3d, XYPLANE_GRID_COLOR, 
				    graphic3d_state.O,
				    graphic3d_state.longitude,
				    graphic3d_state.latitude,
				    graphic3d_state.rotation_angle);
	      gr3d_partial_upgrade (&graphic_pipeline_3d, XZPLANE_GRID_COLOR, 
				    graphic3d_state.O,
				    graphic3d_state.longitude,
				    graphic3d_state.latitude,
				    graphic3d_state.rotation_angle);
	      gr3d_partial_upgrade (&graphic_pipeline_3d, YZPLANE_GRID_COLOR, 
				    graphic3d_state.O,
				    graphic3d_state.longitude,
				    graphic3d_state.latitude,
				    graphic3d_state.rotation_angle);        
	    }
	  if ( (graphic3d_state.show_tics) )
	    {
	      gr3d_partial_upgrade (&graphic_pipeline_3d, TICS_COLOR, 
				    graphic3d_state.O,
				    graphic3d_state.longitude,
				    graphic3d_state.latitude,
				    graphic3d_state.rotation_angle);
	    }
	} /* end if ((graphic_state_changed)) */
    }
} 

static grexitstatus upgrade_3dpipeline (void)
{
  int id, update_graphic_state = graphic3d_state.basic_prop_to_update;
  update_flag uf;

  gr3d_delete (&trashcan_3d);
  trashcan_3d = gr3d_new(0, NUMBER_OF_3DOBJECTS - 1);
  for (id = 0; id < NUMBER_OF_3DOBJECTS; id++)
    {
      uf = math3d_input.math_object_flag[id];
      if (uf == WRONG_UPDATE)
	return GR_FAILURE; /* Out of memory */
      if (uf == TO_UPDATE)
	{
	  update_graphic_state = 1;
	  gr3d_remove_items_with_color (&graphic_pipeline_3d, id, -1.0, NULL);
	  if (par3d_add_to_gr3d_object (&graphic_pipeline_3d, math3d_input.math_object[id]) ==
	      GR_FAILURE)
	    return GR_FAILURE; /* Out of memory */
	  if ( (graphic3d_state.show_discontinuities) )
	    remove_discontinuities_from_3dmathobj(id);
	}
    }
  graphic3d_state.basic_prop_to_update = update_graphic_state;
  set_basic_3d_graphic_state(); /* graphic3d_state.basic_prop_to_update set to 0 */
  if ((update_graphic_state))
    {
      if ( (graphic3d_state.show_main_directions) )
	{
	  erase_main_directions_3d();
	  draw_main_directions_3d();
	}

      if (graphic3d_state.grid_step > 0.0)
	{
	  if ( (graphic3d_state.show_grid) )
	    {
	      /* Draw grid if requested */
	      erase_3d_grid();
	      draw_3d_grid();
	    }
	  if ( (graphic3d_state.show_tics) )
	    {
	      /* Draw tics if requested */
	      erase_3d_tics();
	      draw_3d_tics();
	    }
	}
    }
  /* 
     Now we compute the local coordinates of the 
     objects in the graphic pipeline, we set the
     update flags to UP_TO_DATE (if needed)...
  */
  compute_local_3dcoordinates(0, update_graphic_state);
  /*
    ...and then we sort the pipeline.
  */
  if ((update_graphic_state))
    gr3d_sort_shapes (&graphic_pipeline_3d);
  return GR_OK;
}

grexitstatus upgrade_pipeline (void)
{
  if (gmode == GRAPHIC_2D)
    return upgrade_2dpipeline();
  else     /* gmode == GRAPHIC_3D */
    return upgrade_3dpipeline();
}

double get_half_side_of_the_draw (void)
{
  return (gmode == GRAPHIC_2D ? graphic2d_state.R : graphic3d_state.R);
}

void set_longitude_and_update (double longitude)
{
  if (gmode == GRAPHIC_3D)
    {
      graphic3d_state.longitude = longitude;
      gr3d_view_from_direction (&graphic_pipeline_3d,
				graphic3d_state.O,
				graphic3d_state.longitude,
				graphic3d_state.latitude,
				graphic3d_state.rotation_angle);
    }
}

double get_longitude (void)
{
  return (gmode == GRAPHIC_3D ? graphic3d_state.longitude : DUMMY_LONGITUDE * CONV_FACTOR_D2R);
}


void set_latitude_and_update (double latitude)
{
  if (gmode == GRAPHIC_3D)
    {
      graphic3d_state.latitude = latitude;
      gr3d_view_from_direction (&graphic_pipeline_3d,
				graphic3d_state.O,
				graphic3d_state.longitude,
				graphic3d_state.latitude,
				graphic3d_state.rotation_angle);
    }
}

double get_latitude (void)
{
  return (gmode == GRAPHIC_3D ? graphic3d_state.latitude : DUMMY_LATITUDE * CONV_FACTOR_D2R);
}

void set_rotation_angle_and_update (double rotation_angle)
{
  if (gmode == GRAPHIC_2D)
    {
      graphic2d_state.rotation_angle = rotation_angle;
      gr2d_compute_userdef_coordinates (&graphic_pipeline_2d,
					graphic2d_state.O,
					graphic2d_state.rotation_angle);
    }
  else
    {
      graphic3d_state.rotation_angle = rotation_angle;
      gr3d_view_from_direction (&graphic_pipeline_3d,
				graphic3d_state.O,
				graphic3d_state.longitude,
				graphic3d_state.latitude,
				graphic3d_state.rotation_angle);
    }
}

double get_rotation_angle (void)
{
  return (gmode == GRAPHIC_2D ? graphic2d_state.rotation_angle : graphic3d_state.rotation_angle);
}

void translate_origin_and_update (grint dx, grint dy, grint dz)
{
  if (gmode == GRAPHIC_2D)
    {
      graphic2d_state.O.x += (graphic2d_state.grid_step / 10) * dx;
      graphic2d_state.O.y += (graphic2d_state.grid_step / 10) * dy;

      if ( (graphic2d_state.show_main_directions) )
	{
	  erase_main_directions_2d();
	  draw_main_directions_2d();
	}
      if ( graphic2d_state.grid_step > 0.0 ) 
	{
	  if ( (graphic2d_state.show_grid) )
	    {
	      erase_2d_grid();
	      draw_2d_grid();
	    }
	  if ( (graphic2d_state.show_tics) )
	    {
	      erase_2d_tics();
	      draw_2d_tics();
	    }
	}
      gr2d_compute_userdef_coordinates (&graphic_pipeline_2d,
					graphic2d_state.O,
					graphic2d_state.rotation_angle);
    }
  else
    {
      graphic3d_state.O.x += (graphic3d_state.grid_step / 10) * dx;
      graphic3d_state.O.y += (graphic3d_state.grid_step / 10) * dy;
      graphic3d_state.O.z += (graphic3d_state.grid_step / 10) * dz;
      gr3d_view_from_direction (&graphic_pipeline_3d,
				graphic3d_state.O,
				graphic3d_state.longitude,
				graphic3d_state.latitude,
				graphic3d_state.rotation_angle);
    }
}

void reset_origin_position_and_update (void)
{
  if (gmode == GRAPHIC_2D)
    {
      graphic2d_state.O = graphic2d_state.origin;
      if ( (graphic2d_state.show_main_directions) )
	{
	  erase_main_directions_2d();
	  draw_main_directions_2d();
	}
      if ( graphic2d_state.grid_step > 0.0 ) 
	{
	  if ( (graphic2d_state.show_grid) )
	    {
	      erase_2d_grid();
	      draw_2d_grid();
	    }
	  if ( (graphic2d_state.show_tics) )
	    {
	      erase_2d_tics();
	      draw_2d_tics();
	    }
	}
      gr2d_compute_userdef_coordinates (&graphic_pipeline_2d,
					graphic2d_state.O,
					graphic2d_state.rotation_angle);
    }
  else
    {
      graphic3d_state.O = graphic3d_state.origin;
      gr3d_view_from_direction (&graphic_pipeline_3d,
				graphic3d_state.O,
				graphic3d_state.longitude,
				graphic3d_state.latitude,
				graphic3d_state.rotation_angle);
    }
}

void retrieve_origin_position (double* ox, double* oy, double* oz)
{
  if (gmode == GRAPHIC_2D)
    {
      *ox = graphic2d_state.O.x;
      *oy = graphic2d_state.O.y;
      *oz = 0.0;
    }
  else
    {
      *ox = graphic3d_state.O.x;
      *oy = graphic3d_state.O.y;
      *oz = graphic3d_state.O.z;
    }
}

double get_grid_step (void)
{
  return (gmode == GRAPHIC_2D ? graphic2d_state.grid_step : graphic3d_state.grid_step);
}

void toggle_show_main_directions (int boolean_value)
{
  if (gmode == GRAPHIC_2D)
    {
      if (boolean_value != graphic2d_state.show_main_directions)
	{
	  graphic2d_state.show_main_directions = boolean_value; 
	  if ( (graphic2d_state.show_main_directions) )
	    {
	      draw_main_directions_2d();
	      gr2d_partial_upgrade (&graphic_pipeline_2d, X_COLOR, 
				    graphic2d_state.O,
				    graphic2d_state.rotation_angle);
	      gr2d_partial_upgrade (&graphic_pipeline_2d, Y_COLOR, 
				    graphic2d_state.O,
				    graphic2d_state.rotation_angle);
	      gr2d_partial_upgrade (&graphic_pipeline_2d, Z_COLOR, 
				    graphic2d_state.O,
				    graphic2d_state.rotation_angle);
	      gr2d_sort_shapes (&graphic_pipeline_2d);
	    }
	  else
	    erase_main_directions_2d();
	}
    }
  else
    {
      if (boolean_value != graphic3d_state.show_main_directions)
	{
	  graphic3d_state.show_main_directions = boolean_value; 
	  if ( (graphic3d_state.show_main_directions) )
	    {
	      draw_main_directions_3d();
	      gr3d_partial_upgrade (&graphic_pipeline_3d, X_COLOR, 
				    graphic3d_state.O,
				    graphic3d_state.longitude,
				    graphic3d_state.latitude,
				    graphic3d_state.rotation_angle);
	      gr3d_partial_upgrade (&graphic_pipeline_3d, Y_COLOR, 
				    graphic3d_state.O,
				    graphic3d_state.longitude,
				    graphic3d_state.latitude,
				    graphic3d_state.rotation_angle);
	      gr3d_partial_upgrade (&graphic_pipeline_3d, Z_COLOR, 
				    graphic3d_state.O,
				    graphic3d_state.longitude,
				    graphic3d_state.latitude,
				    graphic3d_state.rotation_angle);
	      gr3d_sort_shapes (&graphic_pipeline_3d);
	    }
	  else
	    erase_main_directions_3d();
	}
    }
}

int get_show_main_directions (void)
{
  return (gmode == GRAPHIC_2D ? graphic2d_state.show_main_directions : graphic3d_state.show_main_directions);
}

void toggle_show_grid (int boolean_value)
{
  double adj_grid_step;
  
  if (gmode == GRAPHIC_2D)
    {
      if (boolean_value != graphic2d_state.show_grid)
	{
	  graphic2d_state.show_grid = boolean_value; 
	  if ( (graphic2d_state.show_grid) )
            {
              /* 
                 First we update the grid step
                 and then we draw the grid
               */
              graphic2d_state.grid_step = (100.0 * graphic2d_state.R) / (DEFAULT_DIVISOR_2D * graphic2d_state.zoom);
              adj_grid_step =  adjust_grid_step(graphic2d_state.grid_step);
	      if (adj_grid_step > 0.0)
		{
		  /* Remark: If the computed grid step is zero,
		     then the request to show the grid should be 
		     simply ignored */
		  graphic2d_state.grid_step = adj_grid_step;            
		  draw_2d_grid();
		  gr2d_partial_upgrade (&graphic_pipeline_2d, GRID_COLOR, 
					graphic2d_state.O,
					graphic2d_state.rotation_angle);
		  gr2d_sort_shapes (&graphic_pipeline_2d);
		}
	    }
	  else
	    erase_2d_grid();
	}
    }
  else
    {
      if (boolean_value != graphic3d_state.show_grid)
	{
	  graphic3d_state.show_grid = boolean_value; 
	  if ( (graphic3d_state.show_grid) )
            {
              /* 
                 First we update the grid step
                 and then we draw the grid
               */    
              graphic3d_state.grid_step = (100.0 * graphic3d_state.R) / (DEFAULT_DIVISOR_3D * graphic3d_state.zoom);
              adj_grid_step =  adjust_grid_step(graphic3d_state.grid_step);
	      if (adj_grid_step > 0.0)
		{
		  /* Remark: If the computed grid step is zero,
		     then the request to show the grid should be 
		     simply ignored */
		  graphic3d_state.grid_step = adj_grid_step;              
		  draw_3d_grid();
		  gr3d_partial_upgrade (&graphic_pipeline_3d, XYPLANE_GRID_COLOR, 
					graphic3d_state.O,
					graphic3d_state.longitude,
					graphic3d_state.latitude,
					graphic3d_state.rotation_angle);
		  gr3d_partial_upgrade (&graphic_pipeline_3d, XZPLANE_GRID_COLOR, 
					graphic3d_state.O,
					graphic3d_state.longitude,
					graphic3d_state.latitude,
					graphic3d_state.rotation_angle);
		  gr3d_partial_upgrade (&graphic_pipeline_3d, YZPLANE_GRID_COLOR, 
					graphic3d_state.O,
					graphic3d_state.longitude,
					graphic3d_state.latitude,
					graphic3d_state.rotation_angle);                
		  gr3d_sort_shapes (&graphic_pipeline_3d);
		}
	    }
	  else
	    erase_3d_grid();
	}
    }
}

int get_show_grid (void)
{
  return (gmode == GRAPHIC_2D ? graphic2d_state.show_grid : graphic3d_state.show_grid);
}

void toggle_show_tics (int boolean_value)
{
  double adj_grid_step;

  if (gmode == GRAPHIC_2D)
    {
      if (boolean_value != graphic2d_state.show_tics)
	{
	  graphic2d_state.show_tics = boolean_value; 
	  if ( (graphic2d_state.show_tics) )
            {
              /* 
                 First we update the grid step
                 and then we draw the tics
               */
              graphic2d_state.grid_step = (100.0 * graphic2d_state.R) / (DEFAULT_DIVISOR_2D * graphic2d_state.zoom);
              adj_grid_step =  adjust_grid_step(graphic2d_state.grid_step);
	      if (adj_grid_step > 0.0)
		{
		  /* Remark: If the computed grid step is zero,
		     then the request to show the tics should be 
		     simply ignored */
		  graphic2d_state.grid_step = adj_grid_step;            
		  draw_2d_tics();
		  gr2d_partial_upgrade (&graphic_pipeline_2d, TICS_COLOR, 
					graphic2d_state.O,
					graphic2d_state.rotation_angle);
		  gr2d_sort_shapes (&graphic_pipeline_2d);
		}
	    }
	  else
	    erase_2d_tics();
	}
    }
  else
    {
      if (boolean_value != graphic3d_state.show_tics)
	{
	  graphic3d_state.show_tics = boolean_value; 
	  if ( (graphic3d_state.show_tics) )
            {
              /* 
                 First we update the grid step
                 and then we draw the tics
               */    
              graphic3d_state.grid_step = (100.0 * graphic3d_state.R) / (DEFAULT_DIVISOR_3D * graphic3d_state.zoom);
              adj_grid_step =  adjust_grid_step(graphic3d_state.grid_step);
	      if (adj_grid_step > 0.0)
		{
		  /* Remark: If the computed grid step is zero,
		     then the request to show the tics should be 
		     simply ignored */
		  graphic3d_state.grid_step = adj_grid_step;              
		  draw_3d_tics();
		  gr3d_partial_upgrade (&graphic_pipeline_3d, TICS_COLOR, 
					graphic3d_state.O,
					graphic3d_state.longitude,
					graphic3d_state.latitude,
					graphic3d_state.rotation_angle);
		  gr3d_sort_shapes (&graphic_pipeline_3d);
		}
	    }
	  else
	    erase_3d_tics();
	}
    }
}

int get_show_tics (void)
{
  return (gmode == GRAPHIC_2D ? graphic2d_state.show_tics : graphic3d_state.show_tics);
}

void toggle_show_discontinuities (int boolean_value)
{
  int id;

  if (gmode == GRAPHIC_2D)
    {
      if (boolean_value != graphic2d_state.show_discontinuities)
	{
	  graphic2d_state.show_discontinuities = boolean_value; 
	  if ( (graphic2d_state.show_discontinuities) )
	    {
	      for (id = 0; id < NUMBER_OF_2DOBJECTS; id++)
		remove_discontinuities_from_2dmathobj(id);
	    }
	  else
	    {
	      ignore_discontinuities_2d();
	      for (id = 0; id < NUMBER_OF_2DOBJECTS; id++)
		{
		  gr2d_partial_upgrade (&graphic_pipeline_2d, id, 
					graphic2d_state.O,
					graphic2d_state.rotation_angle);
		}
	      gr2d_sort_shapes (&graphic_pipeline_2d);
	    }
	}
    }
  else
    {
      if (boolean_value != graphic3d_state.show_discontinuities)
	{
	  graphic3d_state.show_discontinuities = boolean_value; 
	  if ( (graphic3d_state.show_discontinuities) )
	    {
	      for (id = 0; id < NUMBER_OF_3DOBJECTS; id++)
		remove_discontinuities_from_3dmathobj(id);
	    }
	  else
	    {
	      ignore_discontinuities_3d();
	      for (id = 0; id < NUMBER_OF_3DOBJECTS; id++)
		{
		  gr3d_partial_upgrade (&graphic_pipeline_3d, id, 
					graphic3d_state.O,
					graphic3d_state.longitude,
					graphic3d_state.latitude,
					graphic3d_state.rotation_angle);
		}
	      gr3d_sort_shapes (&graphic_pipeline_3d);
	    }
	}
    } /* gmode == GRAPHIC_3D */
}

int get_show_discontinuities (void)
{
  return (gmode == GRAPHIC_2D ? graphic2d_state.show_discontinuities : graphic3d_state.show_discontinuities);
}

void set_zoom_factor (double zoom)
{
  if (gmode == GRAPHIC_2D)
    graphic2d_state.zoom = zoom; 
  else
    graphic3d_state.zoom = zoom; 
}

double get_zoom_factor (void)
{
  if (gmode == GRAPHIC_2D)
    return graphic2d_state.zoom; 
  else
    return graphic3d_state.zoom; 
}

void set_smoke_density (double smoke_density)
{
  if (gmode == GRAPHIC_3D)
    graphic3d_state.smoke_density = smoke_density; 
}

double get_smoke_density (void)
{
  if (gmode == GRAPHIC_3D)
    return graphic3d_state.smoke_density; 
  else
    return 0.0;
}

grexitstatus save_pipeline (const char* filepath)
{
  FILE* fp;

  if ( !(fp = fopen (filepath, "w")) )
    return GR_FAILURE;
  if (gmode == GRAPHIC_2D)
    {
      gr2d_object tmp = gr2d_new (0, NUMBER_OF_2DOBJECTS-1);
      
      if ( gr2d_add_object (&tmp, graphic_pipeline_2d) < graphic_pipeline_2d.len )
	{
          gr2d_delete(&tmp);
	  fclose(fp);
	  return GR_FAILURE;
	}
      gr2d_remove_items_with_color (&tmp, X_COLOR, -1.0, NULL);
      gr2d_remove_items_with_color (&tmp, Y_COLOR, -1.0, NULL);
      gr2d_remove_items_with_color (&tmp, GRID_COLOR, -1.0, NULL);
      gr2d_remove_items_with_color (&tmp, TICS_COLOR, -1.0, NULL);
      if ( gr2d_save (tmp, fp) >= 0 )
        {
          gr2d_delete(&tmp);
	  return (fclose(fp) == 0 ? GR_OK : GR_FAILURE);
        }
      else
        {
          gr2d_delete(&tmp);
	  fclose(fp);
	  return GR_FAILURE;
        }
    }
  else
    {
      gr3d_object tmp = gr3d_new (0, NUMBER_OF_3DOBJECTS-1);
      
      if ( gr3d_add_object (&tmp, graphic_pipeline_3d) < graphic_pipeline_3d.len )
	{
          gr3d_delete(&tmp);
	  fclose(fp);
	  return GR_FAILURE;
	}
      gr3d_remove_items_with_color (&tmp, X_COLOR, -1.0, NULL);
      gr3d_remove_items_with_color (&tmp, Y_COLOR, -1.0, NULL);
      gr3d_remove_items_with_color (&tmp, Z_COLOR, -1.0, NULL);
      gr3d_remove_items_with_color (&tmp, XYPLANE_GRID_COLOR, -1.0, NULL);    
      gr3d_remove_items_with_color (&tmp, XZPLANE_GRID_COLOR, -1.0, NULL);    
      gr3d_remove_items_with_color (&tmp, YZPLANE_GRID_COLOR, -1.0, NULL);    
      gr3d_remove_items_with_color (&tmp, TICS_COLOR, -1.0, NULL);
      if ( gr3d_save (tmp, fp) >= 0 )
        {
          gr3d_delete(&tmp);
	  return (fclose(fp) == 0 ? GR_OK : GR_FAILURE);
        }
      else
        {
          gr3d_delete(&tmp);
	  fclose(fp);
	  return GR_FAILURE;
        }
    }
}

static grexitstatus load_into_2dpipeline (gr2d_object* ptmp_obj,
					  int native_format_used)
{
  int id;

  if (!ptmp_obj)
    return GR_FAILURE;

  graphic2d_state.basic_prop_to_update = 1;
  if ((native_format_used))
    {
      /*
	Replace the contents of the graphic pipeline
	with the objects loaded from the file.
      */
      gr2d_delete (&graphic_pipeline_2d);
      graphic_pipeline_2d = *ptmp_obj;
    }
  else
    {
      for (id = 0; id < NUMBER_OF_2DOBJECTS; id++)
	{
	  /* 
	     Items are moved from the object pointed to by PTMP_OBJ
	     to the graphic pipeline. The current contents of
	     the graphic pipeline are not erased.
	  */
            if ( gr2d_remove_items_with_color (ptmp_obj, id, -1.0, 
                                               &graphic_pipeline_2d) < 0)
              {
                handle_fatal_error ("The program has not enough memory to keep on working\n" \
                                    "please quit now.\n");
                return GR_FAILURE;
              }
	  /*
	    For the objects loaded from file the DEPTH has to
	    be properly set.
	  */
	  gr2d_set_depth (&graphic_pipeline_2d, id, 2+id);
	}
      gr2d_delete (ptmp_obj);
    }
  gr2d_delete (&trashcan_2d);
  trashcan_2d = gr2d_new(0, NUMBER_OF_2DOBJECTS - 1);
  if ( (graphic2d_state.show_discontinuities) )
    {
      for (id = 0; id < NUMBER_OF_2DOBJECTS; id++)
	remove_discontinuities_from_2dmathobj(id);
    }
  set_basic_2d_graphic_state(); /* graphic2d_state.basic_prop_to_update set to 0 */
  if ( (graphic2d_state.show_main_directions) )
    {
      erase_main_directions_2d();
      draw_main_directions_2d();
    }
  if (graphic2d_state.grid_step > 0.0)
    {
      if ( (graphic2d_state.show_grid) )
	{
	  /* Draw grid if requested */
	  erase_2d_grid();
	  draw_2d_grid();
	}
      if ( (graphic2d_state.show_tics) )
	{
	  /* Draw tics if requested */
	  erase_2d_tics();
	  draw_2d_tics();
	}
    }
  /* 
     Now we compute the local coordinates of the 
     objects in the graphic pipeline, we set all
     update flags to TO_UPDATE for later use in
     upgrade_pipeline(),...
  */
  compute_local_2dcoordinates(1, 1);
  /*
    ...and then we sort the pipeline.
  */
  gr2d_sort_shapes (&graphic_pipeline_2d);
  return GR_OK;
}

static grexitstatus load_into_3dpipeline (gr3d_object* ptmp_obj,
					  int native_format_used)
{
  int id;
 
  if (!ptmp_obj)
    return GR_FAILURE;

  graphic3d_state.basic_prop_to_update = 1;
  if ((native_format_used))
    {
      /*
	Replace the contents of the graphic pipeline
	with the objects loaded from the file.
      */
      gr3d_delete (&graphic_pipeline_3d);
      graphic_pipeline_3d = *ptmp_obj;
    }
  else
    {
      for (id = 0; id < NUMBER_OF_3DOBJECTS; id++)
	{
	  /* 
	     Items are moved from the object pointed to by PTMP_OBJ
	     to the graphic pipeline. The current contents of
	     the graphic pipeline are not erased.
	  */
            if ( gr3d_remove_items_with_color (ptmp_obj, id, -1.0, 
                                               &graphic_pipeline_3d) < 0 )
              {
                handle_fatal_error ("The program has not enough memory to keep on working\n" \
                                    "please quit now.\n");
                return GR_FAILURE;
              }
	}
      gr3d_delete (ptmp_obj);
    }
  gr3d_delete (&trashcan_3d);
  trashcan_3d = gr3d_new(0, NUMBER_OF_3DOBJECTS - 1);
  if ( (graphic3d_state.show_discontinuities) )
    {
      for (id = 0; id < NUMBER_OF_3DOBJECTS; id++)
	remove_discontinuities_from_3dmathobj(id);
    }
  set_basic_3d_graphic_state(); /* graphic3d_state.basic_prop_to_update set to 0 */
  if ( (graphic3d_state.show_main_directions) )
    {
      erase_main_directions_3d();
      draw_main_directions_3d();
    }

  if (graphic3d_state.grid_step > 0.0)
    {
      if ( (graphic3d_state.show_grid) )
	{
	  /* Draw grid if requested */
	  erase_3d_grid();
	  draw_3d_grid();
	}
      if ( (graphic3d_state.show_tics) )
	{
	  /* Draw tics if requested */
	  erase_3d_tics();
	  draw_3d_tics();
	}
    }

  /* 
     Now we compute the local coordinates of the 
     objects in the graphic pipeline, we set all
     update flags to TO_UPDATE for later use in
     upgrade_pipeline(),...
  */
  compute_local_3dcoordinates(1, 1);
  /*
    ...and then we sort the pipeline.
  */
  gr3d_sort_shapes (&graphic_pipeline_3d);
  return GR_OK;
}

const char* mgr_filename (const char* new_filepath)
{
  static char filepath[BUFF_SIZE] = {'\0'};

  if ((new_filepath))
    {
      size_t l = strlen(new_filepath);

      if (l > BUFF_SIZE - 1)
	l = BUFF_SIZE - 1;
      strncpy (filepath, new_filepath, l);
      filepath[l] = '\0';
    }
  return (const char*) filepath;
}

grexitstatus load_into_pipeline (const char* filepath,
				 int file_in_native_format)
{
  FILE* fp;
  grexitstatus rv;

  if ( !(fp = fopen (filepath, "r")) )
    return GR_FAILURE;

  if ((file_in_native_format))
    {
      if (gmode == GRAPHIC_2D)
	{
	  gr2d_object tmp;

	  tmp = gr2d_new (0, NUMBER_OF_2DOBJECTS-1);
	  if ( gr2d_load (&tmp, fp) < 0 )
	    {
	      fclose (fp);
	      gr2d_delete (&tmp);
	      return GR_FAILURE;
	    }
	  else
	    {
	      fclose (fp);
	      rv = load_into_2dpipeline (&tmp, 1);
	      if (rv == GR_OK)
		{
		  mgr_filename (filepath);
		  mark_all_ids_as_used();
		}
	    }
	}
      else
	{
	  /* gmode == GRAPHIC_3D */
	  gr3d_object tmp;

	  tmp = gr3d_new (0, NUMBER_OF_3DOBJECTS-1);
	  if ( gr3d_load (&tmp, fp) < 0 )
	    {
	      fclose (fp);
	      gr3d_delete (&tmp);
	      return GR_FAILURE;
	    }
	  else
	    {
	      fclose (fp);
	      rv = load_into_3dpipeline (&tmp, 1);    
	      if (rv == GR_OK)
		{
		  mgr_filename (filepath);
		  mark_all_ids_as_used();
		}
	    }
	}
    } /* end  if ((file_in_native_format)) */
  else
    {
      /* The file to be loaded is in GNUPlot format */
      int dim = 0;
      gr2d_object tmp2d;
      gr3d_object tmp3d;

      tmp2d = gr2d_new (0, NUMBER_OF_2DOBJECTS-1);
      tmp3d = gr3d_new (0, NUMBER_OF_3DOBJECTS-1);
      dim = load_objects_from_file (filepath, fp, &tmp2d, &tmp3d);
      fclose (fp);
      if (dim < 0)
	{
	  gr2d_delete (&tmp2d);
	  gr3d_delete (&tmp3d);
	  return (dim == -2 ? GR_EXCEPTION : GR_FAILURE);
	}
      else if (dim == 2)
	rv = load_into_2dpipeline (&tmp2d, 0);
      else
	rv = load_into_3dpipeline (&tmp3d, 0);    
    }
  return rv;
}

const gr2d_object* get_pointer_to_2dpipeline (void)
{
  return &graphic_pipeline_2d;
}

const gr3d_object* get_pointer_to_3dpipeline (void)
{
  return &graphic_pipeline_3d;
}

char* get_formatted_equations (print_specs settings, 
			       cursor_position* start_position, 
			       int first_equation, int* until_equation)
{
  char *fmt_text = NULL;
  char *caption, *fmt_caption = NULL;
  char *xeq, *fmt_xeq = NULL;
  char *yeq, *fmt_yeq = NULL;
  char *zeq, *fmt_zeq = NULL;
  fmt_error_code caption_err, xeq_err, yeq_err, zeq_err;
  double end_y;
  repr_type type;
  grmathtype mtype;
  int id;


  /* First check if the plotted data comes from a MGR file */
  if ( *(caption = (char*) mgr_filename (NULL)) != '\0' )
    {
      /* 
	 If yes, then you should just format the CAPTION,
	 which points to the path of the MGR file.
      */
      *until_equation = 
	(gmode == GRAPHIC_2D ? NUMBER_OF_2DOBJECTS : NUMBER_OF_3DOBJECTS);
      caption_err = Missing_space;
      caption = xstrrealloc (NULL, 6+xstrlen(caption));
      strcpy (caption, "     ");
      strcat (caption, mgr_filename (NULL));
      fmt_caption = write_text_to_area (caption, settings, 1, 
					*start_position, 
					&caption_err, 
					&end_y);
      xstrrealloc (caption, 0);
      if (caption_err == No_error)
	{
	  start_position->y = end_y;
	  add_line_to_text (&fmt_text, 0, fmt_caption, strlen(fmt_caption));
	  xfree (&fmt_caption);
	}
      else
	{
	  xfree (&fmt_caption);
	  *until_equation = first_equation;
	}
    }
  else if (gmode == GRAPHIC_2D)
    {
      par2d_object obj;

      *until_equation = NUMBER_OF_2DOBJECTS;
      for (id = first_equation; id < NUMBER_OF_2DOBJECTS; id++)
	{
	  obj = math2d_input.math_object[id];
	  caption_err = xeq_err = yeq_err = Missing_space;
	  if (xstrlen(obj.x_eq) > 0 && xstrlen(obj.y_eq) > 0)
	    {
	      type = get_representation_type (id);
	      mtype = obj.type;
	      caption = xstrrealloc (NULL, 6+xstrlen(obj.caption));
	      sprintf (caption, "%3d. ", id+1);
	      strcat (caption, obj.caption);
	      fmt_caption = write_text_to_area (caption, settings, 1, 
						*start_position, &caption_err, 
						&end_y);
	      xstrrealloc (caption, 0);
	      if (type == CARTESIAN)
		{
		  /* Omit the X equation */

		  if (caption_err == No_error)
		    xeq_err = No_error; 
		  /*
		    else
		    xeq_err keeps the value Missing_space, 
		  */
		  /* fmt_xeq remains set to NULL */
		}
	      else
		{
		  /* Print the X equation */
		  if (caption_err == No_error)
		    {
		      start_position->y = end_y;
		      xeq = xstrrealloc (NULL, 14+strlen(obj.x_eq));		  
		      if (mtype == GRMATH_CURVE)
			strcat (xeq, "     x(t)= ");
		      else
			strcat (xeq, "     x(s,t)= ");
		      strcat (xeq, obj.x_eq);
		      fmt_xeq = write_text_to_area (xeq, settings, 1, 
						    *start_position, &xeq_err, 
						    &end_y);
		      xstrrealloc (xeq, 0);
		    }
		  /*
		    else
		    xeq_err keeps the value Missing_space, 
		    and fmt_xeq remains set to NULL.
		  */
		}
	      if (xeq_err == No_error)
		{
		  start_position->y = end_y;
		  if (type == CARTESIAN)
		    {
		      char* converted_eq;

		      converted_eq = convert (obj.y_eq, "t", "x");
		      yeq = xstrrealloc (NULL, 14+strlen(converted_eq));
		      strcat (yeq, "     y(x)= ");
		      strcat (yeq, converted_eq);
		      free ((void*)converted_eq);
		    }
		  else
		    {
		      yeq = xstrrealloc (NULL, 14+strlen(obj.y_eq));
		      if (mtype == GRMATH_CURVE)
			strcat (yeq, "     y(t)= ");
		      else
			strcat (yeq, "     y(s,t)= ");
		      strcat (yeq, obj.y_eq);
		    }
		  fmt_yeq = write_text_to_area (yeq, settings, 1, 
						*start_position, &yeq_err, 
						&end_y);
		  xstrrealloc (yeq, 0);
		}
	      /*
		else
		yeq_err keeps the value Missing_space, 
		and fmt_yeq remains set to NULL.
	      */
	      if (yeq_err == No_error)
		{
		  start_position->y = end_y;
		  add_line_to_text (&fmt_text, 0, fmt_caption, strlen(fmt_caption));
		  add_line_to_text (&fmt_text, 0, fmt_xeq, xstrlen(fmt_xeq)); /* Does not do anything if fmt_xeq == NULL */
		  add_line_to_text (&fmt_text, 0, fmt_yeq, strlen(fmt_yeq));
		  xfree (&fmt_caption);
		  xfree (&fmt_xeq);
		  xfree (&fmt_yeq);
		}
	      else
		{
		  xfree (&fmt_caption);
		  xfree (&fmt_xeq);
		  xfree (&fmt_yeq);
		  *until_equation = id;		      
		  break;
		}
	    } /* if ((obj.x_eq) && (obj.y_eq)) */
	  else
	    {
	      /* obj.x_eq and obj.y_eq are empty */
	      const grchar* obj_caption = get_object_caption (gmode, id);

	      if (obj_caption)
		{
		  /* The object comes from a GDAT file */
		  caption = xstrrealloc (NULL, 6+xstrlen(obj_caption));
		  sprintf (caption, "%3d. ", id+1);
		  strcat (caption, obj_caption);
		  fmt_caption = write_text_to_area (caption, settings, 1, 
						    *start_position, 
						    &caption_err, 
						    &end_y);
		  xstrrealloc (caption, 0);
		  if (caption_err == No_error)
		    {
		      start_position->y = end_y;
		      add_line_to_text (&fmt_text, 0, fmt_caption, strlen(fmt_caption));
		      xfree (&fmt_caption);
		    }
		  else
		    {
		      xfree (&fmt_caption);
		      *until_equation = id;		      
		      break;
		    }
		} /* end if (obj_caption) */
	    }
	} /* end for */
    }
  else
    {
      /* gmode == GRAPHIC_3D */

      par3d_object obj;

      *until_equation = NUMBER_OF_3DOBJECTS;
      for (id = first_equation; id < NUMBER_OF_3DOBJECTS; id++)
	{
	  obj = math3d_input.math_object[id];
	  caption_err = xeq_err = yeq_err = zeq_err = Missing_space;
	  if (xstrlen(obj.x_eq) > 0 && xstrlen(obj.y_eq) > 0 && 
	      xstrlen(obj.z_eq) > 0)
	    {
	      type = get_representation_type (id);
	      mtype = obj.type;
	      caption = xstrrealloc (NULL, 6+xstrlen(obj.caption));
	      sprintf (caption, "%3d. ", id+1);
	      strcat (caption, obj.caption);
	      fmt_caption = write_text_to_area (caption, settings, 1, 
						*start_position, &caption_err, 
						&end_y);
	      xstrrealloc (caption, 0);
	      if (type == CARTESIAN)
		{
		  /* Omit X and Y equation */

		  if (caption_err == No_error)
		    yeq_err = xeq_err = No_error; 
		  /*
		    else
		    yeq_err and xeq_err keep the value Missing_space 
		  */
		  /* fmt_xeq and fmt_yeq remain set to NULL */
		}
	      else
		{
		  /* Print X and Y equation */

		  if (caption_err == No_error)
		    {
		      start_position->y = end_y;
		      xeq = xstrrealloc (NULL, 14+strlen(obj.x_eq));
		      if (mtype == GRMATH_CURVE)
			strcat (xeq, "     x(t)= ");
		      else
			strcat (xeq, "     x(s,t)= ");
		      strcat (xeq, obj.x_eq);
		      fmt_xeq = write_text_to_area (xeq, settings, 1, 
						    *start_position, &xeq_err, 
						    &end_y);
		      xstrrealloc (xeq, 0);
		    }
		  /*
		    else
		    xeq_err keeps the value Missing_space, 
		    and fmt_xeq remains set to NULL.
		  */
		  if (xeq_err == No_error)
		    {
		      start_position->y = end_y;
		      yeq = xstrrealloc (NULL, 14+strlen(obj.y_eq));
		      if (mtype == GRMATH_CURVE)
			strcat (yeq, "     y(t)= ");
		      else
			strcat (yeq, "     y(s,t)= ");
		      strcat (yeq, obj.y_eq);
		      fmt_yeq = write_text_to_area (yeq, settings, 1, 
						    *start_position, &yeq_err, 
						    &end_y);
		      xstrrealloc (yeq, 0);
		    }
		  /*
		    else
		    yeq_err keeps the value Missing_space, and fmt_yeq remains set to NULL.
		  */
		}
	      if (yeq_err == No_error)
		{
		  start_position->y = end_y;
		  if (type == CARTESIAN)
		    {
		      char* converted_eq;

		      converted_eq = convert (obj.z_eq, "st", "xy");
		      zeq = xstrrealloc (NULL, 14+strlen(converted_eq));
		      strcat (zeq, "     z(x,y)= ");
		      strcat (zeq, converted_eq);
		      free ((void*)converted_eq);
		    }
		  else
		    {
		      zeq = xstrrealloc (NULL, 14+strlen(obj.z_eq));
		      if (mtype == GRMATH_CURVE)
			strcat (zeq, "     z(t)= ");
		      else
			strcat (zeq, "     z(s,t)= ");
		      strcat (zeq, obj.z_eq);
		    }
		  fmt_zeq = write_text_to_area (zeq, settings, 1, 
						*start_position, &zeq_err, 
						&end_y);
		  xstrrealloc (zeq, 0);
		}
	      /*
		else
		zeq_err keeps the value Missing_space, and fmt_zeq remains set to NULL.
	      */	      
	      if (zeq_err == No_error)
		{
		  start_position->y = end_y;
		  add_line_to_text (&fmt_text, 0, fmt_caption, strlen(fmt_caption));
		  add_line_to_text (&fmt_text, 0, fmt_xeq, xstrlen(fmt_xeq)); /* Does not do anything if fmt_xeq == NULL */
		  add_line_to_text (&fmt_text, 0, fmt_yeq, xstrlen(fmt_yeq)); /* Does not do anything if fmt_yeq == NULL */
		  add_line_to_text (&fmt_text, 0, fmt_zeq, strlen(fmt_zeq));
		  xfree (&fmt_caption);
		  xfree (&fmt_xeq);
		  xfree (&fmt_yeq);
		  xfree (&fmt_zeq);
		}
	      else
		{
		  xfree (&fmt_caption);
		  xfree (&fmt_xeq);
		  xfree (&fmt_yeq);
		  xfree (&fmt_zeq);		  
		  *until_equation = id;		      
		  break;
		}
	    } /* if ((obj.x_eq) && (obj.y_eq) && (obj.z_eq)) */
	  else
	    {
	      /* obj.x_eq, obj.y_eq and obj.z_eq are empty */
	      const grchar* obj_caption = get_object_caption (gmode, id);

	      if (obj_caption)
		{
		  /* The object comes from a GDAT file */
		  caption = xstrrealloc (NULL, 6+xstrlen(obj_caption));
		  sprintf (caption, "%3d. ", id+1);
		  strcat (caption, obj_caption);
		  fmt_caption = write_text_to_area (caption, settings, 1, 
						    *start_position, 
						    &caption_err, 
						    &end_y);
		  xstrrealloc (caption, 0);
		  if (caption_err == No_error)
		    {
		      start_position->y = end_y;
		      add_line_to_text (&fmt_text, 0, fmt_caption, strlen(fmt_caption));
		      xfree (&fmt_caption);
		    }
		  else
		    {
		      xfree (&fmt_caption);
		      *until_equation = id;		      
		      break;
		    }
		} /* end if (obj_caption) */
	    }
	} /* end for */
    } /* gmode == GRAPHIC_3D */
  return fmt_text;
}
