/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<errno.h>
#include<assert.h>
#include<math.h>
#include<gtk/gtk.h>
#include"callbacks.h"
#include"gui.h"
#include"cairo_driver.h"
#include"controls.h"
#include"gnuplot.h"
#include"utils.h"

#ifdef _USE_GTK3_
#define gtk_widget_hide_all gtk_widget_hide
#endif

static double r_round (double x)
{
  /* 
     Round x to the nearest integer.
     If the fractional part of x is exactly equal     
     to 1/2, then round x away from zero.

     This function works like round(), but it
     supports also compilers which are not C99-compliant.
  */
  if (x >= 0.0)
    {
      if (x - floor (x) < 0.5)
	return floor (x);
      else
	return ceil (x);
    }
  else
    {
      if (x - floor (x) <= 0.5)
	return floor (x);
      else
	return ceil (x);
    }  
}

static grexitstatus create_math_object (par2d_object* p2d, par3d_object* p3d, 
					pointers_to_fields* infields)
{
  char *xeq_temp, *yeq_temp, *zeq_temp;
  const char *x_eq, *y_eq, *z_eq;
  const char *inf_s, *sup_s, *nsteps_s;
  const char *inf_t, *sup_t, *nsteps_t;
  const char* caption;
  double Inf_s, Sup_s, Inf_t, Sup_t;
  gruint Nsteps_s, Nsteps_t;
  grstyle style;
  int id, cartesian_mode_is_active;
  const char* errmsg;
  grexitstatus rv = GR_OK;

  xeq_temp = yeq_temp = zeq_temp = NULL;
  inf_s = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->inf_s_entry));
  sup_s = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->sup_s_entry));
  nsteps_s = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->nsteps_s_entry));
  inf_t = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->inf_t_entry));
  sup_t = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->sup_t_entry));
  nsteps_t = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->nsteps_t_entry));
  caption = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->caption_entry));
  cartesian_mode_is_active = 
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(infields->cart_radiob)) == TRUE ? 1 : 0;

  if ( gstrcmp ((const grchar*)inf_s, NULL) == 0 )
    Inf_s = 0.0;
  else
    {
      if ( (errmsg = str_2_double ((const grchar*)inf_s, &Inf_s)) )
	{
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("y, Min: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("s, Min: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  return GR_FAILURE;
	}
    }

  if ( gstrcmp ((const grchar*)sup_s, NULL) == 0 )
    Sup_s = 0.0;
  else
    {
      if ( (errmsg = str_2_double ((const grchar*)sup_s, &Sup_s)) )
	{
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("y, Max: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("s, Max: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  return GR_FAILURE;
	}
    }

  if ( gstrcmp ((const grchar*)inf_t, NULL) == 0 )
    Inf_t = 0.0;
  else
    {
      if ( (errmsg = str_2_double ((const grchar*)inf_t, &Inf_t)) )
	{
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("x, Min: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("t, Min: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  return GR_FAILURE;
	}
    }

  if ( gstrcmp ((const grchar*)sup_t, NULL) == 0 )
    Sup_t = 0.0;
  else
    {
      if ( (errmsg = str_2_double ((const grchar*)sup_t, &Sup_t)) )
	{
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("x, Max: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("t, Max: ", errmsg,
				 "\n\nA correction is required to proceed.");
	    }
	  return GR_FAILURE;
	}
    }

  if ( gstrcmp ((const grchar*)nsteps_s, NULL) == 0 )
    Nsteps_s = 0;
  else
    {
      if ( str_2_ulng ((const grchar*)nsteps_s, &Nsteps_s) != STR_2_OK )
	{
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("y, Number of subintervals: ", 
				 "Invalid number\n",
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("s, Number of subintervals: ", 
				 "Invalid number\n",
				 "\n\nA correction is required to proceed.");
	    }
	  return GR_FAILURE;
	}
    }

  if ( gstrcmp ((const grchar*)nsteps_t, NULL) == 0 )
    Nsteps_t = 0;
  else
    {
      if ( str_2_ulng ((const grchar*)nsteps_t, &Nsteps_t) != STR_2_OK )
	{
	  if ( (cartesian_mode_is_active) )
	    {
	      notify_math_error ("x, Number of subintervals: ", 
				 "Invalid number\n",
				 "\n\nA correction is required to proceed.");
	    }
	  else
	    {
	      notify_math_error ("t, Number of subintervals: ", 
				 "Invalid number\n",
				 "\n\nA correction is required to proceed.");
	    }
	  return GR_FAILURE;
	}
    }

  if ( gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(infields->cart_radiob)) == TRUE )
    {
      graphic_mode mode = get_graphic_mode();
      
      if (mode == GRAPHIC_2D)
	{
	  y_eq = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->y_entry));
	  cartesian_to_parametric (mode, y_eq, &xeq_temp, &yeq_temp, &zeq_temp);
	}
      else
	{
	  z_eq = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->z_entry));
	  cartesian_to_parametric (mode, z_eq, &xeq_temp, &yeq_temp, &zeq_temp);	  
	}
      x_eq = (const char*) xeq_temp;
      y_eq = (const char*) yeq_temp;
      z_eq = (const char*) zeq_temp;
    }
  else
    {
      x_eq = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->x_entry));
      y_eq = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->y_entry));
      z_eq = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->z_entry));
    }

  style = infields->style;
  id = get_current_object();
  if ( (p2d) )
    {
      par2d_object tmp2d;

      tmp2d = par2d_new(X_EVAL_TMP, Y_EVAL_TMP);
      tmp2d.color = id;
      tmp2d.style = style;
      rv = par2d_define (&tmp2d,
                         (grchar*)x_eq, (grchar*) y_eq,
			 Inf_s, Sup_s, Nsteps_s,
			 Inf_t, Sup_t, Nsteps_t,
			 style, id, (grchar*) caption);
      par2d_clear (&tmp2d);
      /* The one above is a trick to catch the case of an empty definition */
      if (rv != GR_FAILURE)
	{
	  rv = par2d_define (p2d, (grchar*)x_eq, (grchar*) y_eq,
			     Inf_s, Sup_s, Nsteps_s,
			     Inf_t, Sup_t, Nsteps_t,
			     style, id, (grchar*) caption);
	}
      xfree (&xeq_temp);
      xfree (&yeq_temp);
      xfree (&zeq_temp);
    }

  if (rv == GR_FAILURE)
    return rv;

  if ((p3d))
    {
      par3d_object tmp3d;

      tmp3d = par3d_new(X_EVAL_TMP, Y_EVAL_TMP, Z_EVAL_TMP);
      tmp3d.color = id;
      tmp3d.style = style;
      rv = par3d_define (&tmp3d,
                         (grchar*)x_eq, (grchar*) y_eq, (grchar*) z_eq,
			 Inf_s, Sup_s, Nsteps_s,
			 Inf_t, Sup_t, Nsteps_t,
			 style, id, (grchar*) caption);
      par3d_clear (&tmp3d);
      /* The one above is a trick to catch the case of an empty definition */
      if (rv != GR_FAILURE)
	{
	  rv = par3d_define (p3d, (grchar*)x_eq, (grchar*) y_eq, (grchar*) z_eq,
			     Inf_s, Sup_s, Nsteps_s,
			     Inf_t, Sup_t, Nsteps_t,
			     style, id, (grchar*) caption);
	}
      xfree (&xeq_temp);
      xfree (&yeq_temp);
      xfree (&zeq_temp);
    }
  return rv;
}

static grexitstatus save_input_fields (pointers_to_fields* infields)
{
  char *xeq_temp, *yeq_temp, *zeq_temp;
  const char *x_eq, *y_eq, *z_eq;
  const char *inf_s, *sup_s, *nsteps_s;
  const char *inf_t, *sup_t, *nsteps_t;
  const char* caption;
  int cartesian_mode_is_active;
  grexitstatus rv;

  xeq_temp = yeq_temp = zeq_temp = NULL;
  if ( gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(infields->cart_radiob)) == TRUE )
    {
      graphic_mode mode = get_graphic_mode();
      
      if (mode == GRAPHIC_2D)
	{
	  y_eq = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->y_entry));
	  cartesian_to_parametric (mode, y_eq, &xeq_temp, &yeq_temp, &zeq_temp);
	}
      else
	{
	  z_eq = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->z_entry));
	  cartesian_to_parametric (mode, z_eq, &xeq_temp, &yeq_temp, &zeq_temp);	  
	}
      x_eq = (const char*) xeq_temp;
      y_eq = (const char*) yeq_temp;
      z_eq = (const char*) zeq_temp;
      cartesian_mode_is_active = 1;
    }
  else
    {
      x_eq = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->x_entry));
      y_eq = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->y_entry));
      z_eq = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->z_entry));
      cartesian_mode_is_active = 0;
    }
  if ( ((x_eq) && !is_string_blank(x_eq)) || 
       ((y_eq) && !is_string_blank(y_eq)) || 
       ((z_eq) && !is_string_blank(z_eq)) )
    {
      inf_s = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->inf_s_entry));
      sup_s = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->sup_s_entry));
      nsteps_s = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->nsteps_s_entry));
      inf_t = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->inf_t_entry));
      sup_t = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->sup_t_entry));
      nsteps_t = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->nsteps_t_entry));
      caption = (const char*) gtk_entry_get_text (GTK_ENTRY(infields->caption_entry));
    }
  else
    {
      x_eq = y_eq = z_eq = NULL;
      inf_s = sup_s = inf_t = sup_t = nsteps_s = nsteps_t = caption = NULL;
    }
  rv = define_math_object (x_eq, y_eq, z_eq, 
			   inf_s, sup_s, nsteps_s,
			   inf_t, sup_t, nsteps_t, 
			   infields->style, caption,
			   cartesian_mode_is_active);
  xfree (&xeq_temp);
  xfree (&yeq_temp);
  xfree (&zeq_temp);
  return rv;
}

static void set_input_fields (pointers_to_fields* infields, mathobj_description desc)
{
  char* eq;
  int rv;
  graphic_mode mode = get_graphic_mode();
  char buff_nsteps_s[LABEL_SIZE] = "";
  char buff_nsteps_t[LABEL_SIZE] = "";
  unsigned long value;

  gtk_widget_show_all (infields->x_hbox);
  gtk_widget_show_all (infields->y_hbox);
  gtk_widget_show_all (infields->z_hbox);
  gtk_widget_show_all (infields->s_frame);
  if (desc.type == GRMATH_EMPTY)
    {
      // If the description is empty, then preserve
      // the current representation mode. In addition,
      // initialize EQ to "". 
      eq = xstrdup ("");
      rv = (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(infields->cart_radiob)) == TRUE ? 0 : -1);
    }
  else
    {
      rv = parametric_to_cartesian (mode, desc.type,
				    desc.x_eq, desc.y_eq, desc.z_eq,
				    &eq);
    }
  if (rv == 0)
    {
      /* DESC is the description of a cartesian object */
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(infields->cart_radiob), TRUE);      
      if (mode == GRAPHIC_2D)
	{
	  gtk_widget_hide_all (infields->x_hbox);
	  gtk_widget_hide_all (infields->z_hbox);	  
	  gtk_label_set_text (GTK_LABEL(infields->y_label), "y(x) =   ");
	  gtk_widget_hide_all (infields->s_frame);
	  gtk_frame_set_label (GTK_FRAME(infields->t_frame), "Variation Interval of x");	 
	  gtk_entry_set_text (GTK_ENTRY(infields->y_entry), 
			      (const gchar*)eq);
	}
      else
	{
	  gtk_widget_hide_all (infields->x_hbox);
	  gtk_widget_hide_all (infields->y_hbox);	  
	  gtk_label_set_text (GTK_LABEL(infields->z_label), "z(x,y) = ");
	  gtk_frame_set_label (GTK_FRAME(infields->s_frame), "Variation Interval of x");
	  gtk_frame_set_label (GTK_FRAME(infields->t_frame), "Variation Interval of y");
	  gtk_entry_set_text (GTK_ENTRY(infields->z_entry), 
			      (const gchar*)eq);	  
	}
      xfree (&eq);
    }
  else
    {
      /* DESC is the description of a parametric object */
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(infields->param_radiob), TRUE);      
      gtk_label_set_text (GTK_LABEL(infields->y_label), "y(s,t) = ");
      gtk_label_set_text (GTK_LABEL(infields->z_label), "z(s,t) = ");
      gtk_frame_set_label (GTK_FRAME(infields->s_frame), "Variation Interval of s");
      gtk_frame_set_label (GTK_FRAME(infields->t_frame), "Variation Interval of t");	 
      gtk_entry_set_text (GTK_ENTRY(infields->x_entry), 
			  (const gchar*)desc.x_eq);
      gtk_entry_set_text (GTK_ENTRY(infields->y_entry), 
			  (const gchar*)desc.y_eq);
      gtk_entry_set_text (GTK_ENTRY(infields->z_entry), 
			  (const gchar*)desc.z_eq);
    }

  gtk_entry_set_text (GTK_ENTRY(infields->inf_s_entry), 
		      (const gchar*)desc.inf_s);
  gtk_entry_set_text (GTK_ENTRY(infields->sup_s_entry), 
		      (const gchar*)desc.sup_s);
  if ( str_2_ulng (desc.nsteps_s, &value) == STR_2_OK && value == 0 )
    {
      sprintf (buff_nsteps_s, "%d", DEFAULT_NSTEPS_ST); 
      gtk_entry_set_text (GTK_ENTRY(infields->nsteps_s_entry), 
			  (const gchar*)buff_nsteps_s);      
    }
  else
    {
      gtk_entry_set_text (GTK_ENTRY(infields->nsteps_s_entry), 
			  (const gchar*)desc.nsteps_s);
    }

  gtk_entry_set_text (GTK_ENTRY(infields->inf_t_entry), 
		      (const gchar*)desc.inf_t);
  gtk_entry_set_text (GTK_ENTRY(infields->sup_t_entry), 
		      (const gchar*)desc.sup_t);
  if ( str_2_ulng (desc.nsteps_t, &value) == STR_2_OK && value == 0 )
    {
      if (mode == GRAPHIC_2D)
	sprintf (buff_nsteps_t, "%d", DEFAULT_NSTEPS);
      else
	sprintf (buff_nsteps_t, "%d", DEFAULT_NSTEPS_ST); 
      gtk_entry_set_text (GTK_ENTRY(infields->nsteps_t_entry), 
			  (const gchar*)buff_nsteps_t);      
    }
  else
    {
      gtk_entry_set_text (GTK_ENTRY(infields->nsteps_t_entry), 
			  (const gchar*)desc.nsteps_t);
    }

  gtk_entry_set_text (GTK_ENTRY(infields->caption_entry), 
		      (const gchar*)desc.caption);
  infields->style = (desc.style == GRSTYLE_NONE ? DEFAULT_STYLE : desc.style);
  switch (infields->style)
    {
    case GRSTYLE_DOTTED:
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (infields->dotted_radiob),
				    TRUE);
      break;
    case GRSTYLE_FILLED:
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (infields->filled_radiob),
				    TRUE);
      break;
    case GRSTYLE_STROKED:
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (infields->stroked_radiob),
				    TRUE);
      break;
    default: /* This should never be the case */
      handle_fatal_error ("Fatal error: invalid style value,\n" \
			  "please quit now.\n"); 
    }
}

static void restore_input_fields (pointers_to_fields* infields)
{
  mathobj_description desc;

  desc = retrieve_math_object ();
  set_input_fields (infields, desc);
}

gboolean on_close (GtkWidget* wd, GdkEvent* ev, gpointer user_data)
{
  return FALSE; /* Emit destroy signal */
}

void on_display (GtkWidget* wd, gpointer user_data)
{
  pointers_to_fields* infields = (pointers_to_fields*)user_data;

  if (save_input_fields (infields) == GR_FAILURE)
    {
      show_warning_message ( get_last_math_error() );
    }
  else if ( upgrade_pipeline () == GR_FAILURE)
    {
      handle_fatal_error ("The program has not enough memory to keep on working\n" \
			  "please quit now.\n");
    }
  else
    {
      graphic_mode graph_mode = get_graphic_mode();

      refresh_lists_available_ids (graph_mode);
      reinit_captions (graph_mode);
      mgr_filename("");
      set_legend();
      gtk_notebook_set_current_page (GTK_NOTEBOOK(infields->notebook), 1);
    }  
}

void on_back_input_mask (GtkWidget* wd, gpointer user_data)
{
  gtk_notebook_set_current_page (GTK_NOTEBOOK(user_data), 0);
}

static int dry_run = 0;

void on_set_2dmode (GtkWidget* wd, gpointer user_data)
{
  pointers_to_fields* fields = (pointers_to_fields*)user_data;
  int current_selection;
  double rotangle, zoom;
  gboolean show_axes, show_grid, show_tics, show_jumps;

  if ( gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(wd)) == TRUE )
    {
      if (!dry_run)
        {
          if (save_input_fields (fields) == GR_FAILURE)
            {
              show_warning_message ( get_last_math_error() );
              dry_run = 1;
              switch_3dgmode_button_on ();
            }
          else
            {
              current_selection = get_current_object ();
              set_graphic_mode (GRAPHIC_2D);
              if (current_selection >= NUMBER_OF_2DOBJECTS)
                {
                  set_current_object (NUMBER_OF_2DOBJECTS - 1);
                  gtk_combo_box_set_active (GTK_COMBO_BOX(fields->selection_combo_box),
                                            NUMBER_OF_2DOBJECTS - 1);
                }
              else
                set_current_object (current_selection);
              restore_input_fields (fields);
              gtk_widget_set_sensitive (fields->z_entry, FALSE); /* This is ok also when using cartesian representations */

              if ( gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(fields->cart_radiob)) == TRUE )
                {
                  /* gtk_widget_hide_all (fields->x_hbox); */
                  gtk_widget_show_all (fields->y_hbox);	  
                  gtk_widget_hide_all (fields->z_hbox);	  
                  gtk_label_set_text (GTK_LABEL(fields->y_label), "y(x) =   ");
                  gtk_widget_hide_all (fields->s_frame);
                  gtk_frame_set_label (GTK_FRAME(fields->t_frame), "Variation Interval of x");	 
                }

              gtk_widget_set_sensitive (fields->long_spinb, FALSE);
              gtk_spin_button_set_value (GTK_SPIN_BUTTON(fields->long_spinb), DEFAULT_LONGITUDE); 

              gtk_widget_set_sensitive (fields->lat_spinb, FALSE);
              gtk_spin_button_set_value (GTK_SPIN_BUTTON(fields->lat_spinb), DEFAULT_LATITUDE);

              rotangle = r_round(get_rotation_angle() / CONV_FACTOR_D2R);
              gtk_spin_button_set_value (GTK_SPIN_BUTTON(fields->rotangle_spinb), rotangle);

              zoom = get_zoom_factor();
              gtk_range_set_value (GTK_RANGE(fields->zoom_hscale), zoom);

              gtk_widget_set_sensitive (fields->smoke_hscale, FALSE);
              gtk_range_set_value (GTK_RANGE(fields->smoke_hscale), DEFAULT_SMOKE_DENSITY);

              gtk_button_set_label (GTK_BUTTON(fields->axes_checkb),
                                    "x and y directions");
              show_axes = get_show_main_directions();
              gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(fields->axes_checkb), show_axes);

              show_grid = get_show_grid();
              gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(fields->grid_checkb), show_grid);

              show_tics = get_show_tics();
              gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(fields->tics_checkb), show_tics);

              gtk_widget_set_sensitive (fields->jumps_checkb, TRUE);
              show_jumps = get_show_discontinuities();
              gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(fields->jumps_checkb), show_jumps);

              gtk_widget_set_sensitive (fields->zimage_up_button, FALSE);
              gtk_widget_set_sensitive (fields->zimage_down_button, FALSE);

              set_legend();
            }
        }
      else
        dry_run = 0;
    }
}

void on_set_3dmode (GtkWidget* wd, gpointer user_data)
{
  pointers_to_fields* fields = (pointers_to_fields*)user_data;
  int current_selection;
  double longitude, latitude, rotangle, zoom, smoke_density;
  gboolean show_axes, show_grid, show_tics;

  if ( gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(wd)) == TRUE )
    {
      if (!dry_run)
        {
          if (save_input_fields (fields) == GR_FAILURE)
            {
              show_warning_message ( get_last_math_error() );
              dry_run = 1;
              switch_2dgmode_button_on();
            }
          else
            {
              current_selection = get_current_object ();
              set_graphic_mode (GRAPHIC_3D);
              if (current_selection >= NUMBER_OF_3DOBJECTS)
                {
                  set_current_object (NUMBER_OF_3DOBJECTS - 1);
                  gtk_combo_box_set_active (GTK_COMBO_BOX(fields->selection_combo_box),
                                            NUMBER_OF_3DOBJECTS - 1);
                }
              else
                set_current_object (current_selection);
              restore_input_fields (fields);
              gtk_widget_set_sensitive (fields->z_entry, TRUE);

              if ( gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(fields->cart_radiob)) == TRUE )
                {
                  /* gtk_widget_hide_all (fields->x_hbox); */
                  gtk_widget_hide_all (fields->y_hbox);	  
                  gtk_widget_show_all (fields->z_hbox);
                  gtk_label_set_text (GTK_LABEL(fields->z_label), "z(x,y) = ");
                  gtk_widget_show_all (fields->s_frame);	  	  
                  gtk_frame_set_label (GTK_FRAME(fields->s_frame), "Variation Interval of x");
                  gtk_frame_set_label (GTK_FRAME(fields->t_frame), "Variation Interval of y");	  	
                }

              gtk_widget_set_sensitive (fields->long_spinb, TRUE);
              longitude = r_round (get_longitude() / CONV_FACTOR_D2R);
              gtk_spin_button_set_value (GTK_SPIN_BUTTON(fields->long_spinb), longitude); 
              gtk_widget_set_sensitive (fields->lat_spinb, TRUE);
              latitude = r_round (get_latitude() / CONV_FACTOR_D2R);
              gtk_spin_button_set_value (GTK_SPIN_BUTTON(fields->lat_spinb), latitude);

              rotangle = r_round (get_rotation_angle() / CONV_FACTOR_D2R);
              gtk_spin_button_set_value (GTK_SPIN_BUTTON(fields->rotangle_spinb), rotangle);

              zoom = get_zoom_factor();
              gtk_range_set_value (GTK_RANGE(fields->zoom_hscale), zoom);

              gtk_widget_set_sensitive (fields->smoke_hscale, TRUE);
              smoke_density = get_smoke_density();
              gtk_range_set_value (GTK_RANGE(fields->smoke_hscale), smoke_density);

              gtk_button_set_label (GTK_BUTTON(fields->axes_checkb),
                                    "x, y and z directions");
              show_axes = get_show_main_directions();
              gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(fields->axes_checkb), show_axes);

              show_grid = get_show_grid();
              gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(fields->grid_checkb), show_grid);

              show_tics = get_show_tics();
              gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(fields->tics_checkb), show_tics);

              gtk_widget_set_sensitive (fields->jumps_checkb, FALSE);
              gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(fields->jumps_checkb), DEFAULT_SHOW_DISC);

              gtk_widget_set_sensitive (fields->zimage_up_button, TRUE);
              gtk_widget_set_sensitive (fields->zimage_down_button, TRUE);

              set_legend();
            }
        }
      else
        dry_run = 0;
    }
}

void on_changed_selection (GtkWidget* wd, gpointer user_data)
{
  pointers_to_fields* infields = (pointers_to_fields*)user_data;
  graphic_mode mode;
  gint current_selection, new_selection, limit;

  new_selection = gtk_combo_box_get_active (GTK_COMBO_BOX(wd));
  if (new_selection < 0)
    {
      new_selection = DEFAULT_CURRENT_OBJECT; /* This is 0 */
      gtk_combo_box_set_active (GTK_COMBO_BOX(wd), new_selection);
    }
  mode = get_graphic_mode ();
  limit = (mode == GRAPHIC_2D ? NUMBER_OF_2DOBJECTS : NUMBER_OF_3DOBJECTS);
  if (new_selection >= limit)
    {
      new_selection = limit - 1;
      gtk_combo_box_set_active (GTK_COMBO_BOX(wd), new_selection);
    }
  current_selection = get_current_object ();
  if (new_selection != current_selection)
    {
      if (save_input_fields (infields) == GR_FAILURE)
	{
	  show_warning_message ( get_last_math_error() );
	  gtk_combo_box_set_active (GTK_COMBO_BOX(wd), current_selection);
	}
      else
	{
	  set_current_object (new_selection);
	  restore_input_fields (infields);
	}
    } /* end  if (new_selection != current_selection) */
}

void on_set_cartesian (GtkWidget* wd, gpointer user_data)
{
  pointers_to_fields* fields = (pointers_to_fields*)user_data;
  char buff_s[LABEL_SIZE] = "";
  char buff_t[LABEL_SIZE] = "";
  graphic_mode mode;

  if ( gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(wd)) == TRUE )
    {
      /* Switch from parametric to cartesian mode */
      mode = get_graphic_mode();
      if (mode == GRAPHIC_2D)
	{
	  gtk_widget_hide_all (fields->x_hbox);
	  gtk_widget_hide_all (fields->z_hbox);	  
	  gtk_label_set_text (GTK_LABEL(fields->y_label), "y(x) =   ");
	  gtk_widget_hide_all (fields->s_frame);
	  gtk_frame_set_label (GTK_FRAME(fields->t_frame), "Variation Interval of x");	 
	  gtk_entry_set_text (GTK_ENTRY(fields->y_entry), "");

	  sprintf (buff_t, FORMAT_STRING_INT, DEFAULT_NSTEPS);
	  gtk_entry_set_text (GTK_ENTRY(fields->nsteps_t_entry), 
			      (const gchar*)buff_t);      
	  sprintf (buff_t, FORMAT_STRING_DBL, DEFAULT_INF);
	  gtk_entry_set_text (GTK_ENTRY(fields->inf_t_entry), 
			      (const gchar*)buff_t);      
	  sprintf (buff_t, FORMAT_STRING_DBL, DEFAULT_SUP);
	  gtk_entry_set_text (GTK_ENTRY(fields->sup_t_entry), 
			      (const gchar*)buff_t);      
	}
      else
	{
	  gtk_widget_hide_all (fields->x_hbox);
	  gtk_widget_hide_all (fields->y_hbox);	  
	  gtk_label_set_text (GTK_LABEL(fields->z_label), "z(x,y) = ");
	  gtk_frame_set_label (GTK_FRAME(fields->s_frame), "Variation Interval of x");
	  gtk_frame_set_label (GTK_FRAME(fields->t_frame), "Variation Interval of y");
	  gtk_entry_set_text (GTK_ENTRY(fields->z_entry), "");

	  sprintf (buff_s, FORMAT_STRING_INT, DEFAULT_NSTEPS_ST); 
	  gtk_entry_set_text (GTK_ENTRY(fields->nsteps_s_entry), 
			      (const gchar*)buff_s); 
	  sprintf (buff_s, FORMAT_STRING_DBL, DEFAULT_INF);
	  gtk_entry_set_text (GTK_ENTRY(fields->inf_s_entry), 
			      (const gchar*)buff_s);      
	  sprintf (buff_s, FORMAT_STRING_DBL, DEFAULT_SUP);
	  gtk_entry_set_text (GTK_ENTRY(fields->sup_s_entry), 
			      (const gchar*)buff_s);      

	  sprintf (buff_t, FORMAT_STRING_INT, DEFAULT_NSTEPS_ST); 
	  gtk_entry_set_text (GTK_ENTRY(fields->nsteps_t_entry), 
			      (const gchar*)buff_t);      
	  sprintf (buff_t, FORMAT_STRING_DBL, DEFAULT_INF);
	  gtk_entry_set_text (GTK_ENTRY(fields->inf_t_entry), 
			      (const gchar*)buff_t);      
	  sprintf (buff_t, FORMAT_STRING_DBL, DEFAULT_SUP);
	  gtk_entry_set_text (GTK_ENTRY(fields->sup_t_entry), 
			      (const gchar*)buff_t); 
	}
      gtk_entry_set_text (GTK_ENTRY(fields->caption_entry), "");

      if (DEFAULT_STYLE == GRSTYLE_DOTTED)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (fields->dotted_radiob),
				      TRUE);
      else if (DEFAULT_STYLE == GRSTYLE_STROKED)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (fields->stroked_radiob), 
				      TRUE);
      else
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (fields->filled_radiob),
				      TRUE);
    }
}

void on_set_parametric (GtkWidget* wd, gpointer user_data)
{
  pointers_to_fields* fields = (pointers_to_fields*)user_data;
  char buff_s[LABEL_SIZE] = "";
  char buff_t[LABEL_SIZE] = "";
  graphic_mode mode;

  if ( gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(wd)) == TRUE )
    {
      gtk_entry_set_text (GTK_ENTRY(fields->x_entry), "");
      gtk_entry_set_text (GTK_ENTRY(fields->y_entry), "");
      /* Switch from cartesian to parametric mode */
      mode = get_graphic_mode();
      if (mode == GRAPHIC_2D)
	{
	  gtk_widget_show_all (fields->x_hbox);
	  gtk_widget_show_all (fields->z_hbox);	  
	  gtk_label_set_text (GTK_LABEL(fields->y_label), "y(s,t) = ");
	  gtk_label_set_text (GTK_LABEL(fields->z_label), "z(s,t) = ");
	  gtk_widget_show_all (fields->s_frame);
	  gtk_frame_set_label (GTK_FRAME(fields->s_frame), "Variation Interval of s");
	  gtk_frame_set_label (GTK_FRAME(fields->t_frame), "Variation Interval of t");	 

	  sprintf (buff_t, FORMAT_STRING_INT, DEFAULT_NSTEPS);
	  gtk_entry_set_text (GTK_ENTRY(fields->nsteps_t_entry), 
			      (const gchar*)buff_t);      
	}
      else
	{
	  gtk_widget_show_all (fields->x_hbox);
	  gtk_widget_show_all (fields->y_hbox);	  
	  gtk_label_set_text (GTK_LABEL(fields->y_label), "y(s,t) = ");
	  gtk_label_set_text (GTK_LABEL(fields->z_label), "z(s,t) = ");
	  gtk_frame_set_label (GTK_FRAME(fields->s_frame), "Variation Interval of s");
	  gtk_frame_set_label (GTK_FRAME(fields->t_frame), "Variation Interval of t");
	  gtk_entry_set_text (GTK_ENTRY(fields->z_entry), "");

	  sprintf (buff_s, FORMAT_STRING_INT, DEFAULT_NSTEPS_ST); 
	  gtk_entry_set_text (GTK_ENTRY(fields->nsteps_s_entry), 
			      (const gchar*)buff_s); 

	  sprintf (buff_t, FORMAT_STRING_INT, DEFAULT_NSTEPS_ST); 
	  gtk_entry_set_text (GTK_ENTRY(fields->nsteps_t_entry), 
			      (const gchar*)buff_t);      
	}
      sprintf (buff_s, FORMAT_STRING_DBL, DEFAULT_INF);
      gtk_entry_set_text (GTK_ENTRY(fields->inf_s_entry), 
			  (const gchar*)buff_s);      
      sprintf (buff_s, FORMAT_STRING_DBL, DEFAULT_SUP);
      gtk_entry_set_text (GTK_ENTRY(fields->sup_s_entry), 
			  (const gchar*)buff_s);      

      sprintf (buff_t, FORMAT_STRING_DBL, DEFAULT_INF);
      gtk_entry_set_text (GTK_ENTRY(fields->inf_t_entry), 
			  (const gchar*)buff_t);      
      sprintf (buff_t, FORMAT_STRING_DBL, DEFAULT_SUP);
      gtk_entry_set_text (GTK_ENTRY(fields->sup_t_entry), 
			  (const gchar*)buff_t);      
      
      gtk_entry_set_text (GTK_ENTRY(fields->caption_entry), "");

      if (DEFAULT_STYLE == GRSTYLE_DOTTED)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (fields->dotted_radiob),
				      TRUE);
      else if (DEFAULT_STYLE == GRSTYLE_STROKED)
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (fields->stroked_radiob), 
				      TRUE);
      else
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (fields->filled_radiob),
				      TRUE);
    }
}

void on_set_dotted_style (GtkWidget* wd, gpointer user_data)
{
  grstyle *pstyle = (grstyle*)user_data;
  
  *pstyle = GRSTYLE_DOTTED;
}

void on_set_stroked_style (GtkWidget* wd, gpointer user_data)
{
  grstyle *pstyle = (grstyle*)user_data;
  
  *pstyle = GRSTYLE_STROKED;
}

void on_set_filled_style (GtkWidget* wd, gpointer user_data)
{
  grstyle *pstyle = (grstyle*)user_data;
  
  *pstyle = GRSTYLE_FILLED;
}

#define TYPE_UNKNOWN 0
#define TYPE_IS_PS   1
#define TYPE_IS_PDF  2
#define TYPE_IS_SVG  3
#define TYPE_IS_MGR  4
#define TYPE_IS_MDEF 5
#define TYPE_IS_GDAT 6

/*
   The following function recognizes the tyoe of a file
   using its extension.
*/

static
int identify_file_type (const gchar* filename)
{
  gchar* ptr = NULL;
    
  ptr = g_strrstr (filename, ".ps");
  if ((ptr) && *(ptr+3) == '\0')
    return TYPE_IS_PS;
    
  ptr = g_strrstr (filename, ".eps");
  if ((ptr) && *(ptr+4) == '\0')
    return TYPE_IS_PS;
    
  ptr = g_strrstr (filename, ".pdf");
  if ((ptr) && *(ptr+4) == '\0')
    return TYPE_IS_PDF;    
    
  ptr = g_strrstr (filename, ".svg");
  if ((ptr) && *(ptr+4) == '\0')
    return TYPE_IS_SVG;

  ptr = g_strrstr (filename, ".mgr");
  if ((ptr) && *(ptr+4) == '\0')
    return TYPE_IS_MGR;

  ptr = g_strrstr (filename, ".mdef");
  if ((ptr) && *(ptr+5) == '\0')
    return TYPE_IS_MDEF;

  ptr = g_strrstr (filename, ".gdat");
  if ((ptr) && *(ptr+5) == '\0')
    return TYPE_IS_GDAT;
    
  return TYPE_UNKNOWN;
}

static
grexitstatus load_3d (const char* filename, pointers_to_fields* infields)
{
  FILE* fp;
  par3d_object obj;
  grmathtype obj_type;
  mathobj_description desc;

  obj = par3d_new(X_EVAL_TMP, Y_EVAL_TMP, Z_EVAL_TMP);
  if ( !(fp = fopen (filename, "r")) )
    {
      par3d_clear (&obj);
      return GR_FAILURE;
    }
  if ( (obj_type = par3d_load (&obj, fp)) == GRMATH_BAD )
    {
      par3d_clear (&obj);
      fclose(fp);
      return GR_FAILURE;
    }
  if ( (fclose(fp)) )
    {
      par3d_clear (&obj);
      return GR_FAILURE;    
    }
  desc = get_description_3dmath_object (&obj);
  set_input_fields (infields, desc);
  par3d_clear (&obj);
  return GR_OK;
}

static
grexitstatus load_2d (const char* filename, pointers_to_fields* infields)
{
  FILE* fp;
  par2d_object obj;
  grmathtype obj_type;
  mathobj_description desc;

  obj = par2d_new(X_EVAL_TMP, Y_EVAL_TMP);
  if ( !(fp = fopen (filename, "r")) )
    {
      par2d_clear (&obj);
      return GR_FAILURE;
    }
  if ( (obj_type = par2d_load (&obj, fp)) == GRMATH_BAD )
    {
      par2d_clear (&obj);
      fclose(fp);
      return GR_FAILURE;
    }
  if ( (fclose(fp)) )
    {
      par2d_clear (&obj);
      return GR_FAILURE;
    }
  desc = get_description_2dmath_object (&obj);
  set_input_fields (infields, desc);
  par2d_clear (&obj);
  return GR_OK;
}

void on_load_definition (GtkWidget* wd, gpointer user_data)
{
  GtkWidget* dialog;
  pointers_to_fields* infields = (pointers_to_fields*)user_data;
  GtkFileFilter *filter_mdef;
  GtkWidget* top_win = gtk_widget_get_ancestor (wd, GTK_TYPE_WINDOW);
  gchar* filename;
  int filetype;
  graphic_mode mode;
  grexitstatus status;
  static gchar* current_folder = NULL;

  dialog = gtk_file_chooser_dialog_new ("Load definition of a mathematical object from a file",
					GTK_WINDOW(top_win), 
					GTK_FILE_CHOOSER_ACTION_OPEN, 
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, 
					NULL);
    
  /* Add filters for MDEF files */  
  filter_mdef = gtk_file_filter_new();
  gtk_file_filter_set_name (filter_mdef, "GTKmathplot definition files (*.mdef)");
  gtk_file_filter_add_pattern (filter_mdef, "*.mdef");  
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(dialog), filter_mdef);
  /* Set current directory */
  if ((current_folder))
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(dialog), current_folder);   
  else
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(dialog), g_get_home_dir());   
  if ( gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT )
    {
      current_folder = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER(dialog));
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER(dialog));
      filetype = identify_file_type (filename);
      if (filetype == TYPE_IS_MDEF)
	{
	  errno = 0;
	  mode = get_graphic_mode();
	  if (mode == GRAPHIC_2D)
	    status = load_2d ((const char*)filename, infields);
	  else
	    status = load_3d ((const char*)filename, infields);
	  if (status == GR_FAILURE)
	    {
	      if ((errno))
		{
		  gchar* message = g_strjoin (" ", "Cannot load file:",
					      g_strerror (errno), NULL);
		  
		  show_warning_message (message);
		  g_free (message);
		}
	      else
		show_warning_message ("Cannot load file: invalid format");
	    }
	}
      else
	show_warning_message ("Cannot proceed: GTKmathplot definition files\n"\
			      "should have extension .mdef");
    }
  gtk_widget_destroy (dialog);  
}

static
grexitstatus save_3d (const char* filename, pointers_to_fields* infields)
{
  FILE* fp;
  par3d_object obj;
  grexitstatus status;

  obj = par3d_new(X_EVAL_TMP, Y_EVAL_TMP, Z_EVAL_TMP);
  if ( (status = create_math_object (NULL, &obj, infields)) == GR_FAILURE )
    {
      par3d_clear (&obj);
      return GR_EXCEPTION;
    }
  if ( !(fp = fopen (filename, "w")) )
    {
      par3d_clear (&obj);
      return GR_FAILURE;
    }
  if ( (status = par3d_save (obj, fp)) == GR_FAILURE )
    {
      par3d_clear (&obj);
      fclose(fp);
      return GR_FAILURE;
    }
  par3d_clear (&obj);
  if ( (fclose(fp)) )
    return GR_FAILURE;
  else
    return GR_OK;
}

static
grexitstatus eval_save_3d (const char* filename, pointers_to_fields* infields)
{
  FILE* fp;
  par3d_object obj;
  grexitstatus status;

  obj = par3d_new(X_EVAL_TMP, Y_EVAL_TMP, Z_EVAL_TMP);
  if ( (status = create_math_object (NULL, &obj, infields)) == GR_FAILURE )
    {
      par3d_clear (&obj);
      return GR_EXCEPTION;
    }
  if (obj.type == GRMATH_CURVE && obj.style == GRSTYLE_FILLED)
    obj.style = GRSTYLE_STROKED;
  if ( !(fp = fopen (filename, "w")) )
    {
      par3d_clear (&obj);
      return GR_FAILURE;
    }
  if ( (status = par3d_eval_save (obj, fp)) == GR_FAILURE )
    {
      par3d_clear (&obj);
      fclose (fp);
      return GR_FAILURE;
    }
  par3d_clear (&obj);
  if ( (fclose(fp)) )
    return GR_FAILURE;
  else
    return GR_OK;
}

static
grexitstatus save_2d (const char* filename, pointers_to_fields* infields)
{
  FILE* fp;
  par2d_object obj;
  grexitstatus status;

  obj = par2d_new(X_EVAL_TMP, Y_EVAL_TMP);
  if ( (status = create_math_object (&obj, NULL, infields)) == GR_FAILURE )
    {
      par2d_clear (&obj);
      return GR_EXCEPTION;
    }
  if ( !(fp = fopen (filename, "w")) )
    {
      par2d_clear (&obj);
      return GR_FAILURE;
    }
  if ( (status = par2d_save (obj, fp)) == GR_FAILURE )
    {
      par2d_clear (&obj);
      fclose (fp);
      return GR_FAILURE;
    }
  par2d_clear (&obj);
  if ( (fclose(fp)) )
    return GR_FAILURE;
  else
    return GR_OK;
}

static
grexitstatus eval_save_2d (const char* filename, pointers_to_fields* infields)
{
  FILE* fp;
  par2d_object obj;
  grexitstatus status;

  obj = par2d_new(X_EVAL_TMP, Y_EVAL_TMP);
  if ( (status = create_math_object (&obj, NULL, infields)) == GR_FAILURE )
    {
      par2d_clear (&obj);
      return GR_EXCEPTION;
    }
  if (obj.type == GRMATH_CURVE && obj.style == GRSTYLE_FILLED)
    obj.style = GRSTYLE_STROKED;
  if ( !(fp = fopen (filename, "w")) )
    {
      par2d_clear (&obj);
      return GR_FAILURE;
    }
  if ( (status = par2d_eval_save (obj, fp)) == GR_FAILURE )
    {
      par2d_clear (&obj);
      fclose (fp);
      return GR_FAILURE;
    }
  par2d_clear (&obj);
  if ( (fclose(fp)) )
    return GR_FAILURE;
  else
    return GR_OK;
}

void on_save_definition (GtkWidget* wd, gpointer user_data)
{
  GtkWidget* dialog;
  pointers_to_fields* infields = (pointers_to_fields*)user_data;
  GtkFileFilter *filter_mdef;
  GtkWidget* top_win = gtk_widget_get_ancestor (wd, GTK_TYPE_WINDOW);
  gchar* filename;
  int filetype;
  graphic_mode mode;
  grexitstatus status;
  static gchar* current_folder = NULL;

  dialog = gtk_file_chooser_dialog_new ("Save definition to a file", GTK_WINDOW(top_win), 
					GTK_FILE_CHOOSER_ACTION_SAVE, 
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
					GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, 
					NULL);
    
  /* Add filters for MDEF files */  
  filter_mdef = gtk_file_filter_new();
  gtk_file_filter_set_name (filter_mdef, "GTKmathplot definition files (*.mdef)");
  gtk_file_filter_add_pattern (filter_mdef, "*.mdef");  
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(dialog), filter_mdef);
  /* Set current directory */
  if ((current_folder))
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(dialog), current_folder);   
  else
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(dialog), g_get_home_dir());   
  /* Ask for confirmation before overwriting an existing file */
  gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER(dialog), TRUE);
  if ( gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT )
    {
      current_folder = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER(dialog));
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER(dialog));
      filetype = identify_file_type (filename);
      if (filetype == TYPE_IS_MDEF)
	{
	  errno = 0;
	  mode = get_graphic_mode();
	  if (mode == GRAPHIC_2D)
	    status = save_2d ((const char*)filename, infields);
	  else
	    status = save_3d ((const char*)filename, infields);
	  if (status == GR_FAILURE)
	    {
	      if ((errno))
		{
		  gchar* message = g_strjoin (" ", 
					      "Cannot save definition to file:",
					      g_strerror (errno), NULL);
		  
		  show_warning_message (message);
		  g_free (message);
		}
	      else
		show_warning_message ("Sorry, it was not possible to save\n" \
				      "the definition to the given file");
	    }
	  if (status == GR_EXCEPTION)
	    show_warning_message ( get_last_math_error() );
	  /* Do nothing if status == GR_OK */
	}
      else
	show_warning_message ("Cannot proceed: GTKmathplot definition files\n"\
			      "should have extension .mdef");
    }
  gtk_widget_destroy (dialog);
}

void on_save_evaluation (GtkWidget* wd, gpointer user_data)
{
  GtkWidget* dialog;
  pointers_to_fields* infields = (pointers_to_fields*)user_data;
  GtkFileFilter *filter_gdat;
  GtkWidget* top_win = gtk_widget_get_ancestor (wd, GTK_TYPE_WINDOW);
  gchar* filename;
  int filetype;
  graphic_mode mode;
  grexitstatus status;
  static gchar* current_folder = NULL;

  dialog = gtk_file_chooser_dialog_new ("Evaluate and save results to a file", GTK_WINDOW(top_win), 
					GTK_FILE_CHOOSER_ACTION_SAVE, 
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
					GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, 
					NULL);
    
  /* Add filters for GDAT files */  
  filter_gdat = gtk_file_filter_new();
  gtk_file_filter_set_name (filter_gdat, "GTKmathplot definition files (*.gdat)");
  gtk_file_filter_add_pattern (filter_gdat, "*.gdat");  
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(dialog), filter_gdat);
  /* Set current directory */
  if ((current_folder))
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(dialog), current_folder);   
  else
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(dialog), g_get_home_dir());   
  /* Ask for confirmation before overwriting an existing file */
  gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER(dialog), TRUE);
  if ( gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT )
    {
      current_folder = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER(dialog));
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER(dialog));
      filetype = identify_file_type (filename);
      if (filetype == TYPE_IS_GDAT)
	{
	  errno = 0;
	  mode = get_graphic_mode();
	  if (mode == GRAPHIC_2D)
	    status = eval_save_2d ((const char*)filename, infields);
	  else
	    status = eval_save_3d ((const char*)filename, infields);
	  if (status == GR_FAILURE)
	    {
	      if ((errno))
		{
		  gchar* message = g_strjoin (" ", 
					      "Cannot save results to file:",
					      g_strerror (errno), NULL);
		  
		  show_warning_message (message);
		  g_free (message);
		}
	      else
		show_warning_message ("Sorry, it was not possible to save\n" \
				      "the results of the evaluation to the given file");
	    }
	  if (status == GR_EXCEPTION)
	    show_warning_message ( get_last_math_error() );
	  /* Do nothing if status == GR_OK */
	}
      else
	show_warning_message ("Cannot proceed: GTKmathplot data files\n"\
			      "should have extension .gdat");
    }
  gtk_widget_destroy (dialog);
}

void on_clear_definition (GtkWidget* wd, gpointer user_data)
{
  pointers_to_fields* infields = (pointers_to_fields*)user_data;

  gtk_entry_set_text (GTK_ENTRY(infields->x_entry), "");
  gtk_entry_set_text (GTK_ENTRY(infields->y_entry), "");
  gtk_entry_set_text (GTK_ENTRY(infields->z_entry), "");
  gtk_entry_set_text (GTK_ENTRY(infields->inf_s_entry), "");
  gtk_entry_set_text (GTK_ENTRY(infields->sup_s_entry), "");
  gtk_entry_set_text (GTK_ENTRY(infields->nsteps_s_entry), "");
  gtk_entry_set_text (GTK_ENTRY(infields->inf_t_entry), "");
  gtk_entry_set_text (GTK_ENTRY(infields->sup_t_entry), "");
  gtk_entry_set_text (GTK_ENTRY(infields->nsteps_t_entry), "");
  gtk_entry_set_text (GTK_ENTRY(infields->caption_entry), "");
  infields->style = DEFAULT_STYLE;
  switch (infields->style)
    {
    case GRSTYLE_DOTTED:
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (infields->dotted_radiob),
				    TRUE);
      break;
    case GRSTYLE_FILLED:
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (infields->filled_radiob),
				    TRUE);
      break;
    default:
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (infields->stroked_radiob),
				    TRUE);
    }
}

gint on_draw (GtkWidget* wd, GdkEventExpose* ev, gpointer user_data)
{
  cairo_t* cr;
  cairo_initializer in = { wd, NULL, NULL, NULL };
  gboolean pdf_or_ps_surface;
  int r;

  cr = new_cairo_context (in, &r, &pdf_or_ps_surface);
  if (!cr)
    {
      handle_fatal_error ("The program can not draw on the screen,\n" \
			  "please quit now.\n");      
    }
  draw (cr, r, pdf_or_ps_surface);
  cairo_destroy (cr);
  return FALSE; /* Parent widget must be redrawn */
}

void on_save_image (GtkWidget* wd, gpointer user_data)
{
  GtkWidget* dialog;
  GtkFileFilter *filter_ps, *filter_eps, *filter_pdf, *filter_svg, *filter_mgr;  
  GtkWidget* top_win = GTK_WIDGET (user_data);
  gchar* filename;
  cairo_t* cr;
  cairo_initializer in = { NULL, NULL, NULL, NULL };
  int r, filetype;
  gboolean pdf_or_ps_surface;
  grexitstatus status;
  static gchar* current_folder = NULL;
    
  dialog = gtk_file_chooser_dialog_new ("Save image to a file", GTK_WINDOW(top_win), 
					GTK_FILE_CHOOSER_ACTION_SAVE, 
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
					GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, 
					NULL);
    
  /* Add filters for MGR, PS, EPS, PDF and SVG files */  
  filter_pdf = gtk_file_filter_new();
  gtk_file_filter_set_name (filter_pdf, "PDF files (*.pdf)");
  gtk_file_filter_add_pattern (filter_pdf, "*.pdf");      
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(dialog), filter_pdf);

  filter_ps = gtk_file_filter_new();
  gtk_file_filter_set_name (filter_ps, "Postscript files (*.ps)");
  gtk_file_filter_add_pattern (filter_ps, "*.ps");  
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(dialog), filter_ps);
    
  filter_eps = gtk_file_filter_new();
  gtk_file_filter_set_name (filter_eps, "Encapsulated Postscript files (*.eps)");
  gtk_file_filter_add_pattern (filter_eps, "*.eps");  
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(dialog), filter_eps);
    
  filter_svg = gtk_file_filter_new();
  gtk_file_filter_set_name (filter_svg, "Standard Vector Graphics files (*.svg)");
  gtk_file_filter_add_pattern (filter_svg, "*.svg");      
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(dialog), filter_svg);

  filter_mgr = gtk_file_filter_new();
  gtk_file_filter_set_name (filter_mgr, "GTKmathplot graphic files (*.mgr)");
  gtk_file_filter_add_pattern (filter_mgr, "*.mgr");  
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(dialog), filter_mgr);

  /* Set current directory */
  if ((current_folder))
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(dialog), current_folder);   
  else
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(dialog), g_get_home_dir());   
    
  /* Ask for confirmation before overwriting an existing file */
  gtk_file_chooser_set_do_overwrite_confirmation (GTK_FILE_CHOOSER(dialog), TRUE);

  if ( gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT )
    {
      current_folder = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER(dialog));
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER(dialog));
      filetype = identify_file_type (filename);
      switch (filetype)
      {
      case TYPE_IS_MGR:
	errno = 0;
	status = save_pipeline ((const char*)filename);
	if (status == GR_FAILURE)
	  {
	      if ((errno))
		{
		  gchar* message = g_strjoin (" ", 
					      "Cannot save image to file:",
					      g_strerror (errno), NULL);
		  
		  show_warning_message (message);
		  g_free (message);
		}
	      else
		show_warning_message ("Sorry, it was not possible to save\n" \
				      "the image to the given file");
	  }
	/* Do nothing if status == GR_OK */	
	break;
      case TYPE_IS_PS:
	in.ps = g_strdup (filename);
	break;
      case TYPE_IS_PDF:
	in.pdf = g_strdup (filename);
	break;
      case TYPE_IS_SVG:
	in.svg = g_strdup (filename);
	break;
      default:
	show_warning_message ("Choose either a Postscript (*.ps), Encapsulated Postscript (*.eps),\n" \
			      "PDF (*.pdf), SVG (*.svg), or MGR (*.mgr) file to save the image!");
	break;
      }
      g_free (filename);
    }
  gtk_widget_destroy (dialog);
  if ((in.ps) || (in.pdf) || (in.svg))
    {
      cr = new_cairo_context (in, &r, &pdf_or_ps_surface);
      if (!cr)
        {
          handle_fatal_error ("Export failed, please quit now.\n");      
        }
      draw (cr, r, pdf_or_ps_surface);
      cairo_destroy (cr);
      g_free (in.ps);
      g_free (in.pdf);
      g_free (in.svg);
    }
}

void on_load_image (GtkWidget* wd, gpointer user_data)
{
  GtkWidget* dialog;
  GtkFileFilter *filter_mgr, *filter_gdat;
  GtkWidget* top_win = gtk_widget_get_ancestor (wd, GTK_TYPE_WINDOW);
  gchar* filename;
  int filetype;
  grexitstatus status;
  static gchar* current_folder = NULL;

  dialog = gtk_file_chooser_dialog_new ("Load image from a file", GTK_WINDOW(top_win), 
					GTK_FILE_CHOOSER_ACTION_OPEN, 
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, 
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, 
					NULL);
    
  /* Add filters for MGR and GDAT files */  
  filter_gdat = gtk_file_filter_new();
  gtk_file_filter_set_name (filter_gdat, "GNUPlot graphics files (*.gdat)");
  gtk_file_filter_add_pattern (filter_gdat, "*.gdat");      
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(dialog), filter_gdat);

  filter_mgr = gtk_file_filter_new();
  gtk_file_filter_set_name (filter_mgr, "GTKmathplot graphic files (*.mgr)");
  gtk_file_filter_add_pattern (filter_mgr, "*.mgr");  
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER(dialog), filter_mgr);
  /* Set current directory */
  if ((current_folder))
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(dialog), current_folder);   
  else
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER(dialog), g_get_home_dir());   
  if ( gtk_dialog_run (GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT )
    {
      current_folder = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER(dialog));
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER(dialog));
      filetype = identify_file_type (filename);
      if (filetype == TYPE_IS_MGR || filetype == TYPE_IS_GDAT)
	{
	  int file_in_native_format = (filetype == TYPE_IS_MGR ? 1 : 0);

	  errno = 0;
	  status = load_into_pipeline ((const char*)filename, 
				       file_in_native_format);
	  if (status == GR_FAILURE)
	    {
	      if ((errno))
		{
		  gchar* message = g_strjoin (" ", "Cannot load file:",
					      g_strerror (errno), NULL);
		  
		  show_warning_message (message);
		  g_free (message);
		}
	      else
		show_warning_message ("Cannot load file: invalid format");
	    }
	  else if (status == GR_EXCEPTION)
	    {
	      gchar* message = NULL;
	      graphic_mode mode = get_graphic_mode();

	      message = 
		g_strdup_printf ("Loading failed: it is not possible\n" \
				 "to display more than %d mathematical\n" \
				 "objects at once",
				 (mode == GRAPHIC_2D ? 
				  NUMBER_OF_2DOBJECTS : NUMBER_OF_3DOBJECTS));
	      assert (message != NULL);
	      show_warning_message (message);
	      g_free (message);
	    }
	  else
	    set_legend();
	}
      else
	show_warning_message ("Cannot proceed: file extension unknown");
      gtk_widget_queue_draw (GTK_WIDGET(user_data));
    }
  gtk_widget_destroy (dialog);
}

void on_set_longitude (GtkWidget* wd, gpointer user_data)
{
  double longitude = gtk_spin_button_get_value (GTK_SPIN_BUTTON(wd));

  set_longitude_and_update (longitude * CONV_FACTOR_D2R);
  gtk_widget_queue_draw (GTK_WIDGET(user_data));
}

void on_set_latitude (GtkWidget* wd, gpointer user_data)
{
  double latitude = gtk_spin_button_get_value (GTK_SPIN_BUTTON(wd));

  set_latitude_and_update (latitude * CONV_FACTOR_D2R);
  gtk_widget_queue_draw (GTK_WIDGET(user_data));
}

void on_set_rotation_angle (GtkWidget* wd, gpointer user_data)
{
  double rotation_angle = gtk_spin_button_get_value (GTK_SPIN_BUTTON(wd));

  set_rotation_angle_and_update (rotation_angle * CONV_FACTOR_D2R);
  gtk_widget_queue_draw (GTK_WIDGET(user_data));
}

void on_toggle_show_main_dirs (GtkWidget* wd, gpointer user_data)
{
  int boolean_value;

  boolean_value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wd)) == TRUE ? 1 : 0;
  toggle_show_main_directions (boolean_value);
  gtk_widget_queue_draw (GTK_WIDGET(user_data));
}

void on_toggle_show_grid (GtkWidget* wd, gpointer user_data)
{
  int boolean_value;

  boolean_value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wd)) == TRUE ? 1 : 0;
  toggle_show_grid (boolean_value);
  gtk_widget_queue_draw (GTK_WIDGET(user_data));
}

void on_toggle_show_tics (GtkWidget* wd, gpointer user_data)
{
  int boolean_value;

  boolean_value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wd)) == TRUE ? 1 : 0;
  toggle_show_tics (boolean_value);
  gtk_widget_queue_draw (GTK_WIDGET(user_data));
}

void on_toggle_show_discontinuities (GtkWidget* wd, gpointer user_data)
{
  int boolean_value;

  boolean_value = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wd)) == TRUE ? 1 : 0;
  toggle_show_discontinuities (boolean_value);
  gtk_widget_queue_draw (GTK_WIDGET(user_data));
}

void on_set_zoom_factor (GtkWidget* wd, gpointer user_data)
{
  pointers_to_fields* infields = (pointers_to_fields*)user_data;
  double zoom = gtk_range_get_value (GTK_RANGE(wd));
  int grid_on, tics_on;

  /*
    The next two checks are needed to avoid
    a crash of the program due to repetitive calls
    of toggle_show_grid() and toggle_show_tics().
  */
  grid_on = get_show_grid();
  tics_on = get_show_tics();
  if ((grid_on))
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(infields->grid_checkb), FALSE);
      toggle_show_grid (0);
    }
  if ((tics_on))
    {
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(infields->tics_checkb), FALSE);
      toggle_show_tics (0);
    }
  set_zoom_factor (zoom);
  gtk_widget_queue_draw (GTK_WIDGET(infields->draw_area));
}

void on_set_smoke_density (GtkWidget* wd, gpointer user_data)
{
  double smoke = gtk_range_get_value (GTK_RANGE(wd));
  
  set_smoke_density (smoke);
  gtk_widget_queue_draw (GTK_WIDGET(user_data));
}
 
void on_ximage_up (GtkWidget* wd, GdkEventButton* ev, gpointer user_data)
{
  if (ev->button == 3)
    translate_origin_and_update (-1, 0, 0); 
  else if (ev->button == 1)
    translate_origin_and_update (-10, 0, 0); 
  gtk_widget_queue_draw (wd);
}

void on_ximage_down (GtkWidget* wd, GdkEventButton* ev, gpointer user_data)
{
  if (ev->button == 3)
    translate_origin_and_update (+1, 0, 0); 
  else if (ev->button == 1)
    translate_origin_and_update (+10, 0, 0); 
  gtk_widget_queue_draw (wd);
}

void on_yimage_up (GtkWidget* wd, GdkEventButton* ev, gpointer user_data)
{
  if (ev->button == 3)
    translate_origin_and_update (0, -1, 0); 
  else if (ev->button == 1)
    translate_origin_and_update (0, -10, 0); 
  gtk_widget_queue_draw (wd);
}

void on_yimage_down (GtkWidget* wd, GdkEventButton* ev, gpointer user_data)
{
  if (ev->button == 3)
    translate_origin_and_update (0, +1, 0); 
  else if (ev->button == 1)
    translate_origin_and_update (0, +10, 0); 
  gtk_widget_queue_draw (wd);
}

void on_zimage_up (GtkWidget* wd, GdkEventButton* ev, gpointer user_data)
{
  if (ev->button == 3)
    translate_origin_and_update (0, 0, -1);
  else if (ev->button == 1)
    translate_origin_and_update (0, 0, -10);
  gtk_widget_queue_draw (wd);
}

void on_zimage_down (GtkWidget* wd, GdkEventButton* ev, gpointer user_data)
{
  if (ev->button == 3)
    translate_origin_and_update (0, 0, +1);
  else if (ev->button == 1)
    translate_origin_and_update (0, 0, +10);
  gtk_widget_queue_draw (wd);
}

void on_reset_imagepos (GtkWidget* wd, gpointer user_data)
{
  reset_origin_position_and_update();
  gtk_widget_queue_draw (GTK_WIDGET(user_data));
}

void on_toggle_print_addinfo (GtkWidget* wd, gpointer user_data)
{
  please_print_addinfo (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(wd)));
}

void on_toggle_print_disclaimer (GtkWidget* wd, gpointer user_data)
{
  please_print_disclaimer (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(wd)));
}

void on_set_paper_format (GtkWidget* wd, gpointer user_data)
{
  set_paper_format (gtk_combo_box_get_active (GTK_COMBO_BOX(wd)));
}

void on_set_orientation (GtkWidget* wd, gpointer user_data)
{
  set_orientation (gtk_combo_box_get_active (GTK_COMBO_BOX(wd)));
}

void on_set_color_2d (GtkWidget* wd, gpointer user_data)
{
  int object_id = GPOINTER_TO_INT (user_data);
  int color = gtk_combo_box_get_active (GTK_COMBO_BOX(wd));

  if (color < 0 || color >= Nr_Colors_2d)
    {
      show_warning_message ("You have chosen an invalid color,\n"\
			    "your change will be ignored");      
    }
  else
    {
      set_color_for_2dobject (object_id, (cairo_color_2d) color);
      issue_redraw_command();
    }
}

void on_set_color_3d (GtkWidget* wd, gpointer user_data)
{
  int object_id = GPOINTER_TO_INT (user_data);
  int color = gtk_combo_box_get_active (GTK_COMBO_BOX(wd));

  if (color < 0 || color >= Nr_Colors_3d)
    {
      show_warning_message ("You have chosen an invalid color,\n"\
			    "your change will be ignored");      
    }
  else
    {
      set_color_for_3dobject (object_id, (cairo_color_3d) color);
      issue_redraw_command();
    }
}

void on_set_symbol_2d (GtkWidget* wd, gpointer user_data)
{
  int object_id = GPOINTER_TO_INT (user_data);
  int symbol = gtk_combo_box_get_active (GTK_COMBO_BOX(wd));

  if (symbol < 0 || symbol >= Nr_Symbols)
    {
      show_warning_message ("You have chosen an invalid symbol,\n"\
			    "your change will be ignored");      
    }
  else
    {
      set_symbol_for_2dobject (object_id, (cairo_symbol) symbol);
      issue_redraw_command();
    }
}

void on_set_symbol_3d (GtkWidget* wd, gpointer user_data)
{
  int object_id = GPOINTER_TO_INT (user_data);
  int symbol = gtk_combo_box_get_active (GTK_COMBO_BOX(wd));

  if (symbol < 0 || symbol >= Nr_Symbols)
    {
      show_warning_message ("You have chosen an invalid symbol,\n"\
			    "your change will be ignored");      
    }
  else
    {
      set_symbol_for_3dobject (object_id, (cairo_symbol) symbol);
      issue_redraw_command();
    }
}
