/*
    GTKmathplot - a simple GTK+ based program
    to plot mathematical functions.
    Copyright (C) 2012, 2013  Ivano Primi  <ivprimi@libero.it>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include<stdio.h>
#include<stdlib.h>
#include<string.h>
#include<cairo.h>
#include<cairo-ps.h>
#include<cairo-pdf.h>
#include<cairo-svg.h>
#include"gr2d.h"
#include"gr3d.h"
#include"utils.h"
#include"cairo_driver.h"

#define DISCLAIMER "Created by GTKmathplot - Copyright (C) 2012, 2013 Ivano Primi <ivprimi [at] libero [dot] it>"

const double fn_size = 10.0;
const double mini_fn_size = 12.0;

cairo_paper_format_prop supported_paper_formats[NSUPPORTED_PAPER_FORMATS] = {
  {"A3", 842.0, 1191.0},
  {"A4", 595.0, 842.0},
  {"Letter", 612.0, 792.0},
  {"Legal", 612.0, 1008.0}
};

const gchar* supported_orientations[NSUPPORTED_ORIENTATIONS] = {
  "Portrait",
  "Landscape"
};

static cairo_paper_format current_paper_format = DEFAULT_PAPER_FORMAT;

static cairo_orientation current_orientation = DEFAULT_ORIENTATION;

static gboolean print_addinfo = TRUE;

static gboolean print_disclaimer = TRUE;

static print_specs export_settings;

static cursor_position start_point;

static const char* list_2d_colors[Nr_Colors_2d] = {
  "Black",
  "Dark Red",
  "Nigth Blue",
  "Frog Green", 
  "Dark Cyan", 
  "Dark Violet",
  "Vermilion", 
  "Cream", 
  "Light Blue", 
  "Lilla",
  "Grass Green", 
  "Dark Gray", 
  "Black Smoke"
};

static const char* list_3d_colors[Nr_Colors_3d] = {
  "Red",
  "Sea Green",     
  "Violet",
  "Orange",
  "Pompei Red", 
  "Green",
  "Cyan",       
  "Blue",       
  "Gray",       
  "Bluish",
  "Cyanish", 
  "Greenish",
  "Almost Black"
}; 

typedef struct {
  double r, g, b;
} rgb_value;

static const
rgb_value palette_for_2dg[Nr_Colors_2d] = {
  RGB_0_2D,  
  RGB_1_2D,
  RGB_2_2D,
  RGB_3_2D,
  RGB_4_2D,
  RGB_5_2D,
  RGB_6_2D,
  RGB_7_2D,
  RGB_8_2D,
  RGB_9_2D,
  RGB_10_2D,
  RGB_11_2D,
  RGB_12_2D
};

static const
rgb_value palette_for_3dg[Nr_Colors_3d] = {
  RGB_0_3D,  
  RGB_1_3D,
  RGB_2_3D,
  RGB_3_3D,
  RGB_4_3D,
  RGB_5_3D,
  RGB_6_3D,
  RGB_7_3D,
  RGB_8_3D,
  RGB_9_3D,
  RGB_10_3D,
  RGB_11_3D,
  RGB_12_3D
};

/* The following two arrays are initialized in init_cairo_driver() */
static rgb_value paint_for_2dobject[NUMBER_OF_2DOBJECTS+4];

static rgb_value paint_for_3dobject[NUMBER_OF_3DOBJECTS+7];

static const char* list_symbols[Nr_Symbols] = {
  "Plus",
  "Square",
  "Diamond",
  "Cross",
  "Bullet"
};

/* The following two arrays are initialized in init_cairo_driver() */
static cairo_symbol symbol_for_2dobject[NUMBER_OF_2DOBJECTS];

static cairo_symbol symbol_for_3dobject[NUMBER_OF_3DOBJECTS];

static
const char* get_rgb_specification_as_string (double r, double g, double b)
{
  static char color_spec[7+1] = "#000000"; // black

  if (r < 0.0)
    r = 0.0;
  if (r > 1.0)
    r = 1.0;

  if (g < 0.0)
    g = 0.0;
  if (g > 1.0)
    g = 1.0;

  if (b < 0.0)
    b = 0.0;
  if (b > 1.0)
    b = 1.0;

  sprintf (color_spec, "#%02X%02X%02X", 
	   (unsigned int)(255 * r),
	   (unsigned int)(255 * g),
	   (unsigned int)(255 * b));
  return (const char*)color_spec;
}

gchar* get_formatted_caption (const char* caption, graphic_mode mode, 
			      int object_id)
{
    rgb_value* paint;
    const char* color = NULL;
    gchar* format_string = NULL;
    gchar* formatted_caption = NULL;
    gchar* wrapped_caption = NULL;

    if ( is_string_blank(caption) != 0 )
      caption = "";
    wrapped_caption = wrap_string (caption, 4);
    if (mode == GRAPHIC_2D)
      {
	if (object_id < 0 || object_id >= NUMBER_OF_2DOBJECTS)
	  return NULL;
	paint = paint_for_2dobject;
      }
    else
      {
	/* mode == GRAPHIC_3D */
	if (object_id < 0 || object_id >= NUMBER_OF_3DOBJECTS)
	  return NULL;
	paint = paint_for_3dobject;	
      }
    color = get_rgb_specification_as_string (paint[object_id].r,
					     paint[object_id].g,
					     paint[object_id].b);
    if ( (*wrapped_caption) )
      format_string = g_strdup_printf ("<span foreground=\"%s\"> ###</span> %%s ",
				       color);
    else
      format_string = g_strdup_printf ("<span foreground=\"%s\"> ---</span> %%s ",
				       color);      
    if ((format_string))
      {
	formatted_caption = g_markup_printf_escaped (format_string, 
						     wrapped_caption);
	g_free ((gpointer) format_string);
	g_free ((gpointer) wrapped_caption);
      }
    return formatted_caption;
}

gchar* wrap_string (const gchar* str, unsigned indentation)
{
  const unsigned Max_string_length = 20;
  const unsigned Allowance = 10;
  gchar *new_str, *pn;
  const gchar *p;
  size_t n, string_length, nchars_to_add = 0;
  unsigned k;

  string_length = strlen(str);
  nchars_to_add += (indentation + 2) * (string_length / Max_string_length);

  new_str = g_malloc ((string_length + nchars_to_add + 1) * sizeof(gchar));
  pn = new_str;
  p = str;
  /* 
     Copy the whole string, but care to put a newline
     and INDENTATION blanks "almost" every 
     MAX_STRING_LENGTH characters 
  */
  for (n = 0; *p != '\0'; pn++, p++)
    {
      n++;
      /* We are about to copy the Nth character of the string */
      if ( n % Max_string_length == 0 )
	{
	  while (n % Max_string_length < Allowance && *p != '\0' &&
		 !is_special_char(*p))
	  {
	    *pn = *p;
	    pn++;
	    p++;
	    n++;
	  } 
	  /* 
	     Whenever we arrive here, it holds that
	     n % Max_string_length == Allowance, or *p == '\0',
	     or *p is a special character 
	  */
	  if ( (*p) )
	    *pn = *p;
	  else
	    break;
	  pn++;
	  *pn = ' ';
	  pn++;
	  *pn = '\n';
	  for (k = 0; k < indentation; k++)
	    {
	      pn++;
	      *pn = ' ';
	    }
	}
      else
	*pn = *p;
    }
  *pn = '\0';
  return new_str;  
}

void init_cairo_driver (void)
{
  int object_id;

  for (object_id = 0; object_id < NUMBER_OF_2DOBJECTS + 4; object_id++)
    paint_for_2dobject[object_id] = 
      palette_for_2dg[get_default_color_2d (object_id)];

  for (object_id = 0; object_id < NUMBER_OF_2DOBJECTS; object_id++)
    symbol_for_2dobject[object_id] = get_default_symbol (object_id);

  for (object_id = 0; object_id < NUMBER_OF_3DOBJECTS + 7; object_id++)
    paint_for_3dobject[object_id] = 
      palette_for_3dg[get_default_color_3d (object_id)];

  for (object_id = 0; object_id < NUMBER_OF_3DOBJECTS; object_id++)
    symbol_for_3dobject[object_id] = get_default_symbol (object_id);
}

void set_color_for_2dobject (int object_id, cairo_color_2d color)
{
  if (object_id >= 0 && object_id < NUMBER_OF_2DOBJECTS+4 && 
      color >= 0 && color < Nr_Colors_2d)
    paint_for_2dobject[object_id] = palette_for_2dg[color];
}

void set_color_for_3dobject (int object_id, cairo_color_3d color)
{
  if (object_id >= 0 && object_id < NUMBER_OF_3DOBJECTS+7 && 
      color >= 0 && color < Nr_Colors_3d)
    paint_for_3dobject[object_id] = palette_for_3dg[color];
}

void set_symbol_for_2dobject (int object_id, cairo_symbol symbol)
{
  if (object_id >= 0 && object_id < NUMBER_OF_2DOBJECTS && 
      symbol >= 0 && symbol < Nr_Symbols)
    symbol_for_2dobject[object_id] = symbol;
}

void set_symbol_for_3dobject (int object_id, cairo_symbol symbol)
{
  if (object_id >= 0 && object_id < NUMBER_OF_3DOBJECTS && 
      symbol >= 0 && symbol < Nr_Symbols)
    symbol_for_3dobject[object_id] = symbol;
}

const char* get_color_2d_name (cairo_color_2d color)
{
  return (color == Nr_Colors_2d ? NULL : list_2d_colors[color]);
}

const char* get_color_3d_name (cairo_color_3d color)
{
  return (color == Nr_Colors_3d ? NULL : list_3d_colors[color]);
}

const char* get_symbol_name (cairo_symbol symbol)
{
  return (symbol == Nr_Symbols ? NULL : list_symbols[symbol]);
}

/*
  Valid values for object_id
  (see implementation of get_object_label_2d())

  0..NUMBER_OF_2DOBJECTS-1  --> related mathematical shape
  NUMBER_OF_2DOBJECTS       --> Grid
  NUMBER_OF_2DOBJECTS + 1   --> Tics
  NUMBER_OF_2DOBJECTS + 2   --> X-Axis
  NUMBER_OF_2DOBJECTS + 3   --> Y-Axis
*/
cairo_color_2d get_default_color_2d (int object_id)
{
  if (object_id >= 0 && object_id < NUMBER_OF_2DOBJECTS)
    return (cairo_color_2d)object_id;
  else
    {
      switch (object_id)
	{
	case NUMBER_OF_2DOBJECTS:
	  return Dark_Gray;
	  break;
	case NUMBER_OF_2DOBJECTS + 1:
	  return Black_Smoke;
	  break;
	case NUMBER_OF_2DOBJECTS + 2:
	case NUMBER_OF_2DOBJECTS + 3:
	  return Grass_Green;
	  break;
	default:
	  fputs ("get_default_color_2d(): Invalid object index,\n", stderr);
	  fputs ("                        the program is going to terminate now!!!\n", stderr);
	  exit (EXIT_FAILURE);
	  break;
	}    
    }
}

/*
  Valid values for object_id:
  (see implementation of get_object_label_3d())

  0..NUMBER_OF_3DOBJECTS-1  --> related mathematical shape
  NUMBER_OF_3DOBJECTS       --> Tics
  NUMBER_OF_3DOBJECTS + 1   --> XY-Grid
  NUMBER_OF_3DOBJECTS + 2   --> XZ-Grid
  NUMBER_OF_3DOBJECTS + 3   --> YZ-Grid
  NUMBER_OF_3DOBJECTS + 4   --> X-Axis
  NUMBER_OF_3DOBJECTS + 5   --> Y-Axis
  NUMBER_OF_3DOBJECTS + 6   --> Z-Axis
*/
cairo_color_3d get_default_color_3d (int object_id)
{
  if (object_id >= 0 && object_id < NUMBER_OF_3DOBJECTS)
    return (cairo_color_3d)object_id;
  else
    {
      switch (object_id)
	{
	case NUMBER_OF_3DOBJECTS:
	  return Almost_Black;
	  break;
	case NUMBER_OF_3DOBJECTS + 1:
	  return Bluish;
	  break;
	case NUMBER_OF_3DOBJECTS + 2:
	  return Cyanish;
	  break;
	case NUMBER_OF_3DOBJECTS + 3:
	  return Greenish;
	  break;
	case NUMBER_OF_3DOBJECTS + 4:
	  return Green;
	  break;
	case NUMBER_OF_3DOBJECTS + 5:
	  return Cyan;
	  break;
	case NUMBER_OF_3DOBJECTS + 6:
	  return Blue;
	  break;
	default:
	  fputs ("get_default_color_3d(): Invalid object index,\n", stderr);
	  fputs ("                        the program is going to terminate now!!!\n", stderr);
	  exit (EXIT_FAILURE);
	}    
    }
}


/*
  Valid values for object_id
  (see implementation of get_object_label_2d())

  0..NUMBER_OF_2DOBJECTS-1  --> related mathematical shape
  NUMBER_OF_2DOBJECTS       --> Grid
  NUMBER_OF_2DOBJECTS + 1   --> Tics
  NUMBER_OF_2DOBJECTS + 2   --> X-Axis
  NUMBER_OF_2DOBJECTS + 3   --> Y-Axis
*/
static
grint object_id_from_color_id_2dv (grint color)
{
  switch (color)
    {
    case GRID_COLOR:
      return NUMBER_OF_2DOBJECTS;
    case TICS_COLOR:
      return NUMBER_OF_2DOBJECTS + 1;
    case X_COLOR:
      return NUMBER_OF_2DOBJECTS + 2;
    case Y_COLOR:
      return NUMBER_OF_2DOBJECTS + 3;
    default:
      return color;
    }
}


/*
  Valid values for object_id:
  (see implementation of get_object_label_3d())

  0..NUMBER_OF_3DOBJECTS-1  --> related mathematical shape
  NUMBER_OF_3DOBJECTS       --> Tics
  NUMBER_OF_3DOBJECTS + 1   --> XY-Grid
  NUMBER_OF_3DOBJECTS + 2   --> XZ-Grid
  NUMBER_OF_3DOBJECTS + 3   --> YZ-Grid
  NUMBER_OF_3DOBJECTS + 4   --> X-Axis
  NUMBER_OF_3DOBJECTS + 5   --> Y-Axis
  NUMBER_OF_3DOBJECTS + 6   --> Z-Axis
*/
static
grint object_id_from_color_id_3dv (grint color)
{
  switch (color)
    {
    case TICS_COLOR:
      return NUMBER_OF_3DOBJECTS;
    case XYPLANE_GRID_COLOR:
      return NUMBER_OF_3DOBJECTS + 1;
    case XZPLANE_GRID_COLOR:
      return NUMBER_OF_3DOBJECTS + 2;
    case YZPLANE_GRID_COLOR:
      return NUMBER_OF_3DOBJECTS + 3;
    case X_COLOR:
      return NUMBER_OF_3DOBJECTS + 4;
    case Y_COLOR:
      return NUMBER_OF_3DOBJECTS + 5;
    case Z_COLOR:
      return NUMBER_OF_3DOBJECTS + 6;
    default:
      return color;
    }
}

/*
  Valid values for object_id:
  any non-negative integer
*/
cairo_symbol get_default_symbol (int object_id)
{
  if (object_id < 0)
    {
      fputs ("get_default_symbol(): Invalid object index,\n", stderr);
      fputs ("                      the program is going to terminate now!!!\n",
	     stderr);
      exit (EXIT_FAILURE);
    }
  else
    return object_id % Nr_Symbols;
}

const double dash_pattern[2] = {5.0, 5.0}; 

#define MIN_LIGHT 0.2

static
double smoke_function (double d, double smoke_density, double R)
{
  double f = d/(2*R) + 0.5;

  if (smoke_density < GR_EPSILON || f > 1.0)
    return 1.0;
  else if (smoke_density > 1-GR_EPSILON || f < 0.0)
    return MIN_LIGHT;
  else if (f <= smoke_density)
    return MIN_LIGHT + (1-MIN_LIGHT) * (1-smoke_density) * (f / smoke_density);
  else /* smoke_density < f <= 1 */ 
    return 1 - smoke_density * (1-MIN_LIGHT) + smoke_density * (1-MIN_LIGHT) * ((f-smoke_density)/(1-smoke_density));
  /* Remark: parentheses are very important here */
}

static cairo_surface_t* draw_on_image (int r, int left_border, int lower_border)
{
  cairo_surface_t *img;
  cairo_t *cr;
  cairo_text_extents_t te;
  double zoom_factor, smoke_density;
  double R, radius, smoke_factor = 1.0;
  graphic_mode mode;
  grint len, color_id, i, nr_math_objects, object_id, clip_area_is_set;
  const gr2d_object* shape_2d = get_pointer_to_2dpipeline ();
  const gr3d_object* shape_3d = get_pointer_to_3dpipeline ();
  gr2d_basic_shape basic_shape_2d;
  gr3d_basic_shape basic_shape_3d;
  grtype type; 
  const grchar *caption;
  double XA, YA, XB, YB, XC, YC;
  rgb_value* paint;

  img = cairo_image_surface_create (CAIRO_FORMAT_ARGB32, 
				    left_border + 2*r, lower_border + 2*r);
  cr = cairo_create (img);
  mode = get_graphic_mode();
  zoom_factor = get_zoom_factor() / 100.0;
  smoke_density = get_smoke_density();
  R = get_half_side_of_the_draw ();
  
  if (mode == GRAPHIC_2D)
    {
      paint = paint_for_2dobject;
      nr_math_objects = NUMBER_OF_2DOBJECTS;
      len = shape_2d->len;
      radius = shape_2d->R;
    }
  else
    {
      paint = paint_for_3dobject;
      nr_math_objects = NUMBER_OF_3DOBJECTS;
      len = shape_3d->len;
      radius = shape_3d->R;
    }

  for (clip_area_is_set = 0, i = 1; i <= len; i++)
    {  
      if ( mode == GRAPHIC_2D)
	{
	  basic_shape_2d = *(shape_2d->list[i]);
	  color_id = basic_shape_2d.color;
	  type = basic_shape_2d.type;
	  caption = basic_shape_2d.caption;
	  XA = basic_shape_2d.XA;
	  YA = basic_shape_2d.YA;
	  XB = basic_shape_2d.XB;
	  YB = basic_shape_2d.YB;
	  XC = basic_shape_2d.XC;
	  YC = basic_shape_2d.YC;
	}
      else
	{
	  basic_shape_3d = *(shape_3d->list[i]);
	  smoke_factor = smoke_function(basic_shape_3d.depth, 
					smoke_density, radius);
	  color_id = basic_shape_3d.color;
	  type = basic_shape_3d.type;
	  caption = basic_shape_3d.caption;
	  XA = basic_shape_3d.XA;
	  YA = basic_shape_3d.YA;
	  XB = basic_shape_3d.XB;
	  YB = basic_shape_3d.YB;
	  XC = basic_shape_3d.XC;
	  YC = basic_shape_3d.YC;
	}
      if (!clip_area_is_set && (color_id != TICS_COLOR || mode != GRAPHIC_2D))
	{
	  /* If the clip area is not set, then set it now, unless   */
	  /* the graphic mode is 2D and we are about to draw a tic, */
	  /* since in this case we do not need a clip area.         */

	  /* Save the original graphic state... */
	  cairo_save(cr); 
	  /* and then set the clip area */
	  cairo_rectangle (cr, left_border, 0.0, 2*r, 2*r);
	  cairo_clip (cr);

	  cairo_new_path (cr);  /* current path is not
				   consumed by cairo_clip() */
	  clip_area_is_set = 1;
	}
      if (color_id == TICS_COLOR && (clip_area_is_set) && mode == GRAPHIC_2D)
	{
	  /*
	    If we are going to draw tics and we are in 2D graphic mode,
	    then we have to restore the clip area to its 
	    default state, so that the tics can be actually drawn.
	  */
	  cairo_restore (cr);
	  clip_area_is_set = 0;
	}

      if (mode == GRAPHIC_2D)
	object_id = object_id_from_color_id_2dv (color_id);
      else
	object_id = object_id_from_color_id_3dv (color_id);
      cairo_set_source_rgb (cr, 
			    smoke_factor * paint[object_id].r, 
			    smoke_factor * paint[object_id].g, 
			    smoke_factor * paint[object_id].b);

      if (type == GR_TRIANGLE)
	{
	  cairo_set_line_width (cr, 0.25);
	  cairo_set_dash (cr, dash_pattern, 0, 0); /* Dashing is disabled */
	  cairo_move_to (cr, left_border + r + zoom_factor * r*XA / R,  
			 r - zoom_factor * r*YA / R);
	  cairo_line_to (cr, left_border + r + zoom_factor * r*XB / R,  
			 r - zoom_factor * r*YB / R);
	  cairo_line_to (cr, left_border + r + zoom_factor * r*XC / R,  
			 r - zoom_factor * r*YC / R);
	  /* cairo_fill (cr); */
	  cairo_fill_preserve(cr);
	  cairo_stroke(cr);
	}
      else if (type == GR_SEGMENT)
	{
          if ( color_id == GRID_COLOR )
	    {
	      cairo_set_line_width (cr, 0.5);
	      cairo_set_dash (cr, dash_pattern, 2, 0);
	    }
	  else
	    {
	      cairo_set_line_width (cr, 2.0);
	      cairo_set_dash (cr, dash_pattern, 0, 0); /* Dashing is disabled */
	    }
	  cairo_move_to (cr,  left_border + r + zoom_factor * r*XA / R,  
			 r - zoom_factor * r*YA / R);
	  cairo_line_to (cr,  left_border + r + zoom_factor * r*XB / R,  
			 r - zoom_factor * r*YB / R);
	  cairo_stroke(cr);
	}
      else
        {
	  /* type == GR_POINT */
          if (object_id < nr_math_objects)
            {
	      cairo_symbol symbol;
	      double x, y;

	      x = left_border + r + zoom_factor * r*XA / R;
	      y = r - zoom_factor * r*YA / R;
	      symbol = (mode == GRAPHIC_2D ? 
			symbol_for_2dobject[object_id]: 
			symbol_for_3dobject[object_id]);
	      switch (symbol)
		{
		case Plus:
		  cairo_set_line_width (cr, 2.0);
		  cairo_set_dash (cr, dash_pattern, 0, 0);/* Dashing disabled */
		  cairo_move_to (cr, x - 4.5, y);
		  cairo_line_to (cr, x + 4.5, y);
		  cairo_move_to (cr, x, y - 4.5);
		  cairo_line_to (cr, x, y + 4.5);
		  cairo_stroke (cr);
		  break;
		case Square:
		  cairo_set_line_width (cr, 2.0);
		  cairo_set_dash (cr, dash_pattern, 0, 0);/* Dashing disabled */
		  cairo_move_to (cr, x - 4.0, y - 4.0);
		  cairo_line_to (cr, x - 4.0, y + 4.0);		  
		  cairo_line_to (cr, x + 4.0, y + 4.0);		  
		  cairo_line_to (cr, x + 4.0, y - 4.0);
		  cairo_close_path (cr);
		  cairo_stroke (cr);		  
		  break;
		case Diamond:
		  cairo_set_line_width (cr, 2.0);
		  cairo_set_dash (cr, dash_pattern, 0, 0);/* Dashing disabled */
		  cairo_move_to (cr, x, y - 4.0);
		  cairo_line_to (cr, x - 4.0, y);		  
		  cairo_line_to (cr, x, y + 4.0);
		  cairo_line_to (cr, x + 4.0, y);
		  cairo_close_path (cr);
		  cairo_stroke (cr);		  		  
		  break;
		case Cross:
		  cairo_set_line_width (cr, 2.0);
		  cairo_set_dash (cr, dash_pattern, 0, 0);/* Dashing disabled */
		  cairo_move_to (cr, x - 4.5, y - 4.5);		  
		  cairo_line_to (cr, x + 4.5, y + 4.5);		  
		  cairo_move_to (cr, x - 4.5, y + 4.5);		  
		  cairo_line_to (cr, x + 4.5, y - 4.5);
		  cairo_stroke (cr);		  
		  break;
		case Bullet:  
		  cairo_arc (cr, x, y, 3.0, 0.0, 360.0 * CONV_FACTOR_D2R);
		  cairo_fill (cr);
		  break;
		default:
		  fputs ("draw_on_image(): Invalid symbol, the program\n", 
			 stderr);
		  fputs ("                 is going to terminate now!!!\n", 
			 stderr);
		  exit (EXIT_FAILURE);
		}
            }
	  else if (color_id == X_COLOR || color_id == Y_COLOR || color_id == Z_COLOR)
	    {
	      /* This is for labeling the x-, y-, and z- axis */
	      if ((caption) && (*caption))
		{
		  cairo_text_extents (cr, caption, &te);
		  cairo_move_to (cr,  
				 left_border + r + zoom_factor * r*XA / R - (te.width / 2 + te.x_bearing),  
				 r - zoom_factor * r*YA / R - (te.height / 2 + te.y_bearing));
		  cairo_show_text (cr, caption);
		}
	    }
          else if (color_id == TICS_COLOR)
	    {
#ifdef _TRACE2_
	      printf ("caption = %s\n", caption);
#endif
	      if ((caption) && (*caption))
		{
		  cairo_text_extents (cr, caption, &te);
		  cairo_move_to (cr,  
				 left_border + r + zoom_factor * r*XA / R - (te.width / 2 + te.x_bearing),  
				 r - zoom_factor * r*YA / R - (te.height / 2 + te.y_bearing));
		  cairo_show_text (cr, caption);
		}
	      else
		{
		  cairo_arc (cr,  
			     left_border + r + zoom_factor * r*XA / R,  
			     r - zoom_factor * r*YA / R, 
			     1.0, 0.0, 360.0 * CONV_FACTOR_D2R);
		  cairo_fill (cr);		  
		}
	    }
	  else
            {
	      /* This is for drawing the grid */
              cairo_arc (cr,  
			 left_border + r + zoom_factor * r*XA / R,  
			 r - zoom_factor * r*YA / R, 
			 1.0, 0.0, 360.0 * CONV_FACTOR_D2R);
              cairo_fill (cr);
            }         
	}
    } /* end for */
  cairo_destroy (cr);
  return img;
}

static
void cairo_print_lines (cairo_t* cr, graphic_mode mode, 
			char* text, int text_to_print_are_equations, 
			print_specs settings,
			cursor_position start_pos)
{
  char *ptr_beg, *ptr_end;
  int newline_found;
  rgb_value* paint;
  grint curr_color_id, new_color_id;
  int object_id;

  paint = (mode == GRAPHIC_2D) ? paint_for_2dobject : paint_for_3dobject;
  for (ptr_beg = text; *ptr_beg != '\0'; ptr_beg = ptr_end)
    {
      if ((text_to_print_are_equations))
	{
	  /*
	    We are going to print the equations of the displayed objects.
	    We assume they have been so formatted, that
	    each couple/triple of parametric equations
	    starts with a positive number identifiying the
	    associated object. This is then used to decide
	    which color has to be employed to write each equation.
	  */
	  new_color_id = str_2_number (ptr_beg, 2);
#ifdef _TRACE_
	  printf (">>\n%s\n<<\n", ptr_beg);
	  printf ("color_id = %ld\n", new_color_id);
#endif      
	  if ((new_color_id))
	    curr_color_id = new_color_id - 1;

	  if (mode == GRAPHIC_2D)
	    object_id = object_id_from_color_id_2dv (curr_color_id);
	  else
	    object_id = object_id_from_color_id_3dv (curr_color_id);
	  cairo_set_source_rgb (cr, 
				paint[object_id].r, 
				paint[object_id].g, 
				paint[object_id].b);
	}
      else
	{
	  /*
	    We are going to print simple text. Use black as color.
	  */
	  cairo_set_source_rgb (cr, 0.0, 0.0, 0.0);
	}
      for (ptr_end = ptr_beg; *ptr_end != '\0' && *ptr_end != '\n'; ptr_end++);
      if (*ptr_end == '\n')
	{
	  *ptr_end = '\0';
	  ptr_end++;
	  newline_found = 1;	  
	}
      else /* *ptr_end == '\0' */
	newline_found = 0;	  
      /* Neglect any additional newline */
      for (; *ptr_end == '\n'; ptr_end++);
      /* Now PTR_END points to the first character after the last newline */
      cairo_move_to (cr, start_pos.x, start_pos.y);
      cairo_show_text (cr, ptr_beg);
      if ((newline_found))
	start_pos.y += (settings.Mh + settings.line_spacing);
    }
}

void draw (cairo_t* cr, int r, gboolean exported_picture)
{
  graphic_mode mode;
  cairo_surface_t *img;
  double o_x, o_y, o_z, grid_step;
  char orig_x[LABEL_SIZE], orig_y[LABEL_SIZE], orig_z[LABEL_SIZE];
  char g_step[LABEL_SIZE];
  cursor_position curr_pos, orig_pos = start_point;

  curr_pos.x = curr_pos.y = 0.0;
  mode = get_graphic_mode();
  cairo_set_line_width (cr, 1); /* default is 2.0! */  
  cairo_set_source_rgb (cr, 1.0, 1.0, 1.0);
  cairo_rectangle (cr, 0.0, 0.0, DEFAULT_WIDTH, DEFAULT_HEIGHT);
  cairo_fill(cr); /* White background */

  if ((exported_picture) && (print_disclaimer))
    {
      cairo_set_source_rgb (cr, 0.0, 0.0, 0.0);
      cairo_select_font_face (cr, "Sans", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_BOLD);
      cairo_set_font_size (cr, fn_size);
      cairo_move_to (cr, DEFAULT_LBORDER, 1.4 * fn_size);
      cairo_show_text (cr, DISCLAIMER);
    }

  if ((exported_picture) && (print_addinfo))
    {
      char *fmt_text;
      int print_terminated, until_equation, first_equation = 0;

      cairo_select_font_face (cr, "Courier", CAIRO_FONT_SLANT_NORMAL,
			      CAIRO_FONT_WEIGHT_NORMAL);
      cairo_set_font_size (cr, mini_fn_size);
      do
      	{
      	  curr_pos = orig_pos;
      	  fmt_text = get_formatted_equations (export_settings, &curr_pos,
					      first_equation, &until_equation);
      	  if ((fmt_text))
      	    {
#ifdef _TRACE_
	      printf ("orig_pos (x,y) = (%f,%f)\n", orig_pos.x, orig_pos.y);
	      printf ("curr_pos (x,y) = (%f,%f)\n", curr_pos.x, curr_pos.y);
	      printf ("first_equation = %d\n", first_equation);
	      printf (" last_equation = %d\n", until_equation-1);
      	      printf ("fmt_text = >>\n%s\n<<\n", fmt_text);
#endif
	      if ( *(mgr_filename(NULL)) != '\0' )
		cairo_print_lines (cr, mode, fmt_text, 0, 
				   export_settings, orig_pos);
	      else
		cairo_print_lines (cr, mode, fmt_text, 1, 
				   export_settings, orig_pos);
      	      xfree (&fmt_text);
      	    }
	  print_terminated = (mode == GRAPHIC_2D) ? 
	    (until_equation == NUMBER_OF_2DOBJECTS) :
	    (until_equation == NUMBER_OF_3DOBJECTS);
      	  if (!print_terminated)
      	    {
	      first_equation = until_equation;
      	      /* Open a new page and start to write from the top left corner */
      	      cairo_show_page(cr);
      	      orig_pos.x = export_settings.left;
      	      orig_pos.y = export_settings.top;
      	    }
      	} while (!print_terminated);
    } /* if ((exported_picture) && (print_addinfo)) */

  /*
    Remark: if (exported_picture) && (print_addinfo) == FALSE then
    curr_pos.y has kept the value 0.0!
  */
  if ((exported_picture))
    {
      if (export_settings.bottom - curr_pos.y <
	  DEFAULT_TBORDER + DEFAULT_SIDE + DEFAULT_BBORDER)
	{
	  /* Open a new page and start to draw from the top left corner */
	  cairo_show_page(cr);
	  curr_pos.y = 0;
	}
    }
  cairo_set_source_rgb (cr, 0.0, 0.0, 0.0);
  cairo_select_font_face (cr, "Sans", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_NORMAL);
  cairo_set_font_size (cr, fn_size);
  retrieve_origin_position (&o_x, &o_y, &o_z);
  grid_step = get_grid_step ();
  o_x = smart_round (o_x, grid_step, 1);
  o_y = smart_round (o_y, grid_step, 1);
  o_z = smart_round (o_z, grid_step, 1);
  sprintf (g_step, STRING_FMT_G, grid_step);
  sprintf (orig_x, STRING_FMT_X, o_x);
  sprintf (orig_y, STRING_FMT_Y, o_y);
  if (mode == GRAPHIC_2D)
    {
      cairo_move_to (cr, 20+DEFAULT_LBORDER, curr_pos.y + 10 + 1.4 * fn_size); 
      cairo_show_text (cr, orig_x);
      cairo_move_to (cr, 20+DEFAULT_LBORDER+10*fn_size, curr_pos.y + 10 + 1.4 * fn_size);
      cairo_show_text (cr, orig_y);
      cairo_move_to (cr, 20+DEFAULT_LBORDER+20*fn_size, curr_pos.y + 10 + 1.4 * fn_size);
      cairo_show_text (cr, g_step);
    }
  else
    {
      sprintf (orig_z, STRING_FMT_Z, o_z);
      cairo_move_to (cr, 20+DEFAULT_LBORDER, curr_pos.y + 10 + 1.4 * fn_size); 
      cairo_show_text (cr, orig_x);
      cairo_move_to (cr, 20+DEFAULT_LBORDER+10*fn_size, curr_pos.y + 10 + 1.4 * fn_size);
      cairo_show_text (cr, orig_y);
      cairo_move_to (cr, 20+DEFAULT_LBORDER+20*fn_size, curr_pos.y + 10 + 1.4 * fn_size);
      cairo_show_text (cr, orig_z);
      cairo_move_to (cr, 20+DEFAULT_LBORDER+30*fn_size, curr_pos.y + 10 + 1.4 * fn_size);
      cairo_show_text (cr, g_step);
    }

  img = draw_on_image (r, DEFAULT_LBORDER, DEFAULT_BBORDER);
  cairo_set_source_surface (cr, img, 0.0, curr_pos.y + DEFAULT_TBORDER);
  cairo_paint (cr);
  cairo_surface_destroy (img);
}

cairo_t* new_cairo_context (cairo_initializer in, int* r, gboolean* pdf_or_ps_surface)
{
  GtkAllocation a;
  cairo_surface_t *surface = NULL;
  cairo_t* cr = NULL;
  FILE* fp;
  gboolean use_paper_format = FALSE;

  *r = 0;
  if ( (in.wd) )
    {
      cr = gdk_cairo_create(gtk_widget_get_window(in.wd));
      gtk_widget_get_allocation (in.wd, &a);
      if (a.width < DEFAULT_WIDTH || a.height < DEFAULT_HEIGHT)
	{
	  cairo_destroy (cr);
	  cr = NULL;
	}
      else
	{
	  *r = DEFAULT_SIDE / 2;
	}
    }
  else if ( (in.ps) )
    {
      /* POSTSCRIPT */
      fp = fopen (in.ps, "w");
      if ( (fp) )
        {
          gchar* ptr = NULL;
            
          ptr = g_strrstr (in.ps, ".eps");
          if ((ptr) && *(ptr+4) == '\0')
            {
              /* ENCAPSULATED POSTSCRIPT */  
              surface = cairo_ps_surface_create (in.ps, 
						 DEFAULT_WIDTH, DEFAULT_HEIGHT);
              fclose (fp);
              cairo_ps_surface_set_eps (surface, TRUE);
            }    
          else
            {
              /* NORMAL POSTSCRIPT */
              surface = cairo_ps_surface_create (in.ps, 
					         supported_paper_formats[current_paper_format].w,
					         supported_paper_formats[current_paper_format].h);              
              fclose (fp);
              cairo_ps_surface_set_eps (surface, FALSE);  
	      ptr = g_strdup_printf ("%%%%Orientation: %s", supported_orientations[current_orientation]);
	      cairo_ps_surface_dsc_comment (surface, ptr);
	      g_free (ptr);
              use_paper_format = TRUE;
	    }
	}
    }
  else if ( (in.pdf) )
    {
      /* PDF */
      fp = fopen (in.pdf, "w");
      if ( (fp) )
	{
	  surface = cairo_pdf_surface_create (in.pdf, 
					      supported_paper_formats[current_paper_format].w, 
					      supported_paper_formats[current_paper_format].h);
	  fclose (fp);
          use_paper_format = TRUE;
	}
    }
  else if ( (in.svg) )
    {
      /* SVG */
      fp = fopen (in.svg, "w");
      if ( (fp) )
	{
	  surface = cairo_svg_surface_create (in.svg, 
					      DEFAULT_WIDTH, DEFAULT_HEIGHT);
	  fclose (fp);
	}
    }
  if ( (surface) )
    {
      /* Only for POSTSCRIPT, PDF and SVG */
      cr = cairo_create (surface);
      if (use_paper_format == TRUE)
        {
	  cairo_text_extents_t te;

	  cairo_select_font_face (cr, "Courier", CAIRO_FONT_SLANT_NORMAL, CAIRO_FONT_WEIGHT_NORMAL);
	  cairo_set_font_size (cr, mini_fn_size);	  
	  cairo_text_extents (cr, "M", &te);
	  export_settings.Mw = te.width + 1.0; /* Account for space between
						  characters */
	  export_settings.Mh = te.height;
	  export_settings.line_spacing = 6.0;
	  export_settings.tab_width = 5;
	  if (current_orientation == Landscape)
	    {
	      cairo_matrix_t matrix;
	      
	      cairo_translate (cr, 0, supported_paper_formats[current_paper_format].h);
	      cairo_matrix_init (&matrix, 0, -1, 1, 0, 0,  0);
	      cairo_transform (cr, &matrix);
	      export_settings.left = 10;
	      export_settings.right = supported_paper_formats[current_paper_format].h - 10;
	      export_settings.top = 10;
	      export_settings.bottom = supported_paper_formats[current_paper_format].w - 10;
	    }
	  else
	    {
	      /* current_orientation == Portrait */
	      export_settings.left = 10;
	      export_settings.right = supported_paper_formats[current_paper_format].w - 10;
	      export_settings.top = 10;
	      export_settings.bottom = supported_paper_formats[current_paper_format].h - 10;
	    }
	  start_point.x = 10;
	  start_point.y = DEFAULT_TBORDER;
#ifdef _TRACE_
	  printf ("Export settings:\n");
	  printf ("Left border  = %f\n", export_settings.left);
	  printf ("Right border = %f\n", export_settings.right);
	  printf ("Upper border = %f\n", export_settings.top);
	  printf ("Lower border = %f\n", export_settings.bottom);
	  printf ("Line spacing = %f\n", export_settings.line_spacing);
	  printf ("Tab width    = %u\n", export_settings.tab_width);
	  printf ("Mw           = %f\n", export_settings.Mw);
	  printf ("Mh           = %f\n", export_settings.Mh);
	  printf ("Start point:\n");
	  printf ("x = %f\n",  start_point.x);
	  printf ("y = %f\n",  start_point.y);
#endif
        }
      cairo_surface_destroy (surface);
      *r = DEFAULT_SIDE / 2;
    }
  *pdf_or_ps_surface = use_paper_format;
  return cr;
}

cairo_paper_format get_paper_format (void)
{
  return current_paper_format;
}

void set_paper_format (cairo_paper_format new_format)
{
  current_paper_format = new_format;
}

cairo_orientation get_orientation (void)
{
  return current_orientation;
}

void set_orientation (cairo_orientation new_orientation)
{
  current_orientation = new_orientation;
}

gboolean are_addinfo_printed (void)
{
  return print_addinfo;
}

void please_print_addinfo (gboolean yesno)
{
  print_addinfo = yesno;
}

gboolean is_disclaimer_printed (void)
{
  return print_disclaimer;
}

void please_print_disclaimer (gboolean yesno)
{
  print_disclaimer = yesno;
}
