/*
 * luks.c - LUKS Encryption
 * Copyright (C) 2008 Michael Gorven <michael@gorven.za.net>
 */

/*
 *  GRUB  --  GRand Unified Bootloader
 *  Copyright (C) 2003,2007  Free Software Foundation, Inc.
 *
 *  GRUB is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  GRUB is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with GRUB.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <grub/types.h>
#include <grub/misc.h>
#include <grub/mm.h>
#include <grub/dl.h>
#include <grub/term.h>
#include <grub/err.h>
#include <grub/lib/arg.h>
#include <grub/disk.h>
#include <grub/crypto.h>
#include <grub/env.h>
#include <grub/normal.h>
#include <grub/command.h>

#define MAX_PASSPHRASE 256
#define MAX_KEYSIZE 64
#define DEFAULT_HASH	"ripemd160"
#define DEFAULT_CIPHER	"aes-cbc"

#define MIN(a, b)       (a < b ? a : b)

#define LUKS_MAGIC_L 6
#define LUKS_CIPHERNAME_L 32
#define LUKS_CIPHERMODE_L 32
#define LUKS_HASHSPEC_L 32
#define UUID_STRING_L 40

#define LUKS_MAGIC        "LUKS\xBA\xBE"
#define LUKS_DIGESTSIZE   20
#define LUKS_SALTSIZE     32
#define LUKS_NUMKEYS      8
#define LUKS_MKD_ITER     10
#define LUKS_KEY_DISABLED 0x0000DEAD
#define LUKS_KEY_ENABLED  0x00AC71F3
#define LUKS_STRIPES      4000

/* First two entries of on disk LUKS header */
struct grub_luks_phdr_id {
  grub_uint8_t magic[LUKS_MAGIC_L];
  grub_uint16_t version;
};

/* On disk LUKS header */
struct grub_luks_phdr {
  grub_uint8_t magic [LUKS_MAGIC_L];
  grub_uint16_t version;
  char cipherName [LUKS_CIPHERNAME_L];
  char cipherMode [LUKS_CIPHERMODE_L];
  char hashSpec[LUKS_HASHSPEC_L];
  grub_uint32_t payloadOffset;
  grub_uint32_t keyBytes;
  grub_uint8_t mkDigest [LUKS_DIGESTSIZE];
  grub_uint8_t mkDigestSalt [LUKS_SALTSIZE];
  grub_uint32_t mkDigestIterations;
  grub_uint8_t uuid [UUID_STRING_L];
  struct {
    grub_uint32_t active;
    grub_uint32_t passwordIterations;
    grub_uint8_t passwordSalt [LUKS_SALTSIZE];
    grub_uint32_t keyMaterialOffset;
    grub_uint32_t stripes;
  } keyblock [LUKS_NUMKEYS];
} __attribute__ ((packed));

typedef struct grub_luks_phdr *grub_luks_phdr_t;

typedef enum
  {
    GRUB_LUKS_MODE_ECB,
    GRUB_LUKS_MODE_CBC_PLAIN,
    GRUB_LUKS_MODE_CBC_ESSIV
  }
luks_mode_t;

struct luks_private
{
  grub_uint32_t offset;
  grub_disk_t source;
  grub_crypto_cipher_handle_t cipher;
  grub_crypto_cipher_handle_t essiv_cipher;
  luks_mode_t mode;
};
typedef struct luks_private *luks_private_t;

struct grub_luks
{
  char *devname, *source;
  luks_private_t private;

  struct grub_luks *next;
};
typedef struct grub_luks *grub_luks_t;

static grub_luks_t luks_list = NULL;
grub_uint8_t n = 0;

void luks_cleanup(void);
grub_err_t grub_read_password(char* prompt, char* buffer, grub_size_t len);
void print_buffer(char *label, grub_uint8_t *buffer, grub_size_t len);
gcry_err_code_t AF_merge(const gcry_md_spec_t *hash, grub_uint8_t *src, grub_uint8_t *dst, grub_size_t blocksize, grub_size_t blocknumbers);
gcry_err_code_t luks_decrypt(grub_crypto_cipher_handle_t cipher,
			luks_mode_t mode,
			grub_uint8_t *data, grub_size_t len, grub_size_t sector,
			grub_crypto_cipher_handle_t essiv_cipher);
gcry_err_code_t AF_split(const gcry_md_spec_t *hash, grub_uint8_t *src,
		    grub_uint8_t *dst, grub_size_t blocksize, grub_size_t blocknumbers);

grub_err_t grub_read_password(char* prompt, char* buffer, grub_size_t len)
{
  char c;
  grub_size_t pos = 0;

  grub_printf(prompt);

  while ((c = GRUB_TERM_ASCII_CHAR (grub_getkey())) != '\n' && c != '\r')
  {
    if (grub_isprint(c) && pos < len-1)
    {
      buffer[pos++] = c;
      grub_putchar('*');
    }
    else if (c == '\b' && pos > 0)
    {
      buffer[--pos] = 0;
      grub_putchar(c);
      grub_putchar(' ');
      grub_putchar(c);
    }

    grub_refresh();
  }

  buffer[pos] = 0;
  grub_putchar('\n');
  grub_refresh();
  
  return GRUB_ERR_NONE;
}

gcry_err_code_t
luks_decrypt(grub_crypto_cipher_handle_t cipher, luks_mode_t mode,
	     grub_uint8_t *data, grub_size_t len,
	     grub_size_t sector, grub_crypto_cipher_handle_t essiv_cipher)
{
  grub_size_t i;
  gcry_err_code_t err;

  switch (mode)
  {
    case GRUB_LUKS_MODE_ECB:
      return grub_crypto_ecb_decrypt(cipher, data, data, len);
      break;

    case GRUB_LUKS_MODE_CBC_PLAIN:
      for (i = 0; i < len; i += GRUB_DISK_SECTOR_SIZE)
      {
	grub_uint32_t iv[(cipher->cipher->blocksize
			  + sizeof (grub_uint32_t) - 1) 
			 / sizeof (grub_uint32_t)];
	grub_memset(iv, 0, cipher->cipher->blocksize);
        iv[0] = grub_cpu_to_le32(sector & 0xFFFFFFFF);
        err = grub_crypto_cbc_decrypt(cipher, data+i, data+i,
				      GRUB_DISK_SECTOR_SIZE, iv);
        if (err)
          return err;
        sector++;
      }
      return GPG_ERR_NO_ERROR;
      break;

    case GRUB_LUKS_MODE_CBC_ESSIV:
      for (i = 0; i < len; i += GRUB_DISK_SECTOR_SIZE)
      {
	grub_uint32_t iv[(cipher->cipher->blocksize
			  + sizeof (grub_uint32_t) - 1)
			 / sizeof (grub_uint32_t)];
        grub_memset(iv, 0, cipher->cipher->blocksize);
        iv[0] = grub_cpu_to_le32(sector & 0xFFFFFFFF);
        err = grub_crypto_ecb_encrypt(essiv_cipher, iv, iv, cipher->cipher->blocksize);
        if (err)
          return err;
        err = grub_crypto_cbc_decrypt(cipher, data+i, data+i,
				      GRUB_DISK_SECTOR_SIZE, iv);
        if (err)
          return err;
        sector++;
      }
      return GPG_ERR_NO_ERROR;
      break;

    default:
      return GPG_ERR_NOT_IMPLEMENTED;
      break;

  }
  return GPG_ERR_INV_ARG;
}

static int
grub_luks_scan_device (const char *name)
{
  grub_disk_t disk;
  grub_err_t err;
  char buffer[sizeof(struct grub_luks_phdr_id)];
  struct grub_luks_phdr_id *header = (struct grub_luks_phdr_id *) buffer;
  grub_luks_t newdev;

  // Try to open disk
  disk = grub_disk_open(name);
  if (!disk)
    return 0;

  // Try to read LUKS header
  err = grub_disk_read(disk, 0, 0, sizeof(buffer), buffer);
  if (err)
    return 0;
  grub_disk_close(disk);

  // Look for LUKS magic sequence
  if (grub_memcmp(header->magic, LUKS_MAGIC, LUKS_MAGIC_L)
      || grub_be_to_cpu16(header->version) != 1)
    return 0;

  newdev = grub_malloc(sizeof(struct grub_luks));
  if (! newdev)
    return grub_errno;

  newdev->devname = grub_xasprintf ("lk%d", n++);
  newdev->source = grub_strdup(name);
  newdev->private = NULL;

  newdev->next = luks_list;
  luks_list = newdev;

  return 0;
}

static int
grub_luks_iterate (int (*hook) (const char *name))
{
  grub_luks_t i;

  for (i = luks_list; i != NULL; i = i->next)
    if (hook (i->devname))
      return 1;

  return GRUB_ERR_NONE;
}

static grub_err_t
grub_luks_open (const char *name, grub_disk_t disk)
{
  grub_luks_t dev;

  // Search for requested device in the list of LUKS devices
  for (dev = luks_list; dev != NULL; dev = dev->next)
    if (grub_strcmp (dev->devname, name) == 0)
      break;

  if (!dev) 
    return grub_error(GRUB_ERR_UNKNOWN_DEVICE, "No such device");

  grub_dprintf("luks", "Opening device %s\n", name);

  // Check if we've already recovered the master key
  if (dev->private)
  {
    grub_dprintf("luks", "Master key already recovered\n");
    grub_disk_t source;
    // Try to open the source disk and populate the requested disk
    source = grub_disk_open(dev->source);
    if (!source)
      return grub_errno;
    disk->data = dev->private;
    disk->total_sectors = grub_disk_get_size (source) - dev->private->offset;
    disk->id = (int) dev;
    disk->has_partitions = 0;
    dev->private->source = source;
    return GRUB_ERR_NONE;
  }

  char passphrase[MAX_PASSPHRASE];
  passphrase[0] = 0;

  // Try to open source disk
  grub_disk_t source = grub_disk_open(dev->source);
  if (!source)
    return grub_errno;

  grub_crypto_cipher_handle_t cipher=NULL, essiv_cipher=NULL;
  const gcry_md_spec_t *hash=NULL, *essiv_hash=NULL;
  grub_size_t keysize;
  /* GCC thinks we may use this variable uninitialised. Silence the warning.  */
  grub_size_t essiv_keysize = 0;
  grub_uint8_t *hashed_key = NULL;
  char buffer[sizeof(struct grub_luks_phdr)];
  struct grub_luks_phdr *header = (struct grub_luks_phdr *) buffer;
  luks_mode_t mode;

  // Read the LUKS header
  {
    grub_err_t err = grub_disk_read(source, 0, 0, sizeof(buffer), buffer);

    if (err)
      return err;
  }

  // Make sure that strings are null terminated
  header->cipherName[LUKS_CIPHERNAME_L-1] = 0;
  header->cipherMode[LUKS_CIPHERMODE_L-1] = 0;
  header->hashSpec[LUKS_HASHSPEC_L-1] = 0;

  const struct gcry_cipher_spec *ciph;
  ciph = grub_crypto_lookup_cipher_by_name (header->cipherName);
  if (!ciph)
    return grub_error (GRUB_ERR_FILE_NOT_FOUND, "Cipher %s isn't available",
		       header->cipherName);

  // Configure the cipher used for the bulk data
  cipher = grub_crypto_cipher_open (ciph);
  if (!cipher)
    return grub_errno;

  keysize = grub_be_to_cpu32(header->keyBytes);

  // Configure the cipher mode
  if (grub_strncmp(header->cipherMode, "ecb", 3) == 0)
    mode = GRUB_LUKS_MODE_ECB;
  else if (grub_strncmp(header->cipherMode, "cbc-plain", 9) == 0
	   || grub_strncmp(header->cipherMode, "plain", 5) == 0)
    mode = GRUB_LUKS_MODE_CBC_PLAIN;

  else if (grub_strncmp(header->cipherMode, "cbc-essiv", 9) == 0)
    {
      mode = GRUB_LUKS_MODE_CBC_ESSIV;
      char *hash_str = header->cipherMode + 10;
      
      // Configure the hash and cipher used for ESSIV
      essiv_hash = grub_crypto_lookup_md_by_name (hash_str);
      if (!essiv_hash)
	{
	  grub_crypto_cipher_close(cipher);
	  return grub_error(GRUB_ERR_FILE_NOT_FOUND, "Couldn't load %s hash",
			  hash_str);
	}
      essiv_cipher = grub_crypto_cipher_open (ciph);
      if (!cipher)
	{
	  grub_crypto_cipher_close(cipher);
	  return grub_errno;
	}
      
      essiv_keysize = essiv_hash->mdlen;
      hashed_key = grub_malloc(essiv_hash->mdlen);
      if (!hashed_key) 
	{
	  grub_crypto_cipher_close(cipher);
	  grub_crypto_cipher_close(essiv_cipher);
	  return grub_errno;
	}
    }
  else 
    {
      grub_crypto_cipher_close(cipher);
      return grub_error(GRUB_ERR_BAD_ARGUMENT, "Unknown cipher mode: %s", header->cipherMode);
    }

  // Configure the hash used for the AF splitter and HMAC
  hash = grub_crypto_lookup_md_by_name (header->hashSpec);
  if (!hash) 
    {
      grub_crypto_cipher_close(cipher);
      grub_crypto_cipher_close(essiv_cipher);
      grub_free (hashed_key);
      return grub_error(GRUB_ERR_FILE_NOT_FOUND, "Couldn't load %s hash", header->hashSpec);
    }

  grub_printf("Attempting to decrypt master key...\n");

  int i;
  grub_size_t length;
  grub_uint8_t candidate_key[MAX_KEYSIZE], candidate_digest[LUKS_DIGESTSIZE], digest[MAX_KEYSIZE];
  grub_uint8_t *split_key = NULL;
  luks_private_t private = NULL;

  split_key = grub_malloc(MAX_KEYSIZE*LUKS_STRIPES);
  if (!split_key)
    {
      grub_crypto_cipher_close(cipher);
      grub_crypto_cipher_close(essiv_cipher);
      grub_free (hashed_key);
      return grub_errno;
    }

  char *test_passphrase;
  if ((test_passphrase = grub_env_get("passphrase")))
    grub_strcpy(passphrase, test_passphrase);
  else 
    {
      // Get the passphrase from the user
      grub_read_password("Enter passphrase: ", passphrase, MAX_PASSPHRASE);
      if (grub_strlen(passphrase)==0)
	{
	  grub_crypto_cipher_close(cipher);
	  grub_crypto_cipher_close(essiv_cipher);
	  grub_free (hashed_key);
	  grub_free (split_key);
	  return grub_error(GRUB_ERR_BAD_ARGUMENT, "Passphrase not supplied");
	}
    }

  //Try to recover master key from each active keyslot
  for (i = 0; i < LUKS_NUMKEYS; i++)
  {
    // Check if keyslot is enabled
    if (grub_be_to_cpu32(header->keyblock[i].active) == LUKS_KEY_ENABLED)
    {
      gcry_err_code_t gcry_err;
      grub_dprintf("luks", "Trying keyslot %d\n", i);

      // Calculate the PBKDF2 of the user supplied passphrase
      gcry_err = grub_crypto_pbkdf2 (hash, (grub_uint8_t *) passphrase,
				     grub_strlen(passphrase),
				     header->keyblock[i].passwordSalt,
				     LUKS_SALTSIZE,
				     grub_be_to_cpu32(header->keyblock[i].passwordIterations),
				     digest, keysize);
      
      if (gcry_err)
	{
	  grub_crypto_cipher_close(cipher);
	  grub_crypto_cipher_close(essiv_cipher);
	  grub_free (hashed_key);
	  grub_free (split_key);
	  return grub_crypto_gcry_error (gcry_err);
	}

      // Set the PBKDF2 output as the cipher key
      gcry_err = grub_crypto_cipher_set_key(cipher, digest, keysize);
      if (gcry_err) {
	  grub_crypto_cipher_close(cipher);
	  grub_crypto_cipher_close(essiv_cipher);
	  grub_free (hashed_key);
	  grub_free (split_key);
	  return grub_crypto_gcry_error (gcry_err);
      }

      // Configure ESSIV if necessary
      if (mode == GRUB_LUKS_MODE_CBC_ESSIV)
      {
        grub_crypto_hash(essiv_hash, hashed_key, digest, keysize);
        grub_crypto_cipher_set_key(essiv_cipher, hashed_key, essiv_keysize);
      }

      length = grub_be_to_cpu32(header->keyBytes) * grub_be_to_cpu32(header->keyblock[i].stripes);

      // Read and decrypt the key material from the disk
      {
	grub_err_t err = grub_disk_read(source, grub_be_to_cpu32(header->keyblock[i].keyMaterialOffset), 0, length, split_key);
	if (err)
	  {
	    grub_crypto_cipher_close(cipher);
	    grub_crypto_cipher_close(essiv_cipher);
	    grub_free (hashed_key);
	    grub_free (split_key);
	    return err;
	  }
      }

      gcry_err = luks_decrypt(cipher, mode, split_key, length, 0, essiv_cipher);
      if (gcry_err)
	{
	  grub_crypto_cipher_close(cipher);
	  grub_crypto_cipher_close(essiv_cipher);
	  grub_free (hashed_key);
	  grub_free (split_key);
	  return grub_crypto_gcry_error (gcry_err);
	}

      // Merge the decrypted key material to get the candidate master key
      gcry_err = AF_merge(hash, split_key, candidate_key, keysize, grub_be_to_cpu32(header->keyblock[i].stripes));
      if (gcry_err)
	{
	  grub_crypto_cipher_close(cipher);
	  grub_crypto_cipher_close(essiv_cipher);
	  grub_free (hashed_key);
	  grub_free (split_key);
	  return grub_crypto_gcry_error (gcry_err);
	}

      // Calculate the PBKDF2 of the candidate master key
      gcry_err = grub_crypto_pbkdf2(hash, candidate_key,
				    grub_be_to_cpu32(header->keyBytes),
				    header->mkDigestSalt, LUKS_SALTSIZE,
				    grub_be_to_cpu32(header->mkDigestIterations),
				    candidate_digest, LUKS_DIGESTSIZE);
      if (gcry_err)
	{
	  grub_crypto_cipher_close(cipher);
	  grub_crypto_cipher_close(essiv_cipher);
	  grub_free (hashed_key);
	  grub_free (split_key);
	  return grub_crypto_gcry_error (gcry_err);
	}

      // Compare the calculated PBKDF2 to the digest stored in the header to see if it's correct
      if (grub_memcmp(candidate_digest, header->mkDigest, LUKS_DIGESTSIZE) == 0)
	{
	  grub_printf("Slot %d opened\n", i);

	  // Set the master key
	  gcry_err = grub_crypto_cipher_set_key(cipher, candidate_key, keysize);
        if (gcry_err)
	  {
	    grub_crypto_cipher_close(cipher);
	    grub_crypto_cipher_close(essiv_cipher);
	    grub_free (hashed_key);
	    grub_free (split_key);
	    return grub_crypto_gcry_error (gcry_err);
	  }

        private = grub_malloc(sizeof(struct luks_private));
        private->cipher = cipher;

        // Configure ESSIV if necessary
        if (mode == GRUB_LUKS_MODE_CBC_ESSIV)
        {
          grub_crypto_hash(essiv_hash, hashed_key, candidate_key, keysize);
          gcry_err = grub_crypto_cipher_set_key(essiv_cipher, hashed_key, essiv_keysize);
          if (gcry_err)
	    {
	      grub_crypto_cipher_close(cipher);
	      grub_crypto_cipher_close(essiv_cipher);
	      grub_free (hashed_key);
	      grub_free (split_key);
	      grub_free (private);
	      return grub_crypto_gcry_error (gcry_err);
	    }
          private->essiv_cipher = essiv_cipher;
        } else {
          private->essiv_cipher = NULL;
        }

        private->offset = grub_be_to_cpu32(header->payloadOffset);
        private->source = source;
	private->mode = mode;

        // Populate requested disk
        disk->total_sectors = grub_disk_get_size (source) - private->offset;
        disk->id = (int) dev;
        disk->has_partitions = 0;
        disk->data = private;
        dev->private = private;
	grub_free(split_key);
	grub_free(hashed_key);
        return GRUB_ERR_NONE;
      }
    }
  }

  return GRUB_ACCESS_DENIED;
}

static void
grub_luks_close (grub_disk_t disk)
{
  luks_private_t private = (luks_private_t) disk->data;
  grub_dprintf("luks", "Closing disk\n");

  if (private) {
    grub_disk_close(private->source);
    disk->data = NULL;
  }
}

static grub_err_t
grub_luks_read (grub_disk_t disk, grub_disk_addr_t sector,
    grub_size_t size, char *buf)
{
  luks_private_t private = (luks_private_t) disk->data;
  grub_err_t err;
  grub_dprintf("luks", "Reading %d sectors from sector 0x%llx with offset of %d\n", size, sector, private->offset);

  err = grub_disk_read(private->source, sector + private->offset, 0, size<<GRUB_DISK_SECTOR_BITS, buf);
  if (err) {
    grub_dprintf("luks", "grub_disk_read failed with error %d\n", err);
    return err;
  }
  return grub_crypto_gcry_error (luks_decrypt(private->cipher,
					      private->mode,
					      (grub_uint8_t *) buf,
					      size<<GRUB_DISK_SECTOR_BITS,
					      sector, private->essiv_cipher));
}

static grub_err_t
grub_luks_write (grub_disk_t disk __attribute ((unused)),
    grub_disk_addr_t sector __attribute ((unused)),
    grub_size_t size __attribute ((unused)),
    const char *buf __attribute ((unused)))
{
  return GRUB_ERR_NOT_IMPLEMENTED_YET;
}

void luks_cleanup()
{
  grub_luks_t dev = luks_list;
  grub_luks_t tmp;

  while (dev != NULL) {
    grub_free(dev->devname);
    grub_free(dev->source);
    if (dev->private) {
      grub_free(dev->private->cipher);
      if (dev->private->essiv_cipher)
        grub_free(dev->private->essiv_cipher);
      grub_free(dev->private);
    }
    tmp = dev->next;
    grub_free(dev);
    dev = tmp;
  }
}

static struct grub_disk_dev grub_luks_dev = {
  .name = "luks",
  .id = GRUB_DISK_DEVICE_LUKS_ID,
  .iterate = grub_luks_iterate,
  .open = grub_luks_open,
  .close = grub_luks_close,
  .read = grub_luks_read,
  .write = grub_luks_write,
  .next = 0
};

GRUB_MOD_INIT(luks)
{
  grub_device_iterate(&grub_luks_scan_device);
  grub_disk_dev_register(&grub_luks_dev);
}

GRUB_MOD_FINI(luks)
{
  grub_disk_dev_unregister(&grub_luks_dev);
  luks_cleanup();
}

/* vi: set et sw=2 sts=2: */
