/* graph_sampler.c

   Written by Frederic Bois
   22 June 2014

   Copyright (c) 2014 Frederic Bois.

   This code is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   See the GNU General Public License at <http://www.gnu.org/licenses/>

   -- Revisions -----
     Logfile:  %F%
    Revision:  %I%
        Date:  %G%
     Modtime:  %U%
      Author:  @a
   -- SCCS  ---------

   Samples either Bayesian networks or general directed graphs by MCMC,
   given specified priors and a likelihood

   Priors are Bernoulli on edges, power on degree and
   beta-binomial on the proportion of E loops (A -> B -> C -> A) over the
   sum of E loops and F loops (A -> B <- C -> A)

   Note on indices: for a square matrix (N x N) with i indexing lines and
   j indexing columns, i and j starting at zero,
   positions z (starting at zero) corresponding to the first line are given by:
     z = i + j * N
   positions z (starting at zero) corresponding to the first column are:
     z = j + i * N
   given a position z (starting at 0) in the flattened matrix (flattened as
   (column 1, column 2...), the corresponding row i (starting at 0) is:
     i = z % N
   the corresponding column j (starting from zero) is:
     j = z / N ("/" being an integer division)
*/


/* ----------------------------------------------------------------------------
   Inclusions
*/

#include "graph_sampler.h"


/* ----------------------------------------------------------------------------
   Global definitions, private
*/

// stuff for basic Bernoulli prior
double **log_hyper_pB;      // point to prior edge log probability matrix
double **log_hyper_qB;      // point to prior edge 1-log probability matrix

// stuff for priors handling
double diff_logprior = 0;

// stuff for likelihood handling and computations
double diff_loglikelihood = 0;

// stuff for posterior handling
double diff_logposterior = 0;

// default input file name
static char vszFileInDefault[] = "script.txt";


/* ----------------------------------------------------------------------------
   AnnounceProgram

   Print a screen explanation of what we are doing.
*/
void AnnounceProgram (void)
{

  printf("\n");
  printf("*******************************\n");
  printf(">>> This is a graph sampler <<<\n");
  printf("*******************************\n");
  printf("\n");

} /* AnnounceProgram */


/* ----------------------------------------------------------------------------
   CleanupMemory

   Releases (some!) pointers.
*/
void CleanupMemory (void)
{

  if (current_degrees)   free(current_degrees);

  if (current_ll_node)   free(current_ll_node);

  if (pdWorkMatrixSizeN) free(pdWorkMatrixSizeN);

} /* CleanupMemory */


/* ----------------------------------------------------------------------------
   GetCmdLineArgs

   Retrieves filenames from the command line arguments passed to
   the program.

   The command line syntax is:

     graph_sampler [input-file [output-files prefix]]

   Missing names are replaced by defaults .
*/
void GetCmdLineArgs (int cArg, char *const *rgszArg, char **pszFileIn,
                     char **pszPrefixOut)
{
  *pszFileIn = *pszPrefixOut = (char *) NULL;

  switch (cArg) { // filenames
    case 3: // output and input file specificed
      *pszPrefixOut = rgszArg[2];
      if (strlen(rgszArg[2]) > MAXFILENAME) {
        printf("Error: output file name must be less that %d "
               "caracter long - Exiting.\n\n", MAXFILENAME);
        exit(0);
      }
      // Fall through!

    case 2: // input file specified
      *pszFileIn = rgszArg[1];
      break;

    case 1: // no file names specified
      *pszFileIn = vszFileInDefault;
      *pszPrefixOut = NULL;
      break;

    default:
      /* ShowHelp ("Usage"); */ /* disabled for now */
      exit (-1);
      break;
  } // switch

} /* GetCmdLineArgs */


/* ----------------------------------------------------------------------------
   InitArrays

   Initialize various arrays if they have not been initialized in input.
*/
void InitArrays (void)
{
  int i, j;
  register int count;

  // if current_adj is not defined it is initialized here to 0
  if (!current_adj) {
    current_adj = InitiMatrix(nNodes, nNodes);
    printf ("Setting current_adj to default value (null matrix).\n");

    for (i = 0; i < nNodes; i++)
      for (j = 0; j < nNodes; j++)
        current_adj[i][j] = 0;
  }

  // initialize tempered sampling variables
  if (bTempered) {
    indexT = 0;  // start hot
    dCZero = 10; // why ?
    dNZero = 50; // why not ? 
    plnPi = InitdVector (nTemperatures);
  }
  else {
    indexT = 0;
    nTemperatures = 1;
  }
  
  // initialize space for the best (maximum probability) adjacency matrix
  if (bsave_best_graph) {
    best_adj = InitiMatrix(nNodes, nNodes);
    for (i = 0; i < nNodes; i++)
      memcpy(best_adj[i], current_adj[i], nNodes * sizeof(int));
  }

  // if hyper_pB is not defined it is initialized here to 0.5 for
  // all elements (except the diagonal for BNs)
  if (!hyper_pB) {
    hyper_pB = InitdMatrix(nNodes, nNodes);
    printf ("Setting hyper_pB to default value (equiprobability).\n\n");

    for (i = 0; i < nNodes; i++) {
      for (j = 0; j < nNodes; j++) {
        if ((i == j) && (bBN || autocycle))
          hyper_pB[i][j] = 0.0;
        else
          hyper_pB[i][j] = 0.5;
      }
    }
  }
  log_hyper_pB = InitdMatrix(nNodes, nNodes);
  log_hyper_qB = InitdMatrix(nNodes, nNodes);

  for (i = 0; i < nNodes; i++) { // ith line
    for (j = 0; j < nNodes; j++) { // jth column
      log_hyper_pB[i][j] = log(hyper_pB[i][j]);
      log_hyper_qB[i][j] = log(1 - hyper_pB[i][j]);
    }
  }

  // initialize matrix edge_requirements to zero by default
  if (bPriorConcordance && !edge_requirements) {
    edge_requirements = InitiMatrix(nNodes, nNodes);
    printf ("Setting edge_requirements to default value (indifference).\n");

    for (i = 0; i < nNodes; i++) {
      for (j = 0; j < nNodes; j++) {
        if ((i == j) && (bBN))
          edge_requirements[i][j] = -1;
        else
          edge_requirements[i][j] = 0;
      }
    }
  }

  // initialize a table of differences between the binomial log
  // probabilities of the total number of edges when going up
  // successive edge counts: P(n=x+1) - P(n=x)
  if (bPriorEdgeCount == TRUE) {
    long   N = nNodes * nNodes;
    double p = expected_n_edges / (double) N; 

    pdiff_binom_P = InitdVector(N);
    for (i = 0; i < N; i++) // going up
      pdiff_binom_P[i] = log( (N-(i+1)+1) / (i+1.0) * p / (1.0-p) );
  }

  // initialize a table of the degree counts and a table for the
  // cumul of those
  if (bsave_the_degree_counts) {
    degree_count    = InitdVector(nNodes + nNodes);
    cumdegree_count = InitdVector(nNodes + nNodes);
    for (i = 0; i < (nNodes+nNodes); i++) {
      degree_count[i] = 0;
      cumdegree_count[i] = 0;
    }
  }

  // initialize motifs cumulants
  if (bsave_the_motifs_probabilities) {
    cum_nEloops = 0;
    cum_nFloops = 0;
  }

  /* initialize a Boolean array recording which nodes have been assigned
     a zero probability of having parents (a column of zero in the binomial
     prior matrix). Those nodes are typically used as special "control"
     nodes for which the likelihood will not be computed */
  if (bBN || bDBN) {
    bAllowed_parents = InitiVector(nNodes);
    for (j = 0; j < nNodes; j++) { // for each column (node)
      i = 0;
      while ((i < nNodes) && (hyper_pB[i][j] == 0)) {
        i++;
      }
      bAllowed_parents[j] = (i == nNodes ? FALSE : TRUE);
    } // end for
  } // end if

  /* initialize a table of parents for each node
     (this is pricy in storage, would be better with a doubly linked
     list joining the 1s of the adjacency matrix) */
  nParents = InitiVector(nNodes);
  index_parents = InitiMatrix(nNodes, nNodes);

  for (j = 0; j < nNodes; j++) {
    nParents[j] = 0;
    for (i = 0; i < nNodes; i++)
      if (current_adj[i][j]) { // we have a parent, store it
        index_parents[j][nParents[j]] = i;
        nParents[j] += 1;
      }
    if (bBN) {
      if (bZellner && (nParents[j] >= nData)) {
        printf ("Error: node %d has more parents than data: "
                "conflict with Zellner's score - Exiting.\n\n", j);
        exit(0);
      }
    }
    else
      if (bDBN && bZellner && (nParents[j] >= nData - 1)) {
        printf ("Error: node %d has more parents than data (minus 1): "
                "conflict with Zellner's score in a DBN - Exiting.\n\n", j);
        exit(0);
      }
  }

  // if BN, initialize the list of node for fast topological sorting
  // and check acyclicity
  // beware: this assumes that nParents is initialized as above
  if ((bBN) && (!IsDAG_w_topo_list_incremental(nNodes, current_adj)))
    lexerr ("initial graph is not a DAG");

  // initialize an edge summation matrix
  if (bsave_the_edge_probabilities) {
    mat_sum = InitdMatrix(nNodes, nNodes);
    if (nBurnin == 0)
      for (i = 0; i < nNodes; i++)  // ith line
        for (j = 0; j < nNodes; j++)  // jth column
          mat_sum[i][j] = current_adj[i][j];
  }

  // initialize data likelihood related stuff 
  if (bData) {

    // initialize a working matrix to be used in InvertMatrix for example
    if (bBN)
      pdWorkMatrixSizeN = InitdMatrix (nNodes, nNodes);
    else // bDBN hopefully
      pdWorkMatrixSizeN = InitdMatrix (nNodes + 1, nNodes + 1);

    // replace missing data by initial imputed values (averages)
    if (bNAData) {

      printf ("Missing data will be imputed.\n\n");

      double average = 0;

      // init a global list of the missing data coordinates
      plistMissing = InitijList ();

      // init missing data indicator array (a flag per node)
      bHasMissing = InitiVector(nNodes);

      for (i = 0; i < nNodes; i++) {
         count = 0; // reset
         for (j = 0; j < nData; j++) {
           // find the average of node i data, skipping NAs
           if (!isnan(pData[i][j])) {
             average = average + pData[i][j];
             count += 1;
           }
         }
         if (count == 0) {
           printf ("Warning: node %d has all data missing.\n", i+1);
           average = 0; // arbitrary
         }
         else {
           average = average / count;
         }

         // impute the average
         for (j = 0; j < nData; j++) {
           if (isnan(pData[i][j])) {
             QueueijListItem (plistMissing, i, j); // store location in queue
             bHasMissing[i] = TRUE;
             pData[i][j] = average;
           }
         } 
      } // for i

    } // if bNAData
  } // if bData

} /* InitArrays */


/* ----------------------------------------------------------------------------
   Impute

   Impute a node's missing data by MCMC sampling, the importance ratio is 
   computed on the node's Markov blanket.
*/
void Impute (void)
{
  register int child;
  double curr_llike, new_llike, diff_llike, sd = 0.5, data_diff;
  PLISTELEMIJ ple = plistMissing->Head;
  static BOOL bPrint;
  static long iPrint_at = 0;
  static long iPrint_interval;
  static double* pTmp;

  if (!pTmp) { // initializations
    pTmp = InitdVector(nNodes);
    iPrint_interval = (nRuns > 1E3 ? nRuns / 1E3 : 1);
  }

  // scan the missing data list, find start of parent's missing data
  while (ple && (ple->iVal != parent))
    ple = ple->next;

  if (ple == plistMissing->Head) { // start of the list
    // check whether we should print
    if (iter > iPrint_at) {
      bPrint = 1;
      iPrint_at += iPrint_interval;
    }
    else {
      bPrint = 0;
    }
  } 

  do { // for each of the parent missing data
    if (bPrint)
    fprintf (pImputedFile, "%ld\t%d\t%d\t%g\t", 
             iter, parent, ple->jVal, pData[parent][ple->jVal]);

    // sample a new value, remember the difference to undo eventually
    data_diff = sd * (Randoms() - 0.5); // should be an adjustable kernel
    pData[parent][ple->jVal] += data_diff; 

    // likelihood of the children in parent's Markov blanket
    // for parent:
    curr_llike = current_ll_node[parent];
    if (bBN) {
      if (bZellner)
        pTmp[parent] = ZLoglikelihood_node (parent, pData);
      else {
        if (bDirichlet)
          pTmp[parent] = DLoglikelihood_node (parent, pData);
        else
          pTmp[parent] = GLoglikelihood_node (parent, pData);
      }
    }
    else { // bDBN, hopefully
      if (bZellner)
        pTmp[parent] = ZLoglikelihood_node_DBN (parent, pData);
      else {
        if (bDirichlet)
          pTmp[parent] = DLoglikelihood_node_DBN (parent, pData);
        else
          pTmp[parent] = GLoglikelihood_node_DBN (parent, pData);
      }
    }

    new_llike = pTmp[parent];

    // for children:
    for (child = 0; child < nNodes; child++) {
      if (current_adj[parent][child]) {

        curr_llike += current_ll_node[child];

        if (bBN) {
          if (bZellner)
            pTmp[child] = ZLoglikelihood_node (child, pData);
          else {
            if (bDirichlet)
              pTmp[child] = DLoglikelihood_node (child, pData);
            else
              pTmp[child] = GLoglikelihood_node (child, pData);
          }
        }
        else { // bDBN, hopefully
          if (bZellner)
            pTmp[child] = ZLoglikelihood_node_DBN (child, pData);
          else {
            if (bDirichlet)
              pTmp[child] = DLoglikelihood_node_DBN (child, pData);
            else
              pTmp[child] = GLoglikelihood_node_DBN (child, pData);
          }
        }

        new_llike += pTmp[child];
      }
    }
    // printf ("current blanket ll: %g\n", curr_llike);
    // printf ("new blanket ll: %g\n", new_llike);

    // accept of reject
    diff_llike = new_llike - curr_llike;
    if ((diff_llike >= 0) || (log(Randoms()) < diff_llike)) { // accept
      // printf ("accepted\n");
      // update likelihoods
      current_ll_node[parent] = pTmp[parent];
      for (child = 0; child < nNodes; child++) {
        if (current_adj[parent][child])
          current_ll_node[child] = pTmp[child];
      }
      // keep the sampled data...

    if (bPrint)
      fprintf (pImputedFile, "%g\n", new_llike);
    }
    else { // reject the sampled data
      // printf ("rejected\n");
      pData[parent][ple->jVal] -= data_diff;
    if (bPrint)
      fprintf (pImputedFile, "%g\n", curr_llike);
    }

    ple = ple->next;
  } while (ple && (ple->iVal == parent));

} /* Impute */


/* ----------------------------------------------------------------------------
   Logprior_diff
   Computes the difference in density, according to
   2 priors: (eventually) degree and (eventually) motif
   Inputs:
    adjacency_current:  adjacency of the current graph
    parent_node
    child_node
    diff: the edge change proposed
     -1: deletion
     +1: creation
   Outputs:
    *logPdiff: total CHANGE in prior log-density.
*/
void Logprior_diff (int **adjacency_current, int parent_node, int child_node,
                    int diff, double *logPdiff)
{

  if (bPriorConcordance)
    *logPdiff = Logprior_diff_concordance(parent_node, child_node, diff);
  else
    *logPdiff = 0;

  if (bPriorDegreeNode)
    *logPdiff += Logprior_diff_degree(parent_node, child_node, diff);

  // prior on total edge count
  if (bPriorEdgeCount)
    *logPdiff += Logprior_diff_edge_number(diff);

  if (bPriorMotif || bsave_the_motifs_probabilities)
    UpdateCountTriangles(adjacency_current, parent_node, child_node, diff,
                         &diff_nEloops, &diff_nFloops);

  if (bPriorMotif) {
    proposd_motif_prior = LnBB(current_nEloops + diff_nEloops,
                               current_nEloops + current_nFloops +
                               diff_nEloops + diff_nFloops,
                               alpha_motif, beta_motif);

    *logPdiff += proposd_motif_prior - current_motif_prior;
  }

} /* Logprior_diff */


/* ----------------------------------------------------------------------------
   Logprior_diff_bernoulli
   Computes the difference in density, according to
   the beta-binomial (Bernoulli) prior.
   Inputs:
    parent_node
    child_node
    diff: the edge change proposed (DO NOT CALL IT WITH DIFF == 0)
     -1: deletion
     +1: creation
   Returns:
    CHANGE in prior log-density.
*/
double Logprior_diff_bernoulli (int parent_node, int child_node, int diff)
{
  double logPdiff;

  // prior change under Bernoulli
  if (diff == 1)
    logPdiff = log_hyper_pB[parent_node][child_node] -
               log_hyper_qB[parent_node][child_node];
  else
    logPdiff = log_hyper_qB[parent_node][child_node] -
               log_hyper_pB[parent_node][child_node];

  return(logPdiff);

} /* Logprior_diff_bernoulli */


/* ----------------------------------------------------------------------------
   Logprior_full

   Full log-prior for a given graph (specified by its adjacency matrix)
   Inputs:
     N : number of nodes
     adjacency : a pointer to the adjacency matrix
   This assumes that some globals have been set up:
     hyper_pB
     hyper_qB
     nParents
     etc.
   It also sets up useful globals and should be called at only at start, unless
   resetting is wanted.
*/
double Logprior_full (int N, int **adjacency)
{

  int i, j, diff;
  int **budding_adj; // temporary adjacency matrix

  double pr = 0;     // init prior log density
  double cumLD;      // sum of log degrees
  double cumPE;      // temporary for probability of initial edge count

  // Bernoulli prior on edges: always done
  for (i = 0; i < N; i++)
    for (j = 0; j < N; j++)
      if (adjacency[i][j] == 1)
        pr += log_hyper_pB[i][j]; // log(hyper_pB[i][j])
      else
        pr += log_hyper_qB[i][j]; // log(1 - hyper_pB[i][j])

  // concordance prior
  if (bPriorConcordance)
    for (i = 0; i < N; i++)
      for (j = 0; j < N; j++) // count only the disagreements
        if (Logprior_diff_concordance(i, j, (adjacency[i][j] ? 1 : -1)) < 0)
          pr -= lambda_concord;

  // prior on degree distribution
  if (bsave_the_degree_counts || bPriorDegreeNode) {
    // set up the global table for the number of edges for each node
    current_degrees = InitiVector(N);
    for (i = 0; i < N; i++) {
      current_degrees[i] = 0;
    }

    // compute the table for the number of edges for each node
    for (i = 0; i < N; i++) {
      for (j = 0; j < N; j++) // sum over the ith line
        current_degrees[i] += adjacency[i][j];

      for (j = 0; j < N; j++) // sum over the ith column
        if (j != i) // do not count the node itself twice
          current_degrees[i] += adjacency[j][i];
    }
  } // end bsave_the_degree

  if (bPriorDegreeNode) {
    /* get the log-density of the current degrees under the
       power law */
    cumLD = 0; // cumulate the log degrees
    for (i = 0; i < N; i++) {
      if (current_degrees[i] != 0)
        cumLD += log(current_degrees[i]);
    }
    pr += -gamma_degree * cumLD;
  }

  // this is always done:
  current_edge_count = 0;

  // prior on total edge count
  if (bPriorEdgeCount) {
    // initialize also the current edge count using nParents
    // start with P(n=0)
    double p = expected_n_edges / (double) (nNodes * nNodes); 
    cumPE = nNodes * nNodes * log(1 - p);
  }

  // part of the loop is always done to keep track of the number of edges
  // could be conditional on a flag
  for (i = 0; i < nNodes; i++) {
    for (j = 0; j < nParents[i]; j++) {
      if (bPriorEdgeCount)
        cumPE += Logprior_diff_edge_number(+1);
      current_edge_count++;
    }
  }

  if (bPriorEdgeCount)
    pr += cumPE;

  // now for motifs
  if (bsave_the_motifs_probabilities || bPriorMotif) {
    /* counters for the two loop types; global initialize !
       E loops are A->B->C->A
       F (frustrated) loops are A->B->C<-A
       they will be initalized in the following at the values for the
       current graph
       other types of motifs could be added */
    current_nEloops = 0;
    current_nFloops = 0;

    /* to count the loops we simply reconstruct the adjacency matrix given
       (starting from an empty matrix) and count the loops as they are
       being formed */
    budding_adj = InitiMatrix(N, N); // start empty
    for (i = 0; i < N; i++)
      for (j = 0; j < N; j++)
        budding_adj[i][j] = 0;

    // note: i is always parent of j

    diff = 1; /* be explicit: we are only looking at nodes linked to
                 each other */
    for (i = 0; i < N; i++) { // for each node
      for (j = 0; j < N; j++) { // for each potential child
        if (adjacency[i][j] == 1) { // skip the zeros...
          UpdateCountTriangles(budding_adj, i, j, diff,
                               &diff_nEloops, &diff_nFloops);
          budding_adj[i][j] = 1;
          current_nEloops += diff_nEloops;
          current_nFloops += diff_nFloops;
        } // end if
      } // end for j
    } // end for i
  } // end bsave_the_motifs_probabilities

  if (bPriorMotif) {
    // compute prior component and store it globally 'cause it's expensive
    current_motif_prior = LnBB(current_nEloops,
                               current_nEloops + current_nFloops,
                               alpha_motif, beta_motif);
    pr += current_motif_prior;
  }

  return (pr);

} /* Logprior_full */


/* ----------------------------------------------------------------------------
   ReadScript_Bison

   Read the simulation settings from a script file. The syntax is defined
   using lex and yacc. Meaningful input is then checked and default values
   are specified.
*/
void ReadScript_Bison (char *const filename)
{
  int i, j;
  extern FILE *yyin;

  yyin = fopen(filename, "r");
  if (yyin) {
    printf("Reading from file %s.\n\n", filename);
  }
  else
    lexerr("no input file");

  // set default values for scalar predefined variables
  bNormalGamma   = bDirichlet  = bZellner  = 0;

  lambda_concord = 1;
  gamma_degree   = 1;
  alpha_motif    = 1;
  beta_motif     = 1;

  gamma_zellner      = 1;
  alpha_normal_gamma = 1.5;
  beta_normal_gamma  = 1000;

  iter = 1000000000;
  rdm_gen_name = mt19937; // if gsl is available, use Mersenne twister

  // printf("starting script reading...\n");
  yyparse();

  fclose (yyin);

  // check nNodes value
  if (nNodes == 0)
    lexerr ("nNodes cannot be zero");

  // if N too large you have to switch to longs
  if (nNodes > sqrt(INT_MAX))
    lexerr (" nNodes too large for 'int' indexing");

  // check graph type inconsistencies
  if (bBN) {
    if (autocycle)
      lexerr ("autocycles are forbidden in a BN");
    if (bDBN)
      lexerr ("BN and DBN are both set to True");
  }

  // check priors
  if (bBN) {
    if (bPriorConcordance && edge_requirements)
      for (i = 0; i < nNodes; i++)
        if (edge_requirements[i][i] == 1)
          lexerr ("a concordance prior on BNs cannot require an "
                  "edge on the diagonal");

    if (hyper_pB)
      for (i = 0; i < nNodes; i++)
        if (hyper_pB[i][i] > 0)
          lexerr ("a Bernoulli prior on BNs has to have a null diagonal");

    if (bPriorMotif)
      lexerr ("the motifs currently implemented in graph_sampler "
              "are incompatible with BNs");
  }

  if (hyper_pB) {
    if (bPriorConcordance) {
      printf ("Warning: potential conflict: concordance and ");
      printf ("Bernoulli priors are both defined.\n");
    }

    for (i = 0; i < nNodes; i++)
      if ((hyper_pB[i][i] > 0) && (!autocycle))
        lexerr ("a Bernoulli prior has to have a null diagonal if "
                "autocycle is False");
  }

  // check data and likelihood
  bData = (nData > 0);

  if (bData && !pData)
    lexerr ("nData > 0 but data values are not provided");

  if ((!bBN && !bDBN && bData)) {
    printf ("Warning: the data provided will not be used (bBN and bDBN "
            "are False).\n\n");
    bData = FALSE;
  }

  if (!(bDirichlet || bNormalGamma || bZellner) && (bData)) {
    printf ("Warning: the data provided will not be used "
            "(data likelihood is not defined).\n\n");
    bData = FALSE;
  }

  // likelihoods cannot be defined in the absence of data
  if ((bDirichlet || bNormalGamma || bZellner) && (!bData))
    lexerr ("Likelihood specified in the absence of data");

  // one only one type of likelihood is allowed
  if ((bDirichlet + bNormalGamma + bZellner) > 1)
    lexerr ("Multiple specifications of likelihood are not allowed");

  // check discrete data
  if (bDirichlet) {
    if (!pDataLevels)
      lexerr ("Dirichlet score requires that data levels be specified");

    for (i = 0; i < nNodes; i++)
      for (j = 0; j < nNodes; j++)
        if (pData[i][j] != (int) pData[i][j])
          lexerr ("Dirichlet score requires integer data");
  }

  // miscellaneous initializations
  InitRandoms (rdm_gen_name, seed);

  bsave_some_graphs = (n_saved_adjacency > 0);

  // printf ("done reading script.\n");

} /* ReadScript_Bison */


/* ----------------------------------------------------------------------------
   SampleTemperature

  Assumes that the number of attempted temperature jumps is equal to
  the number of iterations...
*/
void SampleTemperature (void)
{
  int    i, indexT_new;
  double dPjump;
  #define MINUSLN2 -0.6931471805599452862268

  // Robbins-Monro updating of the temperature pseudo prior
  for (i = 0; i < nTemperatures; i++) {
    if (i == indexT)
      plnPi[i] -= dCZero / (iter + dNZero);
    else
      plnPi[i] += dCZero / (nTemperatures * (iter + dNZero));
  }

  // update population count of current temperature
  // pCountTemp[indexT]++; // this is for reporting only!

  // propose a new inverse temperature
  if (indexT == 0)
    indexT_new = 1;     // move up
  else {
    if (indexT == nTemperatures - 1)
      indexT_new = indexT - 1;   // move down
    else
      if (Randoms() > 0.5)       // move randomly
        indexT_new = indexT + 1;
      else
        indexT_new = indexT - 1;
  }

  // compute importance ratio
  dPjump = (pInvTemperatures[indexT_new] - pInvTemperatures[indexT]) * 
           (bData ? current_logprior : current_logposterior) +
           plnPi[indexT_new] - plnPi[indexT] +
           ((indexT_new == 0) || (indexT_new == nTemperatures - 1) ? 
            0 : MINUSLN2) -
           ((indexT     == 0) || (indexT     == nTemperatures - 1) ? 
            0 : MINUSLN2);

  // test the proposed temperature jump
  if (log(Randoms()) <= dPjump)
    indexT = indexT_new;  // jump, else stay at indexT

}  /* SampleTemperature */


/* ----------------------------------------------------------------------------
   SetPriorHyperParam

   Set the hyper parameters of the priors by looking at some general features
   of the data.
*/
void SetPriorHyperParam (void)
{
  printf("SetPriorHyperParam to do...\n\n");
  /* for example the lambda should be commensurate or larger than the raw data
     variance . Actually check the precision story.
     The variance (precision) of the reg param should be commensurate or
     larger with the range of values those reg params can take (???) etc */

} /* SetPriorHyperParam */


/* ----------------------------------------------------------------------------
   UndoDiff

   Undoes the change in global number of parents of node "child".
*/
void UndoDiff (int parent, int child, int diff)
{
  if (diff < 0) {
    // just add parent that was removed by increasing the count of parents
    nParents[child] += 1;
  }
  else {
    // remove parent that was added
    nParents[child] -= 1;
  }

} /* UndoDiff */


/* ----------------------------------------------------------------------------
   UpdateBestGraph

   Update the motifs accounting tables.
*/
void UpdateBestGraph (void)
{
  int i;

  if (bsave_best_graph && (iter > nBurnin - 1)) {
    if (dBestPosterior < current_logposterior) {
      dBestPosterior  = current_logposterior;
      dBestPrior      = current_logprior;
      dBestLikelihood = current_loglikelihood;
      for (i = 0; i < nNodes; i++)
        memcpy(best_adj[i], current_adj[i], nNodes * sizeof(int));
    }
  }

} /* UpdateBestGraph */


/* ----------------------------------------------------------------------------
   UpdateEdgeP

   Just do that.
*/
void UpdateEdgeP (void)
{
  int i, j;
  static BOOL bInited = FALSE;

  if (iter >= nBurnin) {
    if (bInited == FALSE) {
      for (i = 0; i < nNodes; i++)
        for (j = 0; j < nNodes; j++)
          mat_sum[i][j] = current_adj[i][j];
      n_at_targetT = (indexT == (nTemperatures - 1) ? 1 : 0);
      bInited = TRUE;
    }
    else { 
      for (i = 0; i < nNodes; i++)
       for (j = 0; j < nNodes; j++)
          mat_sum[i][j] += current_adj[i][j];

      if (indexT == (nTemperatures - 1))
        n_at_targetT++;
    }
  }
  //
} /* UpdateEdgeP */


/* ----------------------------------------------------------------------------
*/
int main (int nArgs, char *const *rgszArg)
{
  BOOL   bEdge;

  int    diff_location;
  char   *szFileIn, *szPrefixOut;

  AnnounceProgram ();

  GetCmdLineArgs (nArgs, rgszArg, &szFileIn, &szPrefixOut);

  ReadScript_Bison (szFileIn);

  InitArrays ();

  InitOutputs (szPrefixOut);

  // compute the prior of the initial network,
  // that initializes also book-keeping for fast computations of priors
  current_logprior = Logprior_full (nNodes, current_adj);

  if (isnan (current_logprior) || !isfinite (current_logprior))
    lexerr ("initial network has prior with null probability");

  if ((bBN || bDBN) && bData) {
    // SetPriorHyperParam(); disabled, not developed
    current_loglikelihood = Loglikelihood_full(nNodes, pData);
  }
  else
    current_loglikelihood = 0;

  current_logposterior = current_logprior + current_loglikelihood;

  dBestPrior      = current_logprior;
  dBestLikelihood = current_loglikelihood;
  dBestPosterior  = current_logposterior;

  /* -------------------
     The sampler is here
  */

  if (bTempered)
    printf ("Doing %ld tempered MCMC iterations.\n\n...\n\n", nRuns);
  else
    printf ("Doing %ld iterations.\n\n...\n\n", nRuns);

  // initialize parent and child node for systematic scan
  parent = -1;
  child  =  0;

  for (iter = 0; iter < nRuns; iter++) {
    /* ! be careful to stay fast in this loop ! */

    label_Redo_it:

    // if BN sought but the proposed graph was not a DAG, come back here

    // to create a proposal graph take 2 nodes, scanning systematically
    if (parent < (nNodes - 1)) {
      parent = parent + 1;
    }
    else {
      parent = 0;

      if (child < (nNodes - 1))
        child = child + 1;
      else {
        child = 0;

        // at a start of an adjacency matrix updating cycle 
        // test and eventually update the temperature
        // (note: this is NOT reached if autocycle is false)
        if (bTempered && (iter > 0))
          SampleTemperature ();
      }
    }

    // if child == 0 and parent has missing data impute the parent's data
    if ((bNAData) && (child == (!parent?1:0)) && (bHasMissing[parent]))
      Impute ();

    // in case of BN or no autocycle: skip the diagonal
    if ((!autocycle) && (parent == child)) {
      if (parent < (nNodes - 1)) {
        parent = parent + 1;
      }
      else {
        parent = 1;
        child  = 0;

        // at a start of an adjacency matrix updating cycle 
        // test and eventually update the temperature
        // (note: this is reached if autocycle is false)
        if (bTempered && (iter > 0))
          SampleTemperature ();
      }
    }

    // sample a move from the baseline Bernoulli prior
    bEdge = (Randoms () < hyper_pB[parent][child]); // 0 or 1
    if (bEdge == current_adj[parent][child]) {
      diff = 0;
    }
    else {
      if (bEdge == 1) {
        diff = 1;

        // for BNs we have to check two things
        if (bBN) {
          // that it's still a DAG
          if (!Check_DAG_Edge (current_adj, parent, child))
            goto label_Redo_it; // forget it completely

          // for Zellner likelihood there should not be more parents than data
          if (bZellner && (nParents[child] >= nData))
            goto label_Redo_it; // forget it completely
        } // if bBN
        else {
          // for DBNs
          if (bDBN && bZellner && (nParents[child] >= nData - 1))
            goto label_Redo_it; // forget it completely
        }
      }
      else {
        diff = -1; // removing an edge to a DAG gives a DAG
      }
    }

    // compute log-probabilities proposal - log-probabilities current
    // take temperature into account
    if (diff != 0) {
      Logprior_diff (current_adj, parent, child, diff, &diff_logprior);

      if (bData)
        Loglikelihood_diff (parent, child, diff, pData, &diff_loglikelihood);

      diff_logposterior = diff_logprior + diff_loglikelihood;

      if (bTempered) // elevate to power 1/temperature
        diff_logposterior *= pInvTemperatures[indexT];
    }
    else { // no change
      diff_logposterior = 0;
    }

    // check acceptation
    if ((diff_logposterior >= 0) || 
        (log(Randoms ()) < diff_logposterior)) { // accept

      UpdateDegrees_if_accept ();

      current_edge_count = current_edge_count + diff;
      UpdateMotifs_if_accept ();

      /* find flattened location of the sampled edge;
         go down columns, to be compatible with R;
         must start from 1 to leave 0 as indicator of no change by convention;
         then signed by the difference between adjacency matrices */
      diff_location = (parent + child * nNodes + 1) * diff;

      // eventually write the location to output file;
      // this is inlined for speed
      if (bsave_the_chain) {
        SaveChain (diff_location);
        if (bTempered) // eventually save the current inverse temperature 
          SaveInverseTemperature();
      }

      /* now we can update the graph adjacency matrice,
         the parenthood of child has been already changed */
      if (diff != 0) {
        current_adj[parent][child] = !(current_adj[parent][child]);

        // eventually update the total prior
        if (bTempered || bsave_best_graph || bsave_some_graphs) {
          current_logprior += diff_logprior +
                              Logprior_diff_bernoulli (parent, child, diff);
        }

        // eventually update likelihood and posterior
        if (bData) {
          current_ll_node[child] += diff_loglikelihood;

          if (bTempered || bsave_best_graph || bsave_some_graphs) {
            current_loglikelihood +=  diff_loglikelihood;
            current_logposterior   = current_logprior + current_loglikelihood;
          }
        }

        // eventuall update the best graph
        if (bsave_best_graph)
          UpdateBestGraph (); // eventually not at the lowest temperature!

      } // end diff != 0
    } // end of accept

    else { // reject

      // no differences between adjacency matrices: just write out zero
      if (bsave_the_chain) {
        SaveChain (0);
        if (bTempered) // save the current inverse temperature
          SaveInverseTemperature();
      }

      UpdateDegrees_if_reject ();
      UpdateMotifs_if_reject ();

      /* no update of adjacency is needed,
         but the change in parenthood of child needs to be undone */
      UndoDiff (parent, child, diff);

    } // end of reject

    // cumulate the adjacency matrices, i.e. cumulate edge counts
    if (bsave_the_edge_probabilities && (indexT == (nTemperatures - 1)))
      UpdateEdgeP ();

    // save eventually the graph, eventually not at the lowest temperature
    SaveGraph ();

  } // end iter
  // End of sampler

  // final results if asked for:

  // edge probability matrix
  SaveEdgeP (pEdgeFile);

  // best graph
  SaveBestGraph ();

  // save cumulated degree counts
  SaveDegreeCounts ();

  // motifs probabilities
  // SaveMotifsP(stdout); is a screen alternative
  SaveMotifsP (pMotifFile);

  CloseOutputs (szPrefixOut);

  CleanupMemory ();

  printf ("Done.\n\n");

  return (1);

} /* end */
