/* matrices.c

   Written by Frederic Bois
   22 June 2014

   Copyright (c) 2014 Frederic Bois.

   This code is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   See the GNU General Public License at <http://www.gnu.org/licenses/> 

   -- Revisions -----
     Logfile:  %F%
    Revision:  %I%
        Date:  %G%
     Modtime:  %U%
      Author:  @a
   -- SCCS  ---------

   A bunch of matrix utilities
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "lexerr.h"
#include "matrices.h"


/* ----------------------------------------------------------------------------
   Cholesky

   Does the Cholesky decomposition of an Hermitian matrix M:
   Compute the matrix L such that M = LL*.

   Returns 0 if successful, -1 otherwise.
   Matrix M lower triangle is destroyed in the process.
*/
int Cholesky (double **M, double **prgdComponent, int dim)
{
  register int i, j, k;
  double dSum;

  for (i = 0; i < dim; i++)
    for (j = 0; j < dim; j++) 
      prgdComponent[i][j] = 0.0;

  for (i = 0; i < dim; i++) {
    for (j = i; j < dim ; j++) {
      dSum = M[i][j];
      for (k = i - 1; k >= 0 ; k--)
        dSum = dSum - M[i][k] * M[j][k];

      if (i == j) {
      	if (dSum <= 0.0) {
          printf ("Warning: input matrix for Cholesky is not "
                  "positive definite\n"); 
          printf ("\ndSum = %g\n", dSum);
          return 0;
        }
        else
          prgdComponent[i][i] = sqrt(dSum);
      }
      else 
        M[j][i] = dSum / prgdComponent[i][i];
    } // end for j
  }
  
  for (i = 0; i < dim ; i++)
    for (j = i+1; j < dim ; j++)
      prgdComponent[j][i] = M[j][i];

  // success
  return 1;

} /* Cholesky */


/* ----------------------------------------------------------------------------
   InitdMatrix

   initializes a rectangular matrix of doubles.

   The pointer to the matrix is returned if no error occurs, otherwise exit.
   It is an error to call it with null dims.
*/
double **InitdMatrix (long dim1, long dim2)
{
  register long i;
  double **rgp;

  if ((dim1 == 0) || dim2 == 0)
    lexerr ("at least one zero dimension in InitdMatrix");

  if (!(rgp = (double **) malloc(dim1 * sizeof(double *))))
    lexerr ("out of memory in InitdMatrix");

  for (i = 0; i < dim1; i++)
    if (!(rgp[i] = (double *) malloc(dim2 * sizeof(double))))
      lexerr ("out of memory in InitdMatrix");

  return (rgp);

} /* InitdMatrix */


/* ----------------------------------------------------------------------------
   InitdVector

   initializes a vector of double.

   The pointer to the vector is returned if no error occurs, otherwise NULL
   is returned. It is an error to call it with dim = 0.
*/
double *InitdVector (long dim)
{
  double *rgp;

  if (dim == 0)
    lexerr ("dimension zero asked in InitdVector");

  if (!(rgp = (double *) malloc(dim * sizeof(double))))
    lexerr ("out of memory in InitdVector");
  else
    return (rgp);

} /* InitdVector */


/* ----------------------------------------------------------------------------
   InitiMatrix

   initializes a rectangular matrix of integers.

   The pointer to the matrix is returned if no error occurs, otherwise exit.
   It is an error to call it with null dims.
*/
int **InitiMatrix (long dim1, long dim2)
{
  register long i;
  int **rgp;

  if ((dim1 == 0) || dim2 == 0)
    lexerr ("at least one zero dimension in InitiMatrix");

  if (!(rgp = (int **) malloc(dim1 * sizeof(int *))))
    lexerr ("out of memory in InitiMatrix");

  for (i = 0; i < dim1; i++)
    if (!(rgp[i] = (int *) malloc(dim2 * sizeof(int))))
      lexerr ("out of memory in InitiMatrix");

  return (rgp);

} /* InitiMatrix */


/* ----------------------------------------------------------------------------
   InitiVector

   initializes a vector of integers.

   The pointer to the vector is returned if no error occurs, otherwise NULL
   is returned. It is an error to call it with dim = 0.
*/
int *InitiVector (long dim)
{
  int *rgp;

  if (dim == 0)
    lexerr ("dimension zero asked in InitiVector");

  if (!(rgp = (int *) malloc(dim * sizeof(int))))
    lexerr ("out of memory in InitiVector");
  else
    return (rgp);

} /* InitiVector */


/* ----------------------------------------------------------------------------
   InvertMatrix

   Inverts matrix M using Cholesky decomposition.
   Matrix M is replaced by its inverse.
*/
void InvertMatrix (double **M, int dim)
{
  register int i, j, k;
  double dSum;
  
  Cholesky (M, pdWorkMatrixSizeN, dim);

  // invert pdWorkMatrixSizeN
  for (i = 0; i < dim; i++) {
    for (j = 0; j <= i; j++) {
      dSum = (i == j ? 1.0 : 0.0); 
      for (k = i - 1; k >= j; k--) 
        dSum -= pdWorkMatrixSizeN[i][k] * M[j][k]; 
      M[j][i] = dSum / pdWorkMatrixSizeN[i][i];
    }
  } 

  // multiply by t(pdWorkMatrixSizeN)
  for (i = dim-1; i >= 0; i--) {
    for (j = 0; j <= i; j++) {
      dSum = (i < j ? 0.0 : M[j][i]); 
      for (k = i + 1; k < dim; k++) 
        dSum -= pdWorkMatrixSizeN[k][i] * M[j][k];
      M[i][j] = M[j][i] = dSum / pdWorkMatrixSizeN[i][i];
    }
  }
  
} /* InvertMatrix */


/* ----------------------------------------------------------------------------
   LU_decomposition

   Does the LU decomposition of a matrix M:
   Compute the matrices L and U such that M = LU.

   Given a matrix M[1..dim][1..dim], this routine replaces it by the LU 
   decomposition of a rowwise permutation of itself. M and dim are input. 
   M is output 

   d is output as +/- 1 depending on whether the number of row interchanges was 
   even or odd, respectively.
   
   Returns 1 if successful.
   Matrix M lower triangle is destroyed in the process.
*/
int LU_decomposition (double **M, int dim, double *d)
{
  register int i, j, k;
  int imax;
  double big, temp, sum, dum;
  static double *pdTmp; // stores the implicit scaling of each row

  if (!pdTmp)
    pdTmp = InitdVector(dim);

  *d = 1.0;  // No row interchanges yet
  for (i = 0; i < dim; i++) { 
    // Loop over rows to get the implicit scaling information
    big = 0.0;
    for (j = 0; j < dim; j++) {
      temp = fabs(M[i][j]);
      if (temp > big)
        big = temp;
    }
    if (big == 0.0) // checking for no nonzero largest element
      lexerr ("singular matrix in routine LU_decomposition");

    pdTmp[i] = 1.0 / big; // save the scaling
  } // for i

  for (j = 0; j < dim; j++) { // loop over columns of Crout's method
  
    for (i = 0; i < j; i++) {
      sum = M[i][j];
      for (k = 0; k < i; k++) {
        sum -= M[i][k] * M[k][j];
      }
      M[i][j] = sum;
    }

    big = 0.0; // Initialize for the search for largest pivot element

    for (i = j; i < dim; i++) {
      sum = M[i][j];
      for (k = 0; k < j; k++) {
        sum -= M[i][k] * M[k][j];
      }
      M[i][j] = sum;

      dum = pdTmp[i] * fabs(sum); 
      // Is the figure of merit for the pivot better than the best so far?
      if (dum >= big) {
        big = dum;
        imax = i;
      }
    } // for i

    // Do we need to interchange rows?
    if (j != imax) { // Yes, do so...
      for (k = 0; k < dim; k++) {
        dum = M[imax][k];
        M[imax][k] = M[j][k];
        M[j][k] = dum;
      }
      *d = -(*d); // change the parity of d
      pdTmp[imax] = pdTmp[j]; // interchange the scale factor
    } // if j

    /* if the pivot element is zero the matrix is singular 
       (at least to the precision of the algorithm). 
       For some applications on singular matrices, 
       it is desirable to substitute TINY for zero */
    if (M[j][j] == 0.0) {
      M[j][j] = TINY;
    }

    if (j < (dim-1)) { // finally, divide by the pivot element
      dum = 1.0 / (M[j][j]);
      for (i = j+1; i < dim; i++)
        M[i][j] *= dum;
    }
  }

  // success
  return 1;

} /* LU_decomposition */


/* ----------------------------------------------------------------------------
   QR_decomposition

*/
int QR_decomposition (double **a, int n, double *c, double *d, int *sing)
{
  int i,j,k;
  float scale=0.0,sigma,sum,tau;

  #define SIGN(a,b) ((b) >= 0.0 ? fabs(a) : -fabs(a))

  *sing=0;
  for (k=1;k<n;k++) {
    for (i=k;i<=n;i++) {
      if (scale < fabs(a[i][k]))
        scale=fabs(a[i][k]);
    }
    if (scale == 0.0) {
      *sing = 1;
      c[k] = d[k] = 0.0;
    } else {
      for (i=k; i<=n; i++) 
        a[i][k] /= scale;
      for (sum = 0.0,i = k; i <= n; i++) 
        sum += sqrt(a[i][k]);
      sigma=SIGN(sqrt(sum),a[k][k]);
      a[k][k] += sigma;
      c[k] = sigma*a[k][k];
      d[k] = -scale*sigma;
      for (j=k+1;j<=n;j++) {
        for (sum=0.0,i=k;i<=n;i++) 
          sum += a[i][k]*a[i][j];
        tau=sum/c[k];
        for (i=k;i<=n;i++) a[i][j] -= tau*a[i][k];
      }
    }
  }

  d[n]=a[n][n];
  if (d[n] == 0.0)
    *sing=1;

  // success
  return 1;

} /* QR_decomposition */


/* ----------------------------------------------------------------------------
   PrintdMatrix

   Print a square double matrix of size dim.
*/
void PrintdMatrix (FILE *pFile, int dim, double **pdMat)
{
  register int i, j;

  for (i = 0; i < dim; i++) {
    for (j = 0; j < dim-1; j++) {
      fprintf(pFile, "%g\t", pdMat[i][j]);
    }
    fprintf(pFile, "%g\n", pdMat[i][j]);
  }
  fprintf(pFile, "\n");

} /* PrintdMatrix */


/* ----------------------------------------------------------------------------
   PrintiMatrix

   Print a square integer matrix of size dim.
*/
void PrintiMatrix (FILE *pFile, int dim, int **piMat)
{
  register int i, j;

  for (i = 0; i < dim; i++) {
    for (j = 0; j < dim-1; j++) {
      fprintf(pFile, "%d\t", piMat[i][j]);
    }
    fprintf(pFile, "%d\n", piMat[i][j]);
  }
  fprintf(pFile, "\n");

} /* PrintiMatrix */


/* ----------------------------------------------------------------------------
   PrintSortediMatrix

   Print a square integer matrix of size dim in the order of rows and columns
   specified by pIndex.
*/
void PrintSortediMatrix (FILE *pFile, int dim, int **piMat, int *pindex)
{
  int i, j;

  for (i = 0; i < dim; i++) {
    for (j = 0; j < dim-1; j++) {
      fprintf(pFile, "%d ", piMat[pindex[i]][pindex[j]]);
    }
    fprintf(pFile, "%d\n", piMat[pindex[i]][pindex[j]]);
  }
  fprintf(pFile, "\n");

} /* PrintSortediMatrix */


/* end */
