/* 
    Copyright 2002 Cyril Picard

    This file is part of the GEDCOMParser library 
    (developed within the Genealogy Free Software Tools project).

    The GEDCOMParser library is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    The GEDCOMParser library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with the GEDCOMParser library ; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#ifndef _GEDCOMPARSER_SOURCE_RECORD_HH_
#define _GEDCOMPARSER_SOURCE_RECORD_HH_

#include <map>
#include <string>
#include <vector>

#include "GEDCOMParser/AttachmentManager.hh"
#include "GEDCOMParser/ChangelogManager.hh"
#include "GEDCOMParser/UserReferenceManager.hh"
#include "GEDCOMParser/SourceRepositoryCitation.hh"
#include "GEDCOMParser/DateManagement/DateValue.hh"
#include "GEDCOMParser/Visitor/GEDCOMElement.hh"

///
namespace GEDCOMParser
{
  class GEDCOMVisitor;
  /// @memo Implements the GEDCOM 5.5 SOURCE_RECORD structure
  class SourceRecord:public GEDCOMElement,
    ///
    public AttachmentManager,
    public ChangelogManager, public UserReferenceManager
  {
  public:
    /// @memo Implements the DATA members of SOURCE_RECORD structure
    class Data:public GEDCOMElement, public AttachmentManager
    {
    public:
      /// @memo Implements the EVENT members of SOURCE_RECORD structure
    class Event:public GEDCOMElement
      {
      public:
	Event (void):_event (), _date (0), _plac ()
	{
	  return;
	}
	Event (Event const &e):_event (e._event), _date (0), _plac (e._plac)
	{
	  if (e._date != 0)
	    {
	      _date = new GEDCOMParser::DateManagement::DateValue (*e._date);
	    }
	  return;
	}
	virtual ~ Event (void)
	{
	  delete _date;
	  _date = 0;
	};
	///@name Accessors (set) 
	//@{    
	///     
	void setEvent (std::string const &);
	///     
	void setDate (std::string const &);
	///     
	void setPlace (std::string const &);
	//@}

	///@name Accessors (get)
	//@{            
	///     
	virtual std::string const &getEvent (void) const;
	///     
	virtual std::string const getDate (void) const;
	///     
	GEDCOMParser::DateManagement::
	  DateValue const *const getDateObject (void) const;
	///
	virtual std::string const &getPlace (void) const;
	virtual void Accept (GEDCOMParser::GEDCOMVisitor *);

	//@}    
      protected:
	std::string _event;
	GEDCOMParser::DateManagement::DateValue * _date;
	std::string _plac;
      };
      ///@name typedefs
      //@{
      /// @doc handling of many events (source record declaration)
      typedef std::vector < SmartPtr < Event > >Events_t;
      //@}
      virtual ~ Data (void)
      {
      };

      ///@name Accessors (set) 
      //@{    
      ///      
      GEDCOMParser::SourceRecord::Data::Event *
	addEvent (GEDCOMParser::SourceRecord::Data::Event * const data = 0);
      ///      
      void setAgnc (std::string const &);
      //@}

      ///@name Accessors (get)
      //@{    
      ///      
      Events_t const &getEvents (void) const;
      ///      
      virtual std::string const &getAgnc (void) const;
      virtual void Accept (GEDCOMParser::GEDCOMVisitor *);

      //@}
    protected:
      Events_t _events;
      std::string _agnc;
    };
    SourceRecord (std::string const &id = "");
    virtual ~ SourceRecord (void);
    ///@name Accessors (set) 
    //@{    
    ///      
    void setId (std::string const &);
    ///      
    GEDCOMParser::SourceRecord::Data * setData (SourceRecord::Data *
						const data = 0);
    ///      
    void contAuth (std::string const &);
    ///      
    void concAuth (std::string const &);
    ///      
    void contTitle (std::string const &);
    ///      
    void concTitle (std::string const &);
    ///      
    void setAbbr (std::string const &);
    ///      
    void contPubl (std::string const &);
    ///      
    void concPubl (std::string const &);
    ///      
    void contText (std::string const &);
    ///      
    void concText (std::string const &);
    ///      
    GEDCOMParser::SourceRepositoryCitation *
      setSourceRepositoryCitation (SourceRepositoryCitation * const data = 0);
    ///      
    void setRin (std::string const &);
    //@}

    ///@name Accessors (get)
    //@{   
    ///      
    virtual std::string const &getId (void) const;
    ///      
    SmartPtr < Data > const &getData (void) const;
    ///      
    virtual std::string const &getAuth (void) const;
    ///      
    virtual std::string const &getTitle (void) const;
    ///      
    virtual std::string const &getAbbr (void) const;
    ///      
    virtual std::string const &getPubl (void) const;
    ///      
    virtual std::string const &getText (void) const;
    ///      
    SmartPtr < SourceRepositoryCitation >
      const &getSourceRepositoryCitation (void) const;
    ///      
    virtual std::string const &getRin (void) const;
    //@}   
    virtual void Accept (GEDCOMParser::GEDCOMVisitor *);
  protected:
    std::string _id;
    SmartPtr < Data > _data;
    std::string _auth;
    std::string _titl;
    std::string _abbr;
    std::string _publ;
    std::string _text;
    SmartPtr < SourceRepositoryCitation > _src_rep_citation;
    std::string _rin;
  };
  ///@name typedefs
  //@{
  /// @doc handling of many source records
  typedef std::map < std::string, SmartPtr < SourceRecord > >SourceRecords_t;
  //@}
};
#endif
