# -*- coding: utf-8 -*-
# Copyright (C) 2010, 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Useful functions not related to the task of getting medium URL.
"""


import re
import warnings

from getmediumurl.compat import parse_qs, urlparse, unquote, basestring, xrange

__all__ = ("deprecated", "escape_regex", "check_single_percent_s",
           "parse_urlpart", "parse_content_type")


def deprecated(message):
    """Warn on use of deprecated function."""
    warnings.warn(message, DeprecationWarning, stacklevel=3)


#: A `frozenset` of characters having special meaning in regexes.
_REGEX_SPECIAL = frozenset(r".^$*+?{}\[]|()")


def escape_regex(string):
    """Escape characters in `string` with special meaning in regexes."""
    return "".join(("\\" if char in _REGEX_SPECIAL else "") + char
                   for char in string)


def check_single_percent_s(string):
    """Check if `string` is formatted by single ``%s``.

    If `string` is a string containing exactly single ``%s``, with all
    other percent characters replaced by ``%%``, then `True` is
    returned.  Otherwise `False` is returned.
    """
    if not isinstance(string, basestring):
        return False
    i = 0
    length = len(string)
    was_percent_s = False
    while i < length:
        i = string.find("%", i)
        if i == -1:
            break
        elif i == length - 1:
            return False
        elif string[i + 1] == "%":
            i += 2
        elif string[i + 1] == "s":
            if not was_percent_s:
                was_percent_s = True
                i += 2
            else:
                return False
        else:
            return False
    return was_percent_s


def parse_urlpart(url, part):
    """Parse `part` of an `url` into a dictionary."""
    return parse_qs(unquote(getattr(urlparse(url), part)))


def make_sequence(objects):
    """Return the argument or an iterable with it."""
    # Strings are usually used as single objects, not sequences of
    # characters.
    if isinstance(objects, basestring):
        return (objects,)
    # Try iterating it, if it works then it isn't a single object.
    try:
        iter(objects)
    except TypeError:
        return (objects,)
    else:
        return objects


# Using syntax from https://tools.ietf.org/html/rfc2616.

class _SyntaxError(ValueError):
    """Exception raised by parsing functions on invalid input."""


_SEPARATORS = "()<>@,;:\\\"/[]?={} \t"
_TOKEN = re.compile("^([" + "".join(chr(i) for i in xrange(32, 127)
                                    if chr(i) not in _SEPARATORS) + "]+)(.*)$")
_QUOTED_STRING = re.compile("^\"((?:[^\"\\\\]|\\\\.)*)\"(.*)$")


def _strip_slashes(string):
    r"""Return *string* with ``\x`` replaced by ``x``."""
    i = 0
    out = ""
    while i < len(string):
        if string[i] == "\\":
            if i == len(string) - 1:
                raise _SyntaxError(string)
            i += 1
        out += string[i]
        i += 1
    return out


def _tokenize_content_type(string):
    """Iterate tokens from *string*."""
    while string:
        string = string.lstrip()
        match = _TOKEN.match(string)
        if match:
            yield "token", match.group(1)
            string = match.group(2)
            continue
        match = _QUOTED_STRING.match(string)
        if match:
            value = _strip_slashes(match.group(1))
            yield "quoted-string", value
            string = match.group(2)
            continue
        if string[0] in _SEPARATORS:
            yield "separator", string[0]
            string = string[1:]
            continue
        raise SyntaxError(string)


def parse_content_type(content_type):
    """Return a pair of type/subtype string and dictionary of
    parameters for an HTTP Content-Type header.
    """
    try:
        tokens = list(_tokenize_content_type(content_type))
        if len(tokens) < 3:
            raise _SyntaxError(tokens)
        type_, slash, subtype = tokens[:3]
        if (type_[0] != "token" or slash != ("separator", "/")
            or subtype[0] != "token"):
            raise _SyntaxError(tokens)
        params = {}
        tokens = tokens[3:]
        while tokens:
            if len(tokens) < 4:
                raise _SyntaxError(tokens)
            if (tokens[0] != ("separator", ";") or tokens[1][0] != "token"
                or tokens[2] != ("separator", "=")
                or tokens[3][0] not in ("token", "quoted-string")):
                raise _SyntaxError(tokens)
            params[tokens[1][1]] = tokens[3][1]
            tokens = tokens[4:]
    except _SyntaxError:
        return ("", {})
    return (type_[1] + "/" + subtype[1], params)
