# -*- coding: utf-8 -*-
# Copyright (C) 2010, 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Tube8.com plugin for GetMediumURL."""


from __future__ import unicode_literals

import json

from urlreader.propcache import cachedproperty

from getmediumurl.compat import unquote
from getmediumurl.format import Format
from getmediumurl.htmlplugin import HTMLPlugin, HTMLMedium


__all__ = ("Tube8",)


class Tube8Medium(HTMLMedium):

    """A video on Tube8.com."""

    url_format = "http://www.tube8.com/%s/"
    website_name = "Tube8.com"

    @cachedproperty
    def _flashvars(self):
        """Variables used by the player."""
        flashvars = self.get_content_substring('var flashvars = {', '};')
        return json.loads('{' + flashvars + '}')

    @cachedproperty
    def _author_a(self):
        """Element for author profile link."""
        for element in self.page.xpath("//li/strong/.."):
            if " ".join(element[0].text.split(" ")) == "Submitted by:":
                for subelement in element.xpath("a"):
                    if "user" in subelement.get("href"):
                        return subelement

    @cachedproperty
    def author_name(self):
        """Author profile name."""
        return self._author_a.text

    @cachedproperty
    def author_url(self):
        """Author profile URL."""
        return self._author_a.get("href")

    @cachedproperty
    def description(self):
        """Video description."""
        for element in self.page.getiterator("li"):
            if element[0].text == "Description:":
                return element[0].tail.strip()

    @cachedproperty
    def title(self):
        """Video title."""
        return self._flashvars["video_title"].replace("+", " ")

    @cachedproperty
    def thumbnail(self):
        """Video thumnail URL."""
        return self._flashvars["image_url"]

    def __iter__(self):
        """Return an interable of supported medium formats."""
        yield Format(unquote(self._flashvars["video_url"]),
                           mime_type="video/x-flv")


class Tube8(HTMLPlugin):

    """Plugin for Tube8.com."""

    url_format = "http://www.tube8.com/%s/"
    medium_class = Tube8Medium
