# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Test Matcher using custom plugins.
"""


import unittest

from getmediumurl import Matcher
from getmediumurl.medium import Medium
from getmediumurl.plugin import Plugin


__all__ = (
    "PluginLoadingTestCase",
    "UselessPlugin",
    "BlankPlugin",
    "UnmatchedPlugin",
    "MatchedPlugin",
    )


class UselessPlugin(Plugin):

    """Unusable plugin.

    `get_file_url` returns `None`.
    """

    @property
    def url(self):
        """Return `None`."""

    @classmethod
    def match(cls, url, urlreader=None):
        """Empty method."""


class BlankPlugin(Plugin):

    """Plugin matching everything to nothing, disabled."""

    @classmethod
    def match(cls, url, urlreader=None):
        """Match anything."""
        return cls(url, urlreader)

    @classmethod
    def disabled(cls):
        """Iterate reasons for disabling this plugin."""
        return ("This space intentionally left blank.",)


class UnmatchedPlugin(Plugin):

    """Plugin which does not match to any URL."""

    @property
    def url(self):
        """Return matched URL."""
        return ""


class MatchedMedium(Medium):

    """Medium for `MatchedPlugin`."""

    website_name = "An unnamed website"


class MatchedPlugin(UnmatchedPlugin, BlankPlugin):

    """Plugin matching anything to an empty string."""

    def __iter__(self):
        """Iterate media."""
        yield MatchedMedium(self)

    @classmethod
    def disabled(cls):
        """Don't disable it."""
        return ()


class PluginLoadingTestCase(unittest.TestCase):

    """Test loading plugins."""

    def is_usable(self):
        """Test loading only usable plugins."""
        plugins = self.matcher.plugins.values()
        self.assert_(Plugin not in plugins)
        self.assert_(UselessPlugin not in plugins)
        self.assert_(BlankPlugin not in plugins)
        self.assert_(UnmatchedPlugin not in plugins)
        self.assert_(MatchedPlugin in plugins)

    def test_is_usable(self):
        """Test loading only usable plugins"""
        self.matcher = Matcher((__name__,))
        self.is_usable()

    def test_is_usable_single(self):
        """Test loading only usable plugins from single module"""
        self.matcher = Matcher(__name__)
        self.is_usable()

    def test_class_loading(self):
        """Test loading specific class"""
        plugins = Matcher(MatchedPlugin).plugins.values()
        self.assertEqual(plugins, [MatchedPlugin])

    def test_module_loading(self):
        """Test loading a module"""
        import tests_getmediumurl.matching_plugins
        self.matcher = Matcher(tests_getmediumurl.matching_plugins)
        self.is_usable()

    def test_non_plugin(self):
        """Test that non-plugin classes are not loaded"""
        matcher = Matcher(PluginLoadingTestCase)
        self.assert_(PluginLoadingTestCase not in matcher.plugins)

    def test_invalid_module_loading(self):
        """Test loading a non-existant module"""
        matcher = Matcher("tests.does_not_exist")
        self.assert_(not matcher.plugins)

    def test_get_website_name(self):
        """Test website with website_name attribute"""
        matcher = Matcher(__name__)
        self.assertEquals(matcher.match("").get_medium().website,
                          "An unnamed website")
