# -*- coding: utf-8 -*-
# Copyright (C) 2010, 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Shared plugin code.
"""


import re

from getmediumurl.utils import escape_regex, check_single_percent_s
from getmediumurl.clsutils import overrides


__all__ = ("Plugin",)


class Plugin(object):

    """
    Common base class of GetMediumURL plugins.

    The `match_re` and `url_format` class attributes should be
    overridden in subclasses.  To be useful, plugins must override
    also the `get_file_url` method.
    """

    #: Regular expression matched on URL to determine if the plugin can
    #: be used for this URL, with `mediumid` group passed to the
    #: constructor.
    match_re = None

    #: A string where ``%s`` will be replace by `mediumid` to make the
    #: URL matched by `match_re`.
    url_format = None

    #: An attribute set to `True` or `False` to use default `has_high_quality`.
    high_quality = False

    #: `True` if the plugin can match without doing HTTP requests.  If
    #: `None`, then `disabled` will set it to `True` if the `match`
    #: method is not overridden.
    #:
    #: Change it if `match` is overridden and doesn't do any HTTP
    #: requests or `match_re` probably matches unneeded pages.
    fast = None

    def __init__(self, mediumid, matcher):
        """Make a plugin object for use with specific medium.

        :Parameters:
          mediumid
            a plugin-specific string identifying the specific medium
          matcher
            an instance of `getmediumurl.matcher.Matcher` used to match
            this plugin
        """
        #: A string which combined with the plugin class uniquely
        # identifies the media found on the URL matched.  It may have
        # any length (but usually not longer than the URL given to the
        # matcher).
        self.mediumid = mediumid
        #: A subclass of `getmediumurl.reader.URLReader`.
        self.urlreader = matcher.urlreader
        #: The `getmediumurl.matcher.Matcher` used to match this plugin.
        self.matcher = matcher

    def __iter__(self):
        """Iterate media found by the plugin."""
        return iter(())

    def get_medium(self):
        """Return single medium or raise exception."""
        found = None
        for medium in self:
            if found is not None:
                raise ValueError("requested single medium, but more found")
            found = medium
        if found is None:
            raise ValueError("requested single medium, but none found")
        return found

    @classmethod
    def disabled(cls):
        """Iterate over strings describing why this plugin is not usable.

        The strings state e.g. that a required module is not found, or
        the plugin does not have necessary code.

        Plugins which can be used must return no such strings.

        This implementation checks if fields needed by other methods
        are set, unless these methods are overridden, and if
        `formats` or `get_file_url` is replaced.
        """
        if not overrides("match", cls, Plugin) \
                and cls.match_re is None:
            if cls.url_format is None:
                yield "match not overridden, match_re " \
                    "and url_format not specified"
            elif not cls._prepare_match_re():
                yield "match not overridden, match_re " \
                    "not specified and could not determine it " \
                    "from url_format"
        if not overrides("__iter__", cls, Plugin):
            yield "does not find media"
        if not overrides("match", cls, Plugin) \
                and cls.fast is None:
            cls.fast = True
        else:
            cls.fast = False

    @classmethod
    def match(cls, url, matcher):
        """Return plugin's object handling this URL, or `None`.

        This implementation uses a regular expression object in class
        attribute `match_re` with `mediumid` group passed to the
        `__init__` method.

        The `matcher` argument is the same as in `__init__`.
        """
        if cls.match_re is None:
            if not cls._prepare_match_re():
                raise ValueError("Tried matching plugin with match_re of None")
        match = cls.match_re.match(url)
        if not match:
            return None
        return cls(match.group("mediumid"), matcher)

    @classmethod
    def _prepare_match_re(cls):
        """Set `match_re` to value obtained from `url_format`.

        If `url_format` is a string containing exactly single ``%s``,
        with all other percent characters replaced by ``%%``, then it
        will make it a regex matching whole line, with ``mediumid``
        named group in place of ``%s``.

        Returns `True` on success, `False` otherwise.
        """
        if not check_single_percent_s(cls.url_format):
            return False
        cls.match_re = re.compile(u"^%s$"
                                  % escape_regex(cls.url_format)
                                  .replace(u"%s", u"(?P<mediumid>.+)"))
        return True
