# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
Classes for media formats.
"""


__all__ = ("Format", "SimpleFormat")


#: A dictionary of file extensions used by default `Format.file_extension`.
_EXTENSIONS = {
    # TODO: add others when needed.
    u"": u"",
    u"video/flv": u".flv",
    u"video/mp4": u".mp4",
    u"video/webm": u".webm",
}


class Format(object):

    """Base class for formats.

    Instances are ordered, the ones representing larger images or
    videos being greater.  Non-visual or unknown formats are the
    smallest.
    """

    @property
    def width(self):
        """Video/image width in pixels or `None` if non-visual or unknown."""
        assert self is not None
        return None

    @property
    def height(self):
        """Video/image height in pixels or `None` if non-visual or unknown."""
        assert self is not None
        return None

    @property
    def mime_type(self):
        """MIME Content-Type typical for this medium format.

        If unknown, an empty string is returned.
        """
        assert self is not None
        return u""

    @property
    def url(self):
        """URL from which the medium can be obtained or `None` if unknown."""
        assert self is not None
        return None

    @property
    def area(self):
        """Area in pixels, or 0 if non-visual or unknown."""
        if self.width is not None and self.height is not None:
            return self.width * self.height
        else:
            return 0

    @property
    def file_extension(self):
        """File extension typical for files of this type."""
        # TODO: maybe use an external MIME database for it?
        return _EXTENSIONS[self.mime_type]

    def __cmp__(self, other):
        """Compare two formats.

        Return -1 if `self` is lower-quality, 0 if they are equal, 1
        if `self` is higher-quality than `other`.
        """
        # TODO: compare the codecs used, prefer ones better supported
        # by free software and typically used with higher-quality
        # media.
        return self.area - other.area

    def __lt__(self, other):
        """``self < other``"""
        return self.__cmp__(other) < 0

    def __le__(self, other):
        """``self <= other``"""
        return self.__cmp__(other) <= 0

    def __gt__(self, other):
        """``self > other``"""
        return self.__cmp__(other) > 0

    def __ge__(self, other):
        """``self >= other``"""
        return self.__cmp__(other) >= 0


class SimpleFormat(Format):

    """Format with all data specified for its constructor."""

    def __init__(self, url=None, mime_type=u"", width=None, height=None):
        """Make a format object with known data.

        The arguments are returned by the properties.
        """
        super(SimpleFormat, self).__init__()
        self._url = url
        self._mime_type = mime_type
        self._width = width
        self._height = height

    @property
    def width(self):
        """Video/image width in pixels or `None` if non-visual or unknown."""
        return self._width

    @property
    def height(self):
        """Video/image height in pixels or `None` if non-visual or unknown."""
        return self._height

    @property
    def mime_type(self):
        """MIME Content-Type typical for this medium format.

        If unknown, an empty string is returned.
        """
        return self._mime_type

    @property
    def url(self):
        """URL from which the medium can be obtained."""
        return self._url
