# -*- coding: utf-8 -*-
# Copyright (C) 2010, 2011  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
GetMediumURL plugin for Dailymotion videos.
"""


import re
import urllib

from getmediumurl.htmlplugin import HTMLPlugin, HTMLMedium
from getmediumurl.compat import HAS_JSON
from getmediumurl.format import SimpleFormat


__all__ = ("Dailymotion",)


# URL substring used to determine video codec and size.
_FORMAT = re.compile(u"/(?P<codec>H264|FLV)-(?P<width>[0-9]+)"
                     u"x(?P<height>[0-9]+)/")


class DailymotionMedium(HTMLMedium):

    """Data about a Dailymotion video."""

    raw_url_format = u"http://www.dailymotion.com/video/%s"
    website_name = u"Dailymotion"
    high_quality = True

    @property
    def author_name(self):
        """Return author profile name."""
        return self.get_content_substring('<Attribute name="owner">',
                                          '</Attribute>')

    @property
    def author_url(self):
        """Return author profile URL."""
        return self.make_absolute("/" + self.author_name)

    @property
    def description(self):
        """Return video description."""
        element = self.page.find("//div[@id='video_description']")
        text = u""
        for subelement in element.getiterator():
            if subelement.text is not None:
                text += subelement.text
            if subelement.tail is not None:
                text += subelement.tail
        return u" ".join(text.split())

    @property
    def language(self):
        """Return video language."""
        return self.get_content_substring('<Attribute name="language">',
                                          '</Attribute>')

    @property
    def title(self):
        """Return video title."""
        return self.page.xpath("//span[@class='title']/text()")[0]

    def __iter__(self):
        """Iterate over formats of the video."""
        from getmediumurl.compat import json
        sequence = self.get_content_substring(
            start='.addVariable("sequence",  "',
            end='");')
        raw_data = urllib.unquote(sequence).replace(r"\'", "'")
        data = json.loads(raw_data)
        urls = {}
        for obj in data:
            for obj in obj.get(u"layerList", tuple()):
                for obj in obj.get(u"sequenceList", tuple()):
                    for obj in obj.get(u"layerList", tuple()):
                        try:
                            urls = obj[u"param"][u"videoPluginParameters"]
                        except KeyError:
                            continue
        for url in urls.itervalues():
            try:
                match = _FORMAT.search(url)
            except TypeError:
                # Don't parse unneeded booleans.
                continue
            if not match:
                continue
            codec = match.group(u"codec")
            width = int(match.group(u"width"))
            height = int(match.group(u"height"))
            if codec == u"H264":
                mime_type = u"video/mp4"
                if width == 848:
                    width, height = 720, 400
            elif codec == u"FLV":
                mime_type = u"video/flv"
                if width == 320:
                    height = 178
            else:
                raise AssertionError("unknown codec %s" % codec)
            yield SimpleFormat(url=url, mime_type=mime_type,
                               width=width, height=height)


class Dailymotion(HTMLPlugin):

    """Plugin for Dailymotion videos."""

    match_re = re.compile(r"^http://www\.dailymotion\.[a-z]+/video/"
                          r"(?P<mediumid>[a-z0-9]+)")
    medium_class = DailymotionMedium

    @classmethod
    def disabled(cls):
        """Determine if this plugin can be used."""
        if not HAS_JSON:
            yield "missing json or simplejson dependency"
        for reason in super(Dailymotion, cls).disabled():
            yield reason
