# -*- coding: utf-8 -*-
# Copyright (C) 2010  Michał Masłowski  <mtjm@mtjm.eu>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""
GetMediumURL plugin for Dailymotion videos.
"""


import re
import urllib

from getmediumurl.htmlplugin import HTMLPlugin
from getmediumurl.compat import HAS_JSON


__all__ = ("Dailymotion",)


class Dailymotion(HTMLPlugin):

    """Plugin for Dailymotion videos."""

    match_re = re.compile(r"^http://www\.dailymotion\.[a-z]+/video/"
                          r"(?P<mediumid>[a-z0-9]+)")
    raw_url_format = u"http://www.dailymotion.com/video/%s"
    website_name = u"Dailymotion"
    high_quality = True

    @property
    def author_name(self):
        """Return author profile name."""
        return self.get_content_substring('<Attribute name="owner">',
                                          '</Attribute>')

    @property
    def author_url(self):
        """Return author profile URL."""
        return self.make_absolute("/" + self.author_name)

    @property
    def description(self):
        """Return video description."""
        for element in self.page.getiterator("div"):
            if element.get("id") == "video_description":
                text = u""
                for subelement in element.getiterator():
                    if subelement.text is not None:
                        text += subelement.text
                    if subelement.tail is not None:
                        text += subelement.tail
                return u" ".join(text.split())

    @property
    def language(self):
        """Return video language."""
        return self.get_content_substring('<Attribute name="language">',
                                          '</Attribute>')

    @property
    def title(self):
        """Return video title."""
        for element in self.page.getiterator("span"):
            if element.get("class") == "title":
                return element.text

    def get_file_url(self, high_quality=False):
        """Get video URL."""
        from getmediumurl.compat import json
        sequence = self.get_content_substring(
            start='so.addVariable("sequence",  "',
            end='");').decode("ascii")
        raw_data = urllib.unquote(sequence).replace(r"\'", "'")
        data = json.loads(raw_data)
        urls = []
        for obj in data:
            for obj in obj.get(u"layerList", tuple()):
                for obj in obj.get(u"sequenceList", tuple()):
                    for obj in obj.get(u"layerList", tuple()):
                        try:
                            for value in obj[u"param"][
                                u"videoPluginParameters"].itervalues():
                                urls.append(value)
                        except KeyError:
                            continue
        # TODO: find <format>-<width>x<height> in urls, choose format.
        if high_quality:
            return urls[0]
        else:
            return urls[-1]

    @classmethod
    def disabled(cls):
        """Determine if this plugin can be used."""
        if not HAS_JSON:
            yield "missing json or simplejson dependency"
        for reason in super(Dailymotion, cls).disabled():
            yield reason
