/* PluginAppletViewer.java - manages embeddable applet windows
   Copyright (C) 2003, 2006  Thomas Fitzsimmons <fitzsim@redhat.com>

   This file is part of GCJ Applet Viewer.

   GCJ Applet Viewer is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   GCJ Applet Viewer is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with GCJ Applet Viewer; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

package gnu.gcjwebplugin;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.MalformedURLException;
import java.nio.charset.Charset;
import java.util.HashMap;


/**
 * PluginAppletViewer communicates through pipes with a web browser
 * plugin.  A PluginAppletViewer manages applet windows that may be
 * embedded into web pages.
 */
class PluginAppletViewer
{
  // A mapping of instance IDs to PluginAppletWindows.
  static HashMap appletWindows = new HashMap ();

  private static BufferedReader pluginInputStream;
  private static BufferedWriter pluginOutputStream;

  static void start(InputStream inputStream, OutputStream outputStream)
    throws MalformedURLException, IOException
  {
    // Set up input and output pipes.  Use UTF-8 encoding.
    pluginInputStream =
      new BufferedReader(new InputStreamReader(inputStream,
                                               Charset.forName("UTF-8")));
    pluginOutputStream =
      new BufferedWriter(new OutputStreamWriter(outputStream,
                                                Charset.forName("UTF-8")));

    write("running");

    // Read first message.
    String message = read();

    PluginAppletWindow currentWindow = null;

    while (true)
      {
	if (message.startsWith("instance"))
	  {
	    // Read applet instance identifier.
	    String key = message.substring(9);

	    if (appletWindows.get(key) == null)
	      appletWindows.put(key, new PluginAppletWindow());

	    currentWindow = (PluginAppletWindow) appletWindows.get(key);
	  }
	else if (message.startsWith("tag"))
	  {
	    int pos = message.indexOf(' ', 4);
	    String documentbase = message.substring(4, pos);
        String tag = message.substring(pos + 1);
        currentWindow.setParser(tag, documentbase);
	  }
	else if (message.startsWith("handle"))
	  {
	    long handle = Long.parseLong(message.substring(7));

	    currentWindow.setHandle(handle);
	  }
	else if (message.startsWith("width"))
	  {
	    int width = Integer.parseInt(message.substring(6));

	    currentWindow.setSize(width, currentWindow.getHeight());
	  }
	else if (message.startsWith("height"))
	  {
	    int height = Integer.parseInt(message.substring(7));

	    currentWindow.setSize(currentWindow.getWidth(), height);
	  }
	else if (message.startsWith("destroy"))
	  {
	    appletWindows.remove(currentWindow);
	    currentWindow.dispose();
	  }

	// Read next message.
	message = read();
      }
  }

  /**
   * Write string to plugin.
   * 
   * @param message the message to write
   *
   * @exception IOException if an error occurs
   */
  static void write(String message) throws IOException
  {
    pluginOutputStream.write(message, 0, message.length());
    pluginOutputStream.newLine();
    pluginOutputStream.flush();

    System.err.println("  PIPE: applet viewer wrote: " + message);
  }

  /**
   * Read string from plugin.
   *
   * @return the read string
   *
   * @exception IOException if an error occurs
   */
  static String read() throws IOException
  {
    String message = pluginInputStream.readLine();

    System.err.println("  PIPE: applet viewer read: " + message);

    if (message == null || message.equals("shutdown"))
      {
	// Close input/output channels to plugin.
	pluginInputStream.close();
	pluginOutputStream.close();

	System.err.println("gcjappletviewer: exiting plugin applet viewer");
	System.exit(0);
      }

    return message;
  }
}
