/* AppletTag.java - representation of an HTML APPLET tag
   Copyright (C) 2003, 2004, 2005  Thomas Fitzsimmons <fitzsim@redhat.com>

   This file is part of GCJ Applet Viewer.

   GCJ Applet Viewer is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   GCJ Applet Viewer is distributed in the hope that it will be
   useful, but WITHOUT ANY WARRANTY; without even the implied warranty
   of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with GCJ Applet Viewer; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

package gnu.gcjwebplugin;

import java.awt.Dimension;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StreamTokenizer;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.StringTokenizer;

class AppletTag
{
  String name = "";
  String code = "";
  String codebase = "";
  String archivesList = "";
  ArrayList archives = new ArrayList();
  HashMap parameters = new HashMap();

  // documentbase is not specified in an applet tag but we keep it
  // here anyway, for convenience.
  URL documentbase;

  // Same goes for the HTML file's character encoding.
  String encoding;

  // Default constructor needed by derived classes.
  AppletTag()
  {
  }

  // Construct an applet tag object from a .class file and the user's
  // current directory.
  AppletTag(String code, String archives, List parameters,
            Dimension dimensions, String encoding)
    throws MalformedURLException, IOException
  {
    this.encoding = encoding;
    // Calculate code.
    this.code =
      code.substring(code.lastIndexOf(File.separatorChar) + 1, code.length());

    // When a class file is given on the command line, codebase is
    // always set to the empty string and documentbase is the
    // directory in which the class file resides.
    String tmpCodeBase =
      code.substring(0, code.lastIndexOf(File.separatorChar) + 1);
    if (tmpCodeBase.startsWith(File.separator))
      this.documentbase = new URL("file", "", tmpCodeBase);
    else
      // documentbase is the current working directory.
      this.documentbase =
	new URL("file", "",
	        System.getProperty("user.dir") + File.separator + tmpCodeBase);

    archivesList = archives;
    // Parse archives.
    parseArchives();

    // Parse parameters, a list of comma-delimited key-value pairs.
    Iterator pairs = parameters.iterator();
    while (pairs.hasNext())
      {
	StringTokenizer paramTokenizer =
	  new StringTokenizer((String) pairs.next(), ",");
	this.parameters.put(paramTokenizer.nextToken().trim().toLowerCase(),
	                    paramTokenizer.nextToken().trim());
      }
    this.parameters.put("width", Integer.toString(dimensions.width));
    this.parameters.put("height", Integer.toString(dimensions.height));
  }

  // Construct an applet tag object from an HTML stream.
  AppletTag(StreamTokenizer tagTokenizer, URL documentbase, String encoding)
    throws IOException
  {
    this.documentbase = documentbase;
    this.encoding = encoding;

    while (tagTokenizer.nextToken() != '>')
      {
	if (tagTokenizer.ttype == StreamTokenizer.TT_EOF)
	  break;

	if (tagTokenizer.ttype == StreamTokenizer.TT_WORD)
	  {
	    if (tagTokenizer.sval.equals("name"))
	      {
		name = parseAttributeString(tagTokenizer);
		this.parameters.put("name", name);
	      }
	    else if (tagTokenizer.sval.equals("code"))
	      code = parseAttributeString(tagTokenizer);
	    else if (tagTokenizer.sval.equals("codebase"))
	      codebase = parseAttributeString(tagTokenizer);
	    else if (tagTokenizer.sval.equals("archive"))
	      archivesList = parseAttributeString(tagTokenizer);
	    else if (tagTokenizer.sval.equals("width"))
	      {
		tagTokenizer.nextToken();
		tagTokenizer.nextToken();
		if (tagTokenizer.ttype == StreamTokenizer.TT_NUMBER)
		  this.parameters.put("width",
				      Integer.toString((int) tagTokenizer.nval));
		else
		  this.parameters.put("width", tagTokenizer.sval);
	      }
	    else if (tagTokenizer.sval.equals("height"))
	      {
		tagTokenizer.nextToken();
		tagTokenizer.nextToken();
		if (tagTokenizer.ttype == StreamTokenizer.TT_NUMBER)
		  this.parameters.put("height",
				      Integer.toString((int) tagTokenizer.nval));
		else
		  this.parameters.put("height", tagTokenizer.sval);
	      }
	  }
      }
  }

  // Parse a case-sensitive element.
  static String parseAttributeString(StreamTokenizer tagTokenizer)
    throws IOException
  {
    String value = "";

    // Read "=".
    tagTokenizer.nextToken();

    // Read non-alphabetic characters that may appear in file names as
    // word characters so that an unquoted file name is parsed as a
    // single token.
    tagTokenizer.wordChars('!', '!');
    tagTokenizer.wordChars('#', '&');
    tagTokenizer.wordChars('(', '/');
    tagTokenizer.wordChars('{', '~');
    tagTokenizer.wordChars('[', '^');

    tagTokenizer.lowerCaseMode(false);

    tagTokenizer.nextToken();
    value = new String(tagTokenizer.sval);

    // Reset character meanings.
    tagTokenizer.ordinaryChar('!');
    tagTokenizer.ordinaryChars('#', '&');
    tagTokenizer.ordinaryChars('(', '/');
    tagTokenizer.ordinaryChars('{', '~');
    tagTokenizer.ordinaryChars('[', '^');

    tagTokenizer.lowerCaseMode(true);
    return value;
  }

  // Parse a comma-delimited list of archives.
  void parseArchives() throws IOException
  {
    StringTokenizer tagTokenizer = new StringTokenizer(archivesList, ",");

    while (tagTokenizer.hasMoreTokens())
      archives.add(prependCodebase(tagTokenizer.nextToken().trim()));
  }

  // Prepend the full codebase to basename and return a URL object
  // representing the result.  The full codebase is the codebase
  // appended to the documentbase.
  URL prependCodebase(String basename) throws MalformedURLException
  {
    URL fullcodebase;

    // If no codebase was specified default to documentbase.
    if (codebase.equals(""))
      {
	if (documentbase.getFile().endsWith(File.separator))
	  fullcodebase = documentbase;
	else
	  {
	    String dirname = documentbase.getFile();

	    // Determine dirname for file by stripping everything
	    // past the last file separator.
	    dirname =
	      dirname.substring(0, dirname.lastIndexOf(File.separatorChar) + 1);

	    fullcodebase =
	      new URL(documentbase.getProtocol(), documentbase.getHost(),
	              documentbase.getPort(), dirname);
	  }
      }
    else
      {
	// codebase was specified
	if (codebase.endsWith(File.separator))
	  fullcodebase = new URL(documentbase, codebase);
	else
	  fullcodebase = new URL(documentbase, codebase + File.separator);
      }
    
    return new URL(fullcodebase, basename);
  }

  // Parse an applet parameter.
  void parseParam(StreamTokenizer tagTokenizer) throws IOException
  {
    String key = null;
    String value = null;

    while (tagTokenizer.nextToken() != '>')
      {
	if (tagTokenizer.ttype == StreamTokenizer.TT_WORD)
	  {
	    if (tagTokenizer.sval.equals("name"))
	      {
		tagTokenizer.nextToken();
		tagTokenizer.nextToken();
		key = new String(tagTokenizer.sval).toLowerCase();
	      }
	    else if (tagTokenizer.sval.equals("value"))
	      value = parseParamValue(tagTokenizer);
	  }

	if (key != null && value != null)
	  parameters.put(key, value);
      }
  }

  // Parse the value of a parameter.
  static String parseParamValue(StreamTokenizer tagTokenizer)
    throws IOException
  {
    String value = "";

    // Read "=".
    tagTokenizer.nextToken();

    // Read non-alphabetic characters that may appear in
    // file names as word characters so that an unquoted
    // file name is parsed as a single token.
    tagTokenizer.wordChars('!', '!');
    tagTokenizer.wordChars('#', '&');
    tagTokenizer.wordChars('(', '/');
    tagTokenizer.wordChars('{', '~');
    tagTokenizer.wordChars('[', '^');

    tagTokenizer.lowerCaseMode(false);

    tagTokenizer.nextToken();
    if (tagTokenizer.ttype == StreamTokenizer.TT_NUMBER)
      {
	// Remove nval's scale.
	int unscaled_nval = (int) tagTokenizer.nval;

	// See if nval represents an integer value.
	if (Double.compare(tagTokenizer.nval, (double) unscaled_nval) == 0)
	  value = Integer.toString(unscaled_nval);
	else
	  value = Double.toString(tagTokenizer.nval);
      }
    else
      value = new String(tagTokenizer.sval);

    // Reset character meanings.
    tagTokenizer.ordinaryChar('!');
    tagTokenizer.ordinaryChars('#', '&');
    tagTokenizer.ordinaryChars('(', '/');
    tagTokenizer.ordinaryChars('{', '~');
    tagTokenizer.ordinaryChars('[', '^');

    tagTokenizer.lowerCaseMode(true);

    return value;
  }

  static URL locationToURL(String location) throws IOException
  {
    URL tmpDocumentBase = null;
    
    try
      {
	// Try parsing location as a URL.
	tmpDocumentBase = new URL(location);

	// If no file was specified in the URL the assume the user
	// meant the root page.
	if (tmpDocumentBase.getFile().equals(""))
	  tmpDocumentBase = new URL(location.concat(File.separator));
      }
    catch (MalformedURLException e)
      {
	// location is not a URL.  See if it is an HTML file.
	String path;

	if (location.startsWith(File.separator))
	  path = new File(location).getCanonicalPath();
	else
	  path =
	    new File(System.getProperty("user.dir") + File.separator
	             + location).getCanonicalPath();

	tmpDocumentBase = new URL("file", "", path);
      }
    return tmpDocumentBase;
  }

  // Return a list of applet tag objects representing the applet tags
  // that appear in the file or URL "location."
  static List parseAppletTags(String location, String encoding)
    throws IOException
  {
    ArrayList tags = new ArrayList();
    URL tmpDocumentBase = locationToURL(location);
    InputStream input = tmpDocumentBase.openStream();
    StreamTokenizer tagTokenizer =
      new StreamTokenizer(new InputStreamReader(input));
    AppletTag currentTag =
      parseNextTag(tagTokenizer, tmpDocumentBase, encoding);
    
    while (currentTag != null)
      {
	tags.add(currentTag);
	currentTag = parseNextTag(tagTokenizer, tmpDocumentBase, encoding);
      }
    
    return tags;
  }

  // Parse the next applet tag in the tagTokenizer stream.
  static AppletTag parseNextTag(StreamTokenizer tagTokenizer,
                                URL documentbase, String encoding)
    throws IOException
  {
    AppletTag currentTag = null;
    int token;

    tagTokenizer.lowerCaseMode(true);
    tagTokenizer.ordinaryChar('/');
    tagTokenizer.wordChars('_', '_');

    token = tagTokenizer.nextToken();
    while (token != StreamTokenizer.TT_EOF)
      {
	// Start of an HTML tag.
	if (token == '<')
	  {
	    token = tagTokenizer.nextToken();
	    if (token == StreamTokenizer.TT_WORD)
	      {
		// An APPLET tag.
		if (tagTokenizer.sval.equals("applet"))
		  currentTag =
		    new AppletTag(tagTokenizer, documentbase, encoding);
		else if (tagTokenizer.sval.equals("embed"))
		  currentTag =
		    new EmbedTag(tagTokenizer, documentbase, encoding);
		else if (tagTokenizer.sval.equals("object"))
		  currentTag =
		    new ObjectTag(tagTokenizer, documentbase, encoding);
		else if (tagTokenizer.sval.equals("app"))
		  currentTag =
		    new AppTag(tagTokenizer, documentbase, encoding);
		else if (tagTokenizer.sval.equals("param"))
		  {
		    // Parse APPLET parameters only.
		    if (currentTag != null
		        && ! (currentTag instanceof EmbedTag))
		      currentTag.parseParam(tagTokenizer);
		  }
	      }
	    else
	      {
		if (token == '/')
		  {
		    token = tagTokenizer.nextToken();

		    if (token == StreamTokenizer.TT_WORD)
		      {
			if (currentTag instanceof AppletTag)
			  {
			    if (tagTokenizer.sval.equals("applet")
			        || tagTokenizer.sval.equals("embed")
			        || tagTokenizer.sval.equals("object")
			        || tagTokenizer.sval.equals("app"))
			      {
				// Parse archives at this point so
				// that we can generate full archive
				// URLs using the documentbase and
				// codebase fields.
				currentTag.parseArchives();
				return currentTag;
			      }
			  }
		      }
		  }
	      }
	  }
	token = tagTokenizer.nextToken();
      }

    // If we hit EOF, just go ahead with whatever we've got.  Some
    // pages don't properly terminate; besides which our parser is a
    // bit bogus anyway.
    if (currentTag != null)
      {
	currentTag.parseArchives();
	return currentTag;
      }

    return null;
  }

  public String toString()
  {
    return ("  name=" + name + "\n" + "  code=" + code + "\n" + "  codebase="
	    + codebase + "\n" + "  archive=" + archives + "\n"
	    + "  parameters=" + parameters + "\n" + "  documentbase="
	    + documentbase + "\n");
  }

  public static Dimension getSize (AppletTag tag)
  {
    NumberFormat numberFormat;
    Dimension size = new Dimension(320, 200);

    try
      {
	String widthStr = (String) tag.parameters.get("width");

	if (widthStr != null)
	  {
	    if (widthStr.charAt(widthStr.length() - 1) == '%')
	      numberFormat = NumberFormat.getPercentInstance(Locale.US);
	    else
	      numberFormat = NumberFormat.getInstance(Locale.US);

	    // FIXME: Handle percentage somehow.
	    size.width = numberFormat.parse(widthStr).intValue();
	  }
      }
    catch (ParseException e)
      {
	// Use default.
      }

    try
      {
	String heightStr = (String) tag.parameters.get("height");

	if (heightStr != null)
	  {
	    if (heightStr.charAt(heightStr.length() - 1) == '%')
	      numberFormat = NumberFormat.getPercentInstance(Locale.US);
	    else
	      numberFormat = NumberFormat.getInstance(Locale.US);

	    // FIXME: Handle percentage somehow.
	    size.height = numberFormat.parse(heightStr).intValue();
	  }
      }
    catch (ParseException e)
      {
	// Use default.
      }

    return size;
  }
}


class EmbedTag extends AppletTag
{
  EmbedTag(StreamTokenizer tagTokenizer, URL documentbase, String encoding)
    throws IOException
  {
    this.documentbase = documentbase;
    this.encoding = encoding;

    // In an EMBED tag, a parameter is any non-standard attribute.  This
    // is a problem for applets that take parameters named "code",
    // "codebase", "archive", "object", or "type".  The solution is to
    // allow the same attributes, prefixed by "java_".  The presence of
    // a "java_" attribute indicates that the non-prefixed attribute
    // should be interpreted as a parameter.  For example if "java_code"
    // and "code" attributes are present in the EMBED tag then the
    // "code" attribute is interpreted as a parameter.
    boolean codeSet = false;
    boolean codebaseSet = false;
    boolean archiveSet = false;
    boolean objectSet = false;

    boolean javaCodeSet = false;
    boolean javaCodebaseSet = false;
    boolean javaArchiveSet = false;
    boolean javaObjectSet = false;

    while (tagTokenizer.nextToken() != '>')
      {
	if (tagTokenizer.ttype == StreamTokenizer.TT_WORD)
	  {
	    if (tagTokenizer.sval.equals("name"))
	      {
		name = parseAttributeString(tagTokenizer);
		parameters.put("name", name);
	      }
	    else if (tagTokenizer.sval.equals("code"))
	      {
		if (javaCodeSet)
		  {
		    // Interpret non-prefixed attribute as a parameter.
		    String key = new String(tagTokenizer.sval);
		    String value = AppletTag.parseParamValue(tagTokenizer);

		    parameters.put(key, value);
		  }
		else
		  {
		    code = parseAttributeString(tagTokenizer);
		    codeSet = true;
		  }
	      }
	    else if (tagTokenizer.sval.equals("java_code"))
	      {
		if (codeSet)
		  parameters.put("code", code);
		code = parseAttributeString(tagTokenizer);
		javaCodeSet = true;
	      }
	    else if (tagTokenizer.sval.equals("object"))
	      {
		if (javaObjectSet)
		  {
		    // Interpret non-prefixed attribute as a parameter.
		    String key = new String(tagTokenizer.sval);
		    String value = AppletTag.parseParamValue(tagTokenizer);

		    parameters.put(key, value);
		  }
		else
		  {
		    code = parseAttributeString(tagTokenizer);
		    objectSet = true;
		  }
	      }
	    else if (tagTokenizer.sval.equals("java_object"))
	      {
		if (objectSet)
		  parameters.put("object", code);
		code = parseAttributeString(tagTokenizer);
		javaObjectSet = true;
	      }
	    else if (tagTokenizer.sval.equals("codebase"))
	      {
		if (javaCodebaseSet)
		  {
		    // Interpret non-prefixed attribute as a parameter.
		    String key = new String(tagTokenizer.sval);
		    String value = AppletTag.parseParamValue(tagTokenizer);

		    parameters.put(key, value);
		  }
		else
		  {
		    codebase = parseAttributeString(tagTokenizer);
		    codebaseSet = true;
		  }
	      }
	    else if (tagTokenizer.sval.equals("java_codebase"))
	      {
		if (codebaseSet)
		  parameters.put("codebase", codebase);
		codebase = parseAttributeString(tagTokenizer);
		javaCodebaseSet = true;
	      }
	    else if (tagTokenizer.sval.equals("archive"))
	      {
		if (javaArchiveSet)
		  {
		    // Interpret non-prefixed attribute as a parameter.
		    String key = new String(tagTokenizer.sval);
		    String value = AppletTag.parseParamValue(tagTokenizer);

		    parameters.put(key, value);
		  }
		else
		  {
		    archivesList = parseAttributeString(tagTokenizer);
		    archiveSet = true;
		  }
	      }
	    else if (tagTokenizer.sval.equals("java_archive"))
	      {
		if (archiveSet)
		  parameters.put("archive", archivesList);
		archivesList = parseAttributeString(tagTokenizer);
		javaArchiveSet = true;
	      }
	    else if (tagTokenizer.sval.equals("width"))
	      {
		tagTokenizer.nextToken();
		tagTokenizer.nextToken();
		if (tagTokenizer.ttype == StreamTokenizer.TT_NUMBER)
		  this.parameters.put("width",
				      Integer.toString((int) tagTokenizer.nval));
		else
		  this.parameters.put("width", tagTokenizer.sval);
	      }
	    else if (tagTokenizer.sval.equals("height"))
	      {
		tagTokenizer.nextToken();
		tagTokenizer.nextToken();
		if (tagTokenizer.ttype == StreamTokenizer.TT_NUMBER)
		  this.parameters.put("height",
				      Integer.toString((int) tagTokenizer.nval));
		else
		  this.parameters.put("height", tagTokenizer.sval);
	      }
	    else if (! tagTokenizer.sval.equals("align")
	             && ! tagTokenizer.sval.equals("alt")
	             && ! tagTokenizer.sval.equals("hspace")
	             && ! tagTokenizer.sval.equals("mayscript")
	             && ! tagTokenizer.sval.equals("pluginspage")
	             && ! tagTokenizer.sval.equals("title")
	             && ! tagTokenizer.sval.equals("type")
	             && ! tagTokenizer.sval.equals("java_type")
	             && ! tagTokenizer.sval.equals("vspace"))
	      {
		// Interpret this unknown attribute as a parameter.
		String key = new String(tagTokenizer.sval).toLowerCase();
		String value = AppletTag.parseParamValue(tagTokenizer);
		parameters.put(key, value);
	      }
	  }
      }
  }
}


class ObjectTag extends AppletTag
{
  // See comment in EmbedTag for an explanation of these fields.
  boolean codeSet;
  boolean codebaseSet;
  boolean archiveSet;
  boolean objectSet;
  boolean javaCodeSet;
  boolean javaCodebaseSet;
  boolean javaArchiveSet;
  boolean javaObjectSet;

  ObjectTag(StreamTokenizer tagTokenizer, URL documentbase, String encoding)
    throws IOException
  {
    this.documentbase = documentbase;
    this.encoding = encoding;

    while (tagTokenizer.nextToken() != '>')
      {
	if (tagTokenizer.ttype == StreamTokenizer.TT_WORD)
	  {
	    if (tagTokenizer.sval.equals("name"))
	      name = parseAttributeString(tagTokenizer);

	    if (tagTokenizer.sval.equals("width"))
	      {
		tagTokenizer.nextToken();
		tagTokenizer.nextToken();
		if (tagTokenizer.ttype == StreamTokenizer.TT_NUMBER)
		  this.parameters.put("width",
				      Integer.toString((int) tagTokenizer.nval));
		else
		  this.parameters.put("width", tagTokenizer.sval);
	      }
	    else if (tagTokenizer.sval.equals("height"))
	      {
		tagTokenizer.nextToken();
		tagTokenizer.nextToken();
		if (tagTokenizer.ttype == StreamTokenizer.TT_NUMBER)
		  this.parameters.put("height",
				      Integer.toString((int) tagTokenizer.nval));
		else
		  this.parameters.put("height", tagTokenizer.sval);
	      }
	  }
      }
  }

  // Parse an applet parameter.
  void parseParam(StreamTokenizer tagTokenizer) throws IOException
  {
    String paramName = null;
    String value = null;

    while (tagTokenizer.nextToken() != '>')
      {
	if (tagTokenizer.ttype == StreamTokenizer.TT_WORD)
	  {
	    if (tagTokenizer.sval.equals("name"))
	      {
		tagTokenizer.nextToken();
		tagTokenizer.nextToken();
		paramName = new String(tagTokenizer.sval).toLowerCase();
	      }
	    else if (tagTokenizer.sval.equals("value"))
	      value = parseParamValue(tagTokenizer);
	  }

	if (paramName != null && value != null)
	  {
	    if (paramName.equals("code"))
	      {
		if (javaCodeSet)
		  // Interpret non-prefixed attribute as a parameter.
		  parameters.put(paramName, value);
		else
		  {
		    code = value;
		    codeSet = true;
		  }
	      }
	    else if (paramName.equals("java_code"))
	      {
		if (codeSet)
		  parameters.put("code", code);
		code = value;
		javaCodeSet = true;
	      }
	    else if (paramName.equals("object"))
	      {
		if (javaObjectSet)
		  parameters.put(paramName, value);
		else
		  {
		    code = value;
		    objectSet = true;
		  }
	      }
	    else if (paramName.equals("java_object"))
	      {
		if (objectSet)
		  parameters.put("object", code);
		code = value;
		javaObjectSet = true;
	      }
	    else if (paramName.equals("codebase"))
	      {
		if (javaCodebaseSet)
		  parameters.put(paramName, value);
		else
		  {
		    codebase = value;
		    codebaseSet = true;
		  }
	      }
	    else if (paramName.equals("java_codebase"))
	      {
		if (codebaseSet)
		  parameters.put("codebase", codebase);
		codebase = value;
		javaCodebaseSet = true;
	      }
	    else if (paramName.equals("archive"))
	      {
		if (javaArchiveSet)
		  parameters.put(paramName, value);
		else
		  {
		    // Save the archive list for later processing.
		    archivesList = value;
		    archiveSet = true;
		  }
	      }
	    else if (paramName.equals("java_archive"))
	      {
		if (archiveSet)
		  parameters.put("archive", archivesList);
		archivesList = value;
		javaArchiveSet = true;
	      }
	    else if (! paramName.equals("type")
		     && ! paramName.equals("java_type")
	             && ! paramName.equals("mayscript")
	             && ! paramName.equals("scriptable"))
	      parameters.put(paramName, value);
	  }
      }
  }
}


class AppTag extends AppletTag
{
  AppTag(StreamTokenizer tagTokenizer, URL documentbase, String encoding)
    throws IOException
  {
    this.documentbase = documentbase;
    this.encoding = encoding;

    while (tagTokenizer.nextToken() != '>')
      {
	if (tagTokenizer.ttype == StreamTokenizer.TT_WORD)
	  {
	    if (tagTokenizer.sval.equals("class"))
	      code = parseAttributeString(tagTokenizer);
	    else if (tagTokenizer.sval.equals("src"))
	      codebase = parseAttributeString(tagTokenizer);
	    else if (tagTokenizer.sval.equals("width"))
	      {
		tagTokenizer.nextToken();
		tagTokenizer.nextToken();
		if (tagTokenizer.ttype == StreamTokenizer.TT_NUMBER)
		  this.parameters.put("width",
				      Integer.toString((int) tagTokenizer.nval));
		else
		  this.parameters.put("width", tagTokenizer.sval);
	      }
	    else if (tagTokenizer.sval.equals("height"))
	      {
		tagTokenizer.nextToken();
		tagTokenizer.nextToken();
		if (tagTokenizer.ttype == StreamTokenizer.TT_NUMBER)
		  this.parameters.put("height",
				      Integer.toString((int) tagTokenizer.nval));
		else
		  this.parameters.put("height", tagTokenizer.sval);
	      }
	  }
      }
  }
}
