/* gcjwebplugin - web browser plug-in to execute Java (tm) applets
   Copyright (C) 2003, 2004  Michael Koch <konqueror@gmx.de>
                             Thomas Fitzsimmons <fitzsim@redhat.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307  USA

   Linking this library statically or dynamically with other modules is
   making a combined work based on this library.  Thus, the terms and
   conditions of the GNU General Public License cover the whole
   combination.

   As a special exception, the copyright holders of this library give you
   permission to link this library with independent modules to produce an
   executable, regardless of the license terms of these independent
   modules, and to copy and distribute the resulting executable under
   terms of your choice, provided that you also meet, for each linked
   independent module, the terms and conditions of the license of that
   module.  An independent module is a module which is not derived from
   or based on this library.  If you modify this library, you may extend
   this exception to your version of the library, but you are not
   obligated to do so.  If you do not wish to do so, delete this
   exception statement from your version. */


// System includes
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>

// Netscape plugin API includes
#include <npapi.h>
#include <npupp.h>

// GLib includes
#include <glib.h>

// gcjwebplugin includes
#include "config.h"
#include "pluginDebug.h"
#include "pluginMeta.h"

// Support hack to get documentbase.
#include <nsIPluginInstance.h>
#include <nsIPluginInstancePeer.h>
#include <nsIPluginTagInfo2.h>

static NS_DEFINE_IID (kIPluginTagInfo2IID, NS_IPLUGINTAGINFO2_IID);

static NPNetscapeFuncs browserFunctions;

static GMutex *mutex_appletviewer_process = NULL;
static gint send_message_to_appletviewer (gchar const* value);
static gint receive_message_from_appletviewer (gchar* string);
static gboolean callbackViewerRead (GIOChannel* source,
				    GIOCondition condition,
				    gpointer data);

struct GCJPluginData
{
  char *code;
  char *codebase;
  char *archive;
  char *documentbase;
  char *parameters;
  char *width;
  char *height;
  // The xid of the plugin window, encoded in hexadecimal.
  char *xid;
  char *instance_id;
};

static NPError start_appletviewer_process (void);

static GIOChannel *output_to_appletviewer = NULL;
static GIOChannel *input_from_appletviewer = NULL;

static int instance_counter = 0;

static guint viewer_watch = 0;

static void gcjplugindata_new (GCJPluginData ** data);
static void gcjplugindata_destroy (GCJPluginData ** data);

NPError
GCJ_GetValue (NPP instance, NPPVariable variable, void* value)
{
  PLUGIN_DEBUG ("GCJ_GetValue\n");

  switch (variable)
    {
    // This plug-in needs XEmbed support.
    case NPPVpluginNeedsXEmbed:
      PLUGIN_DEBUG ("GCJ_GetValue: Returning Plug-in NeedsXEmbed value\n");
      *((PRBool*) value) = PR_TRUE;
      break;

    default:
      PLUGIN_DEBUG ("GCJ_GetValue: Unknown Plug-in value requested\n");
      return NPERR_GENERIC_ERROR;
      break;
    }
  return NPERR_NO_ERROR;
}

NPError
GCJ_New (NPMIMEType pluginType, NPP instance, uint16 mode,
         int16 argc, char* argn[], char* argv[],
         NPSavedData* saved)
{
  PLUGIN_DEBUG ("GCJ_New\n");

  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;

  NPError tmp = NPERR_NO_ERROR;
  
  g_mutex_lock (mutex_appletviewer_process);
  if (! output_to_appletviewer) // Start appletviewer process if necessary.
    tmp = start_appletviewer_process();
  g_mutex_unlock (mutex_appletviewer_process);

  if (tmp != NPERR_NO_ERROR)
    return tmp;

  GCJPluginData* data = NULL;

  gcjplugindata_new (&data);

  if (!data)
    return NPERR_OUT_OF_MEMORY_ERROR;

  // This little hack gets the current document's documentbase.  It
  // will only work when the plugin is loaded in Mozilla, because it
  // relys on browser-private data.
  nsIPluginInstance* xpcom_instance = (nsIPluginInstance *) (instance->ndata);
  nsIPluginInstancePeer* peer = NULL;
  xpcom_instance->GetPeer (&peer);

  nsresult result;
  nsIPluginTagInfo2* pluginTagInfo2;
  result = peer->QueryInterface (kIPluginTagInfo2IID, (void**) &pluginTagInfo2);

  char const* documentbase;
  pluginTagInfo2->GetDocumentBase (&documentbase);

  data->documentbase = g_strdup (documentbase);

  // Release references.
  NS_RELEASE (peer);
  NS_RELEASE (pluginTagInfo2);

  gchar *applet_tag = g_strdup ("<EMBED ");

  for (int i = 0; i < argc; i++)
    {
      if (!g_ascii_strcasecmp (argn[i], "code"))
	{
	  data->code = g_strdup (argv[i]);
	  applet_tag = g_strconcat (applet_tag,
				    g_strdup_printf ("CODE=\"%s\" ", argv[i]), NULL);
	}
      else if (!g_ascii_strcasecmp (argn[i], "codebase"))
	{
	  data->codebase = g_strdup (argv[i]);
	  applet_tag = g_strconcat (applet_tag,
				    g_strdup_printf ("CODEBASE=\"%s\" ", argv[i]), NULL);
	}
      else if (!g_ascii_strcasecmp (argn[i], "archive"))
	{
	  data->archive = g_strdup (argv[i]);
	  applet_tag = g_strconcat (applet_tag,
				    g_strdup_printf ("ARCHIVE=\"%s\" ", argv[i]), NULL);
	}
      else if (!g_ascii_strcasecmp (argn[i], "width"))
	{
	  data->width = g_strdup (argv[i]);
	  applet_tag = g_strconcat (applet_tag,
				    g_strdup_printf ("WIDTH=\"%s\" ", argv[i]), NULL);
	}
      else if (!g_ascii_strcasecmp (argn[i], "height"))
	{
	  data->height = g_strdup (argv[i]);
	  applet_tag = g_strconcat (applet_tag,
				    g_strdup_printf ("HEIGHT=\"%s\" ", argv[i]), NULL);
	}
      else
        {
          // Escape the parameter value so that line termination
          // characters will pass through the pipe.
          if (argv[i] != '\0')
            applet_tag = g_strconcat (applet_tag, argn[i],
                                      "=\"",
                                      g_strescape (argv[i], NULL),
                                      "\" ", NULL);
        }
    }
  applet_tag = g_strconcat (applet_tag,
			    g_strdup_printf ("></EMBED>"), NULL);

  data->instance_id = g_strdup_printf ("instance applet%d", instance_counter++);

  instance->pdata = data;

  char* tag = g_strdup_printf ("tag %s %s", data->documentbase, applet_tag);

  send_message_to_appletviewer (data->instance_id);
  send_message_to_appletviewer (tag);

  return NPERR_NO_ERROR;
}

static void
gcjplugindata_new (GCJPluginData ** data)
{
  *data = (GCJPluginData*)
    (* browserFunctions.memalloc) (sizeof (struct GCJPluginData));

  memset (*data, 0, sizeof (struct GCJPluginData));
}

NPError
GCJ_Destroy (NPP instance, NPSavedData** save)
{
  PLUGIN_DEBUG ("GCJ_Destroy\n");

  GCJPluginData* data = (GCJPluginData*) instance->pdata;

  send_message_to_appletviewer ("destroy");

  gcjplugindata_destroy (&data);

  return NPERR_NO_ERROR;
}

static void
gcjplugindata_destroy (GCJPluginData ** data)
{
  GCJPluginData *tofree = *data;

  g_free (tofree->code);
  g_free (tofree->codebase);
  g_free (tofree->archive);
  g_free (tofree->documentbase);
  g_free (tofree->parameters);
  g_free (tofree->width);
  g_free (tofree->height);
  g_free (tofree->xid);
  g_free (tofree->instance_id);

  (* browserFunctions.memfree) (tofree);
  tofree = NULL;
}

NPError
GCJ_SetWindow (NPP instance, NPWindow* window)
{
  PLUGIN_DEBUG ("GCJ_SetWindow\n");

  if (instance == NULL)
    return NPERR_INVALID_INSTANCE_ERROR;

  GCJPluginData* data = (GCJPluginData *)instance->pdata;

  if ((window == NULL) || (window->window == NULL))
    {
      PLUGIN_DEBUG ("GCJ_SetWindow: got null window\n");
      return NPERR_NO_ERROR;
    }

  if (data->xid)
    {
      // The window already exists.
      if (atol (data->xid) == (unsigned long) window->window)
	{
	  if (output_to_appletviewer
	      && input_from_appletviewer)
	    {
	      // The window is the same as it was for the last
	      // SetWindow call.
	      char* width = g_strdup_printf ("width %d", window->width);
	      if (g_ascii_strcasecmp (width, data->width))
		{
		  // The width of the plugin window has changed.
		  send_message_to_appletviewer (data->instance_id);
		  send_message_to_appletviewer (data->width);
		}

	      char* height = g_strdup_printf ("height %d", window->height);
	      if (g_ascii_strcasecmp (height, data->height))
		{
		  // The height of the plugin window has changed.
		  send_message_to_appletviewer (data->instance_id);
		  send_message_to_appletviewer (data->height);
		}
	    }
	  else
	    {
	      PLUGIN_DEBUG ("GCJ_SetWindow: "
			    GCJAPPLETVIEWER_EXECUTABLE
			    " not spawned\n");
	    }
	}
      else
	{
	  // The parent window has changed.  We don't handle this,
	  // since it's not supposed to happen.
	  PLUGIN_DEBUG ("GCJ_SetWindow: parent changed\n");
	}
    }
  else
    {
      PLUGIN_DEBUG ("GCJ_SetWindow: setting window\n");
      data->xid = g_strdup_printf ("xid %d", (int)(window->window));

      send_message_to_appletviewer (data->instance_id);
      send_message_to_appletviewer (data->xid);
    }

  return NPERR_NO_ERROR;
}

NPError
GCJ_NewStream (NPP instance, NPMIMEType type, NPStream* stream, NPBool seekable, uint16* stype)
{
  PLUGIN_DEBUG ("GCJ_NewStream\n");

  return NPERR_NO_ERROR;
}

void
GCJ_StreamAsFile (NPP instance, NPStream* stream, const char* filename)
{
  PLUGIN_DEBUG ("GCJ_StreamAsFile\n");
}

NPError
GCJ_DestroyStream (NPP instance, NPStream* stream, NPReason reason)
{
  PLUGIN_DEBUG ("GCJ_DestroyStream\n");

  return NPERR_NO_ERROR;
}

int32
GCJ_WriteReady (NPP instance, NPStream* stream)
{
  PLUGIN_DEBUG ("GCJ_WriteReady\n");

  return 0;
}

int32
GCJ_Write (NPP instance, NPStream* stream, int32 offset, int32 len, void* buffer)
{
  PLUGIN_DEBUG ("GCJ_Write\n");

  return 0;
}

void
GCJ_Print (NPP instance, NPPrint* platformPrint)
{
  PLUGIN_DEBUG ("GCJ_Print\n");
}

int16
GCJ_HandleEvent (NPP instance, void* event)
{
  PLUGIN_DEBUG ("GCJ_HandleEvent\n");

  return 0;
}

void
GCJ_URLNotify (NPP instance, const char* url, NPReason reason, void* notifyData)
{
  PLUGIN_DEBUG ("GCJ_URLNotify\n");
}

jref
GCJ_GetJavaClass (void)
{
  PLUGIN_DEBUG ("GCJ_GetJavaClass\n");

  return 0;
}

static gint
send_message_to_appletviewer (gchar const* name)
{
  GError *err = NULL;
  gsize bytes_written;
  NPError np_error;

  // Send name of attribute to gcjappletviewer.
  gchar *name_buf = g_strdup_printf ("%s\n", name);

  if (g_io_channel_write_chars (output_to_appletviewer,
                                name_buf, -1, &bytes_written, &err)
      != G_IO_STATUS_NORMAL)
    {
      g_printerr ("send_message_to_appletviewer: Error: %s\n", err->message);
      return NPERR_GENERIC_ERROR;
    }

  if (g_io_channel_flush (output_to_appletviewer, &err)
      != G_IO_STATUS_NORMAL)
    {
      g_printerr ("send_message_to_appletviewer: Error: %s\n", err->message);
      return NPERR_GENERIC_ERROR;
    }

  return NPERR_NO_ERROR;
}

static gint
receive_message_from_appletviewer (gchar *str)
{
  GError *err = NULL;

  // Receive message from gcjappletviewer.
  gchar *read_buf;
  if (g_io_channel_read_line (input_from_appletviewer,
                              &read_buf, NULL, NULL, &err)
      != G_IO_STATUS_NORMAL)
    {
      g_printerr ("receive_message_from_appletviewer: Error: %s\n", err->message);
      return NPERR_GENERIC_ERROR;
    }

  PIPE_INPUT_DEBUG (read_buf);

  return NPERR_NO_ERROR;
}

// Factory functions.  Functions prefixed by NP_ provide functionality
// that is common to the plug-in as a whole.  Instance functions
// prefixed by GCJ_ operate on specific instances of GCJPluginData.

// Provides the browser with pointers to the plug-in functions that we
// implement and initializes a local table with browser functions that
// we may wish to call.  Called once, after browser startup and before
// the first plug-in instance is created.
NPError
NP_Initialize(NPNetscapeFuncs* browserTable, NPPluginFuncs* pluginTable)
{
  PLUGIN_DEBUG ("NP_Initialize\n");

  if ((browserTable == NULL) || (pluginTable == NULL))
    return NPERR_INVALID_FUNCTABLE_ERROR;

  // Ensure that the major version of the plug-in API that the browser
  // expects is not more recent than the major version of the API that
  // we've implemented.
  if ((browserTable->version >> 8) > NP_VERSION_MAJOR)
    return NPERR_INCOMPATIBLE_VERSION_ERROR;

  // Ensure that the plug-in function table we've received is large
  // enough to store the number of functions that we may provide.
  if (pluginTable->size < sizeof(NPPluginFuncs))      
    return NPERR_INVALID_FUNCTABLE_ERROR;

  // Ensure that the browser function table is large enough to store
  // the number of browser functions that we may use.
  if (browserTable->size < sizeof(NPNetscapeFuncs))
    return NPERR_INVALID_FUNCTABLE_ERROR;

  // Store in a local table the browser functions that we may use.
  browserFunctions.version = browserTable->version;
  browserFunctions.size = browserTable->size;
  browserFunctions.posturl = browserTable->posturl;
  browserFunctions.geturl = browserTable->geturl;
  browserFunctions.geturlnotify = browserTable->geturlnotify;
  browserFunctions.requestread = browserTable->requestread;
  browserFunctions.newstream = browserTable->newstream;
  browserFunctions.write = browserTable->write;
  browserFunctions.destroystream = browserTable->destroystream;
  browserFunctions.status = browserTable->status;
  browserFunctions.uagent = browserTable->uagent;
  browserFunctions.memalloc = browserTable->memalloc;
  browserFunctions.memfree = browserTable->memfree;
  browserFunctions.memflush = browserTable->memflush;
  browserFunctions.reloadplugins = browserTable->reloadplugins;
  browserFunctions.getvalue = browserTable->getvalue;

  // Return to the browser the plug-in functions that we implement.
  pluginTable->version = (NP_VERSION_MAJOR << 8) + NP_VERSION_MINOR;
  pluginTable->size = sizeof(NPPluginFuncs);
  pluginTable->newp = NewNPP_NewProc(GCJ_New);
  pluginTable->destroy = NewNPP_DestroyProc(GCJ_Destroy);
  pluginTable->setwindow = NewNPP_SetWindowProc(GCJ_SetWindow);
  pluginTable->newstream = NewNPP_NewStreamProc(GCJ_NewStream);
  pluginTable->destroystream = NewNPP_DestroyStreamProc(GCJ_DestroyStream);
  pluginTable->asfile = NewNPP_StreamAsFileProc(GCJ_StreamAsFile);
  pluginTable->writeready = NewNPP_WriteReadyProc(GCJ_WriteReady);
  pluginTable->write = NewNPP_WriteProc(GCJ_Write);
  pluginTable->print = NewNPP_PrintProc(GCJ_Print);
  pluginTable->urlnotify = NewNPP_URLNotifyProc(GCJ_URLNotify);
  pluginTable->getvalue = NewNPP_GetValueProc(GCJ_GetValue);

  mutex_appletviewer_process = g_mutex_new ();
  
  return NPERR_NO_ERROR;
}

static NPError
start_appletviewer_process(void)
{

  // Add install prefix to PATH.
  char *searchpath = getenv("PATH");
  char *newpath = NULL;
  
  if (searchpath != NULL)
    {
      newpath = strcat(searchpath, ":/home/mkoch/local/gcjwebplugin/bin");
    }
  else
    {
      newpath = "/home/mkoch/local/gcjwebplugin/bin";
    }

  setenv("PATH", newpath, 1);
  
  // Create named pipes.
  gchar *pipename_in = g_strdup_printf ("/tmp/gcjwebplugin-%i-in", getpid ());
  gchar *pipename_out = g_strdup_printf ("/tmp/gcjwebplugin-%i-out", getpid ());
  
  if (mkfifo (pipename_in, 0700) == -1
      && errno != EEXIST)
    {
      g_printerr ("GCJ_New: Error: %s\n", strerror (errno));
      return NPERR_GENERIC_ERROR;
    }

  if (mkfifo (pipename_out, 0700) == -1
      && errno != EEXIST)
    {
      g_printerr ("GCJ_New: Error: %s\n", strerror (errno));
      return NPERR_GENERIC_ERROR;
    }
  GError *err = NULL;
  gchar *command_line [3] = {
    GCJAPPLETVIEWER_EXECUTABLE,
    g_strdup_printf ("--plugin=%s,%s",
		     pipename_in,
		     pipename_out),
    NULL
  };

  if (!g_spawn_async (NULL,
		      command_line,
		      NULL,
		      G_SPAWN_SEARCH_PATH,
		      NULL,
		      NULL,
		      NULL,
		      &err))
    {
      g_printerr ("GCJ_New: Error: %s\n", err->message);
      return NPERR_GENERIC_ERROR;
    }

  g_printerr ("Spawned "
	      GCJAPPLETVIEWER_EXECUTABLE
	      " successfully.\n");

  output_to_appletviewer = g_io_channel_new_file (pipename_in, "w", &err);
  if (!output_to_appletviewer)
    {
      g_printerr ("GCJ_New: Error: %s\n", err->message);
      return NPERR_GENERIC_ERROR;
    }
  
  input_from_appletviewer = g_io_channel_new_file (pipename_out, "r", &err);
  if (!input_from_appletviewer)
    {
      g_printerr ("GCJ_New: Error: %s\n", err->message);
      return NPERR_GENERIC_ERROR;
    }

  NPError np_error;
  if ((np_error = receive_message_from_appletviewer ("running")) != NPERR_NO_ERROR)
    return np_error;

  g_printerr ("Got confirmation that gcjappletviewer is running.\n");

  // Add callback to input channel from appletviewer.
  viewer_watch = g_io_add_watch (input_from_appletviewer, G_IO_IN,
				 callbackViewerRead, NULL);
  
  return NPERR_NO_ERROR;
}

// Returns a string describing the MIME type that this plug-in
// handles.
char *
NP_GetMIMEDescription(void)
{
  PLUGIN_DEBUG ("NP_GetMIMEDescription\n");

  return PLUGIN_MIME_DESC;
}

// Returns a value relevant to the plug-in as a whole.  The browser
// calls this function to obtain information about the plug-in.
NPError
NP_GetValue(void* future, NPPVariable variable, void *value)
{
  PLUGIN_DEBUG ("NP_GetValue\n");

  NPError result = NPERR_NO_ERROR;

  switch (variable)
    {
    case NPPVpluginNameString:
      PLUGIN_DEBUG ("NP_GetValue: Returning plug-in NameString value\n");
      *((char**) value) = PLUGIN_NAME " " PACKAGE_VERSION;
      break;

    case NPPVpluginDescriptionString:
      PLUGIN_DEBUG ("NP_GetValue: Returning plug-in DescriptionString value\n");
      *((char**) value) = PLUGIN_DESC;
      break;

    default:
      PLUGIN_DEBUG ("NP_GetValue: Unknown plug-in value requested\n");
      result = NPERR_GENERIC_ERROR;
      break;
    }
  return result;
}

// Shuts down the plug-in.  Called after the last plug-in instance is
// destroyed.
NPError
NP_Shutdown(void)
{
  PLUGIN_DEBUG ("NP_Shutdown\n");

  GError *err = NULL;
  gsize bytes_written;

  if (output_to_appletviewer)
    {
      if (g_io_channel_write_chars (output_to_appletviewer, "shutdown", -1,
				    &bytes_written, &err)
	  != G_IO_STATUS_NORMAL)
	{
	  g_printerr ("send_message_to_appletviewer: Error: %s\n",
		      err->message);
	  return NPERR_GENERIC_ERROR;
	}

      if (g_io_channel_flush (output_to_appletviewer,
			      &err) != G_IO_STATUS_NORMAL)
	{
	  g_printerr ("send_message_to_appletviewer: Error: %s\n",
		      err->message);
	  return NPERR_GENERIC_ERROR;
	}

      if (g_io_channel_shutdown (output_to_appletviewer,
				 TRUE, &err) != G_IO_STATUS_NORMAL)
	{
	  g_printerr ("NP_Shutdown: Error: %s\n", err->message);
	  return NPERR_GENERIC_ERROR;
	}
    }

  if (input_from_appletviewer)
    {
      if (g_io_channel_shutdown (input_from_appletviewer,
				 TRUE, &err) != G_IO_STATUS_NORMAL)
	{
	  g_printerr ("NP_Shutdown: Error: %s\n", err->message);
	  return NPERR_GENERIC_ERROR;
	}
    }

  // Delete named pipes.
  gchar *pipename_in = g_strdup_printf ("/tmp/gcjwebplugin-%i-in", getpid ());
  gchar *pipename_out = g_strdup_printf ("/tmp/gcjwebplugin-%i-out", getpid ());

  unlink (pipename_in);
  unlink (pipename_out);

  g_printerr ("Done shutting down.\n");

  return NPERR_NO_ERROR;
}

static gboolean
callbackViewerRead (GIOChannel *source, GIOCondition condition, gpointer data)
{
  // Read pipe data.
  GError *err = NULL;
  gchar* read_buf;
  if (g_io_channel_read_line (input_from_appletviewer,
                              &read_buf, NULL, NULL, &err)
      != G_IO_STATUS_NORMAL)
    {
      g_printerr ("receive_message_from_appletviewer: Error: %s\n", err->message);
    }

  // FIXME: Add handling of read input here.

  return TRUE;
}
