/*
 *  scheduler.c : Routines for managing screens and scheduling them for
 *                being displayed.
 *                This file is part of the FreeLCD package.
 *
 *  $Id: scheduler.c,v 1.5 2004/01/25 16:00:11 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file scheduler.c
 * Routines for managing screens and scheduling them for being displayed.
 */

#include <stdio.h>
#include <errno.h>

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_STRING_H
# if !STDC_HEADERS && HAVE_MEMORY_H
#  include <memory.h>
# endif
# include <string.h>
#endif

#if !HAVE_SNPRINTF
# define snprintf(A,B,C,D)  printf(A,C,D)
#endif

#if HAVE_PTHREAD_H
# include <pthread.h>
#endif

#if HAVE_ASSERT_H
# include <assert.h>
#else
# define assert(FOO)
#endif

#include "scheduler.h"

#include "server/connections.h"
#include "server/drivers.h"
#include "common/xmalloc.h"
#include "common/field_types.h"
#include "common/xmlt.h"
#include "common/debug.h"
#include "renderers/charlayout.h"
#include "renderers/charrenderer.h"
#include "renderers/layout_tags.h"

typedef enum
{
  ERR_NONE = 0, 
  ERR_DATA_INCOMPLETE = 1, 
  ERR_DUPLICATE_NAMES = 2, 
  ERR_UNKNOWN_SCREEN_ID = 3
}
error_code;

static const char* error_description[] =
{
  "SNAFU",
  "not all data fields have a name and type",
  "duplicate data field names",
  "no screen with that ID has been registered yet"
};

typedef enum
{
  VALUE = 100, HRS, MIN, SEC, MSEC, DAY, MON, YEAR
}
data_tag_t;

static data_tag_t data_tags[] =
{ 
  VALUE, HRS, MIN, SEC, MSEC, DAY, MON, YEAR 
};

static attr_t data_attrs[] =
{
  NAME
};

static dict_pair data_tag_pairs[] = 
{
    { "day",   &data_tags[5] },
    { "hrs",   &data_tags[1] },
    { "min",   &data_tags[2] },
    { "mon",   &data_tags[6] },
    { "msec",  &data_tags[4] },
    { "sec",   &data_tags[3] },
    { "value", &data_tags[0] },
    { "year",  &data_tags[7] }
};

static dict_pair data_attr_pairs[] = 
{
    { "name", &data_attrs[0] }
};

static dictionary data_tag_dict =
    { data_tag_pairs, sizeof (data_tag_pairs) / sizeof (dict_pair) };

static dictionary attr_tag_dict =
    { data_attr_pairs, sizeof (data_attr_pairs) / sizeof (dict_pair) };

/* The dictionary for field types is defined in common/field_types.c */
extern dictionary field_type_dict;


/** A screen is identified by this combination of a name, and the network
 * address of the client that submitted it. */
typedef struct
{
  void       *originator;
  const char *name;
}
screen_id;

/** A screen that will be displayed on a device. */
typedef struct
{
  screen_id      id;          /**< A unique identifier. */
  sch_importance importance;  /**< The importance of this screen. */
  int            repetition;  /**< The number of times to display it. */
  int            counter;     /**< The number of times already displayed. */
  cl_layout      *c_layout;   /**< Screen layout for character devices. */
  /* gl_layout      *g_layout; */
  dictionary     field_dict;  /**< Contents of the data fields. */
}
screen;

/** All information associated with a device driver. */
typedef struct
{
  cc_canvas  *c_canvas;      /**< Canvas for character devices. */
  gr_canvas  *g_canvas; 
  const char *device_name;   /**< Name of the device. */
  void       *driver_handle; /**< Driver handle as returned by
                                  drv_create_handle() */
}
sch_driver;

/** All currently registered screens. */
static slist      screenlist;
/** All currently registered drivers. */
static slist      driverlist;

/*-------------------------------------------------------- _free_screen --*/
/** Free all resources of a #screen.
 * \param _scr Opaque pointer to a #screen.
 */
static void
_free_screen (void *_scr)
{
  screen *scr = (screen*)_scr;
  size_t index_;

  free ((char*) scr->id.name);
  
  for (index_ = 0; index_ < scr->field_dict.size; ++index_)
    {
      free ((void*) scr->field_dict.dict[index_].key);
      free (scr->field_dict.dict[index_].value);
    }
  
  if (scr->c_layout)
    cl_free_layout (scr->c_layout);

  free (scr);
}

/*-------------------------------------------------------- _free_driver --*/
/** Free all resources of a #sch_driver.
 * \param _scr Opaque pointer to a #sch_driver.
 */
static void
_free_driver (void *_drv)
{
  sch_driver *drv = (sch_driver*)_drv;
  size_t index_;

  free ((char*) drv->device_name);

  free (drv);
}

/*------------------------------------------------------- _count_fields --*/
/** Increase a counter if a field tag was found in a data definition.
 * \param _data Pointer to a counter.
 * \param node The XML node that will be tested.
 */
static void
_count_fields (void *_data, xml_node *node)
{
  size_t *data = (size_t*)_data;
  
  if (node->tag == FIELD)
    ++*data;
}

/*-------------------------------------------------------- _fill_fields --*/
/** Copy a field's type and name to a dictionary.
 * \param _data Opaque pointer to a pointer to a dictionary entry.
 * \param node XML definition of the data field.
 */
static void
_fill_fields (void *_data, xml_node *node)
{
  dict_pair **data = (dict_pair**)_data;
  
  if (node->tag == FIELD)
    {
      const char *value = xmlt_get_attrib (node, TYPE);
      const char *key   = xmlt_get_attrib (node, NAME);
      void       *lookup; 
      field      *new_field;

      if (value == NULL)
        return;
      
      new_field = xmalloc (sizeof (field));
      memset (new_field, 0, sizeof (field));
      
      /* Unknown data types are silently ignored; hopefully we can remain
       * reasonably forwards and backwards compatible. */
      lookup = dict_lookup (&field_type_dict, value);
      if (lookup == NULL)
        new_field->type = F_NONE;
      else
        new_field->type = *(field_type*)lookup;
      
      (*data)->key = key ? xstrdup (key) : 0;
      (*data)->value = new_field;
      ++*data;
    }
}

static int
_value_from_subnode (xml_node *node, int subnode_tag)
{
  const char *text;
  int temp;
  xml_node *subnode = xmlt_find (node, NULL, subnode_tag);
  
  if (subnode == NULL)
    {
      debug ("  cannot get value from subnode: no subnode found");
      return 0;
    }
  
  text = xmlt_get_first_cdata (subnode);
  if (text == NULL)
    {
      debug ("  cannot get value from subnode: subnode empty");
      return 0;
    }
  
  temp = strtol (text, 0, 10);
  if (errno == ERANGE || errno == EINVAL)
    {
      debug ("  cannot get value from subnode: subnode content out of range");
      return 0;
    }

  return temp;
}

/*--------------------------------------------------------- _parse_data --*/
static void
_parse_data (void *_data, xml_node *node)
{
  dictionary *dict = (dictionary*)_data;

  if (node->tag == VALUE)
    {
      const char *id = xmlt_get_attrib (node, NAME);
      field *lookup;
      
      if (id == NULL)
        return;
      
      lookup= (field *)dict_lookup (dict, id);
      
      if (lookup == NULL)
        return;
          
      lookup->defined = 1;
      lookup->changed = 0;
      lookup->valid = 1;
      
      switch (lookup->type)
        {
        case F_NONE:
          debug_2 ("  %s -> illegal", id);
          lookup->valid = 0;
          return;

        case F_LABEL:
        case F_SCROLLTEXT:
        case F_HEADER:
          {
            const char *text = xmlt_get_first_cdata (node);

            if (text == NULL)
              {
                free (lookup->data.text);
                lookup->data.text = NULL;
                lookup->valid = 0;
              }
            else
              {
                debug_3 ("  %s -> %s", id, text);
                if (   lookup->data.text == NULL 
                    || strcmp (lookup->data.text, text))
                  {
                    debug ("  (change)");
                    lookup->changed = 1;
                    free (lookup->data.text);
                    lookup->data.text = xstrdup (text);
                  }
              }
          }
          break;

        case F_TIME:
          lookup->data.time.hours = _value_from_subnode (node, HRS);
          lookup->data.time.minutes = _value_from_subnode (node, MIN);
          lookup->data.time.seconds = _value_from_subnode (node, SEC);
          lookup->data.time.milliseconds = _value_from_subnode (node, MSEC);
          debug_4 ("  %s -> %i:%i", id, lookup->data.time.hours,
                   lookup->data.time.minutes);
          break;
          
        case F_TIMESPAN:
          lookup->data.timespan.seconds = _value_from_subnode (node, SEC);
          lookup->data.timespan.milliseconds = _value_from_subnode (node, MSEC);
          debug_4 ("  %s -> %i.%i", id, lookup->data.timespan.seconds,
                 lookup->data.timespan.milliseconds);
          break;
          
        case F_DATE:
          lookup->data.date.day = _value_from_subnode (node, DAY);
          lookup->data.date.month = _value_from_subnode (node, MON);
          lookup->data.date.year = _value_from_subnode (node, YEAR);
          debug_4 ("  %s -> %i-%i", id, lookup->data.date.day,
                 lookup->data.date.month);
          break;
          
        case F_SCALAR:
          {
            char *endptr;
            const char *text = xmlt_get_first_cdata (node);
            long value;
              
            lookup->valid = 0;
              
            if (text != NULL && *text != '\0')
              {
                debug_3 ("  %s -> %s", id, text);
                value = strtol (xmlt_get_first_cdata (node), &endptr, 10);

                if (endptr != NULL && *endptr == '\0')
                  {
                    if (lookup->data.scalar != value)
                      {
                        debug ("  (changed)");
                        lookup->changed = 1;
                        lookup->data.scalar = value;
                      }
                  }
              }
          }
        break;
        
        case F_PERCENTAGE:
          {
            char *endptr;
            const char *text = xmlt_get_first_cdata (node);
            double value;
            
            lookup->valid = 0;
            
            if (text != NULL && *text != '\0')
              {
                value = strtod (xmlt_get_first_cdata (node), &endptr);

                if (endptr != NULL && *endptr == '\0')
                  {
                    if (lookup->data.percentage != value)
                      {
                        lookup->changed = 1;
                        lookup->data.percentage = value;
                      }
                  }
              }
          }

          break;

        case F_HISTOGRAM:
          /* FIXME: implement */
          break;

        case F_PIXMAP:
          /* FIXME: implement */
          break;

        default:
          assert (0);
            
        }
    }    
}

/*-------------------------------------------------- _screen_id_compare --*/
static int
_screen_id_compare (const void *_scrn, const void *_id)
{
  const screen    *scrn = (const screen *)_scrn;
  const screen_id *id   = (const screen_id *)_id;

  assert (scrn != NULL);
  assert (id != NULL);

  return    scrn->id.originator == id->originator 
         && !strcmp (scrn->id.name, id->name);
}

/*------------------------------------------------ _driver_name_compare --*/
static int
_driver_name_compare (const void *_drv, const void *_name)
{
  const sch_driver *drv  = (const sch_driver *)_drv;
  const char       *name = (const char *)_name;

  assert (drv != NULL);
  assert (drv->device_name != NULL);
  assert (name != NULL);

  return !strcmp (drv->device_name, name);
}

/*----------------------------------------------------- _update_screens --*/
static void
_update_screens (screen_id *id, sch_importance importance, 
                 int repetition, xml_node *layout)
{
  /** \todo Implement refreshing screen data */
  (void)id;
  (void)importance;
  (void)repetition;
  (void)layout;
}

/*-------------------------------------------------------- _reply_error --*/
static void
_reply_error (void *destination, error_code code)
{
  char   temp[256];
  size_t len;

/*
 * FIXME: snprintf only available in ISO-C99? Uses trusted input anyway,
 *        worth the trouble?
 
  len = snprintf (temp, 256, "ERROR %i : %s\n", 
                  (int)code, error_description[code]);
*/
  
  len = sprintf (temp, "ERROR %i : %s\n", 
                 (int)code, error_description[code]);

  if (len > 0 && len <= 256)
    conn_reply (destination, temp, len);
}

/*------------------------------------------------------------ sch_init --*/
void
sch_init (void)
{
  slist_init (&driverlist);
  slist_init (&screenlist);
}
          
/*------------------------------------------------------ sch_add_screen --*/
void
sch_add_screen (void *originator, const char *name, sch_importance importance,
                int repetition, xml_node *layout)
{
  screen_id    id;
  slist_iter   found;
  dict_pair    *current_pair;
  size_t       i;
  size_t       dictsize;
  
  debug_2 ("add screen %s", name);

  id.originator = originator;
  id.name = name;

  found = slist_find_if (&screenlist, &id, _screen_id_compare);

  if (slist_iter_at_end (found))
    {
      /* This is a new screen. */
      sch_driver     *driver = slist_first (&driverlist);
      screen         *scr    = xmalloc (sizeof (screen));
      drv_dimensions size;

      assert (driver);
      size = drivers_get_dimensions (driver->driver_handle);

      scr->id.originator   = originator;
      scr->id.name         = xstrdup (name);
      scr->importance      = importance;
      scr->repetition      = repetition;
      scr->counter         = 0;
      scr->field_dict.size = 0;

      if (drivers_get_type (driver->driver_handle) == DRV_CHARACTER)
        scr->c_layout     = cl_make_layout (layout, size.width, size.height);
      else
        scr->c_layout     = NULL; /* FIXME : Add check for graphical layout. */

      /* The xx_make_layout function not only creates a layout struct, but
       * also parses the XML. We can use these parsed parts to construct
       * a mapping between field names and their information. First the
       * fields are counted so a correctly sized block can be allocated,
       * and then the dictionary is created. */
      scr->field_dict.size = 0;
      xmlt_for_each (layout, &scr->field_dict.size, _count_fields);

      scr->field_dict.dict = xmalloc (scr->field_dict.size
                                      * sizeof (dict_pair));
      
      current_pair = scr->field_dict.dict;
      xmlt_for_each (layout, &current_pair, _fill_fields);
      
      /* Extra check before sorting: all fields must be provided, and
       * should not be empty. */
      current_pair = scr->field_dict.dict;
      for (i = 0; i < scr->field_dict.size; ++i, ++current_pair)
        {
          if (   current_pair->key == NULL || *(current_pair->key) == '\0'
              || current_pair->value == NULL )
            {
              _reply_error (originator, ERR_DATA_INCOMPLETE);
              _free_screen (scr); 

              return;
            }
        }

      qsort (scr->field_dict.dict, scr->field_dict.size, sizeof (dict_pair),
             dict_pair_compare);

      /* Last check before it's accepted: duplicate names. */
      current_pair = scr->field_dict.dict;
      dictsize = scr->field_dict.size - 1;
      
      for (i = 0; i < dictsize; ++i, ++current_pair)
        {
          if (!strcmp (current_pair->key, (current_pair + 1)->key))
            {
              _reply_error (originator, ERR_DUPLICATE_NAMES);
              _free_screen (scr);
              return;
            }
        }

      /* All is well, add it to the list of screens. */
      slist_append (&screenlist, scr);
    }
  else
    {
      /* This screen was already available in the screen list. */
      screen *scr = (screen *)slist_at_iter (found);
      
      if (importance != scr->importance || repetition != scr->repetition)
        _update_screens (&id, importance, repetition, layout);
    }
}

/*--------------------------------------------------- sch_remove_screen --*/
int
sch_remove_screen (void *originator, const char *name)
{
  slist_iter found;
  screen_id  id;

  debug_2 ("remove screen %s", name);

  id.originator = originator;
  id.name = name;
  
  found = slist_find_if (&screenlist, &id, _screen_id_compare);

  if (slist_iter_at_end (found))
    return 0;

  _free_screen ((screen*) slist_remove_at_iter (&screenlist, found));
  
  return 1;
}

/*------------------------------------------------- sch_register_driver --*/
int
sch_register_driver (void *driver_handle, const char *device_name)
{
  sch_driver *new_driver;
  slist_iter found;

  debug_2 ("sch_register_driver %s", device_name);
  
  found = slist_find_if (&driverlist, device_name, _driver_name_compare);
  if (!slist_iter_at_end (found))
    return 0;

  new_driver = xmalloc (sizeof (sch_driver));
  new_driver->driver_handle = driver_handle;
  new_driver->device_name = xstrdup (device_name);
  
  if (drivers_get_type (driver_handle) == DRV_CHARACTER)
    {
      drv_dimensions dim = drivers_get_dimensions (driver_handle);
      new_driver->c_canvas = cc_create_canvas (dim.width, dim.height);
      new_driver->g_canvas = NULL;
    }
  else
    {
      new_driver->c_canvas = NULL;
      /** \todo Initialize graphical canvas */
      new_driver->g_canvas = NULL;
    }
  
  slist_append (&driverlist, new_driver);  

  return 1;
}

/*----------------------------------------------------- sch_update_data --*/
void
sch_update_data (void *originator, const char *name, xml_node *data)
{
  screen* this_screen;
  screen_id search_id;
  slist_iter this_screen_iter;
  sch_driver *driver;  

  debug_2 ("sch_update_data for %s", name);
  
  search_id.originator = originator;
  search_id.name = name;

  this_screen_iter = 
              slist_find_if (&screenlist, &search_id, _screen_id_compare);

  if (slist_iter_at_end (this_screen_iter))
    {
      debug ("  unknown screen ID");
      _reply_error (originator, ERR_UNKNOWN_SCREEN_ID);
      return;
    }
  
  this_screen = (screen*)slist_at_iter (this_screen_iter);

  if (this_screen == NULL)
    {
      debug ("  unknown screen");
      return;
    }
  
  /* Copy the new values from the XML data into the screen's data fields. */
  xmlt_rescan_document (data, &data_tag_dict, &attr_tag_dict);
  xmlt_for_each (data, &(this_screen->field_dict), _parse_data);
 
  /** \todo Implement driver system. For testing purposes, we always use the
   * first driver in the list. */
  driver = (sch_driver *) slist_first (&driverlist);
  assert (driver != NULL);
  
  /* If anything changed, update the LCD. */
  if (drivers_get_type (driver->driver_handle) == DRV_CHARACTER)
    {
       cr_render (this_screen->c_layout, 
                  &this_screen->field_dict, driver->c_canvas);

       drivers_process_canvas (driver->driver_handle, driver->c_canvas);
    }
  else
    {
      /** \todo Graphical rendering. */
    }
}

/*--------------------------------------------------------- sch_cleanup --*/
void
sch_cleanup ()
{
  slist_delete_special (&driverlist, _free_driver);
  slist_delete_special (&screenlist, _free_screen);
}

