/*
 *  charlayout.c - Character-based layout algorithm.
 *                 This file is part of the FreeLCD package.
 *  
 *  $Id: charlayout.c,v 1.6 2004/01/25 15:52:17 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file charlayout.c 
 * Layout algorithm, optimized for character displays.
 ** \todo Implement aligment and expansion of widgets.
 */

#if HAVE_CONFIG_H
# include "config.h"
#endif

#if HAVE_STRING_H
# include <string.h>
#endif

#if HAVE_ASSERT_H
# include <assert.h>
#else
# define assert(FOO)
#endif

#include "common/xmalloc.h"
#include "common/field_types.h"
#include "common/debug.h"
#include "charrenderer.h"
#include "layout_tags.h"

/*------------------------------------------------------------------------*/
/* XML dictionaries */
 
static tag_t cl_tagindex_[] =
{
  ROW, COL, FIELD, LABEL
};

static dict_pair cl_tags_d[] = 
{
  { "col",   &(cl_tagindex_[1]) },
  { "field", &(cl_tagindex_[2]) },
  { "label", &(cl_tagindex_[3]) },
  { "row",   &(cl_tagindex_[0]) }
};  

static dictionary cd_tags = 
                       { cl_tags_d, sizeof (cl_tags_d) / sizeof (dict_pair) };


static attr_t cl_attrindex_[] =
{
  TYPE, ID, NAME, VALIGN, HALIGN, EXPAND
};

static dict_pair cl_attr_d[] =
{
  { "expand", &(cl_attrindex_[5])  },
  { "halign", &(cl_attrindex_[4])  },
  { "id",     &(cl_attrindex_[1])  },
  { "name",   &(cl_attrindex_[2])  },
  { "type",   &(cl_attrindex_[0])  },
  { "valign", &(cl_attrindex_[3])  }
};

static dictionary cd_attrs = 
                     { cl_attr_d, sizeof (cl_attr_d) / sizeof (dict_pair) };

extern dictionary field_type_dict;


/*------------------------------------------------------------------------
 * Layout data structure type definitions
 *------------------------------------------------------------------------
 */

/** Horizontal widget alignment. */
typedef enum
{
  ALIGN_LEFT, ALIGN_RIGHT, ALIGN_CENTERED
}
align_hor_t;

/** Vertical widget alignment. */
typedef enum
{
  ALIGN_TOP, ALIGN_BOTTOM, ALIGN_MIDDLE
}
align_ver_t;

/*------------------------------------------------------------------------
 * Widget data structure                                                  
 *------------------------------------------------------------------------
 */

/** Widget types, stored in #cl_widget. */
typedef enum
{
  CROW,     /**< Row layout widget. */
  CCOL,     /**< Column layout widget. */
  CFIELD,   /**< Data field. */
  CLABEL    /**< Text label. */
} 
widget_class_t;

/** Size hints for the layout algorithm.
 * The width and height correspond to a penalty, or ugliness. When
 * allocating space to widgets, the layout algorithm attempts to
 * keep the total penalty at a minimum. Or, if it doesn't fit, even
 * with the maximum penalty, the layout is split into smaller parts
 * and the procedure starts all over again. In this case, one layout
 * takes up several screen cycles on the display. 
 */
typedef struct _widget_size_hint
{
  int width;  /**< Width in characters. */
  int height; /**< Height in characters. */
  int penalty; /**< Penalty for these dimensions. */
  
  /** Child layout cache.
   * The \a children_layout field is used to cache the optimum layout
   * of the children for a particular dimension. */
  struct _widget_size_hint **children_layout;
}
widget_size_hint;

/** A widget represents one element of a screen, before it is rendered
 * to a canvas. They can be broadly
 * grouped into layout and content widgets. Layout widgets are rows and
 * columns: they can contain other widgets, and will take care of their
 * layout. Content widges are actually visible on the screen. Their
 * size and position are detemined by the hierarchy of rows and columns
 * above them. 
 */
typedef struct
{
  widget_class_t type; /**< Widget type. */
  field_type subtype;  /**< Widget subtype, only defined for data fields. */
  slist children;      /**< Child widgets, contains #cl_widget pointers. */
  slist layout_hints;  /**< Contains \ref widget_size_hint pointers. */
    
  /** Horizontal expansion factor.
   * After fitting all widgets with their suggested size,
   * the leftover space is distributed along the widgets according to their
   * expansion factor. Zero means that the widget will always be allocated
   * its minimal size. */
  int expand_x;
  int expand_y; /**< Vertical expansion factor. */

  /** Horizontal alignment.
   * If there is still some space left for widgets to move around after
   * expanding them, they will be packed inside the available space at
   * a position determined by \ref align_hor and \ref align_ver. */
  align_hor_t align_hor;
  align_ver_t align_ver;  /**< Vertical alignment. */

  /** Generic data field, the exact use depends on the #type field. */
  void *data;
}
cl_widget;


/** An iterator and the list it belongs to. */
typedef struct
{
  slist      *list;
  slist_iter iter;
}
list_iter_pair;

static
void _widget_calc_hints (void *widget_, void *dummy);

/*-------------------------------------------------------- _widget_init --*/
/** Initialise a widget with zero sizes and empty lists.
 * \param widget Pointer to the widget to initialise.
 * \param type Type of widget. */
static void
_widget_init (cl_widget *widget, widget_class_t type)
{
  widget->type = type;
  slist_init (&widget->children);
  slist_init (&widget->layout_hints);
  widget->expand_x = 0;
  widget->expand_y = 0;
  widget->data = NULL;
}

/*------------------------------------------------------- _hint_compare --*/

/** Compare two layout hints to see if they specify the same width and height.
 * \param _data Pointer to one widget.
 * \param _compare Pointer to the other widget.
 * \return Zero if \a _data and \a _compare are not equal, nonzero if equal. */
static int
_compare_hints (const void* _data, const void* _compare)
{
  widget_size_hint *data    = (widget_size_hint*)_data;
  widget_size_hint *compare = (widget_size_hint*)_compare;

  return (data->width == compare->width && data->height == compare->height);
}

/*---------------------------------------------------------- _hint_init --*/
/** Initialise a layout hint.
 * \param hint Pointer to the hint to initialise. */
static void
_hint_init (widget_size_hint *hint)
{
  hint->width = 0;
  hint->height = 0;
  hint->penalty = 0;
  hint->children_layout = 0;
}

/*----------------------------------------------------------- _hint_dup --*/
/** Make a copy of a layout hint.
 * \param hint Pointer to the hint to copy.
 * \return Pointer to allocated memory that holds a copy of \ref hint. */
static widget_size_hint *
_hint_dup (widget_size_hint *hint)
{
  widget_size_hint *new_hint = xmalloc (sizeof (widget_size_hint));

  new_hint->width = hint->width;
  new_hint->height = hint->height;
  new_hint->penalty = hint->penalty;
  new_hint->children_layout = 0;
  
  return new_hint;
}

/*------------------------------------------------ _layout_strategy_row --*/
static void
_layout_strategy_row (widget_size_hint *hint, widget_size_hint *new_hint)
{
  assert (hint != NULL);
  assert (new_hint != NULL);

  if (hint->height > new_hint->height) 
    new_hint->height = hint->height;
  
  new_hint->width += hint->width;
}

/*------------------------------------------------ _layout_strategy_col --*/
static void
_layout_strategy_col (widget_size_hint *hint, widget_size_hint *new_hint)
{
  assert (hint != NULL);
  assert (new_hint != NULL);

  if (hint->width > new_hint->width) 
    new_hint->width = hint->width;
  
  new_hint->height += hint->height;
}

/*---------------------------------------------- _widget_calc_penalties --*/
static void
_widget_calc_penalties (cl_widget *widget, 
                        void(*strategy)(widget_size_hint*,
                                        widget_size_hint*))
{
  slist_iter     iter;
  list_iter_pair *pairs;
  int            index_; /* "index" was already taken by string.h */
  int            array_size;
  
  assert (widget != NULL);
  assert (strategy != NULL);

  /* All children are instructed to gather their own layout hints. */
  slist_for_each (&widget->children, _widget_calc_hints, 0);

  /* Reserve enough memory for storing the iterators to the children's
   * lists of layout hints, and store them in the pairs array. */
  array_size = slist_length (&widget->children);
  pairs = xmalloc (sizeof (list_iter_pair) * array_size);
  
  iter = slist_begin_iter (&widget->children);
  index_ = 0;
  while (!slist_iter_at_end (iter))
    {
      cl_widget *child = (cl_widget*)slist_iter_and_next (&iter);
      pairs[index_].list = &child->layout_hints;
      pairs[index_].iter = slist_begin_iter (&child->layout_hints);
      ++index_;
    }

  assert (index_ == array_size);

  /* Loop until we have exhausted all layout combinations. */
  for (;;) 
    {
      widget_size_hint new_hint;
      slist_iter found;

      _hint_init (&new_hint);
      
      /* Try the current combination and see what turns up. */
      for (index_ = 0; index_ < array_size; ++index_)
        {
          widget_size_hint *hint;
          hint = (widget_size_hint*)slist_at_iter (pairs[index_].iter);

          if (hint != NULL)
            {
              strategy (hint, &new_hint);
              new_hint.penalty += hint->penalty;
            }
        }

      /* See if the final dimensions were already in the list. */
      found = slist_find_if (&widget->layout_hints, 
                             &new_hint, _compare_hints);

      if (slist_iter_at_end (found))
        {
          /* Not in the list yet. Create a new layout hint and add it to
           * the widget. */

          widget_size_hint *copy = _hint_dup (&new_hint);
          widget_size_hint **array;

          array = xmalloc (sizeof (widget_size_hint*) * array_size);
          for (index_ = 0; index_ < array_size; ++index_)
            array[index_] = slist_at_iter (pairs[index_].iter);
          
          copy->children_layout = array;
          slist_append (&widget->layout_hints, copy);
        }
      else
        {
          /* It's already there. If the penalty of the new hint is less than
           * this one, replace it. */
          
          widget_size_hint *found_hint = slist_at_iter (found);

          if (found_hint->penalty > new_hint.penalty)
            {
              widget_size_hint **array = found_hint->children_layout;

              found_hint->penalty = new_hint.penalty;
              for (index_ = 0; index_ < array_size; ++index_)
                array[index_] = slist_at_iter (pairs[index_].iter);
            }
        }
      
      /* Find the next combination of children layouts. */
      for (index_ = 0; index_ < array_size; ++index_)
        {
          slist_iter_and_next (&pairs[index_].iter);
          
          if (slist_iter_at_end (pairs[index_].iter))
              pairs[index_].iter = slist_begin_iter (pairs[index_].list);
          else
              break;
        }

      /* Exit the loop if the possibilities are exhausted. */
      if (index_ == array_size)
        break;

    } /* for (;;) */
}

static void
_add_layout_hint (slist *list, int width, int height, int penalty)
{
  widget_size_hint *new_hint = xmalloc (sizeof (widget_size_hint));

  new_hint->width   = width;
  new_hint->height  = height;
  new_hint->penalty = penalty;
  slist_append (list, new_hint);
}

/*-------------------------------------------------- _widget_calc_hints --*/
static void
_widget_calc_hints (void *widget_, void *dummy)
{
  cl_widget *widget = (cl_widget*)widget_;
  slist     *hintlist = &widget->layout_hints;
  (void)dummy;
  
  switch (widget->type)
    {
    case CROW:
      _widget_calc_penalties (widget, _layout_strategy_row);
      break;

    case CCOL:
      _widget_calc_penalties (widget, _layout_strategy_col);
      break;

    case CFIELD:
      switch (widget->subtype)
        {
          case F_LABEL:
              _add_layout_hint (hintlist, strlen ((char*)widget->data), 1, 0);
              /* Maybe handle labels here. */
              break;
          
          case F_SCROLLTEXT:
              _add_layout_hint (hintlist, 20, 1, 0);
              _add_layout_hint (hintlist, 10, 1, 2);
              _add_layout_hint (hintlist, 5, 1, 4);
              _add_layout_hint (hintlist, 1, 1, 10);
              break;

          case F_TIME:
              /* Format: 00:00:00.00 */
              _add_layout_hint (hintlist, 11, 1, 0);
              /* Format: 00:00:00 */
              _add_layout_hint (hintlist, 8, 1, 1);
              /* Format: 00:00 */
              _add_layout_hint (hintlist, 5, 1, 4);
              /* Format: 00 */
              _add_layout_hint (hintlist, 2, 1, 10);
              break;

          case F_DATE:
              /* Format: 01 jan 2004 */
              _add_layout_hint (hintlist, 11, 1, 0);
              /* Format: 01 jan\2004 */
              _add_layout_hint (hintlist, 6, 2, 2);
              /* Format: 01-01-2004 */
              _add_layout_hint (hintlist, 10, 1, 1);
              /* Format: 01-01-04 */
              _add_layout_hint (hintlist, 8, 1, 5);
              /* Format: 01 jan */
              _add_layout_hint (hintlist, 6, 1, 6);
              /* Format: 01-01 */
              _add_layout_hint (hintlist, 5, 1, 7);
              /* Format: 01 */
              _add_layout_hint (hintlist, 2, 1, 10);
              break;
              
          case F_SCALAR:
              _add_layout_hint (hintlist, 10, 1, 0);
              _add_layout_hint (hintlist, 8, 1, 2);
              _add_layout_hint (hintlist, 4, 1, 3);
              break;
              
          default:
              debug_2 ("   illegal field %i", widget->subtype);
              assert (0);
        }
      break;

    case CLABEL:
      _add_layout_hint (hintlist, strlen ((char*)widget->data), 1, 0);
      break;

    default:
      assert (0);
    }
}

/*---------------------------------------------- _widget_allocate_space --*/
static int
_widget_allocate_space (cl_widget *widget, int w, int h, int x, int y, 
                        cl_layout* layout)
{
  widget_size_hint *best_hint = NULL;
  unsigned int index_;
  
  slist_iter iter = slist_begin_iter (&widget->layout_hints);
  while (!slist_iter_at_end (iter))
    {
      widget_size_hint *hint = slist_iter_and_next (&iter);
      if (hint->width > w || hint->height > h)
        continue;
      
      if (best_hint == NULL || best_hint->penalty > hint->penalty)
        best_hint = hint;
    }

  if (best_hint == NULL)
    return 0;
  
  index_ = 0;
  iter = slist_begin_iter (&widget->children);
  while (!slist_iter_at_end (iter))
    {
      cl_widget* child = slist_iter_and_next (&iter);
      widget_size_hint *best_layout = best_hint->children_layout[index_];
      assert (best_layout != NULL);
      _widget_allocate_space (child, best_layout->width, best_layout->height,
                              x, y, layout);

      switch (widget->type)
        {
        case CROW:
          x += best_layout->width; 
          break;
          
        case CCOL:
          y += best_layout->height; 
          break;
          
        case CFIELD:
        case CLABEL:
          break;
        }
      
      ++index_;
    }

  if (widget->type != CROW && widget->type != CCOL)
    {
      cl_layout_elem *elem = layout->layout_data + layout->layout_data_size++;
      elem->x = x;
      elem->y = y;
      elem->width = w;
      elem->height = h;
      elem->utf8_data = strdup (widget->data);
      elem->is_label = (widget->type == CLABEL);
    }
  
  return 1;
}
 
/*------------------------------------------------- _is_layout_or_field --*/
static int
_is_layout_or_field (xml_node *node)
{
  int tag = node->tag;
  return tag == ROW || tag == COL || tag == FIELD;
}

/*------------------------------------------------------- _parse_widget --*/
static cl_widget *
_parse_widget (xml_node *node)
{
  cl_widget   *result = xmalloc (sizeof (cl_widget));
  xml_node    *iter   = NULL;
  const char  *data   = NULL;
  const char  *type   = NULL;
  field_type  ftype;

  switch (node->tag)
    {
    case ROW:
      _widget_init (result, CROW); 
      break;
      
    case COL:
      _widget_init (result, CCOL); 
      break;
      
    case FIELD:
      data = xmlt_get_attrib (node, NAME);
      type = xmlt_get_attrib (node, TYPE);

      if (data == NULL || type == NULL)
        return NULL;

      ftype = *((field_type*) dict_lookup (&field_type_dict, type));
      _widget_init (result, CFIELD); 

      result->subtype = ftype;
      result->data = strdup (data);
      result->expand_x = 1;
      result->expand_y = 0;
      break;

    case LABEL:
      data = xmlt_get_first_cdata (node);

      if (data == NULL)
        return NULL;

      _widget_init (result, CLABEL); 
      result->data = strdup (data);
      result->expand_x = 0;
      result->expand_y = 0;
      break;

    default:
      return NULL;
    }

  if (node->tag == ROW || node->tag == COL)
    {
      while ((iter = xmlt_find_if (node, iter, _is_layout_or_field)) != NULL)
        {
          void *child = (void *)_parse_widget (iter);

          if (child == NULL)
            return NULL;

          slist_append (&result->children, child);
        }
    }

  return result; 
}

/*------------------------------------------------------ cl_make_layout --*/
cl_layout *
cl_make_layout (xml_node *layout, int width, int height)
{
  cl_widget *widgets = 0;
  cl_layout *result = xmalloc (sizeof (cl_layout)); 
  xml_node  *node = 0;
  int       blocksize = sizeof (cl_layout_elem) * width * height;
  
  assert (layout);
  assert (width);
  assert (height);
  
  debug_3 ("Create new layout, size %i x %i", width, height);

  result->layout_data_size = 0;
  result->layout_data = xmalloc (blocksize);

  /* The XML data is not scanned for our namespace yet. */
  xmlt_rescan_document (layout, &cd_tags, &cd_attrs);

  /* Make a preliminary layout */
  node = xmlt_find_if (layout, node, _is_layout_or_field);
  if (node == NULL)
    return 0;

  widgets = _parse_widget (node);
  _widget_calc_hints (widgets, 0);
  if (!_widget_allocate_space (widgets, width, height, 0, 0, result))
    {
      /** \todo Handle the case where a layout cannot be fitted
       * inside the dimensions of its canvas, and it needs to be split
       * horizontally or vertically. */
    }
  
  return result;
}

/*------------------------------------------------------ cl_free_layout --*/
void
cl_free_layout (cl_layout *layout)
{
  size_t i;

  assert (layout != NULL);
  
  for (i = 0; i < layout->layout_data_size; ++i)
    free (layout->layout_data[i].utf8_data);
    
  free (layout->layout_data);
  free (layout);
}
