/*
 *  naive.c - A very naive driver. It simply sends the contents of the canvas
 *            to the backend. It does not translate unicode characters outside
 *            the ASCII range. This works just fine for the ncurses backend,
 *            and for some displays as well (although you'll miss a lot of
 *            the display's functionality this way). This file is the right
 *            place to start for writing your own driver.
 *            This file is part of the FreeLCD package.
 *  
 *  $Id: naive.c,v 1.7 2004/01/25 15:50:22 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */


#include <stdlib.h>
#include <string.h>

#include "driver.h"
#include "common/backend.h"
#include "common/xmalloc.h"

typedef enum
{
  WIDTH = 100, HEIGHT
}
tag_enum;

static tag_enum tag[] = { WIDTH, HEIGHT };

static dict_pair tag_dict[] = 
{
  { "height",   &tag[1] },
  { "width",    &tag[0] }
};

static dictionary tags = 
    { tag_dict, sizeof (tag_dict) / sizeof (dict_pair) };


/** This struct is internally used by the naive driver. */ 
typedef struct
{
  drv_dimensions size;
  void           *backend;
}
naive_h;

/*-------------------------------------------------------- drv_get_info --*/
const char *
drv_get_info (const char* field)
{
  if (!strcmp(field, "name"))
    return "naive";

  if (!strcmp(field, "description"))
    return "This driver copies its character buffer directly to the backend.";

  if (!strcmp(field, "version"))
    return "1";

  return 0;
}

/*---------------------------------------------------- drv_create_handle --*/
void *
drv_create_handle (xml_node* config)
{
  naive_h *h;
  const char *num;
  xml_node *node;
  
  (void)config;

  h = xmalloc (sizeof (naive_h));

  xmlt_rescan_document (config, &tags, 0);
  num = NULL;
  node = xmlt_find (config, 0, WIDTH);
  if (node != NULL)
    num = xmlt_get_first_cdata (node);
      
  if (num != NULL)
    h->size.width = atoi (num);
  else
    h->size.width = 20;

  if (h->size.width < 1)
    {
      free (h);
      return NULL;
    }

  num = NULL;
  node = xmlt_find (config, 0, HEIGHT);
  if (node != NULL)
    num = xmlt_get_first_cdata (node);
      
  if (num != NULL)
    h->size.height = atoi (num);
  else
    h->size.height = 4;

  if (h->size.height < 1)
    {
      free (h);
      return NULL;
    }

  h->backend = 0;
  
  return h;
}

/*--------------------------------------------------- drv_destroy_handle --*/
void
drv_destroy_handle (void *handle)
{
  naive_h *h = (naive_h*)handle;
  if (h->backend)
    backend_free_handle (h->backend);
  
  free (h);
}

/*----------------------------------------------------- drv_bind_backend --*/
void
drv_bind_backend (void *handle, void* backend)
{
  naive_h *h = (naive_h*)handle;
  h->backend = backend;
}

/*--------------------------------------------------------- drv_get_type --*/
drv_type
drv_get_type ()
{
  return DRV_CHARACTER;
}

drv_dimensions
drv_get_dimensions (void *handle)
{
  naive_h *h = (naive_h*)handle;

  return h->size;
}

/*--------------------------------------------------- drv_process_canvas --*/
void
drv_process_canvas (void *handle, void *_canvas)
{
  cc_canvas *canvas = (cc_canvas *)_canvas;
  naive_h *h = (naive_h*)handle;
  unsigned int size = canvas->width * canvas->height;
  unsigned int i;

  for (i = 0; i < size; ++i)
    backend_send (h->backend, (char*)&(canvas->elements[i].c),1);
}

/*------------------------------------------------------ drv_get_backend --*/
void *
drv_get_backend (void *handle)
{
  naive_h *h = (naive_h*)handle;

  return h->backend;
}
