/*
 *  driver.h - Interface definition for driver modules.
 *             This file is part of the FreeLCD package.
 *
 *  $Id: driver.h,v 1.6 2004/01/16 23:09:20 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file driver.h
 * The standard interface for display device drivers.
 */


#ifndef _FLCDD_DRIVER_H
#define _FLCDD_DRIVER_H

#include "renderers/charcanvas.h"
#include "common/xmlt.h"

/** Indicates the type of device: character-oriented, or graphical. */
typedef enum
{
  DRV_CHARACTER, DRV_GRAPHICAL
}
drv_type;

/** The dimensions of a device's screen, measured in characters or pixels,
 *  depending on what type it is.
 * \see #drv_type
 */
typedef struct
{
  int width;
  int height;
}
drv_dimensions;
    
/** Make an instance of a driver with a given configuration. For example,
 * you can have one handle to a Matrix Orbital display on ttyS0, and one
 * on ttyS1. These are two instances of the same driver, but with different
 * configurations.
 * \param config The configuration data the driver should use.
 * \return A driver handle that can be passed to one of the other functions.
 */
void*
drv_create_handle (xml_node *config);

/** Release all resources used by a driver.
 * \param handle Driver handle. After calling this function, this handle
 *        cannot be used anymore.
 */
void
drv_destroy_handle (void *handle);

/** Bind a driver handle to a backend (e.g. the serial port). This must
 * be done before the driver can do anything useful.
 * \param handle Driver handle.
 * \param backend Backend handle (\see create_backend() )
 */
void
drv_bind_backend (void *handle, void *backend); 

/** Render the contents of a canvas to the device.
 * \param handle Driver handle.
 * \param canvas Pointer to a canvas. This is either a cc_canvas or a
 *        gc_canvas, depending on the type of device.
 */
void
drv_process_canvas (void *handle, void *canvas);

/** Find out what type of display this driver can control.
 * \return A display type.
 */
drv_type
drv_get_type (void);

/**  Get the dimensions of the connected device. For character-oriented
 * displays, the sizes are given in characters. For graphical displays
 * they are in pixels. 
 * \handle Driver handle.
 * \return The size of the screen of the connected device.
 */
drv_dimensions
drv_get_dimensions (void *handle);

/**  Get information about this driver. The returned char pointer might
 *  need casting to another type, if the field does not contain character
 *  data. All drivers must support at least the following fields:
 *  <ul>
 *   <li>"name", a short descriptive name</li> 
 *   <li>"version", a standard version string, e.g. '1.2'.</li>
 *  </ul>
 *
 * \param field A string, indicating what bit of information is queried.
 * \return A string with the requested information, or NULL if the requested
 *         information is not available.
 */
const char*
drv_get_info (const char *field);

/** Get the backend that is currently bound to the given handle.
 * \param handle Driver handle.
 * \return The handle of the backend.
 */
void*
drv_get_backend (void *handle);

#endif /* Multiple include guard */
