/*
 *  cfontz.c - A driver for CrystalFontz series 632 and 634 devices.
 *             http://www.crystalfontz.com/
 *             This file is part of the FreeLCD package.
 *  
 *  $Id: cfontz.c,v 1.2 2004/01/25 15:49:21 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */


#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>

#include "driver.h"
#include "common/backend.h"

/* Control codes */
#define CF_CURSOR_HOME      1
#define CF_HIDE_DISPLAY     2
#define CF_RESTORE_DISPLAY  3
#define CF_HIDE_CURSOR      4
#define CF_UNDERLINE_CURSOR 5
#define CF_BLOCK_CURSOR     6
#define CF_INVERTING_CURSOR 7
#define CF_BACKSPACE        8
#define CF_BOOT_SCREEN_CTRL 9
#define CF_LINE_FEED        10
#define CF_DELETE           11
#define CF_FORM_FEED        12
#define CF_CARRIAGE_RETURN  13
#define CF_BACKLIGHT_CTRL   14
#define CF_CONTRAST_CTRL    15
#define CF_GOTO_POSITION    17
#define CF_WRAP_ON          23
#define CF_WRAP_OFF         24
#define CF_SET_CUSTOM_CHAR  25
#define CF_REBOOT           26
#define CF_ESCAPE           27
#define CF_DUMP_DATA        30
#define CF_SHOW_INFO        31

#define CF_CUSTOM_CHAR_OFFSET   128
#define CF_UNUSED_CUSTOM_CHAR   0

#include "cfontz-tbl.h"

typedef enum
{
  CF_VERSION_1, CF_VERSION_2
}
firmware_enum;

typedef struct
{
  drv_dimensions  size;             /*<< Display dimensions */
  firmware_enum   firmware;         /*<< Firmware revision (1.x or 2.0) */
  /** Handle to the backend, which is probably a serial or usb port */
  void            *backend;
  cfz_translated  *xtab;            /*<< Unicode character translation table */
  unsigned short  custom_chars[8];  /*<< Custom characters in use */
}
cfontz_h;

typedef enum
{
  WIDTH = 100, HEIGHT, FIRMWARE
}
tag_enum;

static tag_enum tag[] = { WIDTH, HEIGHT, FIRMWARE };

static dict_pair tag_dict[] = 
{
  { "firmware", &tag[2] },
  { "height",   &tag[1] },
  { "width",    &tag[0] }
};

static dictionary tags = 
    { tag_dict, sizeof (tag_dict) / sizeof (dict_pair) };


static firmware_enum fw[] = { CF_VERSION_1, CF_VERSION_2 };

static dict_pair fw_dict[] =
{
  { "1.0",   &fw[0] },
  { "1.2",   &fw[0] },
  { "1.3",   &fw[0] },
  { "2.0",   &fw[1] }
};

static dictionary firmware = 
    { fw_dict, sizeof (fw_dict) / sizeof (dict_pair) };


/*--------------------------------------------------- _find_table_entry --*/
static cfz_translated
_find_table_entry (const cfz_translated *table, wchar_t c)
{
  while (table->index != CFZ_TABLE_END)
    {
      if (table->index == c)
        break;
      
      ++table;
    }

  return *table;
}

/*-------------------------------------------------------- drv_get_info --*/
const char *
drv_get_info (const char* field)
{
  if (!strcmp(field, "name"))
    return "crystalfontz";

  if (!strcmp(field, "description"))
    return "Driver for CrystalFontz 632 and 634 devices.";

  if (!strcmp(field, "version"))
    return "1";

  return 0;
}

/*---------------------------------------------------- drv_create_handle --*/
void *
drv_create_handle (xml_node* config)
{
  int i;
  cfontz_h *h;
  h = malloc (sizeof (cfontz_h));
  if (h)
    {
      xml_node *node;
      const char *num;
      
      h->backend = NULL;
      
      xmlt_rescan_document (config, &tags, 0);
      num = NULL;
      node = xmlt_find (config, 0, WIDTH);
      if (node != NULL)
        num = xmlt_get_first_cdata (node);
          
      if (num != NULL)
        h->size.width = atoi (num);
      else
        h->size.width = 20;

      if (h->size.width < 1)
        {
          free (h);
          return NULL;
        }

      num = NULL;
      node = xmlt_find (config, 0, HEIGHT);
      if (node != NULL)
        num = xmlt_get_first_cdata (node);
          
      if (num != NULL)
        h->size.height = atoi (num);
      else
        h->size.height = 4;

      if (h->size.height < 1)
        {
          free (h);
          return NULL;
        }

      h->firmware = CF_VERSION_1;
      node = xmlt_find (config, 0, FIRMWARE);
      if (node != NULL)
        {
          const char *ver_str = xmlt_get_first_cdata (node);

          if (ver_str != NULL)
            {
              void *lookup = dict_lookup (&firmware, ver_str);

              if (lookup == NULL)
                {
                return NULL;
                }
              else
                h->firmware = *((firmware_enum*) lookup);
            }
        }

      switch (h->firmware)
        {
        case CF_VERSION_1:
            h->xtab = xtbl_v1;
            break;

        case CF_VERSION_2:
            h->xtab = xtbl_v2;
            break;

        default:
            assert (0);
        }
      
      for (i = 0; i < 8; ++i)
        h->custom_chars[i] = CF_UNUSED_CUSTOM_CHAR;
    }
  
  return h;
}

/*--------------------------------------------------- drv_destroy_handle --*/
void
drv_destroy_handle (void *handle)
{
  cfontz_h *h = (cfontz_h*)handle;
  if (h->backend)
    backend_free_handle (h->backend);
  
  free (h);
}

/*----------------------------------------------------- drv_bind_backend --*/
void
drv_bind_backend (void *handle, void* backend)
{
  static char init[] = { CF_HIDE_CURSOR, CF_FORM_FEED, CF_WRAP_OFF,
                         CF_CONTRAST_CTRL, 50 };
  
  cfontz_h *h = (cfontz_h*)handle;
  h->backend = backend;

  sleep (3);

  backend_send (h->backend, init, sizeof (init));
}

/*--------------------------------------------------------- drv_get_type --*/
drv_type
drv_get_type ()
{
  return DRV_CHARACTER;
}

drv_dimensions
drv_get_dimensions (void *handle)
{
  cfontz_h *h = (cfontz_h*)handle;

  return h->size;
}

/*--------------------------------------------------- drv_process_canvas --*/
void
drv_process_canvas (void *handle, void *_canvas)
{
  cc_canvas *canvas = (cc_canvas *)_canvas;
  cfontz_h *h = (cfontz_h*)handle;
  unsigned int row;
  unsigned int col;
  unsigned int i;
  cc_elem *send;
  cfz_translated xlat;
  char buf[160];
  
  for (row = 0; row < canvas->height; ++row)
    {
      for (col = 0; col < canvas->width; ++col)
        {
          send = cc_get_element (canvas, col, row);

          if (send != NULL)
            {
              xlat = _find_table_entry (h->xtab, send->c);
              
              if (xlat.index != CFZ_TABLE_END)
                {
                  if (xlat.custom_char1 != 0)
                    {
                      for (i = 0; i < 8; ++i)
                        {
                          if (h->custom_chars[i] == CF_UNUSED_CUSTOM_CHAR)
                            {
                              h->custom_chars[i] = xlat.custom_char1;
                              break;
                            }
                        }
                    }
                }
            }
        }
    }

  for (i = 0; i < 8; ++i)
    {
      if (h->custom_chars[i] != CF_UNUSED_CUSTOM_CHAR)
        {
          char temp = CF_SET_CUSTOM_CHAR;

          backend_send (h->backend, &temp, 1);
          temp = (char) i;
          backend_send (h->backend, &temp, 1);
          backend_send (h->backend, custom_char[h->custom_chars[i] - 1], 8);
        }
    }
  
  for (row = 0; row < canvas->height; ++row)
    {
      buf[0] = CF_GOTO_POSITION;
      buf[1] = (char) 0;
      buf[2] = (char) row;

      i = 3;

      for (col = 0; col < canvas->width; ++col)
        {
          send = cc_get_element (canvas, col, row);

          if (send != NULL)
            {
              xlat = _find_table_entry (h->xtab, send->c);
              
              if (xlat.index != CFZ_TABLE_END)
                {
                  if (xlat.custom_char1 != 0)
                    {
                      int j;
                      for (j = 0; j < 8; ++j)
                        {
                          if (h->custom_chars[j] == xlat.custom_char1)
                            {
                              buf[i++] = CF_CUSTOM_CHAR_OFFSET + j;
                              break;
                            }
                        }
                    }
                  else
                    buf[i++] = xlat.direct;
                }
            }
        }

      backend_send (h->backend, buf, i);
    }
}

/*------------------------------------------------------ drv_get_backend --*/
void *
drv_get_backend (void *handle)
{
  cfontz_h *h = (cfontz_h*)handle;

  return h->backend;
}
