/* 
 *  socket.c - TCP/IP socket routines.
 *             This file is part of the FreeLCD package.
 *
 *  $Id: socket.c,v 1.4 2003/11/02 21:37:24 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */


#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <netdb.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>

#include "socket.h"

/*------------------------------------------------------- socket_create --*/
socket_t
socket_create (const char *addr, unsigned port)
{
  struct sockaddr_in  s_in;
  socket_t            sock;
  int                 result;
  int                 sock_opt;

  sock = socket (AF_INET, SOCK_STREAM, 0);
  if (sock < 0)
    return -1;

  memset (&s_in, 0, sizeof (s_in));
  s_in.sin_family = PF_INET;
  s_in.sin_port = htons (port);
  s_in.sin_addr.s_addr = htonl (INADDR_ANY);

  result = setsockopt (sock, SOL_SOCKET, SO_REUSEADDR,
                       &sock_opt, sizeof (sock_opt));
  if (result < 0)
    return -2;

  if (addr)
    {
      if (inet_aton (addr, &s_in.sin_addr) == 0)
        return -3;
    }

  result = bind (sock, (struct sockaddr *) (&s_in), sizeof (s_in));
  if (result < 0)
    return -4;

  return sock;
}

/*------------------------------------------------ socket_create_server --*/
socket_t
socket_create_server (unsigned port, fd_set * fdset)
{
  socket_t  sock = socket_create (0, port);
  int       result;

  if (sock < 0)
    return sock;

  result = listen (sock, 10);
  if (result < 0)
    return -4;

  FD_ZERO (fdset);
  FD_SET (sock, fdset);

  return sock;
}

/*--------------------------------------------------------- socket_poll --*/
int
socket_poll (socket_t sock, fd_set * fdset,
             void (*callback_newconn) (socket_t, struct sockaddr_in *sin),
             void (*callback_data) (socket_t, char *, size_t),
             void (*callback_close) (socket_t))
{
  static char         read_buf[4096];
  socket_t            new_sock;
  fd_set              readset = *fdset;
  struct sockaddr_in  s_in;
  struct timeval      tv;
  socklen_t           s_len;
  int                 result;
  int                 i;

  tv.tv_sec = 0;
  tv.tv_usec = 100000;

  result = select (FD_SETSIZE, &readset, 0, 0, &tv);
  if (result < 0)
    return -1;

  for (i = 0; i < FD_SETSIZE; ++i)
    {
      if (FD_ISSET (i, &readset))
        {
          if (i == sock)
            {
              /* New connection on server socket */
              s_len = sizeof (s_in);
              new_sock = accept (sock, (struct sockaddr *) (&s_in), &s_len);
              if (new_sock < 0)
                return -2;

              FD_SET (new_sock, fdset);
              fcntl (new_sock, F_SETFL, O_NONBLOCK);
              callback_newconn (new_sock, &s_in);
            }
          else
            {
              /* Read data from existing connection */
              do
                {
                  result = recv (i, read_buf, 4096, 0);
                  if (result <= 0)
                    {
                      if (result == 0 || errno != EAGAIN)
                        {
                          callback_close (i);
                          close (i);
                          FD_CLR (i, fdset);
                          FD_CLR (i, &readset);
                        }
                    }
                  else 
                    {
                      read_buf[result] = 0;
                      callback_data (i, read_buf, result);
                    }
                }
              while (result > 0);
            }
        }
    }

  return 0;
}

/*--------------------------------------------------------- socket_send --*/
int socket_send (socket_t sock, const char *data, size_t len)
{
  return send (sock, data, len, 0);
}
