/*
 * slist.h - Single linked list.
 *           This file is part of the FreeLCD package.
 *
 * $Id: slist.h,v 1.9 2004/01/16 20:58:54 unicorn Exp $
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA  02111-1307  USA
 *
 * Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/**
 * \file slist.h
 * Implementation of a single linked list.
 * This list only takes data elements of type void*. You will have to
 * cast the elements back to the right type after retrieving them. It
 * is allowed to insert NULL elements.
 */

#ifndef _FLCDD_SLIST_H
#define _FLCDD_SLIST_H 1

#include <stdlib.h>

/** Implementation of a single element in a linked list. */ 
typedef struct _slist_e
{
  /** Pointer to the next element, or NULL if this is the last one. */
  struct _slist_e *next;

  /** Pointer to the actual data that this element represents. */
  void *data;
}
slist_e;

/** A single-linked list. */ 
typedef struct _slist
{
  /** Pointer to the first list element, or NULL for an empty list. */
  slist_e *head;
  /** Pointer to the last list element. */
  slist_e *tail;
}
slist;

/** List iterator. */ 
typedef struct
{
  slist_e *curr; /**< Element at the iterator's position, or NULL. */
  slist_e *prev; /**< Previous element in list. */
}
slist_iter;

/** A range, including Begin, excluding End. */
typedef struct
{
  slist_iter Begin;
  slist_iter End;
}
slist_range;

/** Initialise a list.
 * The elements of the slist struct \a s are set to NULL.
 * \param s List to initialise.
 */
void 
slist_init (slist *s);

/** Free all memory used by the list \a s.
 * Calls free() for every slist_elem struct in the list, and for every data
 * element. 
 * \param s List to be deleted. */
void 
slist_delete (slist *s);

/** Free all memory used by the list in a custom way.
 * Calls free() for every slist_elem struct in \a s, and the user-defined
 * func() for every data element. 
 * \param s List to be deleted.
 * \param func Function that will clean up the data element. */
void 
slist_delete_special (slist *s, void (*func) (void *));

/** Free all memory used by the elements in list \a s.
 * Calls free() for every slist_elem struct in \a s, leaving the data
 * elements alone. 
 * 
 * \code
 * slist my_list;
 * int   *element;
 * int   element_2;
 *
 * element = malloc (sizeof (int));
 * *element = 42;
 * element_2 = 24;
 *
 * slist_init (&my_list);
 * slist_append (&my_list, element);
 * slist_append (&my_list, &element_2);
 *
 * slist_delete_list_only (&my_list);
 *
 * free (element);
 * 
 * \endcode
 *
 * \param s List to be deleted. */
void 
slist_delete_list_only (slist *s);

/** Return the first data element from \a s.
 * \param s List to fetch the first data element from.
 * \return Pointer to the first data element of \a s, or NULL if \a s
 * is empty. */
void *
slist_first (const slist *s);

/** Return the last data element from the list.
 * \param s List to fetch the first data element from.
 * \return Pointer to the last data element of \a s, or NULL if \a s
 * is empty. */
void *
slist_last (const slist *s);

/** Retrieve the data element at the iterator from the list.
 * \param iter Iterator to dereference.
 * \return Pointer to the data element at the iterator's position in the
 * list, or NULL if the iterator is not valid. */
void *
slist_at_iter (slist_iter iter);

/** Insert a new data element at the start of the list.
 * \param s List to add the element \a data to. 
 * \param data Pointer to the data element to be added. */
void 
slist_prepend (slist *s, void *data);

/** Insert a new data element at the end of the list.
 * \param s List to add the element \a data to. 
 * \param data Pointer to the data element to be added. */
void 
slist_append (slist *s, void *data);

/** Insert a new data element right before the iterator.
 * \param s List to add the element \a data to. 
 * \param iter Insertion position.
 * \param data Pointer to the data element to be added. */
void 
slist_insert (slist *s, slist_iter *iter, void *data);

/** Remove the first data element from the list.
 * \param s List to remove the element from.
 * \return Pointer to the removed data element. */
void *
slist_remove_first (slist *s);

/** Remove the last data element from the list.
 * \param s List to remove the element from.
 * \return Pointer to the removed data element. */
void *
slist_remove_last (slist *s);

/** Remove the data element at the iterator.
 * The iterator is no longer valid after this call. 
 * \param s List to remove the element from.
 * \param iter Iterator to use for removal.
 * \return Pointer to the removed data element. */
void *
slist_remove_at_iter (slist *s, slist_iter iter);

/** Create an iterator that points at the first data element of
 *         the list.
 * \param s The list to create the iterator for. 
 * \return An iterator that points to the first data element of the list,
 * or which is invalid if \a s is an empty list. */
slist_iter 
slist_begin_iter (const slist *s);

/** Create an iterator that points right behind the last element of
 *         the list. (It does not point at an actual data element.)
 * \param s The list to create the iterator for.
 * \return An iterator that points right after the last element in list
 * \a s. */
slist_iter
slist_end_iter (const slist *s);

/** Retrieve the data element at the iterator's position, and move
 *         to the next element in the list. 
 * If the iterator was not valid, NULL is returned and /a iter remains
 * unchanged.
 * \param iter The iterator to dereference and move to the next element.
 * \return Pointer to the data element at the iterator's position, or NULL
 * if \a iter is not valid. */
void *
slist_iter_and_next (slist_iter *iter);

/** Call func() for every element in the list.
 * The variable \a userdata can be used to pass extra information to the
 * function \a func.
 * \param s The list to take all elements from.
 * \param func The function that will be called.
 * \param userdata The second argument to \a func. */
void 
slist_for_each (const slist *s, void (*func) (void *data, void *userdata),
                void *userdata);

/** Call func() for every element between the iterators start and end,
 *         including start and excluding end.
 * The variable \a userdata can be used to pass extra information to the
 * function \a func.
 * \param start Points to the begin of the range.
 * \param end Points right after the end of the range.
 * \param func The function that will be called.
 * \param userdata The second argument to \a func. */
void 
slist_for_each_range (slist_iter start, slist_iter end,
                      void (*func) (void *data, void *userdata),
                      void *userdata);

/** Test two iterators for equality.
 * \param a First iterator to be tested.
 * \param b Second iterator to be tested.
 */
int
slist_iter_eq (slist_iter a, slist_iter b);

/** Test if an iterator is at the end of a list.
 * \param iter Iterator to be tested. */
int
slist_iter_at_end (slist_iter iter);

/** Search for a data item in the list.
 * If the item could not be found, and end of list iterator is returned. Use
 * slist_iter_at_end() or slist_iter_eq() to test for this condition.
 * \param s The list to be searched.
 * \param data The element to look for.
 * \return An iterator pointing to the matching element, or right after the
 *         end of the list in case no element was found. */
slist_iter 
slist_find (const slist *s, const void *data);

/** Search for a data item in the list with a user-defined function for
 *        comparing two elements. 
 * If the item could not be found, and end of list iterator is returned. Use
 * \a slist_iter_at_end or \a slist_iter_eq to test for this condition.
 * \param s The list to be searched.
 * \param compare The element to look for.
 * \param func A function that compares two elements, and returns a non-zero
 * value if they are equal.
 * \return An iterator pointing to the matching element, or right after the
 *         end of the list in case no element was found. */
slist_iter 
slist_find_if (const slist *s, const void *compare,
               int (*func) (const void *data, const void *compare));

/** Determine the length of the list
 * \param s The length of this list will be calculated.
 * \return Length of the list. */
size_t
slist_length (const slist *s);

#endif /* Multiple include guard */
