/*
 *  dictionary.h - "Dictionary" data structure, works a bit like std::map.
 *                 This file is part of the FreeLCD package.
 *
 *  $Id: dictionary.h,v 1.6 2004/01/16 20:52:31 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

/** \file dictionary.h
 * Dictionary data structure with binary search.
 * This data structure can associate a string with a void*. You will
 * have to cast elements back to their original type after retrieving them.
 */
#ifndef _FLCDD_DICTIONARY_H
#define _FLCDD_DICTIONARY_H 1

#include "stdlib.h" /* For size_t */

/** A dictionary entry, coupling a key to its value. */
typedef struct
{
  const char *key;   /**< The index string. */
  void       *value; /**< The associated value. */
}
dict_pair;

/** A dictionary, consisting of zero or more \link dict_pair entries\endlink. 
 */
typedef struct
{
  dict_pair *dict; /**< Pointer to an array of dictionary entries. */
  size_t    size;  /**< The number of entries in the dictionary. */
}
dictionary;


/** Look up an entry in a #dictionary and return the associated value.
 * This operation is performed in O(log n) time, where n is the number of
 * entries in the dictionary.
 * \param dict Pointer to the dictionary.
 * \param key The key to look up.
 * \return Pointer to the value associated with \a key, or NULL if
 * \a key was not found in the dictionary. */
void *
dict_lookup (const dictionary *dict, const char *key);

/** Compare two dict_pair structs by their keys. This function is
 * useful for sorting dictionaries that are generated at runtime.
 * \param first Pointer to one dict_pair.
 * \param second Pointer to the other dict_pair.
 * \return <0 if \a first precedes \a second lexicographically, 0 if they are
 *         equal, >0 if \a second precedes \a first. */
int
dict_pair_compare (const void* first, const void* second);

#endif /* Multiple include guard */
