/*
 *  backend.c - Different backends that can be used by the drivers.
 *              This file is part of the FreeLCD package.
 *
 *  $Id: backend.c,v 1.2 2004/01/25 15:45:01 unicorn Exp $
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the
 *  Free Software Foundation; either version 2 of the License, or (at your
 *  option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *  Copyright (c) 2002, 2003, Jeroen van den Berg <unicorn@hippie.nu>
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_STRING_H
# include <string.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif

#include <termios.h>
#include <curses.h>
#include <fcntl.h>

#include "backend.h"

#ifndef HAVE_CFMAKERAW
# define HAVE_CFMAKERAW 0
#endif


typedef enum
{
  NAME = 0, WIDTH, HEIGHT, DEVICE, BAUDRATE
}
tag_e;

tag_e tag[] =
{
  NAME, WIDTH, HEIGHT, DEVICE, BAUDRATE
}; 

dict_pair tag_dict[] =
{
    { "baudrate", &tag[BAUDRATE] },
    { "device", &tag[DEVICE] },
    { "height", &tag[HEIGHT] },
    { "name", &tag[NAME] },
    { "width", &tag[WIDTH] }
};

dictionary tags = { tag_dict, sizeof (tag_dict) / sizeof (dict_pair) };



/** Implementation of a backend handler (the outside world sees just a 
 * void*). */
typedef struct
{
  char opened;  /**< Non-zero if this backend is opened. */
  void *dev_handle;  /**< Device handle, the actual type depends on which
		          send_func() and close_func() are called. */
  int (*send_func) (void *, const char *, size_t); /**< Send some data */
  void (*close_func) (void *); /**< Close this backend. */
}
bck_handle;


#ifdef USE_CURSES

/** Specialised handle for the curses backend. */ 
typedef struct
{
  WINDOW *cont;      /**< Window contents */
  WINDOW *border;    /**< Window borders */
  size_t output;     /**< Characters of output to window */
  size_t max_output; /**< Max number of characters window can hold */
}
cursd;

#endif /* USE_CURSES */


/* ===================
 *  SERIAL
 * =================== 
 */

/*-------------------------------------------------------- _send_serial --*/
static int
_send_serial (void *h, const char *data, size_t len)
{
  int result;

  while (len > 0)
    {
      result = write (*(int*)h, data, len);
      if (result < 0)
        return 0;

      len -= result;
    }
  return 1;
}

/*------------------------------------------------------- _close_serial --*/
static void
_close_serial (void *h) 
{ 
  close (*(int *) h);
  free (h);
}

/*-------------------------------------------------------- _open_serial --*/
static void
_open_serial (bck_handle *h, xml_node *config)
{
  struct termios portset;
  const char* baud = 0;
  const char* device = "/dev/ttyS0";
  xml_node *node;
  int _baudrate;
  int fd;
  
  node = xmlt_find (config, 0, DEVICE);
  if (node)
    device = xmlt_get_first_cdata (node);
  
  node = xmlt_find (config, 0, BAUDRATE);
  if (node)
    baud = xmlt_get_first_cdata (node);

#ifdef WIN32

  /* This extra translation is done to make Win32 installations work
   * out of the box. */

  if (!strcmp ("/dev/ttyS0", device))
    device = "COM1:";

  else if (!strcmp ("/dev/ttyS1", device))
    device = "COM2:";

  else if (!strcmp ("/dev/ttyS2", device))
    device = "COM3:";

  else if (!strcmp ("/dev/ttyS3", device))
    device = "COM4:";

#endif
  
  fd = open (device, O_RDWR | O_NOCTTY | O_NDELAY);
  if (fd < 0)
    return;

  if (baud == 0)
    _baudrate = B9600; /* No baudrate specified, 9600 is a nice default. */
  else
    {
      switch (atoi (baud))
        {
        case 600:
          _baudrate = B600;
          break;
        case 1200:
          _baudrate = B1200;
          break;
        case 1800:
          _baudrate = B1800;
          break;
        case 2400:
          _baudrate = B2400;
          break;
        case 4800:
          _baudrate = B4800;
          break;
        case 9600:
          _baudrate = B9600;
          break;
        case 19200:
          _baudrate = B19200;
          break;
        case 38400:
          _baudrate = B38400;
          break;
        case 57600:
          _baudrate = B57600;
          break;
        case 115200:
          _baudrate = B115200;
          break;
        case 230400:
          _baudrate = B230400;
          break;
        default:
          _baudrate = B9600; /* FIXME: Use safe default, or report error? */
          break;
        }
    }

  /* Set the serial port to raw mode */
  tcgetattr (fd, &portset);

#if HAVE_CFMAKERAW
  cfmakeraw (&portset);
#else
  portset.c_iflag &= ~(  IGNBRK | BRKINT | PARMRK | ISTRIP
                       | INLCR  | IGNCR  | ICRNL  | IXON);
  portset.c_oflag &= ~OPOST;
  portset.c_lflag &= ~(ECHO | ECHONL | ICANON | ISIG | IEXTEN);
  portset.c_cflag &= ~(CSIZE | PARENB);
  portset.c_cflag |= CS8 | CREAD | CLOCAL;
#endif

  /* Set baudrate */
  cfsetospeed (&portset, _baudrate);
  cfsetispeed (&portset, B0);
  tcsetattr (fd, TCSANOW, &portset);

  h->dev_handle = NULL;
  h->send_func = _send_serial;
  h->close_func = _close_serial;
  h->dev_handle = malloc (sizeof (int));
  if (h->dev_handle == NULL)
    exit (EXIT_FAILURE);

  *((int *) h->dev_handle) = fd;
  h->opened = 1;
}



/* ===================
 *  CURSES
 * =================== 
 */

#ifdef USE_CURSES

/*-------------------------------------------------------- _send_curses --*/
static int
_send_curses (void *h, const char *data, size_t len)
{
  cursd *x = h;

  x->output += len;
  if (x->output > x->max_output)
    {
      size_t skip = x->max_output - (x->output - len);
      x->output -= x->max_output;
      wclear (x->cont);
      waddnstr (x->cont, data + skip, len - skip);
    }
  else
    {
      waddnstr (x->cont, data, len);
    }

  touchwin (x->cont);
  wrefresh (x->cont);

  return 1;
}

/*------------------------------------------------------- _close_curses --*/
static void
_close_curses (void *h)
{
  cursd *x = h;

  delwin (x->border);
  delwin (x->cont);
  endwin ();
  curs_set (1);
}

/*-------------------------------------------------------- _open_curses --*/
static void
_open_curses (bck_handle *h, xml_node *config)
{
  cursd *x;
  int width = 20;
  int height = 4;

  int wx;
  int wy;

  const char *num; 
  xml_node *node;

  num = NULL;
  node = xmlt_find (config, 0, WIDTH);
  if (node != NULL)
    num = xmlt_get_first_cdata (node);
  
  if (num != NULL)
    width = atoi (num);

  if (width < 1)
    return;
      
  num = 0;
  node = xmlt_find (config, 0, HEIGHT);
  if (node != NULL)
    num = xmlt_get_first_cdata (node);
  
  if (num != NULL)
    height = atoi (num);
  
  if (height < 1)
    return;
  
  x = malloc (sizeof (cursd));
  if (x == NULL)
    exit (EXIT_FAILURE);

  if (initscr () == 0)
    return;

  wy = (LINES / 2) - (height / 2);
  wx = (COLS / 2) - (width / 2);
  x->cont = newwin (height, width, wy, wx);
  x->border = newwin (height + 2, width + 4, wy - 1, wx - 2);
  
  if (   x->cont == 0 
      || x->border == 0
      || nonl() == ERR
      || noecho() == ERR
      || curs_set (0) == ERR)
    {
      free (x);
      endwin ();
      return;
    }

  box (x->border, 0, 0);
  scrollok (x->cont, 0);
  leaveok (x->cont, 1);
  touchwin (x->border);
  wrefresh (x->border);

  x->output = 0;
  x->max_output = width * height;
  
  h->dev_handle = x;
  h->send_func = _send_curses;
  h->close_func = _close_curses;
  h->opened = 1;
}

#endif  /* USE_CURSES */


/*---------------------------------------------- backend_create_handle --*/
void *
backend_create_handle (xml_node *config, const char* type)
{
  bck_handle *h = malloc (sizeof (bck_handle));

  if (h == NULL)
    exit (EXIT_FAILURE);

  if (type == 0)
    return 0;

  h->opened = 0;
  h->dev_handle = 0;

  xmlt_rescan_document (config, &tags, 0);
  
  if (!strcmp ("serial", type))
    _open_serial (h, config);

#ifdef USE_CURSES
  else if (!strcmp ("curses", type))
    _open_curses (h, config);
#endif
  
  return (void *) h;
}

/*-------------------------------------------------------- backend_send --*/
int
backend_send (void *handle, const char *data, size_t len)
{
  bck_handle *h = (bck_handle *) handle;

  if (h->opened == 0)
    return BACKEND_NOT_OPENED_ERR;

  return h->send_func (h->dev_handle, data, len);
}

/*----------------------------------------------- backend_free_handle --*/
void
backend_free_handle (void *handle)
{
  bck_handle *h = (bck_handle *) handle;

  if (h->opened != 0)
    h->close_func (h->dev_handle);

  free (h->dev_handle);
  free (h);
}
