/*
	Copyright (C) 2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef _ARCHIVER_H
#define _ARCHIVER_H

#include <sys/types.h>
#include <sys/stat.h>
#include <string>

#include "Dicer.h"
#include "Index.h"
#include "ArchiverCallbacks.h"
#include "FilterInBase.h"
#include "NullInFilter.h"
#include "ZLibInFilter.h"

struct archive_header {
	u_int32_t endian_arch;
	u_int16_t format_version;
} __attribute__((packed)); /* Pack this because it gets written to the archive */

class Archiver
{
public:
	Archiver(TArchiverReadCallback read_callback, TArchiverSeek seek);
	~Archiver();
	
	int OpenArchive(bool use_index);
	bool CloseArchive();

  /* By Number */
  int SeekFileNumber(u_int64_t file_number, bool force_index = false, off_t *file_offset = NULL);
  int FileMetadataByNumber(u_int64_t file_number, std::string *metadata);
  int MetaChecksumByNumber(u_int64_t file_number, unsigned char *checksum);
  int FileDataByNumber(u_int64_t file_number, TArchiverFileDataCallback write_callback, void *callback_data);
  int FileChecksumByNumber(u_int64_t file_number, unsigned char *checksum);
  
  /* TODO: By Name */

  /* Helper functions */
  int HelperGetFileSize(std::string &metadata, off_t *file_size);
  int HelperGetFileName(std::string &metadata, std::string *file_name);
  int HelperGetMode(std::string &metadata, bool *is_hardlink, bool *is_compressed, mode_t *file_mode);
  /* The rest of the helpers should return 1 indicating that the file is a hardlink and extra info cannot be extracted. */

  int Index_GetEntry(u_int64_t file_number, off_t *entry_offset);
  
  void PrintSettings();
	
private:
	TArchiverReadCallback m_read;
  TArchiverSeek m_seek;
	Index *m_index;
  
  /* File offset cache */
  /* remember to only store a number or a name not both or at least make sure they point to the same entry */
  u_int64_t m_current_file_number;
  std::string m_current_file_name;
  off_t m_current_file_offset;
  
  /* Metadata cache */
  /* remember to only store a number or a name not both or at least make sure they point to the same entry */
  u_int64_t m_cached_metadata_number;
  std::string m_cached_metadata_filename;
  std::string m_cached_metadata;

  int GetFileMetadata(std::string *metadata);
  int GetMetaChecksum(unsigned char *checksum);
  int GetFileChecksum(unsigned char *checksum);

  int HelperGetEntrySize(std::string &metadata, off_t *entry_size);
};

#endif /* _ARCHIVER_H */

