/*
	Copyright (C) 2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

/*
  NOTE: All programs are derived from flatback.
  
  FIXME: Doesn't handle integer overflow/wraparounds, make sure things are within reasonable limits.

  GENERAL TODO:
    ERRORS
      Need more error messages
      Need a better way of handling errors (use C++ exceptions?)
        Make sure errors are thrown by the libraries that I use, especially the standard template library.
      Need a better way of retreiving error messages instead of just printing to stderr.
        Have each class export an errorstate class?
          errorstate class would tells what the error message is and if an error occurred.
    ENHANCEMENTS
      Support more filesystem information, lik ACLs
    CLEANUP
      Cut back on using char* so I don't have to worry about NULL termination
*/
/*
	Flatback - Creates flat backups. Does not handle incremental backups. Create another wrapper file to do that.
*/

/*
	Includes and global configuration...
*/

#include <string>
#include <iostream>
#include <fstream>

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>

#include <getopt.h>

#include "flatback.h"
#include "Slicer.h"
#include "Archiver.h"

#define HELPSTRING \
"FlatBack version " FLATBACK_VERSION "\n" \
"(C) 2005 Brian Gunlogson\n" \
"./flatback [-szpmkh] -o slice_basename file_list\n" \
"	-o basename, --output=basename #Basename of slice\n" \
"	-s size, --slice=size #Size of slice in bytes (0=single file)\n" \
"	-z level, --zlib=level #Use zlib compression (default level 6)\n" \
"	-p, --pause #Pause between slices\n" \
"	-m, --max-mem-cache=ram_in_bytes #Max RAM to use for compressing files\n" \
"	-k, --ask #Ask for conformation of command line options\n" \
"	-h, --help #Show this help message\n" \
"	file_list #Newline seperated list of files to backup\n"

#define FLATBACK_OPTSTRING "s:m:o:z:pkh"

static const option long_opts[] = {	{ "size", 1, NULL, 's' },
																		{ "output", 1, NULL, 'o' },
																		{ "zlib", 1, NULL, 'z' },
																		{ "pause", 0, NULL, 'p' },
																		{ "max-mem-cache", 1, NULL, 'm' },
																		{ "ask", 0, NULL, 'k' },
																		{ "help", 0, NULL, 'h' },
																		{ NULL, 0, NULL, 0 } };

Slicer *g_slicer = NULL;
Archiver *g_archiver = NULL;

/* Function Prototypes */
void DIE(const char *fmt, ...);
bool pause_function(unsigned int slicenum);
void print_useage();
bool slicer_write_callback(const char *buffer, unsigned int length);
off_t slicer_get_offset();

void DIE(const char *fmt, ...)
{
	va_list ap;
	
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	
	if(g_archiver)
		delete g_archiver;
	if(g_slicer)
		delete g_slicer;

	exit(1);
}

/*
	Function:
		pause_function
		
	Arguments:
		slicenum - Number of completed slice
	
	Returns:
		true - continue
		false - abort
		
	Remarks:
		Callback from Slicer. Prompts user to continue or abort and displays completed slice number
*/
bool pause_function(unsigned int slicenum)
{
	while(1)
	{
		fprintf(stderr, "\n-----------------\n");
		fprintf(stderr, "Slice %i is done.\n", slicenum-1);
		fprintf(stderr, "-----------------\n\n");

		fprintf(stderr, "Press 'c' to continue or 'q' to abort.\n");
		
		char keypress;
		/* FIXME: Replace with something that doesn't require the enter key to be pressed */
		if(fread(&keypress, 1, 1, stdin) != 1)
			DIE("Error in fread(stdin)");
		
		switch(keypress)
		{
			case 'q':
				return false;
			case 'c':
				return true;
			default:
				fprintf(stderr, "\nPressed key was not a vaild choice.\n");
				sleep(1);
		}
	}
	return false;
}

/*
	Function:
		print_usage
		
	Arguments:
		None
	
	Returns:
		Never, terminates.
		
	Remarks:
		Prints command line help message.
*/
void print_useage()
{
	DIE(HELPSTRING);
}

bool slicer_write_callback(const char *buffer, unsigned int length)
{
  if(!g_slicer)
    DIE("g_slicer does not exist!\n");
  
  return g_slicer->AppendData(buffer, length);
}

off_t slicer_get_offset()
{
  if(!g_slicer)
    DIE("g_slicer does not exist!\n");

  return g_slicer->GetArchivePosition();
}

/*
	Entry point
*/
int main(int argc, char **argv)
{
  unsigned int compression_level = 0;
  
	g_slicer = new Slicer;
	if(!g_slicer)
		DIE("Failed to allocate Slicer object!");
	
	g_archiver = new Archiver(slicer_write_callback, slicer_get_offset);
	if(!g_archiver)
		DIE("Failed to allocate Archiver object!");

	bool moreopts = true, b_gave_output = false, confirm_settings = false, quitflag = false;

	while(moreopts)
	{
		unsigned int i;
		
		switch(getopt_long(argc, argv, FLATBACK_OPTSTRING, long_opts, NULL))
		{
			case 'h':
			case ':':
			case '?':
				print_useage();
			case 's':
				for(i = 0; i < strlen(optarg); i++)
					if(!isdigit(*(optarg+i)))
						DIE("Size parameter is not a base 10 positive integer!\n");

				if(!g_slicer->SetSize(atoll(optarg)))
					DIE("Slicer object rejected size argument\n");
				break;
      case 'm':
				for(i = 0; i < strlen(optarg); i++)
					if(!isdigit(*(optarg+i)))
						DIE("Size parameter is not a base 10 positive integer!\n");

        g_archiver->SetMaxMemcache(atoll(optarg));
        break;
			case 'o':
				if(!g_slicer->SetFullName(std::string(optarg)))
					DIE("Slicer object rejected basename argument\n");
				b_gave_output = true;
				break;
			case 'z':
        for(i = 0; i < strlen(optarg); i++)
          if(!isdigit(*(optarg+i)))
            DIE("ZLib parameter is not a base 10 positive integer!\n");
        
        compression_level = atoi(optarg);
				break;
			case 'p':
				g_slicer->SetPauseFunction(&pause_function);
				break;
			case 'k':
				confirm_settings = true;
				break;
			case -1:
				moreopts = false;
				break;
			default:
				DIE("getopt_long returned something unexpected\n");
		}
	}
	
	if(!b_gave_output) {
		fprintf(stderr, "Output basename (-o option) required\n");
		print_useage();
	}
	
	char *filelist = NULL;
	
	if(argc-optind == 1)
		filelist = argv[optind];
	else if(argc-optind == 0)
		DIE("Unable to use STDIN as source for filenames");
	else
		DIE("Only one file argument allowed on the command line");
	
	if(confirm_settings)
	{
		bool ate_keypress = false;
		while(!ate_keypress)
		{
			fprintf(stderr, "\nConfirm Command Line Settings\n\nMain program reports\n=================\n");
			fprintf(stderr, "File List: %s\n", filelist);
			fprintf(stderr, "=================\n\nArchiver Reports\n=================\n");
			g_archiver->PrintSettings();
			fprintf(stderr, "=================\n\nSlicer Reports\n=================\n");
			g_slicer->PrintSettings();
			fprintf(stderr, "=================\n\n");

			fprintf(stderr, "Press 'c' to continue or 'q' to abort.\n");
			
			char keypress;
			/* FIXME: Replace with something that doesn't require the enter key to be pressed */
			if(fread(&keypress, 1, 1, stdin) != 1)
				DIE("Error in fread(stdin)");
			
			switch(keypress)
			{
				case 'q':
					quitflag = true;
				case 'c':
					ate_keypress = true;
					break;
				default:
					fprintf(stderr, "\nPressed key was not a vaild choice.\n");
					sleep(1);
			}
		}
	}
	
	if(!quitflag)
	{
    /* TODO: Maybe make the index optional? The Archiver class already supports it.
       Just change the first argument to CreateArchive to false */
		if(!g_archiver->CreateArchive(true, compression_level))
			DIE("Failed to create archive\n");

		std::ifstream in_list(filelist);

		if(in_list.fail())
			DIE("std::ifstream open failed\n");

		while(in_list.good())
		{
			std::string line;
			if(!std::getline(in_list, line)) {
				if(in_list.bad())
					DIE("std::getline failed\n");
				else
					break;
			}

			printf("Adding '%s' to archive.\n", line.c_str());

			if(!g_archiver->AddToArchive(line.c_str()))
				DIE("Failed to add '%s' to archive!\n", line.c_str());
		}

		if(in_list.bad())
			DIE("There was an error processing the input file\n");

		in_list.close();

		if(!g_archiver->CommitArchive())
			DIE("Failed to commit archive!\n");

    printf("Success!\n");
	}
  
	delete g_archiver;
	delete g_slicer;

	return 0;
}

