/*
 * Fisoco : FInd, Select, Order and COnvert files
 * Copyright 2015-2016-2017 (C) Felicien PILLOT <felicien.pillot@member.fsf.org>
 *
 * This file is part of Fisoco.
 *
 * Fisoco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Fisoco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Fisoco.  If not, see <http://www.gnu.org/licenses/>.
 *
 *//////////////////////////////////////
/***************
 *    cli.cc    *
 *    ~~~~~~    *
 ****************/

#include "../include/cli.h"
#include "../include/global.h"
#include "../include/search.h"

#include <algorithm>
#include <sys/stat.h>
#include <sstream>
#include <fstream>
#include <time.h>

void
Cli::backup (std::string filename)
{
  filename = (filename != "" ? filename : currentDateTime ()) + ".bak.fis";
  ofstream backup_file;
  string path = "/var/tmp/fisoco";
  mkdir (path.c_str(), S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH);
  int i = 1;
  std::list<std::string> list_names = results.get_complete_names ();
  if (list_names.size () != 0)
    {
      backup_file.open (path + "/" + filename);
      backup_file << "# This is generated data" << std::endl
		  << "# Please do not modify" << std::endl;
      for (std::list<std::string>::iterator complete_name = list_names.begin ();
	   complete_name != list_names.end (); complete_name++, i++)
	{
	  backup_file << i << "\t- " << *complete_name << std::endl;
	}
      backup_file.close ();
      std::cout << "List of files saved as " << filename << "." << std::endl;
    }
  else
    std::cout << "Cannot backup: list is empty" << std::endl;
}
 
void
Cli::clean ()
{
  results = List ();
}

/* help ()
 * Displays every help entries, or just the one given in argument
 */
void
Cli::help (std::string argument)
{
  // Check argument
  if (argument == "")
    // Loop through entries
    for (std::string command : command_list)
      printf (" %-7s   %-22s    %-20s\n",
	      entry_map[command].command.c_str (),
	      entry_map[command].arguments.c_str (),
	      entry_map[command].short_desc.c_str ());
  else
    {
      if (argument.size () <= 2)
	  argument = char_map [argument];
      if (std::find (command_list.begin (),
		     command_list.end (),
		     argument) == command_list.end ())
	error ("Cannot give help about unknown command !");
      else
	printf ("\033[0;1m%-7s\033[0m :\n\t%-20s\n",
		entry_map[argument].command.c_str (),
		entry_map[argument].long_desc.c_str ());
    }
}

void
Cli::info (std::list<std::string> command)
{
  std::list<int> files = get_numbers (command);
  std::vector<std::string> infos, keys;
  for (int i : files)
    {
      infos = results.get_infos (i);
      if (infos.size () == 0)
	error ("No information : check the list by typing 'read'");
      else
	{
	  keys = {"Name", "Path"};
	  for (std::string info : infos)
	    for (std::string key : keys)
	      std::cout << "\t -> " << key << " : " << info << std::endl;
	  std::cout << std::endl;
	}
    }
}

void
Cli::load ()
{
  std::cout << gettext ("Scanning backup files...") << std::endl;
  clean ();
  dir_term = "/var/tmp/fisoco";
  search_term = ".bak.fis";
  search ();
  // Give some time to search ()
  for (int i = 0; i <= 3; i++)
    {
      if (results.get_list_size () != 0)
	break;
      else
	sleep (1);
    }
  if (results.get_list_size () != 0)
    {
      read ();
      std::cout << gettext ("Select a file to load : ");
      std::string input;
      std::cin >> input;
      int number;
      std::istringstream iss (input);
      iss >> number;
      std::vector<std::string> infos = results.get_infos (number);
      // Check if number is good
      if (infos.size () == 0)
	error ("Backup file unknown : check the list by typing 'load'");
      // Check if file is a backup
      else if (infos.at (0).find (".bak.fis") == std::string::npos)
	error ("This is not a Fisoco backup file !");
      else
	{
	  debug ("Loading backup file.");
	  // Clear the old list, to refill it with loaded file
	  clean ();
	  string name = infos.at(1) + "/" + infos.at(0); 
	  ifstream backup_file (name);
	  if (backup_file.is_open())
	    {
	      string line;
	      while ( getline (backup_file, line) )
		{
		  std::list<std::string> file = get_file_from_line (line);
		  if (file.size () == 2)
		    // Add path and filename to the new list
		    results.add_file (file.front (), file.back ());
		}
	      backup_file.close();
	      if (results.get_list_size () == 0)
		error ("Nothing loaded. Does backup was empty ? Corrupted ?");
	      else
		std::cout <<
		  gettext ("Backup file ") << number <<
		  gettext ("# has been loaded !") <<
		  gettext (" Type 'read' to see the new list") << std::endl;
	    }
	  else
	    {
	      error ("Unable to open this file");
	      error ("Have you forgotten to run 'load' before this command ?");
	    }
	}
    }
  else
    std::cout << "No backup files found !" << std::endl;
  dir_term = "/";
}

void
Cli::pick (std::list<std::string> command)
{
  std::list<int> numbers = get_numbers (command);
  int cursor = 0;
  int iter = 0;
  int oldN = 0;
  // This remove each file between the given ids
  for (int n : numbers)
    {
      while (iter < n - oldN - 1)
	{
	  results.del_file (cursor);
	  iter++;
	}
      iter = 0;
      oldN = n;
      cursor++;
    }
  // This remove all files after the last given id
  oldN = results.get_list_size () - cursor;
  while (iter < oldN)
    {
      results.del_file (cursor);
      iter++;
    }
}

void
Cli::purify (std::list<std::string> command)
{
  std::list<int> numbers = get_numbers (command);
  int i = 0;
  for (int n : numbers)
    {
      results.del_file (n-i-1);
      i++;
    }
}

void
Cli::read ()
{
  int i = 1;
  std::list<std::string> list_names = results.get_complete_names ();
  if (list_names.size () > 0)
    for (std::list<std::string>::iterator complete_name = list_names.begin ();
	 complete_name != list_names.end (); complete_name++, i++)
      std::cout << i << "\t- " << *complete_name << std::endl;
  else
    std::cout << "List empty : no files searched, founded or selected" << std::endl;
}

/* This function doesn't belong to Cli, it is defined here to avoid warnings
 * from gcc. please_do_work () is called by pthread_create in search ().
 */
static void *please_do_work (void* ptr_caller);
static void
*please_do_work (void* ptr_caller)
{
  Cli* caller = (Cli*) ptr_caller;
  // This causes a segmentation fault
  //  free (ptr_caller);
  Search cli_Worker;
  cli_Worker.do_work (caller);
  return ptr_caller;
}

void
Cli::search ()
{
  debug ("Dir is " + dir_term + "");
  // New list
  results = List ();
  // Create and start a new worker thread
  pthread_t cli_WorkerThread;
  Cli* ptr = (Cli*) malloc (sizeof (Cli*));
  ptr = this;
  pthread_create (&cli_WorkerThread, NULL, please_do_work, ptr);
}
