/* failmalloc - force to fail in allocating memory sometimes */
/*
 * Copyright (C) 2006 Yoshinori K. Okuji <okuji@enbug.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <malloc.h>
#include <stdlib.h>
#include <limits.h>
#include <sys/types.h>
#include <unistd.h>
#include <time.h>

static void failmalloc_install (void);
static void failmalloc_uninstall (void);

static void *(*old_malloc_hook) (size_t, const void *);
static void *(*old_realloc_hook) (void *ptr, size_t size, const void *);
static void *(*old_memalign_hook) (size_t, size_t, const void *);

static double failure_probability = 1.0;
static unsigned long failure_interval = 1;
static unsigned long count = 0;

static int
should_fail (void)
{
  if (failure_interval > 1)
    {
      count++;
      if (count < failure_interval)
        return 0;

      count = 0;
    }

  if (failure_probability < 1.0)
    {
      if (RAND_MAX * failure_probability >= rand ())
        return 1;
    }
  else
    return 1;

  return 0;
}

static void *
failmalloc (size_t size, const void *caller)
{
  void *ret;

  if (should_fail ())
    return NULL;

  failmalloc_uninstall ();
  ret = malloc (size);
  failmalloc_install ();

  return ret;
}

static void *
failrealloc (void *ptr, size_t size, const void *caller)
{
  void *ret;

  if (should_fail ())
    return NULL;

  failmalloc_uninstall ();
  ret = realloc (ptr, size);
  failmalloc_install ();

  return ret;
}

static void *
failmemalign (size_t alignment, size_t size, const void *caller)
{
  void *ret;

  if (should_fail ())
    return NULL;

  failmalloc_uninstall ();
  ret = memalign (alignment, size);
  failmalloc_install ();

  return ret;
}

static void
failmalloc_install (void)
{
  old_malloc_hook = __malloc_hook;
  old_realloc_hook = __realloc_hook;
  old_memalign_hook = __memalign_hook;
  __malloc_hook = failmalloc;
  __realloc_hook = failrealloc;
  __memalign_hook = failmemalign;
}

static void
failmalloc_uninstall (void)
{
  __malloc_hook = old_malloc_hook;
  __realloc_hook = old_realloc_hook;
  __memalign_hook = old_memalign_hook;
}

static void
failmalloc_init (void)
{
  char *val;

  /* Obtain the probability from the environment.  */
  val = getenv ("FAILMALLOC_PROBABILITY");
  if (val)
    {
      double tmp;

      tmp = strtod (val, 0);
      if (tmp >= 0.0 && tmp <= 1.0)
        failure_probability = tmp;
    }

  /* Obtain the interval from the environment.  */
  val = getenv ("FAILMALLOC_INTERVAL");
  if (val)
    {
      unsigned long tmp;

      tmp = strtoul (val, 0, 0);
      if (tmp != ULONG_MAX)
        failure_interval = tmp;
    }

  /* Initialize the random seed with something.  */
  srand (getpid () * 32452843 + time (NULL) * 49979687);

  failmalloc_install ();
}

void (*__malloc_initialize_hook) (void) = failmalloc_init;

