# Copyright 2009 Ben Escoto
#
# This file is part of Explicans.

# Explicans is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# Explicans is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Explicans.  If not, see <http://www.gnu.org/licenses/>.

import unittest, code
from program import *

class AbsoluteReferenceTest(unittest.TestCase):
	"""Test AbsoluteReferences"""
	ar1 = AbsoluteReference(())
	ar2 = AbsoluteReference((4,))
	ar3 = AbsoluteReference((100, 32, 40, 6, 2, 1, 46))
	ar4 = AbsoluteReference((100, 32, 40, 6, 2, 1, 'C'))
	ar5 = AbsoluteReference((1,2,3, ('C',)))
	ar6 = AbsoluteReference((1,2,3, ('R',)))
	ar7 = AbsoluteReference((1,2,3, (21, 3)))
	def testString(self):
		"""Test converting to/from string"""
		for ar in (self.ar1, self.ar2, self.ar3, self.ar4,
				   self.ar5, self.ar6, self.ar7):
			#print "Testing AR: ", ar
			cycled_ar = AbsoluteReference(str(ar))
			assert  cycled_ar == ar, (str(ar), str(cycled_ar))
		temp_ar = AbsoluteReference('(45, 2)')
		assert temp_ar.t == (45, 2), (temp_ar, temp_ar.t)

		try: temp_ar2 = AbsoluteReference('(5, hello)')
		except ARError: pass
		else: assert False, ("Error expected", temp_ar2)

		temp_ar3 = AbsoluteReference('(5, C)')
		assert temp_ar3.t == (5, 'C'), temp_ar3.t



class ProgramTest(unittest.TestCase):
	"""Test a Program"""
	prog1 = """
Reference,Name,Formula
"()",, "array(2)"
(0),, array(4)
"(0, C)",, "6"
"(0, 0)", a, 5
"(0, 1)", b, 7
"(0, 2,)", c, "a+b"
"(1)",, table
"(1, T:C)", col axis, 10
"(1, T:R)", 100, 20
"(1, T:0,1)", col name, 30
"(1, T:0,2)", col name2, 40
"(1, T:1,0)", row name, 50
"(1, T:2,0)", row name2, 60
"(1, T:1,1)", ignored, 70
""" # Sample explicans program
	root_ar = AbsoluteReference(())
	def testStrConversion(self):
		"""Test converting program to string and back"""
		p = Program(self.prog1)
		#print "foo:", p[self.root_ar]
		assert p[self.root_ar].name is None
		assert str(p[self.root_ar].form_str) == "array(2)"
		assert p[AbsoluteReference((0, 1))].name == 'b'
	
		cycled = str(Program(str(p)))
		if sorted(cycled.split('\n')) != sorted(str(p).split('\n')):
			print "Wanted:", str(p)
			print "Got:", cycled
			assert False

	def testChildren(self):
		"""Test returning the children of an absref"""
		children = Program(self.prog1).getchildren(self.root_ar.append(0))
		names = [propset.name for name, propset in children]
		assert names == ['a', 'b', 'c'], names
		form_strings = [propset.form_str for name, propset in children]
		assert form_strings == ['5', '7', 'a+b'], form_strings

	def testColFormula(self):
		"""Test returning a column formula"""
		p = Program(self.prog1)
		ps = p.get_col_ps(AbsoluteReference((0,)))
		assert ps.form_str == '6', ps
		try: p.get_col_ps(AbsoluteReference(()))
		except KeyError: pass
		else: assert False, "Expected Key Error"

	def testTableFormulas(self):
		"""Test table formulas"""
		p = Program(self.prog1)
		ar = AbsoluteReference((1,))
		row_ps, col_ps = p.get_table_axis_propsets(ar)
		assert (row_ps.name, row_ps.form_str) == (100, '20'), row_ps.name
		assert (col_ps.name, col_ps.form_str) == ('col axis', '10'), col_ps.name

		row2_ps = p.get_table_rowname_ps(ar, 2)
		assert row2_ps.name == 'row name2', row2_ps
		assert row2_ps.form_str == '60', row2_ps
		
		col1_ps = p.get_table_colname_ps(ar, 1)
		assert col1_ps.name == 'col name', col1_ps.name
		assert col1_ps.form_str == '30', col1_ps.form_str
		
		value_ps = p.get_table_value_ps(ar, 1, 1)
		assert value_ps.form_str == '70', value_ps.form_str


if __name__ == "__main__": unittest.main()
