/*
 * Input.cs - Curses input handling.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

namespace Curses
{

using System;

/// <summary>
/// <para>The <see cref="T:Curses.Input"/> class manages the
/// input process for curses applications.  Using the methods
/// in this class, applications can read characters from the
/// terminal, install timeout handlers, and trap mouse events
/// on terminals that support a mouse.</para>
/// </summary>
public sealed class Input
{

	// Internal state.

	/// <summary>
	/// <para>Get the next character from the input stream,
	/// without echo.</para>
	/// </summary>
	///
	/// <returns>
	/// <para>Returns the character, coerced to <see langword="int"/>,
	/// or a keycode from the <see cref="T:Curses.KeyCode"/> class,
	/// or -1 if a timeout occurred and <c>Misc.ReportTimeouts</c>
	/// is set to <see langword="true"/>.</para>
	/// </returns>
	///
	/// <remarks>
	/// <para>If timeouts, screen resizes, or mouse events occur
	/// during the call to this method, they will be dispatched
	/// to the applicable event handlers.</para>
	/// </remarks>
	public static int GetNextChar()
			{
				int ch;
				for(;;)
				{
					if(Misc.DockedCursor)
					{
						Misc.DockCursor();
					}
					ch = Curses.Native.CursesHelpGetNextChar
							(Timeout.TimeToFire);
					if(ch == -1)
					{
						// A timeout occurred.
						Timeout.FirePending();

						// Should we report timeouts to the caller?
						if(Misc.ReportTimeouts)
						{
							return -1;
						}
					}
					else if(ch == KeyCode.KEY_MOUSE)
					{
						// A mouse event occurred.
					}
					else if(ch == KeyCode.KEY_RESIZE)
					{
						// A screen resize event occurred.
						Window.Screen.ResizeToMatch();
						if(Resized != null)
						{
							Resized(null, new ResizedEventArgs
										(Curses.Native.CursesHelpGetCols(),
										 Curses.Native.CursesHelpGetLines()));
						}
					}
					else if(ch == KeyCode.KEY_REFRESH)
					{
						// A screen refresh event occurred.
						if(Refreshed != null)
						{
							Refreshed(null, null);
						}
					}
					else if(ch == KeyCode.KEY_MAX)
					{
						// Interrupt or terminate signal received.
						Setup.Exit(0);
					}
					else
					{
						break;
					}
				}
				return ch;
			}

	/// <summary>
	/// <para>Determine if the terminal has a mouse.</para>
	/// </summary>
	///
	/// <value>
	/// <para><see langword="true"/> if the terminal has a mouse,
	/// or <see langword="false"/> otherwise.</para>
	/// </value>
	public static bool HasMouse
			{
				get
				{
					return false;
				}
			}

	/// <summary>
	/// <para>Event that is signalled when a mouse button is pressed.</para>
	/// </summary>
	public static event ButtonEventHandler ButtonPressed;

	/// <summary>
	/// <para>Event that is signalled when a mouse button is released.</para>
	/// </summary>
	public static event ButtonEventHandler ButtonReleased;

	/// <summary>
	/// <para>Event that is signalled when a mouse button is clicked.</para>
	/// </summary>
	public static event ButtonEventHandler ButtonClicked;

	/// <summary>
	/// <para>Event that is signalled when a mouse button
	/// is double-clicked.</para>
	/// </summary>
	public static event ButtonEventHandler ButtonDoubleClicked;

	/// <summary>
	/// <para>Event that is signalled when a mouse button
	/// is triple-clicked.</para>
	/// </summary>
	public static event ButtonEventHandler ButtonTripleClicked;

	/// <summary>
	/// <para>Event that is signalled when the mouse is moved.</para>
	/// </summary>
	public static event MouseMovedEventHandler MouseMoved;

	/// <summary>
	/// <para>Event that is signalled when the screen is resized.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>The curses library will resize all window objects to fit the
	/// new window size determined by the system prior to emitting this
	/// event.  The application may wish to readjust its output to fit
	/// the new size.</para>
	/// </remarks>
	public static event ResizedEventHandler Resized;

	/// <summary>
	/// <para>Event that is signalled when the screen is refreshed.</para>
	/// </summary>
	///
	/// <remarks>
	/// <para>The user can refresh the screen by pressing either
	/// Ctrl-L, Ctrl-R, or the keyword-specific "Refresh" key (if one
	/// is present).</para>
	///
	/// <para>The curses library will refresh the screen with the current
	/// contents of all of its window buffers.  Normally the application
	/// doesn't need to take any other action.</para>
	/// </remarks>
	public static event RefreshedEventHandler Refreshed;

} // class Input

} // namespace Curses
