/*
 * init.c - Helper routines for initializing and terminating curses.
 *
 * Copyright (C) 2001  Southern Storm Software, Pty Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "defs.h"

#ifdef	__cplusplus
extern	"C" {
#endif

volatile int CursesHelpSawInterrupt = 0;
volatile int CursesHelpSawSizeChange = 0;

/*
 * Signal handler for the interrupt key.
 */
static void InterruptHandler(int sig)
{
	CursesHelpSawInterrupt = 1;
	signal(SIGINT, InterruptHandler);
}

/*
 * Signal handler for the "TERM" signal.
 */
static void TerminateHandler(int sig)
{
	CursesHelpSawInterrupt = 1;
	signal(SIGTERM, TerminateHandler);
}

/*
 * Signal handler for the window resize event.
 */
static void WindowSizeHandler(int sig)
{
	CursesHelpSawSizeChange = 1;
	signal(SIGWINCH, WindowSizeHandler);
}

/*
 * Signal handler for the suspend signal.
 */
static void SuspendHandler(int sig)
{
	/* Restore the terminal to the default tty states */
	endwin();

	/* Re-install the signal handler */
	signal(SIGTSTP, SuspendHandler);

	/* Send this process the stop signal to stop it for real */
	kill(getpid(), SIGSTOP);
}

/*
 * Signal handler for the resume signal.
 */
static void ResumeHandler(int sig)
{
	/* Return to visual mode and restore the screen */
	refresh();

	/* Re-install the signal handler */
	signal(SIGCONT, ResumeHandler);
}

int CursesHelpInitialize(void)
{
	/* Initialize the basic curses routines */
	if(initscr() == 0)
	{
		return 0;
	}
	cbreak();
	noecho();
	nonl();
	intrflush(stdscr, FALSE);

	/* Initialize color management */
	start_color();
	use_default_colors();

	/* Enable the keypad */
	keypad(stdscr, TRUE);

	/* Install signal handlers for useful signals */
	CursesHelpSawInterrupt = 0;
	CursesHelpSawSizeChange = 0;
	signal(SIGINT, InterruptHandler);
	signal(SIGTERM, TerminateHandler);
	signal(SIGWINCH, WindowSizeHandler);
	signal(SIGTSTP, SuspendHandler);
	signal(SIGCONT, ResumeHandler);

	/* Ready to go */
	return 1;
}

void CursesHelpTerminate(void)
{
	/* Restore the signals to their default behaviours */
	signal(SIGINT, SIG_DFL);
	signal(SIGTERM, SIG_DFL);
	signal(SIGWINCH, SIG_DFL);
	signal(SIGTSTP, SIG_DFL);
	signal(SIGCONT, SIG_DFL);

	/* Tell the curses library to clean up */
	endwin();
}

#ifdef	__cplusplus
};
#endif
