#!/bin/sh
#
# mklabel.sh - Make label tables for computed goto's.
#
# Copyright (C) 2002  Southern Storm Software, Pty Ltd.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Usage: mklabel.sh awk cvm.h >cvm_labels.h

# Validate the parameters.
if test "x$1" = "x" ; then
	echo "Usage: $0 awk cvm.h >cvm_labels.h" 1>&2
	exit 1
fi
if test "x$2" = "x" ; then
	echo "Usage: $0 awk cvm.h >cvm_labels.h" 1>&2
	exit 1
fi

# Output the "do not edit" header.
echo '/* This file is automatically generated - do not edit */'
echo ''

# Define a null assembly statement.  This is a trick to stop gcc doing
# tail-end combination on the cases within the switch statements.
echo '#if defined(__GNUC__)'
echo '#define VMNULLASM()         __asm__("")'
echo '#else'
echo '#define VMNULLASM()'
echo '#endif'
echo ''

# Allow specific platforms to override Computed GOTO realizations
echo '#ifndef VM_CGOTO_SWITCH'
echo '#define VM_CGOTO_SWITCH(val) goto **(void**)pc;'
echo '#endif'
echo '#ifndef VM_CGOTO_PREFIXSWITCH'
echo '#define VM_CGOTO_PREFIXSWITCH(val) goto **(void**)pc;'
echo '#endif'
echo '#ifndef VM_CGOTO_BREAK'
echo '#define VM_CGOTO_BREAK(val) goto **(void**)pc;'
echo '#endif'
echo '#ifndef VM_CGOTO_BREAKNOEND'
echo '#define VM_CGOTO_BREAKNOEND(val) goto **(void**)pc;'
echo '#endif'

# Are we compiling an interpreter that uses PIC computed goto's?
echo '#if defined(IL_CVM_PIC_TOKEN) || defined(IL_CVM_PIC_DIRECT)'
echo ''

# Output a table of PIC labels, based on an incoming stream
# of #define's for opcodes.  Some awks don't support hex
# string to number conversion, which is why we have to convert
# the opcode value the hard way.
pic_table()
{
	"$1" 'BEGIN{
		      nextval=0
		  }
		  {
		      currval=0;
			  ch=substr($3,3,1);
		  	  if(ch == "0") currval=currval+0; else
		  	  if(ch == "1") currval=currval+1; else
		  	  if(ch == "2") currval=currval+2; else
		  	  if(ch == "3") currval=currval+3; else
		  	  if(ch == "4") currval=currval+4; else
		  	  if(ch == "5") currval=currval+5; else
		  	  if(ch == "6") currval=currval+6; else
		  	  if(ch == "7") currval=currval+7; else
		  	  if(ch == "8") currval=currval+8; else
		  	  if(ch == "9") currval=currval+9; else
		  	  if(ch == "A") currval=currval+10; else
		  	  if(ch == "B") currval=currval+11; else
		  	  if(ch == "C") currval=currval+12; else
		  	  if(ch == "D") currval=currval+13; else
		  	  if(ch == "E") currval=currval+14; else
		  	  if(ch == "F") currval=currval+15;
			  currval=currval*16;
			  ch=substr($3,4,1);
		  	  if(ch == "0") currval=currval+0; else
		  	  if(ch == "1") currval=currval+1; else
		  	  if(ch == "2") currval=currval+2; else
		  	  if(ch == "3") currval=currval+3; else
		  	  if(ch == "4") currval=currval+4; else
		  	  if(ch == "5") currval=currval+5; else
		  	  if(ch == "6") currval=currval+6; else
		  	  if(ch == "7") currval=currval+7; else
		  	  if(ch == "8") currval=currval+8; else
		  	  if(ch == "9") currval=currval+9; else
		  	  if(ch == "A") currval=currval+10; else
		  	  if(ch == "B") currval=currval+11; else
		  	  if(ch == "C") currval=currval+12; else
		  	  if(ch == "D") currval=currval+13; else
		  	  if(ch == "E") currval=currval+14; else
		  	  if(ch == "F") currval=currval+15;
		      while(nextval < currval)
			  {
			      print "	&&'"$2"' - &&COP_NOP_label,"
			      ++nextval
			  }
		      print "	&&" $2 "_label - &&COP_NOP_label,"
		      ++nextval
		  }
		  END{
		      while(nextval < 256)
			  {
			      print "	&&'"$2"' - &&COP_NOP_label,"
			      ++nextval
			  }
		  }' -
	return 0
}

# Output the main label table (PIC).
echo 'static int const main_label_table[256] = {'
grep '^#define[ 	]*COP_' "$2" | grep -v 'COP_PREFIX_' | \
	pic_table "$1" _DEFAULT_MAIN_label
echo '};'
echo ''

# Output the prefix label table (PIC).
echo 'static int const prefix_label_table[256] = {'
grep '^#define[ 	]*COP_PREFIX_' "$2" | \
	pic_table "$1" _DEFAULT_PREFIX_label
echo '};'
echo ''

# Output the helper macros (PIC).
echo '#ifdef IL_CVM_DIRECT'
echo '#define VMSWITCH(val)       VM_CGOTO_SWITCH(val);'
echo '#define VMPREFIXSWITCH(val) VM_CGOTO_PREFIXSWITCH(val);'
echo '#define VMCASE(val)         val##_label'
echo '#define VMDEFAULT           _DEFAULT_MAIN_label'
echo '#define VMPREFIXDEFAULT     _DEFAULT_PREFIX_label'
echo '#define VMBREAK(val)        \
            CVM_DUMP(); \
            VM_CGOTO_BREAK(val)'
echo '#define VMBREAKNOEND        \
            CVM_DUMP(); \
            VM_CGOTO_BREAKNOEND(val)'
echo '#define VMOUTERBREAK'
echo '#else /* !IL_CVM_DIRECT */'
echo '#define VMSWITCH(val)       goto *(&&COP_NOP_label + main_label_table[(val)]);'
echo '#define VMPREFIXSWITCH(val) goto *(&&COP_NOP_label + prefix_label_table[(val)]);'
echo '#define VMCASE(val)         val##_label'
echo '#define VMDEFAULT           _DEFAULT_MAIN_label'
echo '#define VMPREFIXDEFAULT     _DEFAULT_PREFIX_label'
echo '#define VMBREAK(val)        \
            CVM_DUMP(); \
            goto *(&&COP_NOP_label + main_label_table[*pc])'
echo '#define VMBREAKNOEND        \
            CVM_DUMP(); \
            goto *(&&COP_NOP_label + main_label_table[*pc])'
echo '#define VMOUTERBREAK'
echo '#endif /* !IL_CVM_DIRECT */'
echo ''

# Now to handle the non-PIC case of using computed goto's.
echo '#elif defined(IL_CVM_TOKEN) || defined(IL_CVM_DIRECT)'
echo ''

# Output a table of non-PIC labels, based on an incoming stream
# of #define's for opcodes.  Some awks don't support hex
# string to number conversion, which is why we have to convert
# the opcode value the hard way.
non_pic_table()
{
	"$1" 'BEGIN{
		      nextval=0
		  }
		  {
		      currval=0;
			  ch=substr($3,3,1);
		  	  if(ch == "0") currval=currval+0; else
		  	  if(ch == "1") currval=currval+1; else
		  	  if(ch == "2") currval=currval+2; else
		  	  if(ch == "3") currval=currval+3; else
		  	  if(ch == "4") currval=currval+4; else
		  	  if(ch == "5") currval=currval+5; else
		  	  if(ch == "6") currval=currval+6; else
		  	  if(ch == "7") currval=currval+7; else
		  	  if(ch == "8") currval=currval+8; else
		  	  if(ch == "9") currval=currval+9; else
		  	  if(ch == "A") currval=currval+10; else
		  	  if(ch == "B") currval=currval+11; else
		  	  if(ch == "C") currval=currval+12; else
		  	  if(ch == "D") currval=currval+13; else
		  	  if(ch == "E") currval=currval+14; else
		  	  if(ch == "F") currval=currval+15;
			  currval=currval*16;
			  ch=substr($3,4,1);
		  	  if(ch == "0") currval=currval+0; else
		  	  if(ch == "1") currval=currval+1; else
		  	  if(ch == "2") currval=currval+2; else
		  	  if(ch == "3") currval=currval+3; else
		  	  if(ch == "4") currval=currval+4; else
		  	  if(ch == "5") currval=currval+5; else
		  	  if(ch == "6") currval=currval+6; else
		  	  if(ch == "7") currval=currval+7; else
		  	  if(ch == "8") currval=currval+8; else
		  	  if(ch == "9") currval=currval+9; else
		  	  if(ch == "A") currval=currval+10; else
		  	  if(ch == "B") currval=currval+11; else
		  	  if(ch == "C") currval=currval+12; else
		  	  if(ch == "D") currval=currval+13; else
		  	  if(ch == "E") currval=currval+14; else
		  	  if(ch == "F") currval=currval+15;
		      while(nextval < currval)
			  {
			      print "	&&'"$2"',"
			      ++nextval
			  }
		      print "	&&" $2 "_label,"
		      ++nextval
		  }
		  END{
		      while(nextval < 256)
			  {
			      print "	&&'"$2"',"
			      ++nextval
			  }
		  }' -
	return 0
}

# Output the main label table (non-PIC).
echo 'static void *main_label_table[256] = {'
grep '^#define[ 	]*COP_' "$2" | grep -v 'COP_PREFIX_' | \
	non_pic_table "$1" _DEFAULT_MAIN_label
echo '};'
echo ''

# Output the prefix label table (non-PIC).
echo 'static void *prefix_label_table[256] = {'
grep '^#define[ 	]*COP_PREFIX_' "$2" | \
	non_pic_table "$1" _DEFAULT_PREFIX_label
echo '};'
echo ''

# Output the helper macros (non-PIC).
echo '#ifdef IL_CVM_DIRECT'
echo '#define VMSWITCH(val)       VM_CGOTO_SWITCH(val);'
echo '#define VMPREFIXSWITCH(val) VM_CGOTO_PREFIXSWITCH(val);'
echo '#define VMCASE(val)         val##_label'
echo '#define VMDEFAULT           _DEFAULT_MAIN_label'
echo '#define VMPREFIXDEFAULT     _DEFAULT_PREFIX_label'
echo '#define VMBREAK(val)        \
            CVM_DUMP(); \
            VM_CGOTO_BREAK(val)'
echo '#define VMBREAKNOEND        \
            CVM_DUMP(); \
            VM_CGOTO_BREAKNOEND(val)'
echo '#define VMOUTERBREAK'
echo '#else /* !IL_CVM_DIRECT */'
echo '#define VMSWITCH(val)       goto *main_label_table[(val)];'
echo '#define VMPREFIXSWITCH(val) goto *prefix_label_table[(val)];'
echo '#define VMCASE(val)         val##_label'
echo '#define VMDEFAULT           _DEFAULT_MAIN_label'
echo '#define VMPREFIXDEFAULT     _DEFAULT_PREFIX_label'
echo '#define VMBREAK(val)        \
            CVM_DUMP(); \
            goto *main_label_table[*pc]'
echo '#define VMBREAKNOEND        \
            CVM_DUMP(); \
            goto *main_label_table[*pc]'
echo '#define VMOUTERBREAK'
echo '#endif /* !IL_CVM_DIRECT */'
echo ''

# Output the non-goto case of the helper macros.
echo '#else /* IL_CVM_SWITCH */'
echo ''
echo '#define VMSWITCH(val)       switch((val))'
echo '#define VMPREFIXSWITCH(val) switch((val))'
echo '#define VMCASE(val)         case (val)'
echo '#define VMDEFAULT           default'
echo '#define VMPREFIXDEFAULT     default'
echo '#define VMBREAK(val)        VMNULLASM(); break'
echo '#define VMBREAKNOEND        VMNULLASM(); break'
echo '#define VMOUTERBREAK        break'
echo ''
echo '#endif /* IL_CVM_SWITCH */'

# Done.
exit 0
