/**************************************************************************************
  	This file is part of GNU DataExplorer.

    GNU DataExplorer is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    DataExplorer is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with GNU DataExplorer.  If not, see <https://www.gnu.org/licenses/>.

    Copyright (c) 2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023 Winfried Bruegmann
****************************************************************************************/
package gde.device.igc;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.Vector;
import java.util.logging.Logger;

import gde.GDE;
import gde.data.Channel;
import gde.data.Channels;
import gde.data.Record;
import gde.data.RecordSet;
import gde.device.IDevice;
import gde.device.InputTypes;
import gde.exception.DataInconsitsentException;
import gde.exception.DataTypeException;
import gde.exception.DevicePropertiesInconsistenceException;
import gde.exception.MissMatchDeviceException;
import gde.exception.NotSupportedFileFormatException;
import gde.log.Level;
import gde.messages.MessageIds;
import gde.messages.Messages;
import gde.ui.DataExplorer;
import gde.utils.GPSHelper;
import gde.utils.StringHelper;

/**
 * Class to write IGC conform files
 * @author Winfried Brügmann
 */
public class IGCReaderWriter {
	static Logger										log						= Logger.getLogger(IGCReaderWriter.class.getName());

	final static DataExplorer				application		= DataExplorer.getInstance();
	final static Channels						channels			= Channels.getInstance();

	final public static char[]	igcShortDate			= new char[] { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R',
			'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z' };

	/**
	 * query the filename in short IGC filename format
	 * 2.5.1 Short file name style: YMDCXXXF.IGC
	 * Y = Year; value 0 to 9, cycling every 10 years
	 * M = Month; value 1 to 9 then A for 10, B=11, C=12.
	 * D = Day; value 1 to 9 then A=10, B=11, C=12, D=13, E=14, F=15, G=16, H=17, I=18, J=19, K=20, L=21, M=22, N=23, O=24, P=25, Q=26, R=27, S=28, T=29, U=30, V=31.
	 * C = manufacturer's IGC code letter (see table below)
	 * XXX = unique FR Serial Number (S/N); 3 alphanumeric characters
	 * F = Flight number of the day; 1 to 9 then, if needed, A=10 through to Z=35
	 * @param manufacturersIGCCodeLetter
	 * @param threeCharFnId
	 * @param sequenceNumber
	 * @return
	 */
	public static String getIGCShortFileName(char manufacturersIGCCodeLetter, String threeCharFnId, int sequenceNumber) {
		long date = new Date().getTime();
		char year = igcShortDate[Integer.parseInt(new SimpleDateFormat("YY", Locale.getDefault()).format(date))];
		char month = igcShortDate[Integer.parseInt(new SimpleDateFormat("MM", Locale.getDefault()).format(date))];
		char day = igcShortDate[Integer.parseInt(new SimpleDateFormat("dd", Locale.getDefault()).format(date))];
		return String.format("%c%c%c%c%s%c.igc", year, month, day, manufacturersIGCCodeLetter, threeCharFnId, igcShortDate[sequenceNumber]);
	}

	/**
	 * query the filename in short IGC filename format
	 * 2.5.2 Long file name style. This uses a full set of characters in each field, a hyphen separating each field,
	 * the field order being the same as in the short file name. For instance, if the short name for a file from
	 * manufacturer X is 36HXABC2.IGC, the equivalent long file name is 2003-06-17-XXX-ABC-02.IGC.
	 * Long file names may be generated by software that is compatible with long file names, although the DOS
	 * versions of the DATA, CONV and VALI programs must continue to generate and use short file names.
	 * @param threeCharFnId
	 * @param sequenceNumber
	 * @return
	 */
	public static String getIGCLongFileName(String threeCharFnId, int sequenceNumber) {
		return String.format(Locale.ENGLISH, "%s-%s-%s-%02d.igc", StringHelper.getDateAndTime("yyyy-MM-dd"), threeCharFnId, sequenceNumber);
	}

	/**
	 * read the selected IGC file and read/parse
	 * @param filePath
	 * @param device
	 * @param recordNameExtend
	 * @param channelConfigNumber
	 * @return record set created
	 * @throws NotSupportedFileFormatException
	 * @throws MissMatchDeviceException
	 * @throws IOException
	 * @throws DataInconsitsentException
	 * @throws DataTypeException
	 */
	public static RecordSet read(String filePath, IDevice device, String recordNameExtend, Integer channelConfigNumber) throws NotSupportedFileFormatException, IOException, DataInconsitsentException,
			DataTypeException {
		String line = GDE.STRING_STAR, lastLine;
		RecordSet recordSet = null;
		BufferedReader reader; // to read the data
		Channel activeChannel = null;
		int lineNumber = 1;
		int activeChannelConfigNumber = 1; // at least each device needs to have one channelConfig to place record sets
		String recordSetNameExtend = device.getRecordSetStemNameReplacement();
		long timeStamp = -1, actualTimeStamp = -1, startTimeStamp = -1, lastTimeStamp = -1;
		StringBuilder header = new StringBuilder();
		StringBuilder triangles = new StringBuilder();
		StringBuilder error = new StringBuilder();
		StringBuilder albatrossTask = new StringBuilder();
		StringBuilder gpsTriangleRelated = new StringBuilder();
		String date = "000000", time; //16 02 40
		int hour, minute, second;
		int latitude, longitude, altBaro, altGPS;
		int lastLatitude = 0, lastLongitude = 0, lastAltBaro = 0, lastAltGPS = 0;
		int values[] = new int[device.getNumberOfMeasurements(1)-2]; //climb and speed will be calculated
		File inputFile = new File(filePath);
		String dllID = "XXX";
		IgcExtension timeStepExtension = null;
		Vector<IgcExtension> extensions = new Vector<IgcExtension>();
		GDE.getUiNotification().setProgress(0);
		
		GpsTaskResult result = null;

		try {
			if (channelConfigNumber == null)
				activeChannel = IGCReaderWriter.channels.getActiveChannel();
			else
				activeChannel = IGCReaderWriter.channels.get(channelConfigNumber);

			if (activeChannel != null) {
				GDE.getUiNotification().setStatusMessage(Messages.getString(MessageIds.GDE_MSGT0594) + filePath);
				activeChannelConfigNumber = activeChannel.getNumber();

				if (GDE.isWithUi()) {
					IGCReaderWriter.channels.switchChannel(activeChannelConfigNumber, GDE.STRING_EMPTY);
					IGCReaderWriter.application.getMenuToolBar().updateChannelSelector();
					activeChannel = IGCReaderWriter.channels.getActiveChannel();
				}
				String recordSetName = (activeChannel.size() + 1) + ") " + recordSetNameExtend; //$NON-NLS-1$
				int measurementSize = device.getNumberOfMeasurements(activeChannelConfigNumber);
				int dataBlockSize = Math.abs(device.getDataBlockSize(InputTypes.FILE_IO)); // measurements size must not match data block size, there are some measurements which are result of calculation
				log.log(java.util.logging.Level.FINE, "measurementSize = " + measurementSize + "; dataBlockSize = " + dataBlockSize); //$NON-NLS-1$ //$NON-NLS-2$
				if (measurementSize < dataBlockSize) {
					dataBlockSize = measurementSize;
				}

				long approximateLines = inputFile.length() / 35; //B sentence is the most used one and has 35 bytes
				reader = new BufferedReader(new InputStreamReader(new FileInputStream(filePath), "ISO-8859-1")); //$NON-NLS-1$

				//skip all lines before A-Record appears, max 10
				while ((line = reader.readLine()) != null && !line.startsWith("A") && lineNumber < 10) 
					++lineNumber;
				if (lineNumber >= 10) {
					reader.close();
					log.log(Level.SEVERE, filePath + " Check IGC header");
					throw new IOException(filePath + " Check IGC header");
				}
				
				//read all lines which describe the hardware, pilot and plane, save as header
				while ((line = reader.readLine()) != null && !line.startsWith(device.getDataBlockLeader()) && lineNumber < 50) {
					log.log(Level.FINE, line);
					if (line.startsWith("HFDTE")) { //160701	UTC date of flight, here 16th July 2001
						date = line.substring(5).trim();
					}
					//do not care about UTC time offset, the OLE/IGC server will do instead
					//else if (line.startsWith("HFTZNTIMEZONE")) {
					//	timeOffsetUTC = (int) Double.parseDouble(line.substring(14).trim());
					//}
					else if (line.startsWith("H")) {
						header.append(line).append(GDE.LINE_SEPARATOR);
					}
					else if (line.startsWith("LTSK:T:")) { //Albatross task definition
						albatrossTask.append(line.substring(7));
					}
					else if (line.startsWith("C")) {
						if (line.startsWith("WP", 18)) 
							albatrossTask.append(GDE.STRING_MESSAGE_CONCAT).append(line.substring(1));
						gpsTriangleRelated.append(GDE.LINE_SEPARATOR).append(line);
					}
					else if (line.startsWith("LSTAT")) { //Albatrross Statistics
						log.log(Level.OFF, line);
					}
					else if (line.startsWith("E")) { //Albatrross arm, turnpoint
						log.log(Level.OFF, line);
					}
					else if (line.startsWith("A")) { // first line contains manufacturer identifier
						dllID = line.substring(1, 4);
						log.log(Level.FINE, "IGCDLL iddentifier = " + dllID);
					}
					else if (line.startsWith("I")) { // extension specification record
						try {
							final int numExtensions = Integer.parseInt(line.substring(1, 3));
							for (int i = 0; i < numExtensions; i++) {
								IgcExtension extension = new IgcExtension(Integer.parseInt(line.substring(7 * i + 3, 7 * i + 5))-1, Integer.parseInt(line.substring(7 * i + 5, 7 * i + 7)), line.substring(7 * i + 7, 7 * i + 10));
								if (extension.getThreeLetterCode().equals("TDS") || extension.getThreeLetterCode().equals("SUS") )
									timeStepExtension = extension;
								else
									extensions.add(extension);
							}
						}
						catch (Exception e) {
							log.log(Level.SEVERE, e.getMessage(), e);
						}
					}
					++lineNumber;
				}
				if (lineNumber >= 180) {
					reader.close();
					log.log(Level.SEVERE, filePath + " Check IGC header");
					throw new IOException(filePath + " Check IGC header");
				}

				//calculate the start time stamp using the first B record
				int year;
				int month;
				int day;
				if (date.contains(":")) {
					int startIndex = date.indexOf(":");
					year = Integer.parseInt(date.substring(startIndex+5, startIndex+7)) + 2000;
					month = Integer.parseInt(date.substring(startIndex+3, startIndex+5));
					day = Integer.parseInt(date.substring(startIndex+1, startIndex+3));
				}
				else {
					year = Integer.parseInt(date.substring(4, 6)) + 2000;
					month = Integer.parseInt(date.substring(2, 4));
					day = Integer.parseInt(date.substring(0, 2));
				}
				
				if (line == null) 	{	
					reader.close();
					throw new DevicePropertiesInconsistenceException("Error");
				}

				time = line.substring(1, 7);//16 02 40
				while (Integer.parseInt(time) == 0) { 
					log.log(Level.WARNING, String.format("time entry B record zero, line %d", lineNumber));
					line = reader.readLine();
					++lineNumber;
					time = line.substring(1, 7);
				}
				hour = Integer.parseInt(time.substring(0, 2));
				minute = Integer.parseInt(time.substring(2, 4));
				second = Integer.parseInt(time.substring(4, 6));
				startTimeStamp = new GregorianCalendar(year, month - 1, day, hour, minute, second).getTimeInMillis();
				if (timeStepExtension != null) {
					startTimeStamp += Long.parseLong(line.substring(timeStepExtension.start, timeStepExtension.end));
				}
					


				//parse B records B160240 5407121N 00249342W A 00280 00421
				do {
					lastLine = line = line.trim();
					++lineNumber;					
					if (line.length() >= 35 && line.startsWith(device.getDataBlockLeader())) {
						time = line.substring(1, 7); //16 02 40
						if (hour == 23 && hour > Integer.parseInt(time.substring(0, 2))) 
							++day; // switch to next day if 12 -> 0 0r 23 -> 0
						hour = Integer.parseInt(time.substring(0, 2));
						minute = Integer.parseInt(time.substring(2, 4));
						second = Integer.parseInt(time.substring(4, 6));
						actualTimeStamp = new GregorianCalendar(year, month - 1, day, hour, minute, second).getTimeInMillis();
						if (timeStepExtension != null) {
							actualTimeStamp += Long.parseLong(line.substring(timeStepExtension.start, timeStepExtension.end));
						}

						int progress = (int) (lineNumber * 100 / approximateLines);
						if (progress % 5 == 0) GDE.getUiNotification().setProgress(progress);

						if (device.getStateType() == null) {
							reader.close();
							throw new DevicePropertiesInconsistenceException(Messages.getString(MessageIds.GDE_MSGE0043, new Object[] { device.getPropertiesFileName() }));
						}

						try {
							recordSetNameExtend = device.getRecordSetStateNameReplacement(1); // state name
							if (recordNameExtend.length() > 0) {
								recordSetNameExtend = recordSetNameExtend + GDE.STRING_BLANK + GDE.STRING_LEFT_BRACKET + recordNameExtend + GDE.STRING_RIGHT_BRACKET;
							}
						}
						catch (Exception e) {
							reader.close();
							throw new DevicePropertiesInconsistenceException(Messages.getString(MessageIds.GDE_MSGE0044, new Object[] { 0, filePath, device.getPropertiesFileName() }));
						}

						//detect states where a new record set has to be created
						if (recordSet == null || !recordSet.getName().contains(recordSetNameExtend)) {
							//prepare new record set now
							recordSetName = (activeChannel.size() + 1) + ") " + recordSetNameExtend; //$NON-NLS-1$

							recordSet = RecordSet.createRecordSet(recordSetName, device, activeChannel.getNumber(), true, true, true);
							recordSetName = recordSet.getName(); // cut/correct length
							String dateTime = new SimpleDateFormat("yyyy-MM-dd, HH:mm:ss").format(actualTimeStamp); //$NON-NLS-1$
							String recordDescription = device.getName() + GDE.STRING_MESSAGE_CONCAT + Messages.getString(MessageIds.GDE_MSGT0129) + dateTime + GDE.LINE_SEPARATOR + albatrossTask.toString();
							recordSet.setRecordSetDescription(recordDescription);
							//write filename after import to record description
							recordSet.descriptionAppendFilename(filePath.substring(filePath.lastIndexOf(GDE.CHAR_FILE_SEPARATOR_UNIX) + 1));
							if (!activeChannel.getFileDescription().contains("HFGTYGLIDERTYPE"))
								activeChannel.setFileDescription(dateTime.substring(0, 10) + (activeChannel.getFileDescription().length() < 11 ? "" : activeChannel.getFileDescription().substring(10)) 
									+ GDE.LINE_SEPARATOR + header.toString());

							activeChannel.put(recordSetName, recordSet);

							activeChannel.get(recordSetName).setStartTimeStamp(actualTimeStamp);
							activeChannel.setActiveRecordSet(recordSetName);
							activeChannel.applyTemplate(recordSetName, true);

							//values[0] = latitude * 10; // 5 digits after the decimal point only
							//values[1] = longitude * 10;
							//values[2] = altBaro * 1000;
							//values[3] = altGPS * 1000;
							//values[4] = climb calculated;
							//values[5] = speed calculated;
							int i=0;
							for (IgcExtension extension : extensions) {
								if (recordSet.realSize() > 6+i && !recordSet.get(6+i).getName().equals(extension.getThreeLetterCode())) {
									if (log.isLoggable(Level.FINE)) log.log(Level.FINE, String.format("set record with name %s # %d to name %s", recordSet.get(6+i).getName(), 6+i, extension.getThreeLetterCode()));
									recordSet.get(6+i).setName(extension.getThreeLetterCode());
								}
								++i;
							}

						}

						if (timeStamp < actualTimeStamp) {
							//B160240 5407121N 00249342W A 00280 00421
							//I04 36 38 FXA 39 40 SIU 4143TDS 4446ENL
							//1234567 89012345 678901234 5 67890 12345 678 90 123 456
							//B114643 4752040N 01109779E A 00522 00555 035 09 227 225
							if (timeStamp > 0 && actualTimeStamp - timeStamp > 2500) { // 2.5 sec
								log.log(Level.WARNING, String.format(Locale.getDefault(), "High time\t deviation at line %d %s %2d", lineNumber-1, line.substring(1, 7), actualTimeStamp - timeStamp));
								boolean isHighTimeDeviation = actualTimeStamp - timeStamp > 20000; // 20 sec
								boolean isToNextLowDeviation = lastTimeStamp > 0 && lastTimeStamp - timeStamp < 2000; // 2 sec
								log.log(Level.OFF, "isHighTimeDeviation = " + isHighTimeDeviation + " isToNextLowDeviation = " + isToNextLowDeviation);
								if (isHighTimeDeviation && !isToNextLowDeviation) {
									lastTimeStamp = timeStamp;
									log.log(Level.SEVERE, String.format(Locale.getDefault(), "High time\t deviation at line %d %s", lineNumber - 1, line));
									error.append(String.format(Locale.getDefault(), "High time deviation %dms at line %d %s", actualTimeStamp - timeStamp, lineNumber - 1, line)).append(GDE.LINE_SEPARATOR);
									if (hour == 23) 
										hour = 0;
									continue;
								}
							}

							try {
								latitude = Integer.valueOf(line.substring(7, 14));
								latitude = line.substring(14, 15).equalsIgnoreCase("N") ? latitude : -1 * latitude; //$NON-NLS-1$
								if (lastLatitude != 0 && Math.abs(lastLatitude - latitude) > 30)
									log.log(Level.WARNING, String.format(Locale.getDefault(), "High latitude\t deviation at line %d %s %2d", lineNumber-1, line.substring(1, 7), latitude - lastLatitude));
								lastLatitude = latitude;
							}
							catch (Exception e) {
								latitude = values[0];
							}
							try {
								longitude = Integer.valueOf(line.substring(15, 23));
								longitude = line.substring(23, 24).equalsIgnoreCase("E") ? longitude : -1 * longitude; //$NON-NLS-1$
								if (lastLongitude != 0 && Math.abs(lastLongitude - longitude) > 35)
									log.log(Level.WARNING, String.format(Locale.getDefault(), "High longitude\t deviation at line %d %s %2d", lineNumber-1, line.substring(1, 7), longitude - lastLongitude));
								lastLongitude = longitude;
							}
							catch (Exception e) {
								longitude = values[1];
							}
							try {
								altBaro = Integer.valueOf(line.substring(25, 30));
								if (lastAltBaro != 0 && Math.abs(lastAltBaro - altBaro) > 30) {
									log.log(Level.WARNING, String.format(Locale.getDefault(), "High altBaro\t deviation at line %d %s %2d", lineNumber-1, line.substring(1, 7), altBaro - lastAltBaro));
								}
								lastAltBaro = altBaro;
							}
							catch (Exception e) {
								altBaro = values[2];
							}
							try {
								altGPS = Integer.valueOf(line.substring(31, 35));
								if (lastAltGPS != 0 && Math.abs(lastAltGPS - altGPS) > 30) {
									log.log(Level.WARNING, String.format(Locale.getDefault(), "High altGPS\t deviation at line %d %s %2d", lineNumber-1, line.substring(1, 7), altGPS - lastAltGPS));
								}
								lastAltGPS = altGPS;
							}
							catch (Exception e) {
								altGPS = values[3];
							}
							values[0] = latitude * 10; // 5 digits after the decimal point only
							values[1] = longitude * 10;
							values[2] = altBaro * 1000;
							values[3] = altGPS * 1000;

							for (int i = 0; i < extensions.size() && i+4 < values.length; i++) {
								values[i + 4] = extensions.get(i).getValue(line);
							}

							recordSet.addNoneCalculationRecordsPoints(values, actualTimeStamp - startTimeStamp);
							timeStamp = actualTimeStamp;
						}
						else 
							if (actualTimeStamp - timeStamp != 0 && Math.abs(actualTimeStamp - timeStamp) > 1000) {
								log.log(Level.SEVERE, String.format(Locale.getDefault(), "High time\t deviation at line %d %s", lineNumber - 1, line));
								error.append(String.format(Locale.getDefault(), "High time deviation %dms at line %d %s", actualTimeStamp - timeStamp, lineNumber - 1, line)).append(GDE.LINE_SEPARATOR);
							}
					}
					else if (line.startsWith("F")) {
						//skip F RECORD - SATELLITE CONSTELLATION.
						log.log(Level.FINE, "F RECORD - SATELLITE CONSTELLATION = " + line);
					}
					else if (line.startsWith("E")) {
						//skip E RECORD - Albatross enties
						if (line.endsWith("ARM")) {
							log.log(Level.FINE, "E RECORD - Albatross task arm = " + line);
						}
						else if (line.endsWith("TPC"))
							log.log(Level.FINE, "E RECORD - Albatross task turn point condition = " + line);
						else
							log.log(Level.FINE, "E RECORD - Albatross entry = " + line);
					}
					else if (line.startsWith("LISTAT")) {
						log.log(Level.FINE, "L RECORD - Albatross stat = " + line);
						if (line.startsWith("LISTAT{")) {
							
						}
						else if (line.startsWith("LISTAT:LAP:{")) {
							if (result == null)
								result = new GpsTaskResult();
							result.addLap(new GpsLap(line));
						}
					}
					else if (line.startsWith("LSTAT")) {
						log.log(Level.FINE, "L RECORD - Albatross stat = " + line);
						if (result != null) {
							result.add(line);
							
							String taskType = "Sport";
							if (albatrossTask.toString().contains("200,200,70"))
								taskType = "Light";
							else if (albatrossTask.toString().contains("500,500,120"))
								taskType = "SLS";

							log.log(Level.OFF, result.toString(taskType));
							triangles.append(result.toString(taskType));
							
							result = null;
						}
					}
					else if (line.startsWith("LSTART")) {
						log.log(Level.FINE, "L RECORD - Albatross start = " + line);
					}
					else if (line.startsWith("LSECT")) {
						log.log(Level.FINE, "L RECORD - Albatross section = " + line);
					}
					else if (line.startsWith("LHRSTAT")) {
						log.log(Level.FINE, "L RECORD - Albatross task stats = " + line);
					}
					else if (line.startsWith("LPNP")) {
						log.log(Level.FINE, "L RECORD - Albatross lap add penalty = " + line);
					}
					else if (line.startsWith("G")) {
						log.log(Level.FINE, "line number " + lineNumber + " contains security code and is voted as last line! " + lastLine); //$NON-NLS-1$ //$NON-NLS-2$
						break;
					}
					else {
						log.log(Level.WARNING, "line number " + lineNumber + " line length to short or missing " + device.getDataBlockLeader() + " !"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
						continue;
					}
				}
				while ((line = reader.readLine()) != null);

				device.updateVisibilityStatus(activeChannel.get(recordSetName), true);
				if (GDE.isWithUi())
					activeChannel.switchRecordSet(recordSetName);
				else
					activeChannel.setActiveRecordSet(recordSet);
				activeChannel.get(recordSetName).checkAllDisplayable(); // raw import needs calculation of passive records

				reader.close();
				reader = null;
			}
			
			if (error.length() > 10)
				recordSet.setRecordSetDescription(recordSet.getRecordSetDescription() + GDE.LINE_SEPARATOR + error);
			
		}
		catch (FileNotFoundException e) {
			log.log(java.util.logging.Level.WARNING, e.getMessage(), e);
			IGCReaderWriter.application.openMessageDialog(e.getMessage());
		}
		catch (IOException e) {
			log.log(java.util.logging.Level.WARNING, e.getMessage(), e);
			IGCReaderWriter.application.openMessageDialog(e.getMessage());
		}
		catch (Exception e) {
			// check if previous records are available and needs to be displayed
			if (activeChannel != null && activeChannel.size() > 0) {
				String recordSetName = activeChannel.getFirstRecordSetName();
				activeChannel.setActiveRecordSet(recordSetName);
				device.updateVisibilityStatus(activeChannel.get(recordSetName), true);
				activeChannel.get(recordSetName).checkAllDisplayable(); // raw import needs calculation of passive records
				if (GDE.isWithUi()) activeChannel.switchRecordSet(recordSetName);
			}
			// now display the error message
			String msg = filePath + GDE.STRING_MESSAGE_CONCAT + Messages.getString(MessageIds.GDE_MSGE0045, new Object[] { e.getMessage(), lineNumber });
			log.log(java.util.logging.Level.WARNING, msg, e);
			if (GDE.isWithUi())
				IGCReaderWriter.application.openMessageDialog(msg);
		}
		finally {
			GDE.getUiNotification().setProgress(100);
			GDE.getUiNotification().setStatusMessage(GDE.STRING_EMPTY);
		}

		return recordSet;
	}

	/**
	 * write the IGC header and way points
	 * Short file name: 36HXABC2.IGC
	 * Long file name: 2003-06-17-XXX-ABC-02.IGC
	 * where XXX is the manufacturer's three-letter IGC identifier ABC is the IGC serial number/letters of the individual recorder.
	 * The following records are mandatory for an IGC file from an IGC-approved FR:
	 * A - Manufacturer and unique ID for FR
	 * H - Header record
	 * I - FXA addition to B-record, ENL for motor gliders
	 * B - Fix record (lat/long/alt etc.)
	 * F - Satellites used in B record fixes
	 * G - Security record
	 * Use: + for N Lat or E Long     - for S Lat or W Long.
	 * @param device
	 * @param igcFilePath
	 * @param header
	 * @param recordSet
	 * @param ordinalLongitude
	 * @param ordinalLatitude
	 * @param ordinalAltitude
	 * @param startAltitude
	 * @param offsetUTC
	 * @throws Exception
	 */
	public static void write(IDevice device, String igcFilePath, StringBuilder header, RecordSet recordSet, final int ordinalLongitude, final int ordinalLatitude, final int ordinalAltitude,
			final int startAltitude) throws Exception {
		BufferedWriter writer;
		StringBuilder content = new StringBuilder().append(header);
		long startTime = new Date().getTime();

		try {
			GDE.getUiNotification().setStatusMessage(Messages.getString(MessageIds.GDE_MSGT0138, new String[] { GDE.FILE_ENDING_IGC, igcFilePath }));

			if (recordSet != null) {
				int startIndex = GPSHelper.getStartIndexGPS(recordSet, ordinalLatitude, ordinalLongitude);
				Record recordAlitude = recordSet.get(ordinalAltitude);
				SimpleDateFormat sdf = new SimpleDateFormat("HHmmss"); //$NON-NLS-1$
				int offsetHeight = (int) (startAltitude - device.translateValue(recordAlitude, recordAlitude.get(startIndex) / 1000.0));
				char fixValidity = offsetHeight == 0 ? 'A' : 'V';
				long lastTimeStamp = -1, timeStamp;
				long recordSetStartTimeStamp = recordSet.getStartTimeStamp();
				log.log(Level.TIME, "start time stamp = " + StringHelper.getFormatedTime("yyyy-MM-dd HH:mm:ss", recordSetStartTimeStamp));

				for (int i = startIndex; startIndex >= 0 && i < recordSet.get(ordinalLongitude).realSize(); i++) {
					// absolute time as recorded, needs to be converted into UTC
					timeStamp = recordSet.getTime(i) / 10 + recordSetStartTimeStamp;
					if ((timeStamp - lastTimeStamp) >= 950 || lastTimeStamp == -1) {
						content.append(String.format("B%s%s\r\n", sdf.format(timeStamp), device.translateGPS2IGC(recordSet, i, fixValidity, startAltitude, offsetHeight))); //$NON-NLS-1$

						lastTimeStamp = timeStamp;
					}
				}
			}

			writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(igcFilePath), "ISO-8859-1")); //$NON-NLS-1$
			writer.append(content.toString());
			writer.flush();
			writer.close();
			writer = null;
			//recordSet.setSaved(true);
			GDE.getUiNotification().setProgress(100);
		}
		catch (RuntimeException e) {
			IGCReaderWriter.log.log(java.util.logging.Level.SEVERE, e.getMessage(), e);
			throw new Exception(Messages.getString(MessageIds.GDE_MSGE0007) + e.getClass().getSimpleName() + GDE.STRING_MESSAGE_CONCAT + e.getMessage());
		}
		finally {
			GDE.getUiNotification().setStatusMessage(GDE.STRING_EMPTY);
		}
		if (log.isLoggable(Level.TIME)) log.log(Level.TIME, "IGC file = " + igcFilePath + " written successfuly" //$NON-NLS-1$ //$NON-NLS-2$
				+ "write time = " + StringHelper.getFormatedTime("ss:SSS", (new Date().getTime() - startTime)));//$NON-NLS-1$ //$NON-NLS-2$
	}
}
