/*
Copyright (C)  2006  Daniele Zelante

This file is part of comf.

comf is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

comf is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with comf; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/*@LICENSE*/
// $Id$

#include <set>

#include "string.hxx"

#include "defs.hxx"
#include "globals.hxx"
#include "exception.hxx"
#include "integer.hxx"
#include "mymacros.hxx"
#include "mystd.hxx"


COMF_NS_BEGIN


char digitletter(unsigned int n)
{
	if(n<10) return '0'+n;
	if(n<36) return 'A'+n-10;
	return '?';
}




size_t itod(int n, char * s, bool plus)
{
	if(n>0)
		
	{
		if(plus) *(s++)='+';
		return utod(unsign(n),s);
	}
		
	if(n<0)
	{
		*(s++)='-';
		return utod(unsign(-n),s);
	}

	*(s++)='0';
	*s=0;

	return 1;
}


std::string itod(int n)
{
	char tmp[sizeof(n)*3+2];
	itod(n,tmp);
	return std::string(tmp);
}



size_t ftod(double x, char * s)
{
	ASSERT(sizeof(x)==8);	
	return sprintf(s,"%.15g",x);
}

std::string ftod(double x)
{
	char str[256];
	ftod(x,str);
	return std::string(str);
}

size_t ftox(double x, char * s)
{
	ASSERT(sizeof(x)==8);	
	return sprintf(s,"%a",x);
}

std::string ftox(double x)
{
	char str[256];
	ftox(x,str);
	return std::string(str);
}



std::string trim(const std::string & s, const char * pattern)
{
	size_t n = s.size();
	size_t j;
	for(j=0;j<n;++j) if(!strchr(pattern,s[j])) break;
	size_t k;
	for(k=n;k>0;--k) if(!strchr(pattern,s[k-1])) break;

	return s.substr(j,k-j);
	
}

double stof(const char * s)
{
	return strtod(s,0);
}

unsigned int stou(const char * s, char ** endptr)
{
	if(isdigit(s[0])) return static_cast<unsigned int>(strtoul(s,endptr,10));
	switch(s[0])
	{
		case '$' : return static_cast<unsigned int>(strtoul(s+1,endptr,16));
		case '%' : return static_cast<unsigned int>(strtoul(s+1,endptr,2));
		case 'o' : return static_cast<unsigned int>(strtoul(s+1,endptr,8));
		default : THROWCOMFEXCEPTIONHERE();
	}
	
	ASSERT(false);
	return 0;
}		

unsigned int stou(const char * s)
{
	char * endptr = 0;
	int n = stou(s,&endptr);
	if(s+strlen(s) != endptr)
		THROWCOMFEXCEPTIONHERE();
	return n;
}

unsigned int stou(const std::string & s)
{
	return stou(s.data());
}

unsigned long stoul(const char * s, char ** endptr)
{
	if(isdigit(s[0])) return strtoul(s,endptr,10);
	switch(s[0])
	{
		case '$' : return strtoul(s+1,endptr,16);
		case '%' : return strtoul(s+1,endptr,2);
		case 'o' : return strtoul(s+1,endptr,8);
		default : THROWCOMFEXCEPTIONHERE();
	}
	
	ASSERT(false);
	return 0;
}		

unsigned long stoul(const char * s)
{
	char * endptr = 0;
	int n = stoul(s,&endptr);
	if(s+strlen(s) != endptr)
		THROWCOMFEXCEPTIONHERE();
	return n;
}

unsigned long stoul(const std::string & s)
{
	return stoul(s.data());
}


std::string ptoa(const void * p)
{
	char sk[sizeof(p)*2+16];
	snprintf(sk,COMF_DIM(sk),"%p",p);
	return std::string(sk);
}


std::string format(const char * frm, const std::vector<std::string> & dta)
{
	std::string str;
	
	size_t pos = 0;
	
	while(*frm)
	{
		if(*frm != '%')
		{
			str += (*frm);
			frm += 1;
			continue;
		};
  
		if(*(frm+1)=='$')
		{
			ASSERT(pos<dta.size());
			str += (dta[pos++]);
			frm += 2;
			continue;
		};

		if(*(frm+1)=='%')
		{
			str += '%';
			frm += 2;
			continue;
		};
	
		ASSERT(false);

	};

	ASSERT(pos==dta.size());

	return str;
}



//*/////////////////// Format

Format::Format(const char * str)
{
	_form = std::string(str);
}

Format::Format(const std::string & str)
{
	_form = str;
}

Format & Format::operator % (const char * str)
{
	_data.push_back(std::string(str));
	return *this;
}

Format & Format::operator % (const std::string & str)
{
	_data.push_back(str);
	return *this;
}

Format::operator std::string () const
{
	return format(_form.data(),_data);
}


std::vector<std::string> parse
(
	const std::string & src,
	bool usespc,
	const char * toks,
	bool toka,
	bool sdla,
	const char * coms
)
{
	std::set<char> tok;
	if(toks)
		for(;*toks;++toks)
			tok.insert(*toks);
	
	if(usespc)
	{
		tok.erase(' ');
		tok.erase('\t');
	}
	
	std::set<char> com;
	if(coms)
		for(;*coms;++coms) 
			if(tok.find(*coms)==tok.end())
				com.insert(*coms);

	
	std::vector<std::string> array;
	std::string build;
	int state = 1;

	CITERATE(std::string,it,src)	
	{
		char current = *it;
		bool curriscom = com.find(current)!=com.end();
		bool currissdl = (current=='\"');
		bool currisspc = strchr(" \t",current);
		
		switch(state)
		{
			case 1 : 
			{
				if(curriscom) return array;
				if(usespc) if(currisspc) break;
				if(tok.find(current)!=tok.end())
				{
					array.push_back(build);
					build.clear();
					if(toka) array.push_back(std::string(1,current));
					break;
				};
				if(currissdl)
				{
					if(sdla) build += current;
					state = 4;
					break;
				};
				build += current;
				state = 2;
			}; break;

			case 2 :
			{
				if(curriscom)
				{
					array.push_back(build);
					return array;
				};
				if(usespc) if(currisspc)
				{
					state = 3;
					break;
				};
				if(tok.find(current)!=tok.end())
				{
					array.push_back(build);
					build.clear();
					if(toka) array.push_back(std::string(1,current));
					state = 1;
					break;
				};
				if(currissdl)
				{
					array.push_back(build);
					build.clear();
					if(sdla) build += current;
					state = 4;
					break;
				};
				build += current;

			}; break;

			case 3 :
			{
				if(curriscom)
				{
					array.push_back(build);
					return array;
				};
				if(usespc) if(currisspc) break;
				if(tok.find(current)!=tok.end())
				{
					array.push_back(build);
					build.clear();
					if(toka) array.push_back(std::string(1,current));
					state = 1;
					break;
				};
				if(currissdl)
				{
					array.push_back(build);
					build.clear();
					if(sdla) build += current;
					state = 4;
					break;
				};

				array.push_back(build);
				build.clear();
				build += current;
				state = 2;
			}; break;

			case 4 :
			{
				if(currissdl)
				{
					state = 3;
					break;
				};
				if(current=='\\')
				{
					state = 5;
					break;
				};
				build += current;
			}; break;

			case 5 :
			{
				build += current;
				state = 4;
			}; break;
		
			default : ASSERT(false);
		};
	};

	if(state==2 || state==3) array.push_back(build);

	return array;
}


void strupr(char * str)
{
	while(*str)
	{
		*str = static_cast<char>(::toupper(*str));
		++str;
	}
}

void strlwr(char * str)
{
	while(*str)
	{
		*str = static_cast<char>(::tolower(*str));
		++str;
	}
}

void makeupper(std::string & str)
{
	size_t n = str.size();
	LOOP(size_t,j,n) str.at(j) = static_cast<char>(::toupper(str.at(j)));
}

void makelower(std::string & str)
{
	size_t n = str.size();
	LOOP(size_t,j,n) str.at(j) = static_cast<char>(::tolower(str.at(j)));
}


std::string toupper(const std::string & a)
{
	std::string b;
	CITERATE(std::string,it,a) b += static_cast<char>(::toupper(*it));
	return b;
}

std::string tolower(const std::string & a)
{
	std::string b;
	CITERATE(std::string,it,a) b += static_cast<char>(::tolower(*it));
	return b;
}


std::string rjus(const std::string & a, std::string::size_type n, char c)
{
	std::string::size_type w = a.size();
	if(w>=n) return a;
	return std::string(n-w,c)+a;
}

std::string ljus(const std::string & a, std::string::size_type n, char c)
{
	std::string::size_type w = a.size();
	if(w>=n) return a;
	return a+std::string(n-w,c);
}

std::string quote(const char * s)
{
	std::string a;
	a.append("\"");

	while(*s)
	{
		switch(*s)
		{
		case '\\' : a.append("\\\\"); break;
		case '\'' : a.append("\\\'"); break;
		case '\"' : a.append("\\\""); break;
		case '\n' : a.append("\\n"); break;
		case '\r' : a.append("\\r"); break;
		
		default:
			if(*s<32)
			{
				char tmp[] = "\\xHH";
				COMF_ASSERT(sizeof(tmp)==5);
				utobase(unsign(*s),tmp+2,16,2);
				a += tmp;
			}
			else
				a += *s;
		}
		++s;
	}

	a.append("\"");
	return a;
}

std::string quote(const std::string & s)
{
	return quote(s.data());
}


std::string escape(const char * s)
{
	std::string a;
	while(*s)
	{
		switch(*s)
		{
		case ' '  : a.append("\\ "); break;
		case '\\' : a.append("\\\\"); break;
		case '\'' : a.append("\\\'"); break;
		case '\"' : a.append("\\\""); break;
		case '\n' : a.append("\\n"); break;
		case '\r' : a.append("\\r"); break;
		
		default:
			if(*s<32)
			{
				char tmp[] = "\\xHH";
				COMF_ASSERT(sizeof(tmp)==5);
				utobase(unsign(*s),tmp+2,16,2);
				a += tmp;
			}
			else
				a += *s;
		}
		++s;
	}
	return a;
}

std::string escape(const std::string & s)
{
	return escape(s.data());
}



COMF_NS_END
