/*
Copyright (C)  2006  Daniele Zelante

This file is part of comf.

comf is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

comf is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with comf; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/*@LICENSE*/
// $Id: bstorage.cxx,v 1.2 2006/01/02 22:08:04 zeldan Exp $

#include "bstorage.hxx"

#include "memory.hxx"
#include "exception.hxx"
#include "mymacros.hxx"



COMF_NS_BEGIN

//**** BIStorage

BIStorage::BIStorage(const ConstSegment & data) : _data(data)
{
	_data.addUser(this);
	_ptr = 0;
}

BIStorage::~BIStorage()
{
	ASSERT(_blocks.empty());
	//if(_ptr != _data.size()) logmsg();
	_data.delUser(this);
}

const char * BIStorage::ptr() const
{
	return _data.cptr() + _ptr;
}

BIStorage & BIStorage::operator >> (Cmd cmd)
{
	switch(cmd)
	{
		case BEGIN:
		{
			size_t base = _ptr;
			size_t z;
			(*this) >> z;
			_blocks.push(base+z);
		}; break;
			
		case END:
		{
			ASSERT(!_blocks.empty());
			size_t shld = _blocks.top();
			if(_ptr != shld) THROWCOMFEXCEPTIONHERE();
			_blocks.pop();
		}; break;

		case ABORT:
		{
			ASSERT(!_blocks.empty());
			size_t shld = _blocks.top();
			_ptr = shld;
			_blocks.pop();
		}; break;

		default: ASSERT(false);
	}
	return *this;
}


void BIStorage::read(void * p, size_t n)
{
	if(_ptr + n > _data.size()) THROWCOMFEXCEPTIONHERE();
	memcpy(p,_data.cptr()+_ptr,n);
	_ptr += n;
}

void BIStorage::skip(size_t n)
{
	if(_ptr + n > _data.size()) THROWCOMFEXCEPTIONHERE();
	_ptr += n;
}
	


BIStorage & BIStorage::operator >> (Serializable & o)
{
	o.serialize(*this);
	return *this;
}

BIStorage & BIStorage::operator >> (unsigned int & n)
{
	read(&n,sizeof(n));
	return *this;
}

BIStorage & BIStorage::operator >> (int & n)
{
	read(&n,sizeof(n));
	return *this;
}

BIStorage & BIStorage::operator >> (unsigned long & n)
{
	read(&n,sizeof(n));
	return *this;
}

BIStorage & BIStorage::operator >> (long & n)
{
	read(&n,sizeof(n));
	return *this;
}

BIStorage & BIStorage::operator >> (bool & n)
{
	read(&n,sizeof(n));
	return *this;
}

BIStorage & BIStorage::operator >> (double & n)
{
	read(&n,sizeof(n));
	return *this;
}


BIStorage & BIStorage::operator >> (std::string & a)
{
	size_t sz;
	(*this) >> sz;
	a = std::string(ptr(),sz);
	skip(sz);
	return *this;
}

#ifdef MAKROZ_HASWIDE
BIStorage & BIStorage::operator >> (std::wstring & a)
{
	size_t sz;
	(*this) >> sz;
	a = std::wstring(reinterpret_cast<const wchar_t *>(ptr()),sz);
	skip(sz*sizeof(wchar_t));
	return *this;
}
#endif


//**** BOStorage

BOStorage::BOStorage(DynamicSegment & data) : _data(data)
{
	_data.addUser(this);
	_ptr = 0;
}

BOStorage::~BOStorage()
{
	ASSERT(_blocks.empty());
	_data.delUser(this);
}

void BOStorage::write(const void * p, size_t n)
{
	_data.resize(_ptr+n);
	memcpy(_data.ptr()+_ptr,p,n);
	_ptr += n;
}


BOStorage & BOStorage::operator << (const Serializable & o)
{
	o.serialize(*this);
	return *this;
}

BOStorage & BOStorage::operator << (Cmd cmd)
{
	switch(cmd)
	{
		case BEGIN:
		{
			_blocks.push(_ptr);
			_ptr += sizeof(size_t);
		}; break;
			
		case END:
		{
			ASSERT(!_blocks.empty());
			size_t prev = _blocks.top();
			_blocks.pop();
			size_t dist = _ptr - prev;
			memcpy(_data.ptr()+prev,&dist,sizeof(size_t));
		}; break;

		default: ASSERT(false);
	}
	return *this;
}

BOStorage & BOStorage::operator << (unsigned int n)
{
	write(&n,sizeof(n));
	return *this;
}

BOStorage & BOStorage::operator << (int n)
{
	write(&n,sizeof(n));
	return *this;
}

BOStorage & BOStorage::operator << (unsigned long n)
{
	write(&n,sizeof(n));
	return *this;
}

BOStorage & BOStorage::operator << (long n)
{
	write(&n,sizeof(n));
	return *this;
}


BOStorage & BOStorage::operator << (bool n)
{
	write(&n,sizeof(n));
	return *this;
}

BOStorage & BOStorage::operator << (double n)
{
	write(&n,sizeof(n));
	return *this;
}


BOStorage & BOStorage::operator << (const std::string & a)
{
	size_t sz = a.size();
	(*this) << sz;
	write(a.data(),sz);
	return *this;
}

#ifdef MAKROZ_HASWIDE
BOStorage & BOStorage::operator << (const std::wstring & a)
{
	size_t sz = a.size();
	(*this) << sz;
	write(a.data(),sz*sizeof(wchar_t));
	return *this;
}
#endif

COMF_NS_END
