/*
**  Inspector.m
**
**  Copyright (c) 2003
**
**  Author: Yen-Ju Chen <yjchenx@hotmail.com>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "Inspector.h"
#include "Utilities.h"
#include "DocumentController.h"
#include <AppKit/AppKit.h>
#include <CodeEditorView/CodeEditorView.h>
#include <CodeEditorView/RulesetManager.h>
#include <CodeEditorView/CEView+Inspector.h>

static Inspector *sharedInspector;

@implementation Inspector

+ (Inspector *) sharedInspector
{
  if (!sharedInspector)
    {
      sharedInspector = [[Inspector alloc] init];
    }
  return sharedInspector;
}

- (void) windowDidBecomeKey: (NSNotification *) not
{
  [self setCodeEditorView: currentCEView];
}

- (void) setCodeEditorView: (CodeEditorView *) ceView
{
  ASSIGN(currentCEView, ceView);
  
  if (currentCEView == nil)
    return;
  
  if (inspectorPanel)
    {
      NSString *filename;
      int lines, glyphs;
      int overall, line, character;
      
      filename = [[[DocumentController sharedDocumentController] currentDocument] fileName];

      if(filename)
        {          
          [filenameField setStringValue: [filename lastPathComponent]];
          [pathField setStringValue: [filename stringByDeletingLastPathComponent]];
          if([showButton superview] == nil)
            {
              [[inspectorPanel contentView] addSubview: showButton];
              RELEASE(showButton);
            }
        }
      else 
        {
          [filenameField setStringValue: @"Untitled"];
          [pathField setStringValue: @""];
          if ([showButton superview] != nil)
            {
              RETAIN(showButton);
              [showButton removeFromSuperview];
            }
        }

      [languageButton selectItemWithTitle: [currentCEView languageName]];
      
      [currentCEView getCursorPosOverall: &overall inLine: &character atLine: &line];
      [overallField setIntValue: overall];
      [charField setIntValue: character];
      [lineField setIntValue: line];

      [currentCEView getNumberOfGlyphs:&glyphs lines: &lines];
      [linesField setIntValue: lines];
      [bytesField setIntValue: glyphs];
      [inspectorPanel setViewsNeedDisplay: YES];
    }
}

- (NSPanel *) inspectorPanel
{
  if (inspectorPanel == nil)
    {
      NSArray *array;

      if(NO == [NSBundle loadNibNamed: @"Inspector.gorm" owner: self])
        {
           NSBeep();
           return nil;
        }
      [showButton setImage: [NSImage imageNamed: @"OpenInWorkspace.tiff"]];
      [showButton setImagePosition: NSImageOnly];
      [languageButton removeAllItems];
      [languageButton addItemWithTitle: _(@"Text")];
      array = [currentCEView supportedLanguages];
      [languageButton addItemsWithTitles: array];
    }
  return inspectorPanel;
}

/* Inspector Panel */
- (void) orderFrontInspectorPanel: (id) sender
{
  if ([self inspectorPanel] == nil)
    {
      NSBeep();
      return;
    }
  [self setCodeEditorView: currentCEView];
  [inspectorPanel makeKeyAndOrderFront: self];
}

- (void) gotoChar: (id) sender
{
  int character = [sender intValue];
  
  if (currentCEView == nil)
    return;

  if(character < 0)
    return;
  if(character < [[currentCEView string] length])
      {
	    [currentCEView setSelectedRange: NSMakeRange(character, 0)];
	    [[currentCEView window] makeKeyAndOrderFront: self];
      }
  else
    NSBeep();
}

- (void) gotoCharAndLine: (id) sender
{
  int line, character;
  
  if (currentCEView == nil)
    return;

  line = [lineField intValue];
  character = [charField intValue];

  if(line < 1 || character < 0)
    return;
  
  [currentCEView gotoLineNum: line charInLine: character];
  [[currentCEView window] makeKeyAndOrderFront: self];
}

- (void) showInWorkspace: (id) sender
{
  NSString *filename, *rootPath;
  BOOL result;

  NSAssert(currentCEView != nil, @"\ncurrentCEView should exist");

  filename = [[[DocumentController sharedDocumentController] currentDocument] fileName];

  rootPath = [filename stringByDeletingLastPathComponent];

  result = [[NSWorkspace sharedWorkspace] selectFile: filename
		                    inFileViewerRootedAtPath: rootPath];

  if (result == NO)
    NSBeep();
}

- (void) setLanguage: (id) sender
{
  NSAssert(currentCEView != nil, @"\ncurrentCEView should exist");
  
  [currentCEView setLanguageName: [sender titleOfSelectedItem]];
}

- (void) dealloc
{
  RELEASE(currentCEView);
  [super dealloc];
}

@end
