#include "mainwindow.h"
#include "ui_mainwindow.h"
#include <cmath>
#include <QDebug>
#include <list>
#include <QFileDialog>
#include <QMessageBox>
#include <math.h>
#include "extras.h"
#include <unistd.h>

#define ALTURA_DA_ESCALA 284

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    char *PROGRAM_DIR=get_absolute_speckle_tool_path();

    lastdir=QDir::currentPath();

    ui->setupUi(this);
    ui->actionSave_Results->setIcon(QIcon(QString(PROGRAM_DIR)+"../share/qualitytest/icons/save.png"));
    ui->actionHelp->setIcon(QIcon(QString(PROGRAM_DIR)+"../share/qualitytest/icons/help.png"));
    ui->actionDoc->setIcon(QIcon(QString(PROGRAM_DIR)+"../share/qualitytest/icons/help.png"));
    ui->actionAbout->setIcon(QIcon(QString(PROGRAM_DIR)+"../share/qualitytest/icons/about.png"));

    char * str=get_absolute_filenamepath();
    ui->statusBar->showMessage(str,40000);
    free(str);str=NULL;

    setWindowTitle("Speckle Quality Test :: "+QString(APP_VERSION));
    setWindowIcon(QIcon(QString(PROGRAM_DIR)+"../share/qualitytest/icons/sqt_icon.ico"));

    check_pushbuttons();

    connect(ui->actionSave_Results, SIGNAL(triggered()),this, SLOT(save_results()));
    connect(ui->actionAbout, SIGNAL(triggered()), this, SLOT(show_about()));
    connect(ui->actionHelp, SIGNAL(triggered()), this, SLOT(show_help()));
    connect(ui->actionDoc, SIGNAL(triggered()), this, SLOT(show_doc()));


    not_first = false;
}

MainWindow::~MainWindow()
{
    delete ui;
}

void MainWindow::setLastDir(const char* dir)
{
    if(dir==NULL)   lastdir=QString("");
    else            lastdir=QString(dir);
}

QList<int> MainWindow::findIntegerDivisors(int number)
{
    QList<int> list;

    for(int i = 2; i <= (number / 2); i++)
    {
        if(number % i == 0)
            list << i;
    }

    return list;

}

IplImage* MainWindow::quadImage(IplImage *image, int horizontal_step,
                                  int vertical_step)
{
    IplImage* result = cvCreateImage(cvGetSize(image), 8, 3);
    cvCopy(image, result);

    // Horizontal Lines
    for(int i = vertical_step; i < result->height; i += vertical_step)
    {
        cvLine(result, cvPoint(0, i), cvPoint(result->widthStep, i),
               cvScalar(0, 255, 0, 0), 1, 8, 0); // cvScalar(B, G, R, alfa);
    }
    // Vertical Lines
    for (int i = horizontal_step; i < result->widthStep; i += horizontal_step)
    {
        cvLine(result, cvPoint(i, 0), cvPoint(i, image->height),
               cvScalar(0, 255, 0, 0), 1, 8, 0);
    }

    return result;
}

void MainWindow::initialize()
{
    integerDivisors_horizontal = findIntegerDivisors(img->width);
    integerDivisors_vertical = findIntegerDivisors(img->height);

    pos_list_horizontal = integerDivisors_horizontal.size()/2;
    pos_list_vertical = integerDivisors_vertical.size()/2;

    ui->lcd_horizontal_value->display(
                integerDivisors_horizontal.at(pos_list_horizontal));
    ui->lcd_vertical_value->display(
                integerDivisors_vertical.at(pos_list_vertical));
    check_pushbuttons();
    preview_quad();
}

// PREVIEW
void MainWindow::preview_quad()
{
    aux = quadImage(img, ui->lcd_horizontal_value->value(),
                    ui->lcd_vertical_value->value());
    cvShowImage("Preview", aux);
}

IplImage* MainWindow::homogeneity_test(int lar_ROI, int alt_ROI)
{
    // Executa MHI. ************************************************************

    Mhi* mhiObj = new Mhi(ui->spinBox_buffer_size->value());

    // Imagens Auxiliares.
    IplImage* result = cvCreateImage(cvGetSize(img), 8, 3);
    IplImage* aux_gray = cvCreateImage(cvGetSize(img), 8, 1);

    // Calcula MHI

    mhiObj->update_mhi(img, result, aux_gray,
                            ui->horizontalSlider_threshold_MHI->value(),
                            ui->doubleSpinBox_Duration_MHI->value());

    //sleep((unsigned int)(10.0/ui->spinBox_fps->value()));

    for(int i = 1; i < (ui->spinBox_buffer_size->value() * 2); i++)
    {
        img = cvLoadImage(filesName.at(i).toLocal8Bit().data());
        mhiObj->update_mhi(img, result, aux_gray,
                                ui->horizontalSlider_threshold_MHI->value(),
                                ui->doubleSpinBox_Duration_MHI->value());
        //sleep((unsigned int)(10.0/ui->spinBox_fps->value()));

    }

    // Calculo das Medias das ROI.**********************************************

    // Variaveis auxiliares do laco.
    int lar_original = result->width, alt_original = result->height;
    unsigned long long int soma;
    double media;

    // Matriz de medias das ROI.
    int NumAltROI=alt_original/alt_ROI;
    int NumLarROI=lar_original/lar_ROI;
    int medias [NumAltROI][NumLarROI];

    // Parametros de criacao do histograma.
    int dim = 1;               // Numero de dimensoes do histograma.
    int hist_size = 256;       // Tamanho do histograma.
    int sizes[] = {hist_size}; // Tamanho do vetor de cada dimensao.
    int type = CV_HIST_ARRAY;  // Tipo de Histograma.
    float xranges[] = { 0, 255 };
    float* ranges[] = { xranges };

    // Cria e calcula o histograma.
    CvHistogram* hist = cvCreateHist( dim, sizes, type, ranges, 1 );

    // Percorre todas as ROI da imagem. medias[0:NumHRoi][0:NumWRoi] -> (x,y) 
    for (int x = 0; x < NumAltROI; x++)
    {
        for(int y = 0; y < NumLarROI; y++)
        {
            soma = 0;
            media = 0.0;

            // Cria o retangulo da ROI. en (x,y)
            CvRect rect = cvRect((lar_ROI * y),(alt_ROI * x),lar_ROI, alt_ROI);

            // Determina a ROI na imagem.
            cvSetImageROI(aux_gray, rect);

            // Calcula o Histograma.
            cvCalcHist( &aux_gray, hist, 0, NULL);

            // Faz a soma das intensidades apartir do histograma.
            for (int i = 0; i < 256; i++)
                soma = soma + (i * cvQueryHistValue_1D(hist, i));

            // calculo da media.
            media = soma / (lar_ROI * alt_ROI);

            // Armazena as medias na matriz
            medias[x][y] = media;

            cvResetImageROI(aux_gray);

            //qDebug() << "ROI (" << x << ", " << y << ") MHI Medio: " << medias[x][y]  ;
        }
    }

    // Calculo da Homogeneidade das ROI. size(homog_matrix) == size(meadias)
    QList<int> homog_list;
    int homog_matrix [NumAltROI][NumLarROI];
    double max = 0, min = 1000000;

    // Percorre a matriz.
    for(int x = 0; x < NumAltROI; x++)
    {
        for (int y = 0; y < NumLarROI; y++)
        {
            // Limpa a Lista.
            homog_list.clear();

            homog_list.append(medias[x][y]);

            if(x != 0)
                homog_list.append(medias[x-1][y]);

            if(x != (NumAltROI - 1))
                homog_list.append(medias[x+1][y]);

            if(y != 0)
                homog_list.append(medias[x][y-1]);

            if(y != (NumLarROI - 1))
                homog_list.append(medias[x][y+1]);

            homog_matrix[x][y] = calc_homogeneity(homog_list);

            if (homog_matrix[x][y] < min)
                min = homog_matrix[x][y];
            if (homog_matrix[x][y] > max)
                max = homog_matrix[x][y];

//            qDebug() << "ROI (" << x << ", " << y << ") homogeneidade: " << homog_matrix[x][y]  ;
        }
    }

    // Exibe o resultado da Homogeneidade. *************************************

    // Percorre todas as ROI da imagem e normaliza o resultado.
    for (int x = 0; x < NumAltROI; x++)
    {
        for(int y = 0; y < NumLarROI; y++)
        {
            // Cria o retangulo da ROI.
            CvRect rect = cvRect((lar_ROI * y),(alt_ROI * x),lar_ROI, alt_ROI);

            // Determina a ROI na imagem.
            cvSetImageROI(aux_gray, rect);

            // Normaliza entre 0 e 1 com uma casa decimal.
            double a = round(10 * (homog_matrix[x][y] - min)/(max - min) ) / 10;
//            qDebug() << "ROI (" << x << "," << y << ") homogeneidade: " << a;
            cvSet(aux_gray, cvScalarAll(255 * a));

            cvResetImageROI(aux_gray);
        }
    }

    // Colore e quadricula o resultado da homogeneidade.
    result = coloreImagem(aux_gray);
    result = quadImage(result, lar_ROI, alt_ROI);

    IplImage* final_result;

    if (result->height > ALTURA_DA_ESCALA)
    {
        // Cria imagem do resultado final com um branco levemente escurecido.
        final_result = cvCreateImage(cvSize(result->width + 114,
                                                      result->height), 8, 3);
    }
    else
    {
        final_result = cvCreateImage(cvSize(result->width + 114,
                                                      ALTURA_DA_ESCALA), 8, 3);
    }

    cvSet(final_result, cvScalarAll(245));

    // Seta a Regiao e cola o resultado da homogeneidade.
    if (result->height > 284)
        cvSetImageROI(final_result,cvRect(0, 0, result->width, result->height));
    else
        cvSetImageROI(final_result,cvRect(0,
                                          ALTURA_DA_ESCALA/2 - result->height/2,
                                          result->width, result->height));

    cvCopy(result, final_result);

    cvResetImageROI(final_result);

    // Seta a Regiao, carrega a imagem da escala e cola no resultado final.
    char *PROGRAM_DIR=get_absolute_speckle_tool_path();
    QString imgfilename=QString(PROGRAM_DIR)+"../share/qualitytest/img/homogeneity_scale.jpg";
    result = cvLoadImage(imgfilename.toLocal8Bit().data());

    if (aux_gray->height > ALTURA_DA_ESCALA)
        cvSetImageROI(final_result, cvRect((aux_gray->width + 10),
                                       (aux_gray->height/2 - result->height/2),
                                       result->width, result->height));
    else
        cvSetImageROI(final_result, cvRect((aux_gray->width + 10), 0,
                                       result->width, result->height));

    cvCopy(result, final_result);

    cvResetImageROI(final_result);

    cvReleaseImage(&aux_gray);
    cvReleaseImage(&result);

    return final_result;
}

double MainWindow::calc_homogeneity(QList<int> list)
{
    // Variaveis de calculo
    unsigned long long int soma = 0;
    double std = 0.0, media = 0.0;

    // Percorre a lista
    for (int i = 0; i < list.size(); i++)
        soma = soma + list.at(i);

    //calculo da media.
    media = soma / list.size();

    //calculo do desvio padrao.
    for (int i =0; i<list.size(); i++)
        std = pow((list.at(i) - media), 2);

    std = sqrt(std / (list.size() - 1));

    return (100 * (std/(media + 1)));
}

/** ColoreImagem:
 * Metodo utilizado para colorir uma imagem do formato iplImage de 8-bits e um
 * canal. Recebe a imagem como parametro, cria uma nova imagem de 8-bits e tres
 * canais na qual sera feito a coloracao artificial da imagem recebida.
 * @return: IplImage*
 * @param: IplImage* -> ponteiro para a imagem que sera colorida.
 * @author: Michel Melo da Silva
 * @date: 2011/02/14
 * @last: 2011/02/14
 */
IplImage* MainWindow::coloreImagem(IplImage *image)
{
    // Cria uma imagem de 8-bits e 3 canais com a mesma resolucao da imagem
    // recebida como parametro.
    IplImage* result = cvCreateImage(cvGetSize(image), 8, 3);

    int red,green,blue;
    unsigned char aux;

    // Percorre todos os dados da imagem.
    for (int y = 0; y < image->imageSize; y++) {

        aux = image->imageData[y];

        if(aux < 64){
            red = 0;
            green = (aux * 4);
            blue = 255;
         } else {
             if(aux < 128){
                red = 0;
                green = 255;
                blue = (255 - (aux - 64) * 4);
            } else {
                if(aux < 192){
                    red = ((aux - 128) * 4);
                    green = 255;
                    blue = 0;
                } else {
                    red = 255;
                    green = (255 - (aux - 192) * 4);
                    blue = 0;
                }
            }
        }

        result->imageData[y*3+2] = red;
        result->imageData[y*3+1] = green;
        result->imageData[y*3+0] = blue;

    }

    return result;
}

/** Contrast Test:
 * Metodo utilizado para verificar o contraste de cada ROI (Region of Interest)
 * da imagem. Retorna uma imagem colorida (RGB) quadriculada onde cada quadrado
 * sera pintato com o valor de contraste encontrado naquela area. O parametro de
 * entrada eh uma imagem colorida.
 * @return: IplImage*
 * @param: IplImage*
 * @author: Michel Melo da Silva
 * @date: 2012/10/17
 * @last: 2012/10/17
 */
IplImage* MainWindow::contrast_test(IplImage *image)
{
    // Copia da imagem original em tons de cinza para processamento.
    IplImage* result = cvCreateImage(cvGetSize(image), 8, 1);
    cvCvtColor(image, result, CV_RGB2GRAY);

    // Variaveis para a ROI.
    int alt_original = image->height;
    int lar_original = image->width;
    int alt_ROI = ui->lcd_vertical_value->value();
    int lar_ROI = ui->lcd_horizontal_value->value();

    // Variaveis para calculo Contraste.
    unsigned long int soma, somaq, pixel;
    double var, std, media;

    // Percorre todas as ROI da imagem.
    for (int x = 0; x < (lar_original/lar_ROI); x++)
    {
        for(int y = 0; y < (alt_original/alt_ROI); y++)
        {
            // Cria o retangulo da ROI.
            CvRect rect = cvRect((lar_ROI * x),(alt_ROI * y),lar_ROI, alt_ROI);

            // Determina a ROI na imagem.
            cvSetImageROI(result, rect);

            // Seta as variaveis.
            soma = 0;
            somaq = 0;
            var = 0.0;
            std = 0.0;
            media = 0.0;

            // Calcula a soma das intensidades (soma )e a soma das intensidades
            // ao quadrado (somaq) dentro da ROI.
            for (int i = rect.y; i < (rect.y + rect.height); i++)
            {
                for (int j = rect.x; j < (rect.x + rect.width); j++)
                {
                    soma += result->imageData[(i * result->widthStep + j)];
                    somaq += pow(result->imageData[(i * result->widthStep + j)],
                                 2);
                }
            }

            //calculo da media.
            media = soma / (lar_ROI * alt_ROI);

            //calculo da variancia.
            var = (somaq / (lar_ROI * alt_ROI)) - pow(media, 2);

            //calculo do desvio padrao.
            std = sqrt(var);

            pixel = round(( (std/media))*255);

//            //Debuga as variaveis.
//            qDebug() << "soma: " << soma;
//            qDebug() << "somaq: " << somaq;
//            qDebug() << "media: " << media;
//            qDebug() << "variancia: " << var;
//            qDebug() << "dp: " << std;
//            qDebug() << "contraste: " << std/media;
//            qDebug() << "pixel: " << (round(((std/media))*255));
//                        qDebug();

            // Colore a ROI na imagem resultado com o valor pixel.
            cvSet(result, cvRealScalar(pixel));

            // Reseta a ROI da imagem.
            cvResetImageROI(result);

        }
    }

    // Converte o resultado para um imagem colorida de 3 canais (RGB) e
    // quadricula a imagem para retorno.
    IplImage* resultcolor = cvCreateImage(cvGetSize(result), 8, 3);
    cvCvtColor(result, resultcolor, CV_GRAY2RGB);
    resultcolor = quadImage(resultcolor, lar_ROI, alt_ROI);

    IplImage* final_result;

    if (resultcolor->height > ALTURA_DA_ESCALA)
    {
        // Cria imagem do resultado final com um branco levemente escurecido.
        final_result = cvCreateImage(cvSize(resultcolor->width + 114,
                                            resultcolor->height), 8, 3);
    }
    else
    {
        final_result = cvCreateImage(cvSize(resultcolor->width + 114,
                                                      ALTURA_DA_ESCALA), 8, 3);
    }

    cvSet(final_result, cvScalarAll(245));

    // Seta a Regiao e cola o resultado da homogeneidade.
    if (resultcolor->height > ALTURA_DA_ESCALA)
        cvSetImageROI(final_result,cvRect(0, 0, resultcolor->width,
                                          resultcolor->height));
    else
        cvSetImageROI(final_result,cvRect(0,
                                          ALTURA_DA_ESCALA/2 - result->height/2,
                                          result->width, result->height));

    cvCopy(resultcolor, final_result);

    cvResetImageROI(final_result);

    // Seta a Regiao, carrega a imagem da escala e cola no resultado final.
    char *PROGRAM_DIR=get_absolute_speckle_tool_path();
    QString imgfilename=QString(PROGRAM_DIR)+"../share/qualitytest/img/contrast_scale.jpg";
    resultcolor = cvLoadImage(imgfilename.toLocal8Bit().data());

    if (final_result->height > ALTURA_DA_ESCALA)
        cvSetImageROI(final_result, cvRect((result->width + 10),
                                     (result->height/2 - resultcolor->height/2),
                                      resultcolor->width,
                                      resultcolor->height));
    else
        cvSetImageROI(final_result, cvRect((result->width + 10), 0,
                                       resultcolor->width,
                                           resultcolor->height));

    cvCopy(resultcolor, final_result);

    cvResetImageROI(final_result);

    // Desaloca a imagem em tons de cinza e retorna o resultado colorido.
    cvReleaseImage(&result);
    cvReleaseImage(&resultcolor);
    return final_result;
}

/** Saturation Test:
 * Metodo utilizado para verificar a saturacao de cada ROI (Region of Interest)
 * da imagem. Retorna uma imagem quadriculada, colorida (RGB), com um circulo no
 * centro de cada ROI, esse circulo sera vermelho caso a ROI esteja saturada,
 * azul caso ela esteja "preta" e verde se esta for normal. Os parametros de
 * entrada sao: uma imagem colorida e a porcentagem para avaliar a ROI.
 * @return: IplImage*
 * @param: IplImage*, integer
 * @author: Michel Melo da Silva
 * @date: 2012/10/16
 * @last: 2012/10/17
 */
IplImage* MainWindow::saturation_test(IplImage *image, int percentagem)
{

    // Copia da imagem original em tons de cinza para processamento.
    IplImage* gray = cvCreateImage(cvGetSize(image), 8, 1);
    cvCvtColor(image, gray, CV_RGB2GRAY);

    // Imagem que sera retornada no resultado.
    IplImage* result = cvCreateImage(cvGetSize(image), 8, 3);
    cvCopy(image,result);

    // Variaveis para construcao da ROI.
    int alt_original = image->height;
    int lar_original = image->width;
    int alt_ROI = ui->lcd_vertical_value->value();
    int lar_ROI = ui->lcd_horizontal_value->value();

    // Percorre todas as ROI da imagem.
    for (int x = 0; x < (lar_original/lar_ROI); x++)
    {
        for(int y = 0; y < (alt_original/alt_ROI); y++)
        {
//            // Pontos:
//            qDebug() << "(" << (lar_ROI * x) << ", " << (alt_ROI * y) << ")";


            // Define a ROI.
            cvSetImageROI(gray, cvRect((lar_ROI * x),(alt_ROI * y),
                                        lar_ROI, alt_ROI));

            // Checa a situacao da cor da ROI.
            int a = color_situation_check(gray, percentagem,
                                          (alt_ROI * lar_ROI));

            // Decisao do que fazer na ROI com base na situacao encontrada.
            switch (a)
            {
                case 1:
                    // Desenha um circulo azul no centro da ROI.
                    cvCircle(result, cvPoint(((lar_ROI * x) + (lar_ROI / 2)),
                                             ((alt_ROI * y) + (alt_ROI / 2))),
                             3, cvScalar(255, 0, 0, 0), 2);
                    break;
                case 2:
                    // Desenha um circulo vermelho no centro da ROI.
                    cvCircle(result, cvPoint((lar_ROI * x) + (lar_ROI / 2),
                                             (alt_ROI * y) + (alt_ROI / 2)),
                             3, cvScalar(0, 0, 255, 0), 2);
                    break;
                case 3:
                    // Desenha um circulo verde no centro da ROI.
                    cvCircle(result, cvPoint((lar_ROI * x) + (lar_ROI / 2),
                                             (alt_ROI * y) + (alt_ROI / 2)),
                             3, cvScalar(0, 255, 0, 0), 2);
                    break;
            }

            // Reseta a ROI da imagem.
            cvResetImageROI(result);
        }
    }

    // Quadricula a image e retorna como uma imagem colorida.
    result = quadImage(result,ui->lcd_horizontal_value->value(),
                       ui->lcd_vertical_value->value());

    IplImage* final_result;

    if (result->height > ALTURA_DA_ESCALA)
    {
        // Cria imagem do resultado final com um branco levemente escurecido.
        final_result = cvCreateImage(cvSize(result->width + 114,
                                                      result->height), 8, 3);
    }
    else
    {
        final_result = cvCreateImage(cvSize(result->width + 114,
                                                      ALTURA_DA_ESCALA), 8, 3);
    }

    cvSet(final_result, cvScalarAll(245));

    // Seta a Regiao e cola o resultado.
    if (result->height > ALTURA_DA_ESCALA)
        cvSetImageROI(final_result,cvRect(0, 0, result->width, result->height));
    else
        cvSetImageROI(final_result,cvRect(0,
                                          ALTURA_DA_ESCALA/2 - result->height/2,
                                          result->width, result->height));

    cvCopy(result, final_result);

    cvResetImageROI(final_result);

    // Seta a Regiao, carrega a imagem da escala e cola no resultado final.
    char *PROGRAM_DIR=get_absolute_speckle_tool_path();
    QString imgfilename=QString(PROGRAM_DIR)+"../share/qualitytest/img/Saturation_scale.jpg";
    result = cvLoadImage(imgfilename.toLocal8Bit().data());

    if (gray->height > ALTURA_DA_ESCALA)
    {
        int X0=gray->width + 10;
        int Y0=gray->height/2 - result->height/2;
        CvRect RRR=cvRect(  X0, Y0, result->width, result->height);

        cvSetImageROI(final_result, RRR);
    }
    else
    {
        int X0=gray->width + 10;
        int Y0=0;
        CvRect RRR=cvRect(  X0, Y0, result->width,      result->height);

        cvSetImageROI(final_result, RRR);
    }


    cvCopy(result, final_result);

    cvResetImageROI(final_result);

    cvReleaseImage(&gray);
    cvReleaseImage(&result);

    return final_result;
}

/** Color Situation Check:
 * Metodo utilizado para verificar a situacao da imagem: se ela esta saturada,
 * preta ou normal. Recebe uma imagem em tons de cinza na qual sera feita a
 * checagem, um inteiro informando qual eh a porcentaem que determina a situacao
 * da imagem e um inteiro informando quantos pixels ha na regiao de interesse.
 * Retorna um inteiro com valor: 1 para imagem preta, 2 para imagem saturada ou
 * 3 para imagem normal.
 * @return: integer
 * @param: IplImage*, integer, integer
 * @author: Michel Melo da Silva
 * @date: 2012/10/16
 * @last: 2012/10/17
 */
int MainWindow::color_situation_check(IplImage *image, int percentagem,
                                      int total_pixel_ROI)
{
    // Preenche a lista com os valores de pixels pretos e saturados.
    QList<int> valores = histogram_threshold(image,
                                ui->spinBox_saturation_black_area->value(),
                                ui->spinBox_saturation_saturated_area->value());

    // Se a porcentagem de pixels preto for maior do que a porcentagem de pixels
    // na imagem retorna o valor 1.
    if (valores.at(0) > ((total_pixel_ROI)*percentagem/100))
        return 1;

    // Se a porcentagem de pixels saturados for maior do que a porcentagem de
    // pixels na imagem retorna o valor 2.
    if (valores.at(1) > ((total_pixel_ROI)*percentagem/100))
        return 2;

    // Se a imagem nao tiver nem saturada nem preta retorna 3.
    return 3;
}

/** Histogram Threshold:
 * Metodo utilizado para calcular o numero de pixels estao na regiao preta e o
 * numero de pixels saturados. Recebe como parametro uma imagem em tons de
 * cinza, o valor que sera considerado limite na regiao preta (este valor e
 * todos a baixo serao considerados regiao preta), o valor limite de saturacao
 * (os pixels com esse valor de intensidade e todos os valores acima serao
 * considerados saturados). O retorno da funcao eh uma lista de inteiros onde o
 * primeiro elemento eh a quantidade de pixels em regiao preta e o segundo
 * elemento sao os pixels saturados.
 * @return: QList<int>
 * @param: IplImage*, integer, integer
 * @author: Michel Melo da Silva
 * @date: 2012/10/16
 * @last: 2012/10/17
 */
QList<int> MainWindow::histogram_threshold(IplImage *image, int threshold_min,
                                           int threshold_max)
{
    // Parametros de criacao do histograma.
    int dim = 1;               // Numero de dimensoes do histograma.
    int hist_size = 256;       // Tamanho do histograma.
    int sizes[] = {hist_size}; // Tamanho do vetor de cada dimensao.
    int type = CV_HIST_ARRAY;  // Tipo de Histograma.

    float xranges[] = { 0, 255 };
    float* ranges[] = { xranges };

    // Cria e calcula o histograma.
    CvHistogram* hist = cvCreateHist( dim, sizes, type, ranges, 1 );
    cvCalcHist( &image, hist, 0, NULL);

    // Lista de inteiros que sera retornada.
    QList<int> list;

    int total = 0;

    // Total recebe o numero de pixels que tem intensidade abaixo do valor min.
    for (int i = 0; i <= threshold_min; i++)
        total += cvQueryHistValue_1D(hist, i);

    // Coloca o numero de pixels pretos na primeira posicao da lista.
    list << total;

    total = 0;

    // Total recebe o numero de pixels que tem intensidade acima do valor max.
    for (int i = threshold_max; i < 256; i++)
        total += cvQueryHistValue_1D(hist, i);

    // Coloca o numero de pixels saturados na segunda posicao da lista.
    list << total;

    // Retorna a lista com duas posicoes.
    return list;
}

void MainWindow::check_pushbuttons()
{
    if(integerDivisors_vertical.isEmpty())
    {
        ui->pushButton_up_horizontal->setEnabled(false);
        ui->pushButton_down_horizontal->setEnabled(false);
        ui->pushButton_up_vertical->setEnabled(false);
        ui->pushButton_down_vertical->setEnabled(false);
    } else
    {
        if(pos_list_horizontal == 0)
            ui->pushButton_down_horizontal->setEnabled(false);
        else
            ui->pushButton_down_horizontal->setEnabled(true);
        if(pos_list_horizontal == (integerDivisors_horizontal.size() - 1))
            ui->pushButton_up_horizontal->setEnabled(false);
        else
            ui->pushButton_up_horizontal->setEnabled(true);
        if(pos_list_vertical == 0)
            ui->pushButton_down_vertical->setEnabled(false);
        else
            ui->pushButton_down_vertical->setEnabled(true);
        if(pos_list_vertical == (integerDivisors_vertical.size() - 1))
            ui->pushButton_up_vertical->setEnabled(false);
        else
            ui->pushButton_up_vertical->setEnabled(true);
    }
}

void MainWindow::on_pushButton_down_horizontal_clicked()
{
    if((pos_list_horizontal - 1) >= 0)
    {
        pos_list_horizontal--;
        ui->lcd_horizontal_value->display(
                    integerDivisors_horizontal.at(pos_list_horizontal));
    }
    preview_quad();
    check_pushbuttons();
}

void MainWindow::on_pushButton_up_horizontal_clicked()
{
    if((pos_list_horizontal + 1) < integerDivisors_horizontal.size())
    {
        pos_list_horizontal++;
        ui->lcd_horizontal_value->display(
                    integerDivisors_horizontal.at(pos_list_horizontal));
    }
    preview_quad();
    check_pushbuttons();
}

void MainWindow::on_pushButton_down_vertical_clicked()
{
    if((pos_list_vertical - 1) >= 0)
    {
        pos_list_vertical--;
        ui->lcd_vertical_value->display(
                    integerDivisors_vertical.at(pos_list_vertical));
    }
    preview_quad();
    check_pushbuttons();
}

void MainWindow::on_pushButton_up_vertical_clicked()
{
    if((pos_list_vertical + 1) < integerDivisors_vertical.size())
    {
        pos_list_vertical++;
        ui->lcd_vertical_value->display(
                    integerDivisors_vertical.at(pos_list_vertical));
    }
    preview_quad();
    check_pushbuttons();
}

void MainWindow::on_pushButton_load_clicked()
{

    if (not_first)
    {
        cvReleaseImage(&img);
        cvReleaseImage(&sat);
        cvReleaseImage(&cont);
        cvReleaseImage(&homo);
        cvReleaseImage(&aux);
    }

    //printf("hito1\n");
    filesName = QFileDialog::getOpenFileNames(this,
                                              tr("Select Images"),
                                              lastdir,
                           tr("Arquivos de Imagem (*.png *.jpg *.jpeg *.bmp)"));

    //printf("hito2\n";

    

    if(!filesName.isEmpty())
    {
        lastdir=QFileInfo(filesName.at(0)).absoluteDir().absolutePath();

        if( ((filesName.size()%2)!=0)||(filesName.size()<8) )
        {
            QString msg = "The number of images should be even and at least 8. ";
            msg.append(QString::number(filesName.size()));
            msg.append(" images.");
            QMessageBox::critical(this, "Error", msg);

        }
        else
        {
            int Nimgs=filesName.size();
            ui->spinBox_buffer_size->setValue(Nimgs/2);
            

                ui->pushButton_qt->setEnabled(true);

                img = cvLoadImage(filesName.at(0).toLocal8Bit().data());

                initialize();

                ui->spinBox_buffer_size->setEnabled(false);
                ui->spinBox_fps->setEnabled(false);
                ui->horizontalSlider_threshold_MHI->setEnabled(false);
                ui->doubleSpinBox_Duration_MHI->setEnabled(false);

            QString msg = "The images were loaded!";
            QMessageBox::information(this, "Loaded", msg);
          
        }
    }

}

// desenha a escala da homogeneidade.
IplImage* MainWindow::scale()
{
    IplImage* aa = cvCreateImage(cvSize(220,20),8,1);
    IplImage* color = cvCreateImage(cvSize(220,20),8,3);

    for (double i = 0; i <= 10 ; i++)
    {

        cvSetImageROI(aa,cvRect(i*20,0,20,20));

        cvSet(aa,cvScalarAll(255 * (i/10)));
        cvResetImageROI(aa);
    }

    color = coloreImagem(aa);
    cvShowImage("palete", color);
    cvSaveImage("1.jpg",color);

    return color;
}

void MainWindow::on_spinBox_buffer_size_valueChanged(int arg1)
{
    ui->label_number_of_images->setText(QString::number(arg1 * 2));

    if(ui->checkBox_MHIduration_auto->isChecked())
        ui->doubleSpinBox_Duration_MHI->setValue((double)arg1/
                                                 ui->spinBox_fps->value());
}

void MainWindow::on_pushButton_qt_clicked()
{

    if(!not_first)
    {
        char *PROGRAM_DIR=get_absolute_speckle_tool_path();
        ui->actionSave_Results->setIcon(QIcon(QString(PROGRAM_DIR)+"../share/qualitytest/icons/save.png"));
    }
    not_first = true;


    // SATURATION TEST *********************************************************
    sat = saturation_test(img, ui->spinBox_saturation_percentage->value());


    // CONTRAST TEST ***********************************************************
    cont = contrast_test(img);


    // HOMOGENEITY TEST ********************************************************
    homo = homogeneity_test(ui->lcd_horizontal_value->value(),ui->lcd_vertical_value->value());

    ui->actionSave_Results->setEnabled(true);


    //cvReleaseImage(&img);
    //filesName.clear();
    //integerDivisors_vertical.clear();
    //integerDivisors_horizontal.clear();

    if (filesName.size()==0)    ui->pushButton_qt->setEnabled(false);

    //pos_list_horizontal = 0;
    //pos_list_vertical = 0;
    check_pushbuttons();

    // Third Step: Set the resolution -> Width x Height
    //ui->lcd_horizontal_value->display(0);
    //ui->lcd_vertical_value->display(0);

    // MHI parameters //////////////////////////////////////////////////////////
    ui->spinBox_fps->setEnabled(true);
    if (filesName.size()==0)    ui->spinBox_buffer_size->setEnabled(true);
    ui->doubleSpinBox_Duration_MHI->setEnabled(true);
    ui->horizontalSlider_threshold_MHI->setEnabled(true);
    ////////////////////////////////////////////////////////////////////////////

    cvShowImage("Saturation Test", sat);
    cvShowImage("Contrast Test", cont);
    cvShowImage("Homogeneity Test", homo);

    QString msg = "Quality test finalized";
    QMessageBox::information(this, "Done", msg);
}

void MainWindow::on_spinBox_fps_valueChanged(int arg1)
{
    if(ui->checkBox_MHIduration_auto->isChecked())
        ui->doubleSpinBox_Duration_MHI->setValue(
                    (double)ui->spinBox_buffer_size->value()/arg1);
}

void MainWindow::on_horizontalSlider_threshold_MHI_valueChanged(int value)
{
    ui->label_sensibility_value->setText(QString::number(value));
}

void MainWindow::save_results()
{
    dir = QFileDialog::getExistingDirectory(this,
                                            tr("Select Folder"),
                                            lastdir);

    if (!dir.isEmpty())
    {
        dir_aux = dir;
        cvSaveImage(dir_aux.append("/saturation_test.jpg").toLocal8Bit().data(),
                    sat);

        dir_aux = dir;
        cvSaveImage(dir_aux.append("/constrast_test.jpg").toLocal8Bit().data(),
                    cont);
        dir_aux = dir;

        cvSaveImage(dir_aux.append("/homogeneity_test.jpg").toLocal8Bit().data(),
                    homo);

        dir_aux = dir;
        cvSaveImage(dir_aux.append("/preview.jpg").toLocal8Bit().data(), aux);

        dir_aux.clear();
        dir.clear();

        QMessageBox::information(this, tr("Images Saved"),
                                 tr("Images Saved Sucessfully!"));
    } else
    {
        QMessageBox::information(this, tr("Images Not Saved"),
                                 tr("Images Not Saved!"));
    }
}

void MainWindow::show_about()
{
    obj_sobre = new DialogSobre(this);
    obj_sobre->exec();
}

void MainWindow::show_help()
{
    obj_ajuda = new DialogAjuda(this);
    obj_ajuda->exec();
}

#include <QDesktopServices>
void MainWindow::show_doc()
{
    char* dir=get_absolute_speckle_tool_path();

    #ifdef __OS_UNIX__
    QString doc = "../share/doc/qualitytest/SpeckleQualityTestV1.0-UserGuide.pdf";
    #endif

    #ifdef __OS_WINDOWS__
    QString doc = "..\\share\\doc\\qualitytest\\SpeckleQualityTestV1.0-UserGuide.pdf";
    #endif

    doc = dir + doc;
    ui->statusBar->showMessage(doc,40000);

    printf("\nLaunching: %s\n",doc.toLocal8Bit().data());

    bool id=QDesktopServices::openUrl(QUrl::fromLocalFile(doc));

    if(id==0)
    {
        QMessageBox *msgBox = new QMessageBox(this);
        msgBox->setText("Problems launching: "+doc);
        msgBox->setWindowTitle("Warning!!!");
        msgBox->setIcon(QMessageBox::Warning);
        msgBox->setWindowModality(Qt::ApplicationModal);
        msgBox->exec();
    }
    else
    {
        printf("Launched.");
    }
}
